function povrch = wli_qfit(Zax, InData, Lambda, Delta)
%WLI_QFIT najde stredy interferencnich prouzku pomoci fitovani
%    kosinem modulovanym parabolou.
%   
%    povrch = wli_fit(Zax, Data, Lambda, [Delta])
%       Zax      - osa z (napr. z interferometru) [micro m]
%       Data     - namerene intentizy
%       Lambda   - stredni vlnova delka zdroje [micro m]
%       Delta    - okoli maxima, ktere fitovat (nepovinne) [micro m]
%
%    Parametr Data muze mit vice rozmeru (napr. z kamery). Vystupem je pak
%    matice stejnych rozmeru jako rozliseni snimace obsahujici zjistene
%    polohy stredu interferencnich prouzku.
%
%    Stredni vlnovou delku zdroje je mozne urcit jako dvojnasobek stredni
%    vlnove delky interferencniho prouzku.
%
%    Pokud se nezada Delta, pouzije se vlnova delka (tj. prumerne dva
%    prouzky na kazdou stranu).
%
%    Pro ladeni (zobrazovani informaci o vypoctu a grafu pro overeni
%    funkcnosti) je treba upravit kod funkce a v kodu ladeni povolit.
%

% Povoleni ladeni (vykresluje grafy a vypisuje informace) - true/false
% Pri povoleni ladeni nepouzivejte paralelni zpracovani (misto parfor nize
% pozijte cyklus for), v opacnem pripade nejsou grafy zobrazovany.
debug=false;

% Paralelni zpracovani je mozne povolit zamenou for za parfor nize
% Duvodem je, ze bez Parallel Computing Toolbox parfor nefunguje spravne,
% pri nedostupnosti tohoto Toolboxu by tedy nemusela tato funkce fungovat.

% zakazat upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

% zmenit rozmery matice s daty (pokud neni 3D)
switch length(find(size(InData)>1))                                         % zjistit pocet rozmeru matice Data
    case 3                                                                  % zpracovavame data z 2D snimace
        Data=InData;
    case 2                                                                  % zpracovavame data z 1D snimace
        Data(:,1,:)=InData;
    case 1                                                                  % zpracovavame data z 1 bodu
        Data(1,1,:)=InData;
    otherwise
        error('Neplatny rozmer matice Data');                               % matici nelze zpracovat
end

% zjistit rozliseni a dalsi parametry
n = size(Data,3);                                                           % spocitat snimky
if n ~= length(Zax)
    error('Matice Data a vektor Zax nemaji stejnou delku.');
end
xRes = size(Data,1);                                                        % zjistit vodorovne rozliseni snimku
yRes = size(Data,2);                                                        % zjistit svisle rozliseni snimku
step = mean(diff(Zax));                                                     % urcit prumerny krok mezi snimky
if nargin<4; Delta=Lambda; end;                                             % pokud neni zadana Delta, nastavit na vlnovou delku

% vypocitat povrch
povrch = zeros(xRes,yRes);                                                  % vyplnit matici nulami
%parfor i=1:xRes                                                             % pro paralelni zpracovani
for i=1:xRes                                                                % bez paralelniho zpracovani
    for j=1:yRes
        
        % najit priblizne prouzek pomoci maxima
        Signal=squeeze(Data(i,j,:));                                        % nacist signal pro zpracovani
        [MaxVal MaxInd]=max(abs(Signal-mean(Signal)));                      % vypocitat maximum
        Center=Zax(MaxInd);                                                 % poloha stredu
        IndMin=round(MaxInd-Delta/step);                                    % minimalni vzorek
        IndMax=round(MaxInd+Delta/step);                                    % maximalni vzorek
        if IndMin>IndMax; temp=IndMax; IndMax=IndMin; IndMin=temp; end;     % pokud je osa klesajici
        if IndMin<1; IndMin=1; end;                                         % pokud jsme mimo vzorky, zacit od kraje
        if IndMax>n; IndMax=n; end;                                         % pokud jsme mimo vzorky, koncit na kraji

        % data pro dalsi zpracovani jen kolem maxima
        IDetail=double(Signal(IndMin:IndMax))';                             % cast prouzku pro fit
        ZaxDetail=Zax(IndMin:IndMax)-Center;                                % cast osy z pro fit
        
        % fitovani pomoci linearni regrese
        U = 4*pi/Lambda*ZaxDetail;                                          % vypocitat argument kosinu
        A= [...
            ones(size(ZaxDetail))' ...
            (ZaxDetail.^2.*cos(U))' ...
            (ZaxDetail   .*cos(U))' ...
            (              cos(U))' ...
            (ZaxDetail.^2.*sin(U))' ...
            (ZaxDetail   .*sin(U))' ...
            (              sin(U))' ...
        ];                                                                  % matice preurcene soustavy
        X = A\IDetail';                                                     % vypocet koeficientu
        
        % vypocet polohy prouzku
        phi0 = atan(X(7)/X(4));                                             % faze kosinu v pocatku
        z0 = -X(3)/(2*X(2));                                                % poloha vrcholu paraboly
        povrch(i,j) = Lambda/(4*pi)*(phi0+2*pi*fix((4*pi/Lambda*z0-phi0)/(2*pi)))+Center;% poloha stredu prouzku

        % ladeni
        if debug                                                            % pokud je povoleno ladeni
            % vypsat informace
            disp(sprintf('bod x=%i/%i, y=%i/%i',i,xRes,j,yRes));
            disp(sprintf('lambda=%d um, delta=%d um',Lambda,Delta));
            disp(sprintf('phi0=%d, z0=%d, z=%d um',phi0,z0+Center,povrch(i,j)));
            
            % vykreslit grafy
            debugfig = figure(); set(gcf, 'Position', get(0,'Screensize')); hold on; % otevrit obrazek
            plot(Zax,Signal,'b');
            plot(ZaxDetail+Center,IDetail,'r');
            plot(ZaxDetail+Center,X(1)+(X(2)/cos(phi0)*(ZaxDetail).^2 + X(3)/cos(phi0)*(ZaxDetail) + X(4)/cos(phi0)).*cos(4*pi/Lambda*ZaxDetail-phi0) ,'k');
            plot(ZaxDetail+Center,X(1)+X(2)/cos(phi0)*(ZaxDetail).^2 + X(3)/cos(phi0)*(ZaxDetail) + X(4)/cos(phi0) ,'g');
            plot([z0+Center z0+Center],[min(IDetail) max(IDetail)] ,'k--','LineWidth',1);
            plot([povrch(i,j) povrch(i,j)],[min(IDetail) max(IDetail)] ,'m--','LineWidth',3);
            plot(ZaxDetail+Center,X(1)-X(2)/cos(phi0)*(ZaxDetail).^2 - X(3)/cos(phi0)*(ZaxDetail) - X(4)/cos(phi0) ,'g');
            xlim([Center-4*Delta Center+4*Delta]);
            xlabel('z/um'); ylabel('I');
            title(sprintf('Pro pokracovani kliknete na obrazek.\n\nbod x=%i/%i, y=%i/%i\nlambda=%d um, delta=%d um\nphi0=%d, z0=%d, z=%d um',i,xRes,j,yRes,Lambda,Delta,phi0,z0+Center,povrch(i,j)));
            legend('signal','zpracovano','fit','obalka','vrchol','detekovano');
            
            % cekat na uzivatele
            disp('Pro pokracovani kliknete na obrazek');                    % vypsat hlasku
            waitforbuttonpress;                                             % cekat na kliknuti
            close(debugfig);                                                % zavrit obrazek
        end
    end
end

% snizit rozmer matice s povrchem (neni-li matice Data 3D)
povrch=squeeze(povrch);

% povolit upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

end