function povrch = wli_fda(ZAxIn, InData, Lambda, Delta, Dz)
%WLI_FDA najde stredy interferencnich prouzku pomoci FDA
%   
%    povrch = wli_fda(Zax, Data, Lambda, Delta, Dz)
%       Zax      - osa z (napr. z interferometru) [micro m]
%       Data     - namerene intentizy
%       Lambda   - stredni vlnova delka zdroje [micro m]
%       Delta    - okoli stredni vlnove delky, ktere zpracovat [micro m]
%       Dz       - okoli maxima signalu, ktere prohledavat
%
%    Parametr Data muze mit vice rozmeru (napr. z kamery). Vystupem je pak
%    matice stejnych rozmeru jako rozliseni snimace obsahujici zjistene
%    polohy stredu interferencnich prouzku.
%
%    Pro ladeni (zobrazovani informaci o vypoctu a grafu pro overeni
%    funkcnosti) je treba upravit kod funkce a v kodu ladeni povolit.
%

% Povoleni ladeni (vykresluje grafy a vypisuje informace) - true/false
% Pri povoleni ladeni nepouzivejte paralelni zpracovani (misto parfor nize
% pozijte cyklus for), v opacnem pripade nejsou grafy zobrazovany.
debug=false;

% Paralelni zpracovani je mozne povolit zamenou for za parfor nize
% Duvodem je, ze bez Parallel Computing Toolbox parfor nefunguje spravne,
% pri nedostupnosti tohoto Toolboxu by tedy nemusela tato funkce fungovat.

% zakazat upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

% zmenit rozmery matice s daty (pokud neni 3D)
switch length(find(size(InData)>1))                                         % zjistit pocet rozmeru matice Data
    case 3                                                                  % zpracovavame data z 2D snimace
        Data=InData;
    case 2                                                                  % zpracovavame data z 1D snimace
        Data(:,1,:)=InData;
    case 1                                                                  % zpracovavame data z 1 bodu
        Data(1,1,:)=InData;
    otherwise
        error('Neplatny rozmer matice Data');                               % matici nelze zpracovat
end

% zjistit rozliseni a dalsi parametry
N = size(Data,3);                                                           % spocitat snimky
if N ~= length(ZAxIn)
    error('Matice Data a vektor Zax nemaji stejnou delku.');
end
xRes = size(Data,1);                                                        % zjistit vodorovne rozliseni snimku
yRes = size(Data,2);                                                        % zjistit svisle rozliseni snimku
step = mean(diff(ZAxIn));                                                   % urcit prumerny krok mezi snimky
if nargin<5; Dz=0; end;                                                     % pokud neni zadane Dz, nastavit na nulu
if nargin<4; Delta=0; end;                                                  % pokud neni zadana Delta, nastavit na nulu
if nargin<3; Lambda=0; end;                                                 % pokud neni zadana Lambda, nastavit na nulu

% vypocitat povrch
povrch = zeros(xRes,yRes);                                                  % vyplnit matici nulami
%parfor i=1:xRes                                                             % pro paralelni zpracovani
for i=1:xRes                                                                % bez paralelniho zpracovani
    for j=1:yRes
        
        % nacist prouzek
        SignalIn=double(squeeze(Data(i,j,:)));                              % nacist signal pro zpracovani
        if Dz ~= 0                                                          % pokud je zadano okoli maxima
            % najit priblizne prouzek pomoci maxima
            [MaxVal MaxInd]=max(abs(SignalIn-mean(SignalIn)));              % vypocitat maximum
            IndMin=round(MaxInd-Dz/step);                                   % minimalni vzorek
            IndMax=round(MaxInd+Dz/step);                                   % maximalni vzorek
            if IndMin>IndMax; temp=IndMax; IndMax=IndMin; IndMin=temp; end; % pokud je osa klesajici
            if IndMin<1; IndMin=1; end;                                     % pokud jsme mimo vzorky, zacit od kraje
            if IndMax>N; IndMax=N; end;                                     % pokud jsme mimo vzorky, koncit na kraji

            % data pro dalsi zpracovani jen kolem maxima
            Signal=SignalIn(IndMin:IndMax);                                 % cast prouzku pro zpracovani
            ZAx=ZAxIn(IndMin:IndMax);                                       % cast osy z pro zpracovani
        else                                                                % bude se zpracovavat cely zaznam
            Signal=SignalIn;                                                % nacist cely signal pro zpracovani
            ZAx=ZAxIn;                                                      % nacist celou osu
        end
        Step = mean(diff(ZAx));                                             % urcit prumerny krok mezi snimky
        n = length(ZAx);                                                    % spocitat snimky
        
        % vypocet polohy prouzku
        Y = fft(fftshift(Signal),n)/n;                                      % vypocitat Fourierovu transformaci preskupeneho signalu
        k = pi/Step*linspace(0,1,floor(n/2)+1);                             % vytvofit osu vlnovych cisel
        AmpSpectrum = 2*abs(Y(1:floor(n/2)+1));                             % amplitudove spektrum
        AmpSpectrum(1) = 0;                                                 % stejnosmerna slozka bude nulova
        PhaSpectrum = unwrap(angle(Y(1:floor(n/2)+1)));                     % fazove spektrum
        if Lambda == 0                                                      % pokud neni zadana vlnova delka
            [~, kCenInd] = max(AmpSpectrum);                                % najit maximalni amplitudu
            LambdaV = 4*pi/k(kCenInd);                                      % vypocitat odpovidajici vlnovou delku
        else
            LambdaV = Lambda;                                               % vlnova delka je zadana, priradit
        end
        kMin = 4*pi/(LambdaV+Delta);                                        % vypocitat minimalni vlnove cislo
        kMax = 4*pi/(LambdaV-Delta);                                        % vypocitat maximalni vlnove cislo
        [~, kminInd] = min(abs(k-kMin));                                    % vypocitat index maximalniho vlnoveho cisla
        [~, kmaxInd] = min(abs(k-kMax));                                    % vypocitat index minimalniho vlnoveho cisla
        if (kmaxInd-kminInd)<2                                              % pokud interval nezahrnuje alespon 3 body
            kminInd=kminInd-1;                                              % posunout minimum
            kmaxInd=kmaxInd+1;                                              % posunout maximum
        end
        kDetail = k(kminInd:kmaxInd);                                       % vybrat vlnova cisla pro detekci
        AmpSpectrumDetail = AmpSpectrum(kminInd:kmaxInd);                   % vybrat cast amplitudoveho spektra pro detekci
        PhaSpectrumDetail = PhaSpectrum(kminInd:kmaxInd);                   % vybrat case fazoveho spektra pro detekci
        Poly = polyfit(kDetail,PhaSpectrumDetail',1);                       % fitovat zavislost faze na vlnovem cisle primkou
        FringeInd = n/2 - Poly(1)/Step;                                     % smernice urduje vzdalenost prouzku od stredu puvodniho signalu, prepocitat na vzorky
        if round(FringeInd)==FringeInd                                      % pokud je stred presne na vzorku
            povrch(i,j)=ZAx(FringeInd);                                     % poloha je poloha vzorku
        else                                                                % poloha neni presne na vzorku
            NearestInd=[floor(FringeInd) ceil(FringeInd)];                  % nejblizsi indexy
            NearestZAx=ZAx(NearestInd);                                     % nejblizsi namerene body
            povrch(i,j)=NearestZAx(1)+(FringeInd-NearestInd(1))*diff(NearestZAx)/diff(NearestInd);% zjistit polohu linearni interpolaci
        end
        
        
        % ladeni
        if debug                                                            % pokud je povoleno ladeni
            % vypsat informace
            disp(sprintf('bod x=%i/%i, y=%i/%i',i,xRes,j,yRes));
            disp(sprintf('lambda=%d, delta=%d, z=%d um',Lambda,Delta,povrch(i,j)));
            
            % vykreslit grafy
            debugfig = figure(); set(gcf, 'Position', get(0,'Screensize')); hold on; % otevrit obrazek
                        
            subplot(4,1,1); hold on;
            title(sprintf('Pro pokracovani kliknete na obrazek.\n\nbod x=%i/%i, y=%i/%i\nlambda=%d um, delta=%d um, z=%d um',i,xRes,j,yRes,Lambda,Delta,povrch(i,j)));
            plot(ZAx,Signal);
            plot([povrch(i,j) povrch(i,j)],[min(Signal) max(Signal)],'k','LineWidth',2);
            xlabel('z [um]'); ylabel('I');
            legend('signal','poloha');

            subplot(3,1,2); hold on;
            plot(k,AmpSpectrum,'b');
            plot(kDetail,AmpSpectrumDetail,'r');
            xlabel('k [1/um]'); ylabel('I');
            xlim([4*pi/0.9 4*pi/0.3]);
            legend('namereno','zpracovano');

            subplot(3,1,3); hold on;
            plot(k,PhaSpectrum,'b');
            plot(kDetail,polyval(Poly,kDetail),'k','LineWidth',2);
            plot(kDetail,PhaSpectrumDetail,'r');
            xlabel('k [1/um]'); ylabel('faze [rad]');
            xlim([4*pi/0.9 4*pi/0.3]);
            legend('namereno','zpracovano','fit');
            
            % cekat na uzivatele
            disp('Pro pokracovani kliknete na obrazek');                    % vypsat hlasku
            waitforbuttonpress;                                             % cekat na kliknuti
            close(debugfig);                                                % zavrit obrazek
        end
    end
end

% snizit rozmer matice s povrchem (neni-li matice Data 3D)
povrch=squeeze(povrch);

% povolit upozorneni
warning('on','MATLAB:mir_warning_maybe_uninitialized_temporary');

end