function povrch = wli_env_hilb(ZaxIn, InData, Delta)
%WLI_ENV_HILB najde stredy interferencnich prouzku pomoci maxima
%obalky vypocitane pomoci Hilbertovy transformace.
%   
%    povrch = wli_env_hilb(Zax, Data, [Delta])
%       Zax      - osa z (napr. z interferometru) [micro m]
%       Data     - namerene intentizy
%       Delta    - jak velke okoli maxima prohledavat [micro m] (nepovinne)
%
%    Parametr Data muze mit vice rozmeru (napr. z kamery). Vystupem je pak
%    matice stejnych rozmeru jako rozliseni snimace obsahujici zjistene
%    polohy stredu interferencnich prouzku.
%
%    Pokud se parametr Delta neuvede, prohledava se cely zaznam (pomale).
%    Hledani okolo maxima je mozne pouzit pouze pokud se stredni hodnota
%    intenzity v prubehu mereni nemeni.
%
%    Pro ladeni (zobrazovani informaci o vypoctu a grafu pro overeni
%    funkcnosti) je treba upravit kod funkce a v kodu ladeni povolit.
%

% Povoleni ladeni (vykresluje grafy a vypisuje informace) - true/false
% Pri povoleni ladeni nepouzivejte paralelni zpracovani (misto parfor nize
% pozijte cyklus for), v opacnem pripade nejsou grafy zobrazovany.
debug=false;

% Paralelni zpracovani je mozne povolit zamenou for za parfor nize
% Duvodem je, ze bez Parallel Computing Toolbox parfor nefunguje spravne,
% pri nedostupnosti tohoto Toolboxu by tedy nemusela tato funkce fungovat.

% zakazat upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

% zmenit rozmery matice s daty (pokud neni 3D)
switch length(find(size(InData)>1))                                         % zjistit pocet rozmeru matice Data
    case 3                                                                  % zpracovavame data z 2D snimace
        Data=InData;
    case 2                                                                  % zpracovavame data z 1D snimace
        Data(:,1,:)=InData;
    case 1                                                                  % zpracovavame data z 1 bodu
        Data(1,1,:)=InData;
    otherwise
        error('Neplatny rozmer matice Data');                               % matici nelze zpracovat
end

% zjistit rozliseni a dalsi parametry
n = size(Data,3);                                                           % spocitat snimky
if n ~= length(ZaxIn)
    error('Matice Data a vektor Zax nemaji stejnou delku.');
end
xRes = size(Data,1);                                                        % zjistit vodorovne rozliseni snimku
yRes = size(Data,2);                                                        % zjistit svisle rozliseni snimku
step = mean(diff(ZaxIn));                                                   % urcit prumerny krok mezi snimky
if nargin<3; Delta=0; end;                                                  % pokud neni zadana Delta, nastavit na nulu

% vypocitat povrch
povrch = zeros(xRes,yRes);
%parfor i=1:xRes                                                             % pro paralelni zpracovani
for i=1:xRes                                                                % bez paralelniho zpracovani
    for j=1:yRes
        
        % nacist prouzek
        SignalIn=double(squeeze(Data(i,j,:)));                              % nacist signal pro zpracovani
        if Delta ~= 0                                                       % pokud je zadano okoli maxima
            % najit priblizne prouzek pomoci maxima
            [MaxVal MaxInd]=max(abs(SignalIn-mean(SignalIn)));              % vypocitat maximum
            IndMin=round(MaxInd-Delta/step);                                % minimalni vzorek
            IndMax=round(MaxInd+Delta/step);                                % maximalni vzorek
            if IndMin>IndMax; temp=IndMax; IndMax=IndMin; IndMin=temp; end; % pokud je osa klesajici
            if IndMin<1; IndMin=1; end;                                     % pokud jsme mimo vzorky, zacit od kraje
            if IndMax>n; IndMax=n; end;                                     % pokud jsme mimo vzorky, koncit na kraji

            % data pro dalsi zpracovani jen kolem maxima
            Signal=SignalIn(IndMin:IndMax)';                                % cast prouzku pro zpracovani
            Zax=ZaxIn(IndMin:IndMax);                                       % cast osy z pro zpracovani
        else                                                                % bude se zpracovavat cely zaznam
            Signal=SignalIn;                                                % nacist cely signal pro zpracovani
            Zax=ZaxIn;                                                      % nacist celou osu
        end

        % vypocet stredu prouzku
        s=length(Signal);                                                   % pocet snimku pro porovnani
        Signal=Signal-mean(Signal);                                         % odecist prumer (stred bude v 0)
        [HilbMaxVal HilbMaxInd] = max(abs(hilbert(Signal)));
        if (HilbMaxInd > s) || (HilbMaxInd < 1)                             % pokud byla nalezena neplatna hodnota
            HilbMaxInd = 1;                                                 % nahradit pocatkem (aby zbytek nezhavaroval)
        end
        povrch(i,j) = Zax(HilbMaxInd);                                      % vypocitat polohu povrchu
        
        % ladeni
        if debug                                                            % pokud je povoleno ladeni
            % vypsat informace
            disp(sprintf('bod x=%i/%i, y=%i/%i',i,xRes,j,yRes));
            disp(sprintf('delta=%d um',Delta));
            disp(sprintf('z=%d um',povrch(i,j)));
            
            % vykreslit grafy
            debugfig = figure(); set(gcf, 'Position', get(0,'Screensize')); hold on; % otevrit obrazek
            plot(ZaxIn,SignalIn-mean(SignalIn),'b');
            plot(Zax,Signal,'r');
            plot(Zax,abs(hilbert(Signal)),'k');
            plot(Zax,-abs(hilbert(Signal)),'k');
            if Delta ~= 0
                xlim([ZaxIn(MaxInd)-4*Delta ZaxIn(MaxInd)+4*Delta]);
            else
                xlim([Zax(HilbMaxInd)-4 Zax(HilbMaxInd)+4]);
            end
            xlabel('z/um'); ylabel('I');
            title(sprintf('Pro pokracovani kliknete na obrazek.\n\nbod x=%i/%i, y=%i/%i\ndelta=%d um\nz=%d um',i,xRes,j,yRes,Delta,povrch(i,j)));
            legend('signal','zpracovano','obalka')
            
            % cekat na uzivatele
            disp('Pro pokracovani kliknete na obrazek');                    % vypsat hlasku
            waitforbuttonpress;                                             % cekat na kliknuti
            close(debugfig);                                                % zavrit obrazek
        end
    end
end

% snizit rozmer matice s povrchem (neni-li matice Data 3D)
povrch=squeeze(povrch);

% povolit upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

end