function povrch = wli_corr_gauss(ZaxIn, InData, ModLam, ModLLc, Delta)
%WLI_CORR_GAUSS najde stredy interferencnich prouzku pomoci korelace.
%Namenera data se porovnavaji s prouzkem vypocitanym za predpokladu, ze
%zdroj ma Gaussovske spektrum.
%   
%    povrch = wli_corr_gauss(Zax, Data, Lambda, LC, [Delta])
%       Zax      - osa z (napr. z interferometru) [micro m]
%       Data     - namerene intentizy
%       Lambda   - stredni vlnova delka zdroje [micro m]
%       LC       - koherencni delka zdroje [micro m]
%       Delta    - jak velke okoli maxima prohledavat [micro m] (nepovinne)
%
%    Parametr Data muze mit vice rozmeru (napr. z kamery). Vystupem je pak
%    matice stejnych rozmeru jako rozliseni snimace obsahujici zjistene
%    polohy stredu interferencnich prouzku.
%
%    Pokud se parametr Delta neuvede, prohledava se cely zaznam (pomale).
%    Hledani okolo maxima je mozne pouzit pouze pokud se stredni hodnota
%    intenzity v prubehu mereni nemeni.
%
%    Pro ladeni (zobrazovani informaci o vypoctu a grafu pro overeni
%    funkcnosti) je treba upravit kod funkce a v kodu ladeni povolit.
%

% Povoleni ladeni (vykresluje grafy a vypisuje informace) - true/false
% Pri povoleni ladeni nepouzivejte paralelni zpracovani (misto parfor nize
% pozijte cyklus for), v opacnem pripade nejsou grafy zobrazovany.
debug=false;

% Paralelni zpracovani je mozne povolit zamenou for za parfor nize
% Duvodem je, ze bez Parallel Computing Toolbox parfor nefunguje spravne,
% pri nedostupnosti tohoto Toolboxu by tedy nemusela tato funkce fungovat.

% zakazat upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

% zmenit rozmery matice s daty (pokud neni 3D)
switch length(find(size(InData)>1))                                         % zjistit pocet rozmeru matice Data
    case 3                                                                  % zpracovavame data z 2D snimace
        Data=InData;
    case 2                                                                  % zpracovavame data z 1D snimace
        Data(:,1,:)=InData;
    case 1                                                                  % zpracovavame data z 1 bodu
        Data(1,1,:)=InData;
    otherwise
        error('Neplatny rozmer matice Data');                               % matici nelze zpracovat
end

% zjistit rozliseni a dalsi parametry
n = size(Data,3);                                                           % spocitat snimky
if n ~= length(ZaxIn)
    error('Matice Data a vektor Zax nemaji stejnou delku.');
end
xRes = size(Data,1);                                                        % zjistit vodorovne rozliseni snimku
yRes = size(Data,2);                                                        % zjistit svisle rozliseni snimku
step = abs(mean(diff(ZaxIn)));                                              % urcit prumerny krok mezi snimky
if nargin<5; Delta=0; end;                                                  % pokud neni zadana Delta, nastavit na nulu

% modelovat prouzek
ModxAx=-1.5*ModLLc:step:1.5*ModLLc;                                         % osa, na ktere se bude prouzek modelovat
ModCen=round(length(ModxAx)/2);                                             % na kterem vzorku je stred prouzku
ModInt=exp(-(2*(ModxAx)/ModLLc).^2).*cos(2.*pi.*(ModxAx)./ModLam);          % model prouzku
ModInN=(ModInt-mean(ModInt))./max(ModInt-mean(ModInt));                     % normovat

% vypocitat povrch
povrch = zeros(xRes,yRes);                                                  % vyplnit matici nulami
%parfor i=1:xRes                                                             % pro paralelni zpracovani
for i=1:xRes                                                                % bez paralelniho zpracovani
    for j=1:yRes
        
        % nacist prouzek
        SignalIn=double(squeeze(Data(i,j,:)));                              % nacist signal pro zpracovani
        if Delta ~= 0                                                       % pokud je zadano okoli maxima
            % najit priblizne prouzek pomoci maxima
            [MaxVal MaxInd]=max(abs(SignalIn-mean(SignalIn)));              % vypocitat maximum
            IndMin=round(MaxInd-Delta/step);                                % minimalni vzorek
            IndMax=round(MaxInd+Delta/step);                                % maximalni vzorek
            if IndMin>IndMax; temp=IndMax; IndMax=IndMin; IndMin=temp; end; % pokud je osa klesajici
            if IndMin<1; IndMin=1; end;                                     % pokud jsme mimo vzorky, zacit od kraje
            if IndMax>n; IndMax=n; end;                                     % pokud jsme mimo vzorky, koncit na kraji

            % data pro dalsi zpracovani jen kolem maxima
            Signal=SignalIn(IndMin:IndMax)';                                % cast prouzku pro zpracovani
            Zax=ZaxIn(IndMin:IndMax);                                       % cast osy z pro zpracovani
        else                                                                % bude se zpracovavat cely zaznam
            Signal=SignalIn;                                                % nacist cely signal pro zpracovani
            Zax=ZaxIn;                                                      % nacist celou osu
        end

        % vypocet stredu prouzku
        s=length(Signal);                                                   % pocet snimku pro porovnani
        Signal=Signal-mean(Signal);                                         % odecist prumer (stred bude v 0)
        Signal=Signal./max(Signal);                                         % normovat (maximum bude 1)
        Corr=xcorr(abs(Signal),abs(ModInN));                                % vypocitat korelaci
        %Corr=xcorr((Signal),(ModInN));                                     % vypocitat korelaci
        [CorrMaxVal CorrMaxInd] = max(Corr);                                % zjistit maximum korelace
        if s<=2*ModCen                                                      % pokud je zaznam kratsi nez delka modelu prouzku
            fringCenter = CorrMaxInd-ModCen;                                % index stredu prouzku
        else                                                                % zaznam je delsi nez model prouzku
            fringCenter = CorrMaxInd-(s-ModCen);                            % index stredu prouzku
        end
        if (fringCenter > s) || (fringCenter < 1)                           % pokud byla nalezena neplatna hodnota
            fringCenter = 1;                                                % nahradit pocatkem (aby zbytek nezhavaroval)
        end
        povrch(i,j) = Zax(fringCenter);                                     % vypocitat polohu povrchu
        
        % ladeni
        if debug                                                            % pokud je povoleno ladeni
            % vypsat informace
            disp(sprintf('bod x=%i/%i, y=%i/%i',i,xRes,j,yRes));
            disp(sprintf('lambda=%d um, lc=%d um, delta=%d um',ModLam,ModLLc,Delta));
            disp(sprintf('z=%d um',povrch(i,j)));
            
            % vykreslit grafy
            debugfig = figure(); set(gcf, 'Position', get(0,'Screensize')); hold on; % otevrit obrazek
            subplot(3,1,1);
            plot(ModxAx,ModInN);
            title(sprintf('Pro pokracovani kliknete na obrazek.\n\nbod x=%i/%i, y=%i/%i\nlambda=%d um, lc=%d um, delta=%d um\n z=%d um',i,xRes,j,yRes,ModLam,ModLLc,Delta,povrch(i,j)));
            xlabel('z/um'); ylabel('I');
            legend('model')
            subplot(3,1,2);
            plot(ZaxIn,SignalIn,'b');
            xlabel('z/um'); ylabel('I');
            legend('signal')
            subplot(3,1,3); hold on;
            plot(ZaxIn,(SignalIn-mean(SignalIn))/max(abs((SignalIn-mean(SignalIn)))),'b');
            plot(Zax,Signal,'r');
            if Signal(fringCenter) > 0
                plot(ModxAx+povrch(i,j),ModInN ,'k');
            else
                plot(ModxAx+povrch(i,j),-ModInN ,'k');
            end
            if Delta ~= 0
                xlim([ZaxIn(MaxInd)-4*Delta ZaxIn(MaxInd)+4*Delta]);
            else
                xlim([ZaxIn(fringCenter)-5 ZaxIn(fringCenter)+5]);
            end
            xlabel('z/um'); ylabel('I');
            legend('signal','zpracovano','model')
            
            % cekat na uzivatele
            disp('Pro pokracovani kliknete na obrazek');                    % vypsat hlasku
            waitforbuttonpress;                                             % cekat na kliknuti
            close(debugfig);                                                % zavrit obrazek
        end
    end
end

% snizit rozmer matice s povrchem (neni-li matice Data 3D)
povrch=squeeze(povrch);

% povolit upozorneni
warning('off','MATLAB:mir_warning_maybe_uninitialized_temporary');

end