%% whitesim_dispersion.m
%
% Simuluje vliv disperze na mereni profilu povrchu
% 

close all; clear all; clc;                                                  % zavrit a smazat vse


%% Nastaveni

% parametry zdroje svetla
sfile = 'Data/LED.txt';                                                     % soubor s namerenym spektrem zdroje
LambdaLaser = 0.633;                                                        % vlnova delka laseru (micro m)

% parametry vypoctu
dMax  = 3;                                                                  % maximalni tloustka skla (micro m)
dStep = 0.1;                                                                % krok tloustky skla  (micro m)
dPlotStep = 1.5;                                                            % krok pro kresleni prouzku (micro m)
xMax  = 7;                                                                  % maximalni vzdalenost od pocatku (micro m)
xStep = 0.01;                                                               % kroku pro vypocet  (micro m)

% parametry vzduchu
AirT = 20;                                                                  % teplota vzduchu (dg C)
Airp = 98;                                                                  % tlak vzduchu (kPa)
AirH = 30;                                                                  % vlhkost vzduchu (%)
AirC = 2000;                                                                % koncentrace CO2 (ppm)

% parametry skla pro vypocet indexu lomu (SCHOTT N-BK7)
B1 = 1.03961212;
B2 = 0.231792344;
B3 = 1.01046945;
C1 = 0.00600069867;
C2 = 0.0200179144;
C3 = 103.560653;


%% Nacist spektrum zdroje

display('Nacitam spektrum ze souboru');
Spectrum = dlmread(sfile);                                                  % nacist spektrum ze souboru
Lambda = Spectrum(:,1).*1e-3;                                               % v prvnim sloupci jsou vlnove delky v nanometrech, prepocitat na mikrometry
Intens = Spectrum(:,2);                                                     % v druhem sloupci intenzity
S = Intens-min(Intens);                                                     % odecist od intenzity minimum


%% Paralelni zpracovani

% Pokud je k dispozici Parallel Computing Toolbox, odkomentujte nasledujici
% radek pro urychleni vypoctu

% if matlabpool('size') == 0; matlabpool; end;                                % spustit zpracovani na vice jadrech


%% Vypocet indexu lomu a disperze

display('Pocitam indexy lomu a disperzi');
nGlass = sqrt( 1 + ...
    (B1.*Lambda.^2)./(Lambda.^2-C1) + ...
    (B2.*Lambda.^2)./(Lambda.^2-C2) + ...
    (B3.*Lambda.^2)./(Lambda.^2-C3));                                       % index lomu skla pro bilou
nGlassLaser = sqrt( 1 + ...
    (B1.*LambdaLaser.^2)./(LambdaLaser.^2-C1) + ...
    (B2.*LambdaLaser.^2)./(LambdaLaser.^2-C2) + ...
    (B3.*LambdaLaser.^2)./(LambdaLaser.^2-C3));                             % index lomu skla pro laser
nAir = zeros(length(Lambda),1);
for i = 1:length(Lambda)                                                    % pro kazdou vlnovou delku bile
    nAir(i) = air_index(Lambda(i)*1000,AirT,Airp,AirH,'ciddor',AirC);       % index lomu vzduchu pro bilou
end
nAirLaser = air_index(LambdaLaser*1000,AirT,Airp,AirH,'ciddor',AirC);       % index lomu vzduchu pro laser


%% Model prouzku (intenzita)

display('Generuji osy');
d = -dMax:dStep:dMax;                                                       % tloustky skla, ktere se budou pocitat
x = -xMax:xStep:xMax;                                                       % osa, na ktere se bude pocitat

display('Pocitam interferenci');
figure(); set(gcf, 'Position', get(0,'Screensize')); subplot(211); hold on; % otevrit obrazek
Inorm = zeros(length(d),length(x));                                         % vyplnit matici nulami
for j=1:length(d)                                                           % pro kazde posunuti
    y=zeros(length(Lambda),length(x));
    for i = 1:length(Lambda)                                                % pro kazdou vlnovou delku
        dx = - 2*d(j)*nGlass(i) + 2*d(j)*nAir(i);                           % vypocet posunuti vlivem disperze
        y(i,:) = cos(4.*pi.*(x+dx)./Lambda(i)).*S(i);                       % spocitat intenzitu
    end
    Inorm(j,:) = sum(y);                                                    % secist intenzity jednotlivych frekvenci
    if mod(d(j),dPlotStep)==0                                               % ma-li se vykreslovat prouzek
        plot(x,Inorm(j,:));                                                 % vykreslit prouzek
    end
end
xlabel('z [\mum]');
ylabel('I');
title('Prouzky pro ruzne tloustky skla');

display('Pocitam posunuti pro laser');
dxLaser = - 2*d*nGlassLaser + 2*d*nAirLaser;                                % vypocitat posunuti pro laser


%% Vypocet poloh prouzku

display('Pocitam polohy prouzku pomoci korelace');
Xcor1     = wli_corr_spectrum(x, Inorm, sfile);

display('Pocitam polohy prouzku pomoci korelace doplnene o fitovani');
Xcor1_fit = wli_corr_spectrum_fit(x, Inorm, sfile);

display('Pocitam polohy prouzku pomoci fitovani modelu (maximum signalu)');
Xqfit_max = wli_qfit(x,Inorm,2*0.262,0.42);

display('Pocitam polohy prouzku pomoci fitovani modelu (maximum obalky)');
Xqfit_env = wli_qfit_env(x,Inorm,2*0.262,0.42);

display('Pocitam polohy prouzku pomoci FDA');
Xfda      = wli_fda(x, Inorm, 0.560,0.060,1);


%% Vykreslit vysledky

display('Vykresluji vysledky');
subplot(212); hold on;
plot(d,(Xcor1    +dxLaser'),'g-');
plot(d,(Xcor1_fit+dxLaser'),'g--');
plot(d,(Xqfit_max+dxLaser'),'r');
plot(d,(Xqfit_env+dxLaser'),'r--');
plot(d,(Xfda     +dxLaser'),'b');
xlim([-dMax dMax]);
xlabel('tloustka skla [\mum]');
ylabel('posun detekovaneho mista [\mum]');
legend('korelace','korelace, fit','fit (maxima)','fit (obalka)','FDA');
title('Zavislost detekovane polohy stredu prouzku na tloustce skla');
