%% surf_calc.m
%
% Toto je ukazkovy soubor pro analyzu profilu povrchu.
% 

close all; clear all; clc;                                                  % zavrit a smazat vse


%% Nastaveni

% Soubor s daty
%
% Soucasti priloh jsou dva soubory s ukazkovymi daty:
%  - data1.mat - povrch obsahuje asi 12.5 mikrometru hluboke lepty
%  - data2.mat - povrch obsahuje lept hluboky asi 50-100 nanometru
% Pri obou merenich nebyla dokonale kompenzovana disperze, takze je mozne
% pri vyberu ruznych metod detekce prouzku sledovat jejich odolnost vuci
% disperzi.

DataFile = 'Data/data1.mat';
% DataFile = 'Data/data2.mat';

% Zdroj laseroveho zareni
LaserLambda0 = 633;                                                         % vlnova delka laseru (ve vakuu)         [nm]

% Nejistoty mereni velicin pro vypocet indexu lomu vzduchu
AirTemperatureUncertainty = 0.2;                                            % nejistota mereni teploty vzduchu       [stupne C]
AirHumidityUncertainty    = 0.5;                                            % nejistota mereni rel. vlhkosti vzduchu [procenta]
AirPressureUncertainty    = 0.5;                                            % nejistota mereni tlaku vzduchu         [kPa]
AirCO2Uncertainty         = 50;                                             % nejistota mereni koncentrace CO2       [ppm]

% Zdroj bileho svetla
WhiteSpectrumFile = 'Data/LED.txt';                                         % cesta k souboru se spektrem zdroje bileho svetla
WhiteLambda0 = 0.530;                                                       % stredni vlnova delka zdroje bileho svetla [mikrometry]
WhiteCoherenceLength = 2.0;                                                 % koherencni delka zdroje bileho svetla [mikrometry]


%% Paralelni zpracovani

% Pokud je k dispozici Parallel Computing Toolbox, odkomentujte nasledujici
% radek pro urychleni vypoctu a v pouzive funkci pro detekci stredu
% interfereniho prouzku odkomentujte radek obsahujici parfor
% (a zakomentujte radek obsahujici for). Duvodem je odlisne cyklu parfor
% v nekterych verzich prostredi MATLAB pri absenci Parallel Computing
% Toolbox, coz vede k nefunkcnosti funkci.

% if matlabpool('size') == 0; matlabpool; end;                                % spustit zpracovani na vice jadrech


%% Nacteni dat ze souboru

load(DataFile);                                                             % nacist data ze souboru

% Nyni mame nacteny nasledujici promenne:
%  - LaserIntensity - intenzita z detektoru interference laseroveho zareni
%  - WhiteIntensity - intenzity z kamery (pro vice bodu)
%  - AirTemperature - vyvoj teploty vzduchu behem mereni       [stupne C]
%  - AirHumidity    - vyvoj rel. vlhkosti vzduchu behem mereni [procenta]
%  - AirPressure    - vyvoj tlaku vzduchu behem mereni         [kPa]
%  - AirCO2         - vyvoj koncentrace CO2 behem mereni       [ppm]


%% Vypocet indexu lomu vzduchu

[AirRefractiveIndex, AirRefractiveIndexUncertainty] = ...
    air_index_edlen(...
        LaserLambda0, AirTemperature, AirPressure, AirHumidity, AirCO2,...
        0, AirTemperatureUncertainty, AirPressureUncertainty, AirHumidityUncertainty, AirCO2Uncertainty...
    );                                                                      % vypocitat index lomu vzduchu vcetne nejistoty
display(sprintf('Index lomu vzduchu je %0.8f +- %0.8f.',...
    AirRefractiveIndex, AirRefractiveIndexUncertainty));                    % vypsat informaci o indexu lomu vzduchu


%% Vypocet vlnove delky laseru

LaserLambda = LaserLambda0/AirRefractiveIndex;                              % vypocitat vlnovou delku laseru
display(sprintf('Vlnova delka laseru je %0.3f nm', LaserLambda));           % vypsat informaci o vlnove delce


%% Vypocet polohy mericiho zrcadla

zAxis = interf2pos(LaserIntensity,LaserLambda/1000);                        % vypocet osy z (v mikrometrech)


%% Vypocet profilu povrchu

% Pro vypocet profilu povrchu je k dispozici nekolik metod detekce stredu
% interferencniho prouzku. Tyto metody jsou popsany v textu prace.

% Odkomentovany nechte prave jeden radek s metodou, kterou se ma pocitat.

% Pro vykreslovani vice informaci povolte v souboru metody ladeni.

% SurfaceProfile = wli_corr_spectrum    (zAxis, WhiteIntensity, WhiteSpectrumFile);                               % korelacni metoda, model ze spektra
SurfaceProfile = wli_corr_spectrum_fit(zAxis, WhiteIntensity, WhiteSpectrumFile);                               % korelacni metoda, model ze spektra, fitovani
% SurfaceProfile = wli_corr_gauss       (zAxis, WhiteIntensity, WhiteLambda0/2, WhiteCoherenceLength);            % korelacni metoda, Gaussovsky model
% SurfaceProfile = wli_corr_gauss_fit   (zAxis, WhiteIntensity, WhiteLambda0/2, WhiteCoherenceLength);            % korelacni metoda, Gaussovsky model, fitovani
% SurfaceProfile = wli_env_hilb         (zAxis, WhiteIntensity);                                                  % vypocet obalky pomoci Hilbertovy transformace
% SurfaceProfile = wli_fda              (zAxis, WhiteIntensity);                                                  % analyza ve frekvencni oblasti
% SurfaceProfile = wli_fda_interp       (zAxis, WhiteIntensity);                                                  % analyza ve frekvencni oblasti, fitovani
% SurfaceProfile = wli_qfit             (zAxis, WhiteIntensity, WhiteLambda0);                                    % fitovani modelu
% SurfaceProfile = wli_qfit_env         (zAxis, WhiteIntensity, WhiteLambda0);                                    % fitovani modelu, maximum obalky
% SurfaceProfile = wli_weight_center    (zAxis, WhiteIntensity, WhiteCoherenceLength/2, WhiteLambda0/4, 0.5, 20); % vazeny prumer poloh maxim


%% Vykresleni profilu povrchu

figure();                                                                   % otevrit novy obrazek
surf(SurfaceProfile);                                                       % vykreslit profil povrchu
axis square;                                                                % osy x a y budou mit stejne meritko
colorbar;                                                                   % zobrazit barevne meritko
title('Profil povrchu');                                                    % vypsat popis obrazku
xlabel('x [px]'); ylabel('y [px]'); zlabel('z [\mum]');                     % vykreslit popisky os
view([125 35]);                                                             % otocit pohled
set(gcf, 'Position', get(0,'Screensize'));                                  % zvetsit obrazek na celou obrazovku
