function [RVn RVDn] = air_index_edlen(...
    Vwavelength,Vtemperature,Vpressure,Vhumidity,VxCO2,...
    VDwavelength,VDtemperature,VDpressure,VDhumidity,VDxCO2)
%AIR_INDEX_EDLEN vypocita index lomu vzduchu a nejistotu pro dane podminky.
%
%    [n Dn] =air_index_edlen(wavelength,temperature,pressure,humidity,xCO2,...
%               [Dwavelength],[Dtemperature],[Dpressure],[Dhumidity],[DxCO2])
%       wavelength  - vlnova delka ve vakuu [nm]
%       temperature - teplota [°C]
%       pressure    - tlak [kPa]
%       humidity    - relativni vlhkost [%]
%       xCO2        - koncentrace CO2 [ppm]
%       D*          - nejistoty velicin, ve stejnych jednotkach (nepovinne)
%
%    Hodnot teploty, tlaku, vlhkosti a koncentrace CO2 muze byt uvedeno
%    vice (vsech vsak musi byt stejne a mely by byt mereny ve stejny
%    okamzik). Pak bude vypocitana nejistota typu A indexu lomu.
%
%    Jestlize jsou zadany hodnoty nejistot jednotlivych velicin, je
%    vypocitana nejisota typu B indexu lomu.
%
%    Jsou-li zadány nejistoty jednotlivych velicin a je-li uvedeno vice
%    mereni, bude vypocitana kombinovana nejistota.
%
%    Presnost metody neni pro bezne laboratorni podminky vyssi nez 3e-8.
%

% vytvorit analyticke promenne
syms wavelength temperature pressure humidity xCO2 Dwavelength Dtemperature Dpressure Dhumidity DxCO2

% analyticky priprava parametru
sigma=1000/wavelength;                                                      % frekvence 1/mikrometr
p=pressure*1000;                                                            % tlak v pascalech
t=temperature;                                                              % teplota ve stupnich Celsia
T=t+273.15;                                                                 % teplota v Kelvinech
x=xCO2/1e6;                                                                 % molarni koncentrace CO2
psat=exp(1.2378847e-5*T^2-1.9121316e-2*T+33.93711047-6.3431645e3/T);        % tlak nasycene vodni pary
f=humidity/100*psat;                                                        % parcialni tlak vody v pascalech

% analyticky vypocet indexu lomu
nm1   = (8342.54+2406147/(130-sigma^2)+15998/(38.9-sigma^2))*1e-8;          % vlnova delka
nm1x  = nm1*(1+(0.529-0.0036*sigma)*(x-0.0004));                            % koncentrace CO2
nm1tp = nm1x*p/96095.43*(1+1e-8*(0.601-0.00972*t)*p)/(1+0.0036610*t);       % teplota, tlak
ntpf  = nm1tp+1-f*(3.7345-0.0401*sigma^2)*1e-10;                            % vlhkost
n     = ntpf;                                                               % vysledek

% analyticky vypocet nejistoty indexu lomu
Dn = sqrt(...
    (diff(n,wavelength)*Dwavelength)^2+...
    (diff(n,temperature)*Dtemperature)^2+...
    (diff(n,pressure)*Dpressure)^2+...
    (diff(n,humidity)*Dhumidity)^2+...
    (diff(n,xCO2)*DxCO2)^2);

% dosazeni hodnot
wavelength=Vwavelength;

% nacist ciselne nejistoty
if nargin == 10                                                             % pokud jsou zadany nejistoty   
    Dwavelength=VDwavelength;
    Dtemperature=VDtemperature;
    Dpressure=VDpressure;
    Dhumidity=VDhumidity;
    DxCO2=VDxCO2;
elseif nargin == 5                                                          % pokud nejsou zadany nejistoty, uvazovat nulove
    Dwavelength=0;
    Dtemperature=0;
    Dpressure=0;
    Dhumidity=0;
    DxCO2=0;
else                                                                        % pokud je zadan spatny pocet argumentu
    error('Neplatny pocet argumentu.');                                     % vypsat chybu
end

N=length(Vtemperature);                                                     % pocet mereni
% pokud pocet mereni jednotlivych velicin nesouhlasi
if length(Vpressure)~=N || length(Vhumidity)~=N || length(VxCO2)~=N
    error('Pocty mereni jednotlivych hodnot nesouhlasi');
end

% vypocet prumerne hodnoty indexu lomu (kvuli rychlosti pocitat numericky zvlast)
for i=1:N                                                                   % pro kazde mereni
    AAtemperature=Vtemperature(i);                                          % zjistit teplotu
    AApressure=Vpressure(i);                                                % zjistit tlak
    AAhumidity=Vhumidity(i);                                                % zjistit vlhkost
    AAxCO2=VxCO2(i);                                                        % zjistit obsah CO2
    
    % numericky priprava parametru
    AAsigma=1000/Vwavelength;                                               % frekvence 1/mikrometr
    AAp=AApressure*1000;                                                    % tlak v pascalech
    AAt=AAtemperature;                                                      % teplota ve stupnich Celsia
    AAT=AAt+273.15;                                                         % teplota v Kelvinech
    AAx=AAxCO2/1e6;                                                         % molarni koncentrace CO2
    AApsat=exp(1.2378847e-5*AAT^2-1.9121316e-2*AAT+33.93711047-6.3431645e3/AAT);% tlak nasycene vodni pary
    AAf=AAhumidity/100*AApsat;                                              % parcialni tlak vody v pascalech

    % numericky vypocet indexu lomu
    AAnm1   = (8342.54+2406147/(130-AAsigma^2)+15998/(38.9-AAsigma^2))*1e-8;% vlnova delka
    AAnm1x  = AAnm1*(1+(0.529-0.0036*AAsigma)*(AAx-0.0004));                % koncentrace CO2
    AAnm1tp = AAnm1x*AAp/96095.43*(1+1e-8*(0.601-0.00972*AAt)*AAp)/(1+0.0036610*AAt);% teplota, tlak
    AAntpf  = AAnm1tp+1-AAf*(3.7345-0.0401*AAsigma^2)*1e-10;                % vlhkost
    Vn(i)   = AAntpf;                                                       % vysledek
end
RVn = mean(Vn);                                                             % vypocitat prumernou hodnotu indexu lomu

% vypocet nejistoty typu A
if N>1                                                                      % pokud je vice nez 1 mereni
    ssum=0;                                                                 % vypocet vyberove smerodatne odchylky aritmetickeho prumeru
    parfor i=1:N
    	ssum=ssum+(RVn-Vn(i))^2;
    end
    sigma=sqrt(ssum/(N*(N-1)));
    VDnA=sigma*tinv(0.95,N);                                                % nejistota typu A
else
    VDnA=0;                                                                 % nejistotu typu A neuvazujeme
end

% vypocet nejistoty typu B
temperature=mean(Vtemperature);
pressure=mean(Vpressure);
humidity=mean(Vhumidity);
xCO2=mean(VxCO2);
VDnB = eval(Dn);                                                            % nejistota typu B

% vypocet kombinovane nejistoty
RVDn = sqrt(VDnA^2 + VDnB^2);                                               % kombinovana nejistota

end