/**
 * @file    tim_tms320f28069.c
 * @author  Matej Turinsky
 * @date    12. 3. 2024
 * @brief   Timer driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_TIM_DRIVER && defined(TMS320F28069)

#include <stddef.h>
#include <stdarg.h>
#include <assert.h>

#include "../../../../common/statuscodes.h"
#include "../../tim.h"
#include "tim_tms320f28069.h"


/** Macro computing EPWMxENCLK bit in the PCLKCR1 register */
#define SYSCTRL_PCLKCR1_EPWMxENCLK(_EPWMx) (((uint32_t)(_EPWMx) - (uint32_t)&EPwm1Regs) / ((uint32_t)&EPwm2Regs - (uint32_t)&EPwm1Regs))

/** Macro computing ECAPxENCLK bit in the PCLKCR1 register */
#define SYSCTRL_PCLKCR1_ECAPxENCLK(_ECAPx) ((((uint32_t)(_ECAPx) - (uint32_t)&ECap1Regs) / ((uint32_t)&ECap2Regs - (uint32_t)&ECap1Regs)) + 8)

/** Macro computing INT3x bit in the PIEIER3 register */
#define PIECTRL_PIEIER3_INT3x(_EPWMx) (((uint32_t)(_EPWMx) - (uint32_t)&EPwm1Regs) / ((uint32_t)&EPwm2Regs - (uint32_t)&EPwm1Regs))

/** Private function declarations */
static uint32_t private_getSYSCLKOUTValue(void);
static uint32_t private_getCounter(tim_dev_t* pDev);
static void private_setCounter(tim_dev_t* pDev, uint32_t counter);
static int private_getEnabled(tim_dev_t* pDev);
static void private_setEnabled(tim_dev_t* pDev, int enabled);
static uint32_t private_getFrequency(tim_dev_t* pDev);
static void private_setFrequency(tim_dev_t* pDev, uint32_t frequency);
static void private_setOverflowCallback(tim_dev_t* pDev, io_callback_t* pCallback);


/**
 * @brief   ePWM general interrupt handler.
 * @param   pDev    Pointer to device handler.
 */
static void epwm_irqHandler(tim_dev_t* pDev)
{
    if((pDev->overflowCallback.function != NULL) && (pDev->pOcCtrlReg->ETSEL.bit.INTSEL == 0x01))
    {
        pDev->overflowCallback.function(pDev->overflowCallback.argument);
    }
    else if((pDev->captureCompareCallback.function != NULL) && (pDev->pOcCtrlReg->ETSEL.bit.INTSEL != 0x01))
    {
        pDev->captureCompareCallback.function(pDev->captureCompareCallback.argument);
    }

    pDev->pOcCtrlReg->ETCLR.all = 0x0D;     // clear event-trigger submodule interrupt flag
    PieCtrlRegs.PIEACK.bit.ACK3 = 1;        // acknowledge PIE that interrupt was been served
}

/**
 * @brief   eCAP general interrupt handler.
 * @param   pDev    Pointer to device handler.
 */
static void ecap_irqHandler(tim_dev_t* pDev)
{
    if((pDev->captureCompareCallback.function != NULL) && ((pDev->pIcCtrlReg->ECFLG.all & 0x1F) != 0))
    {
        pDev->captureCompareCallback.function(pDev->captureCompareCallback.argument);
    }
    else if((pDev->overflowCallback.function != NULL) && (pDev->pIcCtrlReg->ECFLG.bit.CTROVF != 0))
    {
        pDev->overflowCallback.function(pDev->overflowCallback.argument);
    }

    pDev->pIcCtrlReg->ECCLR.all = 0xFF; // clear interrupt flags
    PieCtrlRegs.PIEACK.bit.ACK4 = 1;    // acknowledge PIE that interrupt was been served
}

/**
 * ePWM1 peripheral
 */
#if !DISABLE_EPWM1_ISR
static tim_dev_t* epwm1;
__interrupt void epwm1_interrupt(void)
{
    epwm_irqHandler(epwm1);
}
#endif /* !DISABLE_EPWM1_ISR */

/**
 * ePWM2 peripheral
 */
#if !DISABLE_EPWM2_ISR
static tim_dev_t* epwm2;
__interrupt void epwm2_interrupt(void)
{
    epwm_irqHandler(epwm2);
}
#endif /* !DISABLE_EPWM2_ISR */

/**
 * ePWM3 peripheral
 */
#if !DISABLE_EPWM3_ISR
static tim_dev_t* epwm3;
__interrupt void epwm3_interrupt(void)
{
    epwm_irqHandler(epwm3);
}
#endif /* !DISABLE_EPWM3_ISR */

/**
 * ePWM4 peripheral
 */
#if !DISABLE_EPWM4_ISR
static tim_dev_t* epwm4;
__interrupt void epwm4_interrupt(void)
{
    epwm_irqHandler(epwm4);
}
#endif /* !DISABLE_EPWM4_ISR */

/**
 * ePWM5 peripheral
 */
#if !DISABLE_EPWM5_ISR
static tim_dev_t* epwm5;
__interrupt void epwm5_interrupt(void)
{
    epwm_irqHandler(epwm5);
}
#endif /* !DISABLE_EPWM5_ISR */

/**
 * ePWM6 peripheral
 */
#if !DISABLE_EPWM6_ISR
static tim_dev_t* epwm6;
__interrupt void epwm6_interrupt(void)
{
    epwm_irqHandler(epwm6);
}
#endif /* !DISABLE_EPWM6_ISR */

/**
 * ePWM7 peripheral
 */
#if !DISABLE_EPWM7_ISR
static tim_dev_t* epwm7;
__interrupt void epwm7_interrupt(void)
{
    epwm_irqHandler(epwm7);
}
#endif /* !DISABLE_EPWM7_ISR */

/**
 * ePWM8 peripheral
 */
#if !DISABLE_EPWM8_ISR
static tim_dev_t* epwm8;
__interrupt void epwm8_interrupt(void)
{
    epwm_irqHandler(epwm8);
}
#endif /* !DISABLE_EPWM8_ISR */

/**
 * eCAP1 peripheral
 */
#if !DISABLE_ECAP1_ISR
static tim_dev_t* ecap1;
__interrupt void ecap1_interrupt(void)
{
    ecap_irqHandler(ecap1);
}
#endif /* !DISABLE_ECAP1_ISR */

/**
 * eCAP2 peripheral
 */
#if !DISABLE_ECAP2_ISR
static tim_dev_t* ecap2;
__interrupt void ecap2_interrupt(void)
{
    ecap_irqHandler(ecap2);
}
#endif /* !DISABLE_ECAP2_ISR */

/**
 * eCAP3 peripheral
 */
#if !DISABLE_ECAP3_ISR
static tim_dev_t* ecap3;
__interrupt void ecap3_interrupt(void)
{
    ecap_irqHandler(ecap3);
}
#endif /* !DISABLE_ECAP3_ISR */


/**
 * @brief   Initialization of the timer (ePWM and/or eCAP).
 * @param   pDev    Pointer to the timer device handler.
 * @return  Operation status code.
 */
int tim_init(tim_dev_t* pDev)
{
    assert(pDev != NULL);

    EALLOW;
    if(pDev->pOcCtrlReg != NULL)
    {
        // PIE vector table
    #if !DISABLE_EPWM1_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm1Regs))
        {
            epwm1 = pDev;
            PieVectTable.EPWM1_INT = &epwm1_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM1_ISR */

    #if !DISABLE_EPWM2_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm2Regs))
        {
            epwm2 = pDev;
            PieVectTable.EPWM2_INT = &epwm2_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM2_ISR */

    #if !DISABLE_EPWM3_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm3Regs))
        {
            epwm3 = pDev;
            PieVectTable.EPWM3_INT = &epwm3_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM3_ISR */

    #if !DISABLE_EPWM4_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm4Regs))
        {
            epwm4 = pDev;
            PieVectTable.EPWM4_INT = &epwm4_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM4_ISR */

    #if !DISABLE_EPWM5_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm5Regs))
        {
            epwm5 = pDev;
            PieVectTable.EPWM5_INT = &epwm5_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM5_ISR */

    #if !DISABLE_EPWM6_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm6Regs))
        {
            epwm6 = pDev;
            PieVectTable.EPWM6_INT = &epwm6_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM6_ISR */

    #if !DISABLE_EPWM7_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm7Regs))
        {
            epwm7 = pDev;
            PieVectTable.EPWM7_INT = &epwm7_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM7_ISR */

    #if !DISABLE_EPWM8_ISR
        if((uint32_t)pDev->pOcCtrlReg == (uint32_t)TIM_OC_CTRL_REG_CAST(EPwm8Regs))
        {
            epwm8 = pDev;
            PieVectTable.EPWM8_INT = &epwm8_interrupt;
            goto SKIP_EPWM;
        }
    #endif /* !DISABLE_EPWM8_ISR */

    SKIP_EPWM:
        SysCtrlRegs.PCLKCR1.all |= 1 << SYSCTRL_PCLKCR1_EPWMxENCLK(pDev->pOcCtrlReg);

        // Time-base initialization
        pDev->pOcCtrlReg->TBCTL.bit.FREE_SOFT = 0x03;   // enable free run
        pDev->pOcCtrlReg->TBCTL.bit.PRDLD = 0x01;       // disable shadow period register
        pDev->pOcCtrlReg->TBCTL.bit.HSPCLKDIV = 0x00;   // set HSPCLKDIV to /1
        pDev->pOcCtrlReg->ETPS.bit.INTPRD = 0x01;       // ePWM interrupt event prescaler = 0

        // ADC triggering allowed
        pDev->pOcCtrlReg->ETPS.bit.SOCAPRD = 0x01;      // ADC-SOCA event prescaler = 0
        pDev->pOcCtrlReg->ETSEL.bit.SOCASEL = 0x01;     // trigger ADC-SOCA with counter = 0
        pDev->pOcCtrlReg->ETSEL.bit.SOCAEN = 1;         // enable ADC-SOCA trigger with EPWMx
        pDev->pOcCtrlReg->ETCLR.bit.SOCA = 1;           // clear ADC-SOCA flag

        // Synchronize ePWMs time-bases
//        pDev->pOcCtrlReg->TBCTL.bit.SYNCOSEL = 0x00;    // forward sync pulse
//        pDev->pOcCtrlReg->TBCTL.bit.PHSEN = 0x01;       // enable sync
//        pDev->pOcCtrlReg->TBCTL.bit.SWFSYNC = 0x01;     // force sync

        // OC initialization
        // TODO


    }
    else if(pDev->pIcCtrlReg != NULL)
    {
        // PIE vector table
    #if !DISABLE_ECAP1_ISR
        if((uint32_t)pDev->pIcCtrlReg == (uint32_t)TIM_IC_CTRL_REG_CAST(ECap1Regs))
        {
            ecap1 = pDev;
            PieVectTable.ECAP1_INT = &ecap1_interrupt;
            goto SKIP_ECAP;
        }
    #endif /* !DISABLE_ECAP1_ISR */

    #if !DISABLE_ECAP2_ISR
        if((uint32_t)pDev->pIcCtrlReg == (uint32_t)TIM_IC_CTRL_REG_CAST(ECap2Regs))
        {
            ecap2 = pDev;
            PieVectTable.ECAP2_INT = &ecap2_interrupt;
            goto SKIP_ECAP;
        }
    #endif /* !DISABLE_ECAP2_ISR */

    #if !DISABLE_ECAP3_ISR
        if((uint32_t)pDev->pIcCtrlReg == (uint32_t)TIM_IC_CTRL_REG_CAST(ECap3Regs))
        {
            ecap3 = pDev;
            PieVectTable.ECAP3_INT = &ecap3_interrupt;
            goto SKIP_ECAP;
        }
    #endif /* !DISABLE_ECAP3_ISR */

    SKIP_ECAP:
        SysCtrlRegs.PCLKCR1.all |= 1 << SYSCTRL_PCLKCR1_ECAPxENCLK(pDev->pIcCtrlReg);

        // IC initialization
        //TODO

    }
    else
    {
        EDIS;
        return IO_ERROR;
    }
    EDIS;

    // Initialize callbacks
    io_callback_t emptyCallback = {NULL, NULL};
    pDev->overflowCallback = emptyCallback;
    pDev->captureCompareCallback = emptyCallback;

    return IO_OK;
}

/**
 * @brief   Open timer (ePWM and/or eCAP).
 * @param   pDev    Pointer to the time device handler.
 * @return  Non-zero when error occurred.
 */
int tim_open(tim_dev_t* pDev)
{
    assert(pDev != NULL);

    if(pDev->pOcCtrlReg != NULL)
    {
        pDev->pOcCtrlReg->TBCTR = 0;            // reset counter
        pDev->pOcCtrlReg->TBSTS.all = 0x06;     // clear time-base status register
    }

    return IO_OK;
}

/**
 * @brief   Close timer (ePWM and/or eCAP).
 * @param   pDev    Pointer to the timer device handler.
 * @return  Non-zero when error occurred.
 */
int tim_close(tim_dev_t* pDev)
{
    assert(pDev != NULL);

    if(pDev->pOcCtrlReg != NULL)
    {
        pDev->pOcCtrlReg->TBCTL.bit.CTRMODE = 0x03; // freeze timer
    }

    return IO_OK;
}

/**
 * @brief   IO control function for the timer (ePWM and/or eCAP).
 * @param   pDev    Pointer to the timer device handler.
 * @param   request Command for desired operation.
 * @param   arg     Optional argument of the command.
 * @return  Operation status code.
 */
int tim_ioctl(tim_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    EALLOW;
    switch(request)
    {
        case TIM_IOC_GET_CHANNELS:
            *va_arg(arg, int*) = 1;
            break;

        case TIM_IOC_GET_COUNTER:
            *va_arg(arg, uint32_t*) = private_getCounter(pDev);
            break;

        case TIM_IOC_SET_COUNTER:
            private_setCounter(pDev, (uint32_t)va_arg(arg, unsigned long));
            break;

        case TIM_IOC_GET_ENABLED:
            *va_arg(arg, int*) = private_getEnabled(pDev);
            break;

        case TIM_IOC_SET_ENABLED:
            private_setEnabled(pDev, va_arg(arg, int));
            break;

        case TIM_IOC_GET_FREQUENCY:
            *va_arg(arg, uint32_t*) = private_getFrequency(pDev);
            break;

        case TIM_IOC_SET_FREQUENCY:
            private_setFrequency(pDev, (uint32_t)va_arg(arg, unsigned long));
            break;

        case TIM_IOC_SET_OF_CALLBACK:
            private_setOverflowCallback(pDev, va_arg(arg, io_callback_t*));
            break;

//TODO
//        case TIM_IOC_GET_CC_MODE:
//
//            break;
//
//        case TIM_IOC_SET_CC_MODE:
//
//            break;
//
//        case TIM_IOC_SET_CC_CALLBACK:
//
//            break;
//
//        case TIM_IOC_GET_OC_VALUE:
//
//            break;
//
//        case TIM_IOC_SET_OC_VALUE:
//
//            break;
//
//        case TIM_IOC_GET_IC_VALUE:
//
//            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}



static uint32_t private_getSYSCLKOUTValue(void)
{
    uint32_t clk = 10000000;    // Internal OSC1/OSC2 frequency (default)

    if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 1 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 0)
    {
        clk = XCLKIN;   // predefined symbol for External OSC frequency
    }
    else if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 0 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 1)
    {
        clk = XTAL;     // predefined symbol for Crystal or Resonator frequency
    }

    // compute PLL output (SYSCLKOUT)
    if(SysCtrlRegs.PLLCR.bit.DIV != 0)
    {
        if(SysCtrlRegs.PLLSTS.bit.DIVSEL <= 1)
        {
            clk = clk >> 2;
        }
        else if(SysCtrlRegs.PLLSTS.bit.DIVSEL == 2)
        {
            clk = clk >> 1;
        }
        clk = clk * SysCtrlRegs.PLLCR.bit.DIV;
    }

    return clk;
}
static uint32_t private_getCounter(tim_dev_t* pDev)
{
    if(pDev->pIcCtrlReg != NULL)
    {
        return pDev->pIcCtrlReg->TSCTR;
    }
    else
    {
        return pDev->pOcCtrlReg->TBCTR;
    }
}
static void private_setCounter(tim_dev_t* pDev, uint32_t counter)
{
    if(pDev->pIcCtrlReg != NULL)
    {
        pDev->pIcCtrlReg->TSCTR = counter;
    }
    else
    {
        pDev->pOcCtrlReg->TBCTR = (uint16_t)counter;
    }
}
static int private_getEnabled(tim_dev_t* pDev)
{
    if(pDev->pIcCtrlReg != NULL)
    {
        return (int)pDev->pIcCtrlReg->ECCTL2.bit.TSCTRSTOP;
    }
    else
    {
        return (pDev->pOcCtrlReg->TBCTL.bit.CTRMODE == 0x03) ? 0 : 1;
    }
}
static void private_setEnabled(tim_dev_t* pDev, int enabled)
{
    if(pDev->pIcCtrlReg != NULL)
    {
        pDev->pIcCtrlReg->ECCTL2.bit.TSCTRSTOP = (enabled != 0) ? 0x01 : 0x00;
    }
    else
    {
        pDev->pOcCtrlReg->TBCTL.bit.CTRMODE = (enabled != 0) ? 0x00 : 0x03;
    }
}
static uint32_t private_getFrequency(tim_dev_t* pDev)
{
    uint32_t prescale = 1;
    uint32_t autoreload = 1;
    if(pDev->pIcCtrlReg != NULL)
    {
        // eCAP tick frequency: (SYSCLKOUT / PRESCALE)
        if(pDev->pIcCtrlReg->ECCTL1.bit.PRESCALE != 0)
        {
            prescale = (uint32_t)(pDev->pIcCtrlReg->ECCTL1.bit.PRESCALE << 2);
        }
    }
    else
    {
        // ePWM time-base frequency: (SYSCLKOUT / ((HSPCLKDIV * CLKDIV) * (TBPRD + 1))
        if(pDev->pOcCtrlReg->TBCTL.bit.HSPCLKDIV != 0)
        {
            prescale = (uint32_t)(pDev->pOcCtrlReg->TBCTL.bit.HSPCLKDIV << 2);
        }
        if(pDev->pOcCtrlReg->TBCTL.bit.CLKDIV != 0)
        {
            prescale *= (uint32_t)(1 << pDev->pOcCtrlReg->TBCTL.bit.CLKDIV);
        }
        autoreload += (uint32_t)pDev->pOcCtrlReg->TBPRD;
    }

    return private_getSYSCLKOUTValue() / (prescale * autoreload);
}
static void private_setFrequency(tim_dev_t* pDev, uint32_t frequency)
{
    uint32_t clk = private_getSYSCLKOUTValue();

    if(pDev->pIcCtrlReg != NULL)
    {
        // eCAP tick frequency: (SYSCLKOUT / PRESCALE)
        float psc = (clk / (frequency << 1));
        if((psc - (uint32_t)psc) > 0.5f)
        {
            psc++;
        }
        pDev->pIcCtrlReg->ECCTL1.bit.PRESCALE = ((uint16_t)psc) & 0x1F;
    }
    else
    {
        // ePWM time-base frequency: (SYSCLKOUT / ((HSPCLKDIV * CLKDIV) * (TBPRD + 1))
        uint16_t prescaler = 0;
        while(((float)clk / ((prescaler + 1UL) * 65536UL)) > (float)frequency)
        {
            prescaler++;
        }

        uint16_t clkdiv = 0;
        while(((float)prescaler / (1UL << clkdiv)) > 14.0f)
        {
            clkdiv++;
            if(clkdiv == 7)
            {
                break;
            }
        }

        uint16_t hpsclkdiv = 0;
        if((1UL << clkdiv) < prescaler)
        {
            hpsclkdiv++;
            while(((1UL << clkdiv) * (hpsclkdiv << 1)) < prescaler)
            {
                hpsclkdiv++;
                if(hpsclkdiv == 7)
                {
                    break;
                }
            }
        }

        float tbprd;
        if(hpsclkdiv == 0)
        {
            tbprd = clk / (frequency*(1UL << clkdiv));
        }
        else
        {
            tbprd = clk / (frequency*(1UL << clkdiv)*(hpsclkdiv << 1));
        }
        tbprd--;
        if(tbprd > 65535)
        {
            tbprd = 65535;
        }

        SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 0;      // prepare for ePWM synchronization

        pDev->pOcCtrlReg->TBCTL.bit.CLKDIV = clkdiv;
        pDev->pOcCtrlReg->TBCTL.bit.HSPCLKDIV = hpsclkdiv;
        pDev->pOcCtrlReg->TBPRD = (uint16_t)tbprd;

        SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;      // synchronize all initialized ePWM
    }
}
static void private_setOverflowCallback(tim_dev_t* pDev, io_callback_t* pCallback)
{
    pDev->overflowCallback.function = pCallback->function;
    pDev->overflowCallback.argument = pCallback->argument;

    if(pDev->overflowCallback.function != NULL)
    {
        PieCtrlRegs.PIEIER3.all |= 1 << PIECTRL_PIEIER3_INT3x(pDev->pOcCtrlReg);
        IER |= M_INT3;
        pDev->pOcCtrlReg->ETSEL.bit.INTSEL = 0x01;  // Interrupt <- TBCTR = 0x0000
        pDev->pOcCtrlReg->ETSEL.bit.INTEN = 1;
    }
    else
    {
        PieCtrlRegs.PIEIER3.all &= ~(1 << PIECTRL_PIEIER3_INT3x(pDev->pOcCtrlReg));
        if(PieCtrlRegs.PIEIER3.all == 0)
        {
            IER &= ~M_INT3;
        }
        if(pDev->pOcCtrlReg->ETSEL.bit.INTSEL == 0x01)
        {
            pDev->pOcCtrlReg->ETSEL.bit.INTEN = 0;  // Disable interrupt when: Interrupt <- TBCTR = 0x0000
        }
    }
}

#endif /* USE_TIM_DRIVER && defined(TMS320F28069) */
