/**
 * @file    nanomodbus_port.c
 * @author  Matej Turinsky
 * @date    16. 4. 2024
 * @brief   Source file with nanoMODBUS platform-specific functionalities
 */

#include "nanomodbus_port.h"
#include "host_config.h"
#include "devio.h"


/**
 * @brief   Read platform implementation for nanoMODBUS.
 * @param   buf             pointer to read buffer
 * @param   count           number of bytes to read
 * @param   byte_timeout_ms timeout in milliseconds
 * @param   arg             user variable argument
 * @return  Number of bytes actually read of error code when > 0.
 */
int32_t modbus_read(uint16_t* buf, uint16_t count, int32_t byte_timeout_ms, void* arg)
{
    int32_t toReceive = (int32_t)count;
    char* pRxBuffer = (char*)buf;
    int32_t received;
    uint64_t byteTimeout = (uint64_t)byte_timeout_ms;
    uint64_t rxTimeout = tickCounter + byteTimeout;

    do
    {
        received = (int32_t)io_read(arg, pRxBuffer, (int)toReceive);
        toReceive -= received;
        pRxBuffer += received;

        //Check timeout
        if(byte_timeout_ms > 0UL)
        {
            if(received > 0)
            {
                rxTimeout = tickCounter + byteTimeout;
            }
            else if(tickCounter > rxTimeout)
            {
                break;
            }
        }

    }while(toReceive);

    return (int32_t)count - toReceive;
}

/**
 * @brief   Read platform implementation for nanoMODBUS.
 * @param   buf             pointer to write buffer
 * @param   count           number of bytes to write
 * @param   byte_timeout_ms timeout in milliseconds
 * @param   arg             user variable argument
 * @return  Number of bytes actually wrote of error code when > 0.
 */
int32_t modbus_write(const uint16_t* buf, uint16_t count, int32_t byte_timeout_ms, void* arg)
{
    int32_t toSend = (int32_t)count;
    char* pTxBuffer = (char*)buf;
    int32_t sended;
    uint64_t byteTimeout = (uint64_t)byte_timeout_ms;
    uint64_t txTimeout = tickCounter + byteTimeout;

    do
    {
        sended = (int32_t)io_write(arg, pTxBuffer, (int)toSend);
        toSend -= sended;
        pTxBuffer += sended;

        //Check timeout
        if(byte_timeout_ms > 0UL)
        {
            if(sended > 0)
            {
                txTimeout = tickCounter + byteTimeout;
            }
            else if(tickCounter > txTimeout)
            {
                break;
            }
        }

    }while(toSend);

    return (int32_t)count - toSend;
}

/**
 * @brief   Read coils nanoMODBUS callback.
 * @param   address     address to holding register
 * @param   quantity    number of registers
 * @param   coils_out   pointer to read buffer
 * @param   unit_id     device ID
 * @param   arg         user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_readCoils(uint16_t address, uint16_t quantity, nmbs_bitfield coils_out, uint16_t unit_id, void* arg)
{
    uint16_t regAdd = address - 1U;

    if((regAdd + quantity) < sizeof(coils_t))
    {
        memcpy(coils_out, &modbus_coils.registers[regAdd], quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}

/**
 * @brief   Read discrete inputs nanoMODBUS callback.
 * @param   address     address to holding register
 * @param   quantity    number of registers
 * @param   inputs_out  pointer to read buffer
 * @param   unit_id     device ID
 * @param   arg         user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_readDiscInputs(uint16_t address, uint16_t quantity, nmbs_bitfield inputs_out, uint16_t unit_id, void* arg)
{
    if((address + quantity) < sizeof(discInputs_t))
    {
        memcpy(inputs_out, &modbus_discIn.registers[address], quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}


/**
 * @brief   Read holding registers nanoMODBUS callback.
 * @param   address         address to holding register
 * @param   quantity        number of registers
 * @param   registers_out   pointer to read buffer
 * @param   unit_id         device ID
 * @param   arg             user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_readHoldingRegsCallback(uint16_t address, uint16_t quantity, uint16_t* registers_out, uint16_t unit_id, void* arg)
{
    if((address + quantity) < sizeof(holdingRegs_t))
    {
        memcpy(registers_out, &modbus_holdReg.registers[address], quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}

/**
 * @brief   Read input registers nanoMODBUS callback.
 * @param   address         address to holding register
 * @param   quantity        number of registers
 * @param   registers_out   pointer to read buffer
 * @param   unit_id         device ID
 * @param   arg             user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_readInputRegsCallback(uint16_t address, uint16_t quantity, uint16_t* registers_out, uint16_t unit_id, void* arg)
{
    if((address + quantity) < sizeof(inputRegs_t))
    {
        memcpy(registers_out, &modbus_inputReg.registers[address], quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}

/**
 * @brief   Write multiple coils nanoMODBUS callback.
 * @param   address     address to holding register
 * @param   quantity    number of registers
 * @param   coils       coils new state
 * @param   unit_id     device ID
 * @param   arg         user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_writeMultCoils(uint16_t address, uint16_t quantity, const nmbs_bitfield coils, uint16_t unit_id, void* arg)
{
    if((address + quantity) < sizeof(coils_t))
    {
        // Modbus coils to GPIOA map
        const int pinDisconnectors[2] = {(34-32), (39-32)};//const int pinDisconnectors[2] = GPIOA_PINS_DISCONECTORS;
        const int pinShortRelays[16] = GPIOA_PINS_SHORT_RELAYS;

        // Get current state of GPIOA
        uint32_t port = 0UL;
        if(io_read(arg, &port, 4) != 4)
        {
            return NMBS_ERROR_INVALID_RESPONSE;
        }

        uint16_t i;
        if(address == 0)
        {
            // Disconnectors
            for(i = 0U; i < sizeof(pinDisconnectors)/sizeof(pinDisconnectors[0]); i++)
            {
                port &= ~(1U << pinDisconnectors[i]);
                port |= (coils[0] & (1U << i)) ? (1U << pinDisconnectors[i]) : 0U;
            }
        }

        if((quantity == 2) || (address == 1))
        {
            // ShortRelays
            for(i = 0U; i < sizeof(pinShortRelays)/sizeof(pinShortRelays[0]); i++)
            {
                port &= ~(1U << pinShortRelays[i]);
                port |= (coils[1] & (1U << i)) ? (1U << pinShortRelays[i]) : 0U;
            }
        }

        if(io_write(arg, &port, 4) != 4)
        {
            return NMBS_ERROR_INVALID_RESPONSE;
        }

        memcpy(modbus_coils.registers, (void*)coils, quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}

/**
 * @brief   Write multiple registers nanoMODBUS callback.
 * @param   address     address to holding register
 * @param   quantity    number of registers
 * @param   registers   registers new values
 * @param   unit_id     device ID
 * @param   arg         user variable argument
 * @return  Non-zero value when message failed to send.
 */
nmbs_error modbus_writeMultRegs(uint16_t address, uint16_t quantity, const uint16_t* registers, uint16_t unit_id, void* arg)
{
    if((address + quantity) < sizeof(holdingRegs_t))
    {
        memcpy(&modbus_holdReg.registers[address], (void*)registers, quantity);

        return NMBS_ERROR_NONE;
    }
    else
    {
        return NMBS_ERROR_INVALID_ARGUMENT;
    }
}

