/**
 * @file    adc_cla_tms320f28069.c
 * @author  Matej Turinsky
 * @date    23. 2. 2024
 * @brief   ADC driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_ADC_CLA_DRIVER && defined(TMS320F28069)

#include <stddef.h>
#include <assert.h>
#include <stdarg.h>
#include <string.h>

#include "../../../../common/statuscodes.h"
#include "../../adc.h"
#include "adc_cla_tms320f28069.h"


/** Private typedefs */
typedef struct INTSEL1N2_BITS* pIntSel_t;
typedef struct ADCSOCxCTL_BITS* pSocCtl_t;

/** Private variables */
static bool claInitialized = false;

/** Private function declarations */
static int private_getChannel(adc_cla_dev_t* pDev);
static void private_setChannel(adc_cla_dev_t* pDev, int channel);
static int private_getFrequency(adc_cla_dev_t* pDev, const uint32_t* pFrequency);
static int private_setFrequency(adc_cla_dev_t* pDev, const uint32_t frequency);
static int private_getSamplingMode(adc_cla_dev_t* pDev);
static void private_setSamplingMode(adc_cla_dev_t* pDev, int mode);
static void private_setCallback(io_callback_t* pDevCallback, io_callback_t* pCallback);

/** Simple delay macro */
#define _DELAY(loops) do{ uint32_t cycle = 0UL; while(cycle < loops) cycle++; } while(0)

/**
 * @brief   ADC general interrupt handler.
 * @param   pDev    Pointer to device handler.
 */
static void adc_cla_irqHandler(adc_cla_dev_t* pDev)
{
    uint16_t channelMask = (1U << (pDev->claChannel - 1U));

    // Overflow interrupt
    if(pDev->pClaReg->MIOVF.all & channelMask)
    {
        pDev->flags |= ADC_FLAG_OVERRUN;

        // clear CLA task overflow flag
        pDev->pClaReg->MICLROVF.all |= channelMask;

        if(pDev->errorCallback.function != NULL)
        {
            pDev->errorCallback.function(pDev->errorCallback.argument);
        }
    }
    // Conversion interrupt
    else
    {
        pDev->claTaskDone = true;

        // clear CLA task done flag
        pDev->pClaReg->MICLR.all |= channelMask;

        if(pDev->convCallback.function != NULL)
        {
            pDev->convCallback.function(pDev->convCallback.argument);
        }
    }

    pDev->pCtrlReg->ADCINTFLGCLR.all |= channelMask;    // clear ADC interrupt flag
    PieCtrlRegs.PIEACK.bit.ACK11 = 1;                   // acknowledge PIE that CLA interrupt was served
}

/**
 * ADC peripheral
 */
#if !DISABLE_ADC_CLA1_ISR
static adc_cla_dev_t* adc_cla1;
__interrupt void adc_cla1_interrupt(void)
{
    adc_cla_irqHandler(adc_cla1);
}
#endif /* !DISABLE_ADC_CLA1_ISR */

#if !DISABLE_ADC_CLA2_ISR
static adc_cla_dev_t* adc_cla2;
__interrupt void adc_cla2_interrupt(void)
{
    adc_cla_irqHandler(adc_cla2);
}
#endif /* !DISABLE_ADC_CLA2_ISR */

#if !DISABLE_ADC_CLA3_ISR
static adc_cla_dev_t* adc_cla3;
__interrupt void adc_cla3_interrupt(void)
{
    adc_cla_irqHandler(adc_cla3);
}
#endif /* !DISABLE_ADC_CLA3_ISR */

#if !DISABLE_ADC_CLA4_ISR
static adc_cla_dev_t* adc_cla4;
__interrupt void adc_cla4_interrupt(void)
{
    adc_cla_irqHandler(adc_cla4);
}
#endif /* !DISABLE_ADC_CLA4_ISR */

#if !DISABLE_ADC_CLA5_ISR
static adc_cla_dev_t* adc_cla5;
__interrupt void adc_cla5_interrupt(void)
{
    adc_cla_irqHandler(adc_cla5);
}
#endif /* !DISABLE_ADC_CLA5_ISR */

#if !DISABLE_ADC_CLA6_ISR
static adc_cla_dev_t* adc_cla6;
__interrupt void adc_cla6_interrupt(void)
{
    adc_cla_irqHandler(adc_cla6);
}
#endif /* !DISABLE_ADC_CLA6_ISR */

#if !DISABLE_ADC_CLA7_ISR
static adc_cla_dev_t* adc_cla7;
__interrupt void adc_cla7_interrupt(void)
{
    adc_cla_irqHandler(adc_cla7);
}
#endif /* !DISABLE_ADC_CLA7_ISR */

#if !DISABLE_ADC_CLA8_ISR
static adc_cla_dev_t* adc_cla8;
__interrupt void adc_cla8_interrupt(void)
{
    adc_cla_irqHandler(adc_cla8);
}
#endif /* !DISABLE_ADC_CLA8_ISR */


/**
 * @brief   Initialization of the ADC (with CLA usage).
 * @param   pDev    Pointer to the ADC device handler.
 * @return  Operation status code.
 */
int adc_cla_init(adc_cla_dev_t* pDev)
{
    assert(pDev != NULL);

    // check task function before copying code
    if(pDev->pClaTask == NULL)
    {
        return IO_ERROR;
    }

    EALLOW;
    // copy over the CLA code
    memcpy(&Cla1funcsRunStart, &Cla1funcsLoadStart, (uint32_t)&Cla1funcsLoadSize);

#if !DISABLE_CLA_MATH_TABLES
    // copy over the CLA math tables
    memcpy(&Cla1mathTablesRunStart, &Cla1mathTablesLoadStart, (uint32_t)&Cla1mathTablesLoadSize);
#endif

    SysCtrlRegs.PCLKCR3.bit.CLA1ENCLK = 1;      // enable CLA clock

    // PIE and CLA vector table
    switch(pDev->claChannel)
    {
    #if !DISABLE_ADC_CLA1_ISR
        case 1U:
            adc_cla1 = pDev;
            PieVectTable.CLA1_INT1 = &adc_cla1_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx1 = 1;
            pDev->pClaReg->MVECT1 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA1_ISR */

    #if !DISABLE_ADC_CLA2_ISR
        case 2U:
            adc_cla2 = pDev;
            PieVectTable.CLA1_INT2 = &adc_cla2_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx2 = 1;
            pDev->pClaReg->MVECT2 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA2_ISR */

    #if !DISABLE_ADC_CLA3_ISR
        case 3U:
            adc_cla3 = pDev;
            PieVectTable.CLA1_INT3 = &adc_cla3_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx3 = 1;
            pDev->pClaReg->MVECT3 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA3_ISR */

    #if !DISABLE_ADC_CLA4_ISR
        case 4U:
            adc_cla4 = pDev;
            PieVectTable.CLA1_INT4 = &adc_cla4_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx4 = 1;
            pDev->pClaReg->MVECT4 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA4_ISR */

    #if !DISABLE_ADC_CLA5_ISR
        case 5U:
            adc_cla5 = pDev;
            PieVectTable.CLA1_INT5 = &adc_cla5_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx5 = 1;
            pDev->pClaReg->MVECT5 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA5_ISR */

    #if !DISABLE_ADC_CLA6_ISR
        case 6U:
            adc_cla6 = pDev;
            PieVectTable.CLA1_INT6 = &adc_cla6_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx6 = 1;
            pDev->pClaReg->MVECT6 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA6_ISR */

    #if !DISABLE_ADC_CLA7_ISR
        case 7U:
            adc_cla7 = pDev;
            PieVectTable.CLA1_INT7 = &adc_cla7_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx7 = 1;
            pDev->pClaReg->MVECT7 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA7_ISR */

    #if !DISABLE_ADC_CLA8_ISR
        case 8U:
            adc_cla8 = pDev;
            PieVectTable.CLA1_INT8 = &adc_cla8_interrupt;
            PieCtrlRegs.PIEIER11.bit.INTx8 = 1;
            pDev->pClaReg->MVECT8 = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);
            break;
    #endif /* !DISABLE_ADC_CLA8_ISR */

        default:
            return IO_ERROR;
    }
    IER |= M_INT11;

    // CLA task trigger configuration
    if(!claInitialized)
    {
        claInitialized = true;
        pDev->pClaReg->MPISRCSEL1.all = 0xFFFFF111; // default triggers (all SW only)

        // Default ADC-CLA tasks
        pDev->pClaReg->MVECT1 = (uint16_t)((uint32_t)&AdcCla1Task1 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT2 = (uint16_t)((uint32_t)&AdcCla1Task2 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT3 = (uint16_t)((uint32_t)&AdcCla1Task3 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT4 = (uint16_t)((uint32_t)&AdcCla1Task4 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT5 = (uint16_t)((uint32_t)&AdcCla1Task5 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT6 = (uint16_t)((uint32_t)&AdcCla1Task6 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT7 = (uint16_t)((uint32_t)&AdcCla1Task7 - (uint32_t)&Cla1Prog_Start);
        pDev->pClaReg->MVECT8 = (uint16_t)((uint32_t)&AdcCla1Task8 - (uint32_t)&Cla1Prog_Start);

        // CLA memory configuration
        pDev->pClaReg->MMEMCFG.bit.PROGE = 1;     // Map L3DPSARAM as CLA program RAM
        pDev->pClaReg->MMEMCFG.bit.RAM0E = 1;     // Map L1DPSARAM as CLA data RAM0
        pDev->pClaReg->MMEMCFG.bit.RAM0CPUE = 1;  // CPU can access CLA data RAM0
        pDev->pClaReg->MMEMCFG.bit.RAM1E = 1;     // Map L2DPSARAM as CLA data RAM1
        pDev->pClaReg->MMEMCFG.bit.RAM1CPUE = 1;  // CPU can access CLA data RAM1
        pDev->pClaReg->MMEMCFG.bit.RAM2E = 1;     // Map L0DPSARAM as CLA data RAM2
        pDev->pClaReg->MMEMCFG.bit.RAM2CPUE = 1;  // CPU can access CLA data RAM2
    }
    uint16_t* mvect = (uint16_t*)&pDev->pClaReg->MVECT1;
    uint16_t i;
    for(i = 0; i < 8; i++)
    {
        if((pDev->claChannel - 1U) == i)
        {
            pDev->pClaReg->MPISRCSEL1.all &= ~(0x0FUL << (i << 2));                     // ADCx triggers CLA
            *mvect = (uint16_t)((uint32_t)pDev->pClaTask - (uint32_t)&Cla1Prog_Start);  // occupy CLA MVECT
            break;
        }
        mvect++;
    }

    // CLA flags clear
    pDev->claTaskDone = false;
    pDev->pClaReg->MICLR.all = 0x00FF;        // clear CLA interrupt flags
    pDev->pClaReg->MICLROVF.all = 0x00FF;     // clear CLA overflow flags

    pDev->pClaReg->MIER.all |= (1U << (pDev->claChannel - 1U));   // enable task interrupt

    // ADC initialization
    SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 1;       // enable ADC clock
    _DELAY(100UL);
    pDev->pCtrlReg->ADCCTL1.all |= (0x7 << 5);  // power up ADC
    EDIS;

    // Add delay after power up
    _DELAY(1000000UL);

    EALLOW;
    pDev->pCtrlReg->ADCCTL1.bit.INTPULSEPOS = 1;    // interrupt at start of conversion
    pDev->pCtrlReg->ADCCTL2.bit.ADCNONOVERLAP = 1;  // disable overlapping

    // ADC interrupt configuration
    pIntSel_t intsel = (pIntSel_t)&pDev->pCtrlReg->INTSEL1N2.bit;
    intsel += ((pDev->claChannel - 1U) >> 1);
    if(pDev->claChannel & 0x01)
    {
        intsel->INT1CONT = 0;                               // disable continuous mode
        intsel->INT1SEL = ((pDev->claChannel - 1U) << 1);   // SOC(x*2) generates ADCINTx
        intsel->INT1E = 1;                                  // enable ADCINTx
    }
    else
    {
        intsel->INT2CONT = 0;                               // disable continuous mode
        intsel->INT2SEL = ((pDev->claChannel - 1U) << 1);   // SOCx(x*2) generates ADCINTx
        intsel->INT2E = 1;                                  // enable ADCINTx
    }

    // ADC SOC(x*2)/SOC(x*2+1) configuration
    pSocCtl_t socctl = (pSocCtl_t)&pDev->pCtrlReg->ADCSOC0CTL.bit;
    socctl += ((pDev->claChannel - 1U) << 1);
    socctl->TRIGSEL = 0x00;     // no external trigger
    socctl->ACQPS = 0x06;       // sample window to 7clks (minimum)
    socctl++;
    socctl->TRIGSEL = 0x00;     // no external trigger
    socctl->ACQPS = 0x06;       // sample window to 7clks (minimum)

    // Configure Analog IO
    int n;
    for(n = 2; n < 15; n++)
    {
        if(n != 8)
        {
            GpioCtrlRegs.AIOMUX1.all |= 0x02UL << (n << 2);     // disable GPIO on AIO-n
        }
    }
    EDIS;

    // Initialize callbacks
    io_callback_t emptyCallback = {NULL, NULL};
    pDev->convCallback = emptyCallback;
    pDev->errorCallback = emptyCallback;

    return IO_OK;
}

/**
 * @brief   Open ADC (with CLA usage).
 * @param   pDev    Pointer to the ADC device handler.
 * @return  Non-zero when error occurred.
 */
int adc_cla_open(adc_cla_dev_t* pDev)
{
    assert(pDev != NULL);

    // Start trigger timer
    pDev->timDev = io_open(pDev->timPath);
    if(pDev->timDev == NULL)
    {
        return IO_ERROR;
    }

    EALLOW;
    pDev->pCtrlReg->ADCINTFLGCLR.all = 0x01FF;  // clear interrupt flags
    pDev->pCtrlReg->ADCINTOVFCLR.all = 0x01FF;  // clear overflow flags
    pDev->pCtrlReg->ADCCTL1.bit.ADCENABLE = 1;  // enable ADC
    EDIS;
    pDev->flags = 0;    // clear handler flags

    return IO_OK;
}

/**
 * @brief   Close ADC (with CLA usage).
 * @param   pDev    Pointer to the ADC device handler.
 * @return  Non-zero when error occurred.
 */
int adc_cla_close(adc_cla_dev_t* pDev)
{
    assert(pDev != NULL);

    // Close time
    if(pDev->timDev != NULL)
    {
        io_close(pDev->timDev);
    }

	EALLOW;
    pDev->pCtrlReg->ADCCTL1.bit.ADCENABLE = 0;  // disable ADC
	EDIS;
	
    return IO_OK;
}

/**
 * @brief   IO control function for the ADC (with CLA usage).
 * @param   pDev    Pointer to the ADC device handler.
 * @param   request Command for desired operation.
 * @param   arg     Optional argument of the command.
 * @return  Operation status code.
 */
int adc_cla_ioctl(adc_cla_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    EALLOW;
    switch(request)
    {
        case ADC_IOC_GET_RESOLUTION:
            *va_arg(arg, int*) = 12;
            break;

        case ADC_IOC_GET_CHANNEL:
            *va_arg(arg, int*) = private_getChannel(pDev);
            break;

        case ADC_IOC_SET_CHANNEL:
            private_setChannel(pDev, va_arg(arg, int));
            break;

        case ADC_IOC_GET_FREQUENCY:
            retVal = private_getFrequency(pDev, va_arg(arg, uint32_t*));
            break;

        case ADC_IOC_SET_FREQUENCY:
            retVal = private_setFrequency(pDev, (uint32_t)va_arg(arg, unsigned long));
            break;

        case ADC_IOC_GET_SAMPLING_MODE:
            *va_arg(arg, int*) = private_getSamplingMode(pDev);
            break;

        case ADC_IOC_SET_SAMPLING_MODE:
            private_setSamplingMode(pDev, va_arg(arg, int));
            break;

        case ADC_IOC_SET_CONV_CALLBACK:
            private_setCallback(&pDev->convCallback, va_arg(arg, io_callback_t*));
            break;

        case ADC_IOC_SET_ERROR_CALLBACK:
            private_setCallback(&pDev->errorCallback, va_arg(arg, io_callback_t*));
            break;

        case ADC_IOC_READ_FLAGS:
            *va_arg(pDev, int*) = (int)pDev->flags;
            break;

        case ADC_IOC_CLEAR_FLAGS:
            pDev->flags &= ~((uint16_t)va_arg(arg, int));
            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}

/**
 * @brief   Compute desired number of samples by ADC (with CLA usage).
 * @param   pDev        Pointer to the ADC device handler.
 * @param   pBuffer     Pointer to output buffer where to store data.
 * @param   conversions Number of conversions to do.
 * @return  Number of conversions actually done.
 */
int adc_cla_read(adc_cla_dev_t* pDev, void *pBuffer, int conversions)
{
    assert(pDev != NULL);
    int toConv = conversions;
    void* pHead = pBuffer;
    pSocCtl_t socctl = (pSocCtl_t)&pDev->pCtrlReg->ADCSOC0CTL.bit;
    socctl += ((pDev->claChannel - 1U) << 1);

    // Enable triggering
    io_ioctl(pDev->timDev, TIM_IOC_SET_COUNTER, 0);
    EALLOW;
    socctl->TRIGSEL = pDev->trigSrc;
    EDIS;

    while(toConv)
    {
        if(pDev->claTaskDone)
        {
            pDev->claTaskDone = false;

            toConv--;
        }
        else
        {
            // break loop if overflow occurs
            if(pDev->flags & ADC_FLAG_OVERRUN)
            {
                break;
            }
        }
    }

    // Disable triggering
    EALLOW;
    socctl->TRIGSEL = ADC_CLA_TRIG_SW_ONLY;
    EDIS;

    // copy CLA result to output buffer if provided
    if(pHead != NULL)
    {
        memcpy(pHead, pDev->pClaTaskResult, pDev->claTaskResultSize);
        pHead = (void*)((size_t)pHead + pDev->claTaskResultSize);
    }

    return conversions - toConv;
}


static int private_getChannel(adc_cla_dev_t* pDev)
{
    pSocCtl_t socctl = (pSocCtl_t)&pDev->pCtrlReg->ADCSOC0CTL.bit;
    socctl += ((pDev->claChannel - 1U) << 1);

    return (int)socctl->CHSEL;
}
static void private_setChannel(adc_cla_dev_t* pDev, int channel)
{
    pSocCtl_t socctl = (pSocCtl_t)&pDev->pCtrlReg->ADCSOC0CTL.bit;
    socctl += ((pDev->claChannel - 1U) << 1);

    if(private_getSamplingMode(pDev) == ADC_CLA_SAMPLING_MODE_SIMULTANEOUS)
    {
        socctl->CHSEL = (channel & 0x07);
    }
    else
    {
        socctl->CHSEL = (channel & 0x0F);
    }
}
static int private_getFrequency(adc_cla_dev_t* pDev, const uint32_t* pFrequency)
{
    if(pDev->timDev == NULL)
    {
        return IO_ERROR;
    }
    else if(io_ioctl(pDev->timDev, TIM_IOC_GET_FREQUENCY, pFrequency) != IO_OK)
    {
        return IO_ERROR;
    }
    else
    {
        return IO_OK;
    }
}
static int private_setFrequency(adc_cla_dev_t* pDev, const uint32_t frequency)
{
    if(pDev->timDev == NULL)
    {
        return IO_ERROR;
    }
    else if(io_ioctl(pDev->timDev, TIM_IOC_SET_FREQUENCY, frequency) != IO_OK)
    {
        return IO_ERROR;
    }
    else if(io_ioctl(pDev->timDev, TIM_IOC_SET_ENABLED, 1) != IO_OK)
    {
        return IO_ERROR;
    }
    else
    {
        return IO_OK;
    }
}
static int private_getSamplingMode(adc_cla_dev_t* pDev)
{
    return (pDev->pCtrlReg->ADCSAMPLEMODE.all & (1U << (pDev->claChannel - 1U))) ? ADC_CLA_SAMPLING_MODE_SIMULTANEOUS : ADC_CLA_SAMPLING_MODE_SEQUENTIAL;
}
static void private_setSamplingMode(adc_cla_dev_t* pDev, int mode)
{
    if(mode == ADC_CLA_SAMPLING_MODE_SIMULTANEOUS)
    {
        pDev->pCtrlReg->ADCSAMPLEMODE.all |= (1U << (pDev->claChannel - 1U));
        private_setChannel(pDev, private_getChannel(pDev));     // round ADC channel if needed
    }
    else if(mode == ADC_CLA_SAMPLING_MODE_SEQUENTIAL)
    {
        pDev->pCtrlReg->ADCSAMPLEMODE.all &= ~(1U << (pDev->claChannel - 1U));
    }
}
static void private_setCallback(io_callback_t* pDevCallback, io_callback_t* pCallback)
{
    pDevCallback->function = pCallback->function;
    pDevCallback->argument = pCallback->argument;
}

#endif /* USE_ADC_CLA_DRIVER && defined(TMS320F28069) */
