/**
 * @file    nanomodbus_port.h
 * @author  Matej Turinsky
 * @date    16. 4. 2024
 * @brief   Header file with nanoMODBUS platform-specific functionalities
 */

#pragma once

#ifndef NANOMODBUS_PORT_H_
#define NANOMODBUS_PORT_H_

#include <stdint.h>

#include "nanomodbus.h"


/*
 * Modbus typedefs
 */
// Coils
typedef struct coilsValues {
    uint16_t discontactorP : 1;
    uint16_t discontactorN : 1;
    uint16_t rsvd1 : 14;
    uint16_t shortRelay1  : 1;
    uint16_t shortRelay2  : 1;
    uint16_t shortRelay3  : 1;
    uint16_t shortRelay4  : 1;
    uint16_t shortRelay5  : 1;
    uint16_t shortRelay6  : 1;
    uint16_t shortRelay7  : 1;
    uint16_t shortRelay8  : 1;
    uint16_t shortRelay9  : 1;
    uint16_t shortRelay10 : 1;
    uint16_t shortRelay11 : 1;
    uint16_t shortRelay12 : 1;
    uint16_t shortRelay13 : 1;
    uint16_t shortRelay14 : 1;
    uint16_t shortRelay15 : 1;
    uint16_t shortRelay16 : 1;
} coilsValues_t;
typedef union coils{
    coilsValues_t values;
    uint16_t registers[2];
} coils_t;

// Discrete inputs
typedef struct discInputsValues {
    uint16_t diffCurrent : 1;       ///< differential current sensor input
    uint16_t rsvd1 : 15;
    uint16_t currFailed : 1;        ///< failed to update current flag
    uint16_t voltFailed : 1;        ///< failed to update voltage flag
    uint16_t powerFailed : 1;       ///< failed to update power flag
    uint16_t energyFailed : 1;      ///< failed to update energy flag
    uint16_t diffCurrFailed : 1;    ///< failed to update differential current flag
    uint16_t arcsFailed : 1;        ///< failed to update arc detection flag
    uint16_t irradianceFailed : 1;  ///< failed to update solar irradiance flag
    uint16_t panelTempFailed : 1;   ///< failed to update panel temperature flag
    uint16_t ambientTempFailed : 1; ///< failed to update ambient temperature flag
    uint16_t rsvd2 : 5;
    uint16_t running : 1;           ///< device running flag (toggles when running)
    uint16_t initialized : 1;       ///< device initialized flag (set after initialization)
} discInputsValues_t;
typedef union discInputs {
    discInputsValues_t values;
    uint16_t registers[2];
} discInputs_t;

// Holding registers
typedef struct holdingRegsValues {
    uint64_t localTime;
    float stringEnergy;
    uint16_t arcCounter;
} holdingRegsValues_t;
typedef union holdingRegs {
    holdingRegsValues_t values;
    uint16_t registers[7];
} holdingRegs_t;

// Input registers
typedef struct inputRegsValues {
    float stringVoltage;
    float stringCurrent;
    float stringPower;
    float solarIrradiance;
    float panelTemperature;
    float ambientTemperature;
} inputRegsValues_t;
typedef union inputRegs {
    inputRegsValues_t values;
    uint16_t registers[12];
} inputRegs_t;


/*
 * Modbus variables
 */
// Available modbus registers
extern coils_t modbus_coils;
extern discInputs_t modbus_discIn;
extern holdingRegs_t modbus_holdReg;
extern inputRegs_t modbus_inputReg;

// Timeout millisecond ticker
extern volatile uint64_t tickCounter;


/*
 * nanoMODBUS functions
 */
// Read\Write platform implementations
int32_t modbus_read(uint16_t* buf, uint16_t count, int32_t byte_timeout_ms, void* arg);
int32_t modbus_write(const uint16_t* buf, uint16_t count, int32_t byte_timeout_ms, void* arg);
// Command callbacks
nmbs_error modbus_readCoils(uint16_t address, uint16_t quantity, nmbs_bitfield coils_out, uint16_t unit_id, void* arg);
nmbs_error modbus_readDiscInputs(uint16_t address, uint16_t quantity, nmbs_bitfield inputs_out, uint16_t unit_id, void* arg);
nmbs_error modbus_readHoldingRegsCallback(uint16_t address, uint16_t quantity, uint16_t* registers_out, uint16_t unit_id, void* arg);
nmbs_error modbus_readInputRegsCallback(uint16_t address, uint16_t quantity, uint16_t* registers_out, uint16_t unit_id, void* arg);
nmbs_error modbus_writeMultCoils(uint16_t address, uint16_t quantity, const nmbs_bitfield coils, uint16_t unit_id, void* arg);
nmbs_error modbus_writeMultRegs(uint16_t address, uint16_t quantity, const uint16_t* registers, uint16_t unit_id, void* arg);


#endif /* NANOMODBUS_PORT_H_ */
