/**
 * @file    main.c
 * @author  Matej Turinsky
 * @date    10. 5. 2024
 * @brief   Main file for MT-SD11.
 */

#include <mbedtls/threading.h>

#include <devio.h>
#include <modemtec/twdog.h>

#include <fs.h>
#include <modemtec/plc-core/plc.h>
#include <modemtec/plc-core/dsp.h>

#include "mt49s/autoconf.h"
#include "mt49s/comm.h"
#include "network-test.h"

static comm_handler_t comm;

static devio_t wdog0;

int wdog_init(void)
{
    int result;

    if ((wdog0 = io_open("/dev/wdog0")) == NULL)
    {
        result = -1;
    }
    else if (io_ioctl(wdog0, WD_IOC_START) != IO_OK)
    {
        result = -2;
    }
    else
    {
        result = 0;
    }

    return result;
}

int wdog_reset(void)
{
    int result;

    if (io_ioctl(wdog0, WD_IOC_RESET) != IO_OK)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

void vApplicationDaemonTaskStartupHook(void)
{
    autoconf_t autoconf;

    /* Initialize mbedTLS system calls */
    mbedtls_platform_set_calloc_free(pvPortCalloc, vPortFree);
    mbedtls_threading_set_alt(freertos_mutex_init,
                              freertos_mutex_free,
                              freertos_mutex_lock,
                              freertos_mutex_unlock);

    /* Initialize peripherals */
    io_init();

    /* Initialize watchdog */
    wdog_init();
    twdog_init(wdog_reset, NULL);

    /* Initialize filesystem */
    fs_init();
    fs_mount();

    /* Initialize PLC core */
    plc_init();


    /* Initialize applications */
    comm_init(&comm);
    network_test_init();

    /* Set default PLC core mode */
    plc_set_mode(PLC_MODE_DEFAULT);

    /* Set modem according to auto configuration */
    if ((autoconf_load(&autoconf) == 0) &&
        (autoconf_check(&autoconf) == 0) &&
        (AUTOCONF_FLAG_IS_ENABLED(autoconf.Flags)))
    {
        comm.keep.panid = autoconf.PanID;
        comm.keep.role = AUTOCONF_FLAG_IS_COORDINATOR(autoconf.Flags) ? PLC_ROLE_COORDINATOR : PLC_ROLE_DEVICE;
        comm.keep.check_time = autoconf.ConnCheckTime * 60;

        autoconf_set_port(&autoconf, comm.msp.commDevice);
        comm_set_peer(&comm, autoconf.PeerAddress);
        comm_set_mode(&comm, COMM_MODE_TRANSPARENT);
    }

    comm_start(&comm);
}

int main(__attribute__((unused)) int argc, __attribute__((unused)) char* argv[])
{
    vTaskStartScheduler();
    return -1;
}
