/**
 * @file    if/g3asl.c
 * @author  Roman Mego, Matej Turinsky
 * @date    21. 2. 2024
 * @brief   MT-PLC-SEMx PLC G3 ASL functions.
 */

#include <assert.h>
#include <endian.h>
#include <string.h>

#include <FreeRTOS.h>
#include <event_groups.h>
#include <task.h>

#include <lwip/err.h>
#include <lwip/ip_addr.h>
#include <lwip/ip6.h>
#include <lwip/netif.h>

#include <devio.h>
#include <netutil/g3.h>
#include <modemtec/twdog.h>

#include "modemtec/plc-core/plc.h"
#include "modemtec/plc-core/if.h"
#include "modemtec/plc-core/mem.h"
#include "modemtec/plc-core/pool.h"
#include "modemtec/plc-core/util.h"
#include "modemtec/plc-core/if/g3asl.h"

static err_t plc_g3asl_if_output(struct netif* netif, struct pbuf* p, const ip6_addr_t* ip6addr)
{
    err_t result;
    plc_core_t* core;

    core = (plc_core_t*)netif->state;

    if (io_write(core->sm2400, p->payload, p->len) != p->len)
    {
        result = ERR_IF;
    }
    else
    {
        result = ERR_OK;
    }

    return result;
}

static void plc_g3asl_if_input_task(void* arg)
{
    plc_core_t* core = arg;

    assert(core != NULL);

    io_ioctl(core->sm2400, SM2400G3_IOC_SET_READ_TIMEOUT, 1);

    while ((xEventGroupGetBits(core->xEventGroup) & PLC_NETIF_REMOVE_FLAG) == 0)
    {
        struct pbuf* p = plc_pool_rx_pbuf_alloc();

        if (twdog_reset(TWDOG_TASK_ID_PLC_RX) != 0)
        {
            pbuf_free(p);
            vTaskDelay(10);
            continue;
        }

        if (io_read(core->sm2400, p->payload, p->len) <= 0)
        {
            pbuf_free(p);
            vTaskDelay(10);
            continue;
        }

        if (core->netif.input(p, &core->netif) != ERR_OK)
        {
            pbuf_free(p);
        }
    }

    xEventGroupClearBits(core->xEventGroup, PLC_NETIF_REMOVE_FLAG);
    vTaskDelete(NULL);
}

static err_t plc_g3asl_if_init(struct netif* netif)
{
    err_t result;
    plc_core_t* core;

    assert(netif != NULL);

    netif->name[0] = 'G';
    netif->name[1] = '3';
    netif->output_ip6 = plc_g3asl_if_output;

    netif->mtu = IP6_MIN_MTU_LENGTH;
    netif->flags = NETIF_FLAG_BROADCAST;

    core = (plc_core_t*)netif->state;

    if (twdog_register(TWDOG_TASK_ID_PLC_RX) != 0)
    {
        result = ERR_IF;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((core->xTaskNetif = xTaskCreateStatic(plc_g3asl_if_input_task, "G3 ASL NETIF", PLC_NETIF_STACK_SIZE, core, PLC_NETIF_TASK_PRIORITY, core->xTaskNerifStack, &core->xTaskNetifBuffer)) == NULL)
#else
    else if (xTaskCreate(plc_g3asl_if_input_task, "G3 ASL NETIF", PLC_NETIF_STACK_SIZE, core, PLC_NETIF_TASK_PRIORITY, &core->xTaskNetif) != pdPASS)
#endif
    {
        result = ERR_MEM;
    }
    else
    {
        result = ERR_OK;
    }

    return result;
}

static int plc_g3asl_get_coord_device(sm2400g3_call_devCred_t* arg)
{
    int result;
    plc_device_info_t info;

    if (plc_mem_find_device(arg->devCred->extAddr, &info) == 0)
    {
        arg->devCred->shortAddr = info.address;
        memcpy(arg->devCred->extAddr, info.uid, sizeof(arg->devCred->extAddr));
        memcpy(arg->devCred->psk, info.psk, sizeof(arg->devCred->psk));
        result = 0;
    }
    else
    {
        result = -1;
    }

    return result;
}

static int plc_g3asl_init(plc_core_t* core)
{
    int result;

    /* Register function for coordinator request of PSK for device */
    io_callback_t callback = {
        .function = (int(*)(void*))plc_g3asl_get_coord_device,
        .argument = NULL
    };

    assert(core != NULL);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_REGISTER_DEVCRED_CALL, &callback) != IO_OK)
    {
        result = -1;
    }
    else
    {
        netif_add_noaddr(&core->netif, core, plc_g3asl_if_init, plc_if_ip6_input);
        netif_set_default(&core->netif);
        netif_set_up(&core->netif);
        result = 0;
    }

    return result;
}

static int plc_g3asl_deinit(plc_core_t* core)
{
    assert(core != NULL);

    netif_set_down(&core->netif);

    xEventGroupSetBits(core->xEventGroup, PLC_NETIF_REMOVE_FLAG);
    while (eTaskGetState(core->xTaskNetif) != eDeleted);

    twdog_deregister(TWDOG_TASK_ID_PLC_RX);

    netif_remove(&core->netif);

    return 0;
}

static int plc_g3asl_connect(plc_core_t* core, plc_panid_t panId, plc_role_t role)
{
    int result;
    uint16_t coordAddr;
    sm2400g3_connState_t connState;
    int connectCommand = (role == PLC_ROLE_COORDINATOR) ? SM2400G3_IOC_START_NETWORK : SM2400G3_IOC_JOIN_NETWORK;
    sm2400g3_AslParam_t paramCoordShortAddress = {
        .attr = G3_ASL_IB_COORD_SHORT_ADDRESS,
        .index = 0,
        .pData = &coordAddr,
        .dataLen = sizeof(coordAddr)
    };
    sm2400g3_modemConf_t config = {
        .deviceType = (role == PLC_ROLE_COORDINATOR) ? 1 : 0,
        .panId = panId
    };
    plc_addr_t groupAddress = htobe16(0x8001);
    sm2400g3_AslParam_t paramGroupAddress = {
        .attr = G3_ASL_IB_GROUP_TABLE,
        .index = 0,
        .pData = &groupAddress,
        .dataLen = sizeof(groupAddress)
    };
    uint32_t ip6Addr[4];

    assert(core != NULL);

    if (plc_mem_get_eui64(config.extAddr) != 0)
    {
        result = -1;
    }
    if (plc_mem_get_psk(config.psk) != 0)
    {
        result = -2;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_SET_MODEM_CONF, &config) != IO_OK)
    {
        result = -3;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_SET_ASL_PARAM, &paramGroupAddress) != IO_OK)
    {
        result = -4;
    }
    else if (io_ioctl(core->sm2400, connectCommand, &connState) != IO_OK)
    {
        result = -5;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_IPV6_ADDRESS, &ip6Addr) != IO_OK)
    {
        result = -6;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_ASL_PARAM, &paramCoordShortAddress) != IO_OK)
    {
        result = -7;
    }
    else
    {
        ip_addr_t addr;
        plc_panid_t panId;
        plc_addr_t shortAddr;

        /* Set local address */
        IP_ADDR6(&addr, ip6Addr[0], ip6Addr[1], ip6Addr[2], ip6Addr[3]);
        netif_ip6_addr_set(&core->netif, 0, &addr);
        netif_ip6_addr_set_state(&core->netif, 0, IP6_ADDR_VALID);

        /* Set coordinator address */
        panId = PLC_IP6_PANID(&addr);
        shortAddr = be16toh(coordAddr);
        PLC_IP6(&addr, panId, shortAddr);
        ip_addr_copy(core->coordAddr, addr);

        netif_set_link_up(&core->netif);

        result = 0;
    }

    return result;
}

static int plc_g3asl_disconnect(plc_core_t* core)
{
    int result;
    ip_addr_t addr;

    assert(core != NULL);

    ip_addr_set_zero(&addr);

    netif_set_link_down(&core->netif);
    netif_ip6_addr_set_state(&core->netif, 0, IP6_ADDR_INVALID);
    netif_ip6_addr_set(&core->netif, 0, &addr);
    ip_addr_copy(core->coordAddr, addr);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_LEAVE_NETWORK) != IO_OK)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

static int plc_g3asl_get_version(plc_core_t* core, char* version, size_t bytes)
{
    int result;
    sm2400g3_GetFwVersion_t arg = {
        .pStr = version,
        .bytes = bytes
    };

    assert(core != NULL);
    assert(version != NULL);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_FW_VERSION, &arg) != IO_OK)
    {
        result = -1;
    }
    else
    {
        result = arg.bytes;
    }

    return result;
}

static int plc_g3asl_get_role(plc_core_t* core, plc_role_t* role)
{
    int result;

    uint8_t devType;
    sm2400g3_AslParam_t aslParam = {
        .attr = G3_ASL_IB_DEVICE_TYPE,
        .index = 0,
        .pData = &devType,
        .dataLen = sizeof(devType)
    };

    assert(core != NULL);
    assert(role != NULL);

    /* Read device role parameter from PLC */
    if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_ASL_PARAM, &aslParam) != IO_OK)
    {
        result = -1;
    }
    else
    {
        switch (devType)
        {
            case 0:
                *role = PLC_ROLE_DEVICE;
                break;
            case 1:
                *role = PLC_ROLE_COORDINATOR;
                break;
            default:
                *role = PLC_ROLE_NONE;
                break;
        }

        result = 0;
    }

    return result;
}

static int plc_g3asl_get_neighbor(plc_core_t* core, plc_index_t index, plc_addr_t* addr)
{
    int result;
    uint8_t data[32];
    g3_neighbour_table_t info;
    sm2400g3_MacParam_t mac = {
        .attr = G3_MAC_ID_NEIGHBOUR_TABLE,
        .index = index,
        .pData = &data,
        .dataLen = sizeof(data)
    };

    assert(core != NULL);
    assert(addr != NULL);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_MAC_PARAM, &mac) != IO_OK)
    {
        result = -1;
    }
    else if (mac.dataLen == 0)
    {
        result = -2;
    }
    else if (!g3_parse_neighbour_table(&info, mac.pData, mac.dataLen))
    {
        result = -3;
    }
    else
    {
        *addr = info.short_address;
        result = 0;
    }

    return result;
}

static int plc_g3asl_get_asl_param(plc_core_t* core, plc_aslid_t id, plc_index_t index, void* data, size_t size)
{
    int result;
    sm2400g3_AslParam_t param = {
        .attr = id,
        .index = index,
        .pData = data,
        .dataLen = size
    };

    assert(core != NULL);
    assert(data != NULL);

    /* Reading PSK is forbidden */
    if (param.attr == 0xFE)
    {
        result = -1;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_ASL_PARAM, &param) != IO_OK)
    {
        result = -2;
    }
    else
    {
        result = param.dataLen;
    }

    return result;
}

static int plc_g3asl_set_asl_param(plc_core_t* core, plc_aslid_t id, plc_index_t index, const void* data, size_t size)
{
    int result;
    sm2400g3_AslParam_t param = {
        .attr = id,
        .index = index,
        .pData = (void*)data,
        .dataLen = size
    };

    assert(core != NULL);
    assert(data != NULL);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_SET_ASL_PARAM, &param) != IO_OK)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

static int plc_g3asl_get_mac_param(plc_core_t* core, uint32_t id, uint16_t index, void* data, size_t size)
{
    int result;
    sm2400g3_MacParam_t param = {
        .attr = id,
        .index = index,
        .pData = data,
        .dataLen = size
    };

    assert(core != NULL);
    assert(data != NULL);

    /* Reading GMK is forbidden */
    if (id == G3_MAC_ID_KEY_TABLE)
    {
        result = -2;
    }
    else if (io_ioctl(core->sm2400, SM2400G3_IOC_GET_MAC_PARAM, &param) != IO_OK)
    {
        result = -3;
    }
    else
    {
        result = param.dataLen;
    }

    return result;
}

static int plc_g3asl_set_mac_param(plc_core_t* core, uint32_t id, uint16_t index, const void* data, size_t size)
{
    int result;
    sm2400g3_MacParam_t param = {
        .attr = id,
        .index = index,
        .pData = (void*)data,
        .dataLen = size
    };

    assert(core != NULL);
    assert(data != NULL);

    if (io_ioctl(core->sm2400, SM2400G3_IOC_SET_MAC_PARAM, &param) != IO_OK)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

static const plc_call_t call = {
    .init = plc_g3asl_init,
    .deinit = plc_g3asl_deinit,
    .connect = plc_g3asl_connect,
    .disconnect = plc_g3asl_disconnect,
    .get_version = plc_g3asl_get_version,
    .get_role = plc_g3asl_get_role,
    .get_neighbor = plc_g3asl_get_neighbor,
    .get_asl_param = plc_g3asl_get_asl_param,
    .set_asl_param = plc_g3asl_set_asl_param,
    .get_mac_param = plc_g3asl_get_mac_param,
    .set_mac_param = plc_g3asl_set_mac_param
};

const plc_call_t* plc_g3asl_call(void)
{
    return &call;
}
