/**
 * @file    modemtec/plc-core/plc.h
 * @author  Roman Mego, Matej Turinsky
 * @date    21. 2. 2024
 * @brief   MT-PLC-SEMx PLC core definitions.
 */

#pragma once

#ifndef MODEMTEC_PLC_CORE_PLC_H_
#define MODEMTEC_PLC_CORE_PLC_H_

#if defined(__cplusplus)
extern "C" {
#endif

#include <stddef.h>
#include <stdint.h>

#include <lwip/ip_addr.h>

/**
 * Mode of the PLC core
 */
typedef enum plc_mode
{
    PLC_MODE_NONE,      ///< None or unknown mode
    PLC_MODE_G3_ASL,    ///< G3 PLC in ASL mode (6LoWPAN)
    PLC_MODE_G3_PHY     ///< G3 PLC in PHY mode (IEEE802.15.4)
} plc_mode_t;

/** Default mode of the PLC code */
#define PLC_MODE_DEFAULT    PLC_MODE_G3_ASL

/**
 * Role of the PLC core in the network
 */
typedef enum plc_role
{
    PLC_ROLE_NONE,          ///< None or unknown role
    PLC_ROLE_DEVICE,        ///< Device/Client role
    PLC_ROLE_COORDINATOR    ///< Coordinator role
} plc_role_t;

/**
 * State of the PLC core
 */
typedef enum plc_state
{
    PLC_STATE_NONE,         ///< Unknown state
    PLC_STATE_CONNECTED,    ///< PLC core is connected to the network
    PLC_STATE_CONNECTING,   ///< PLC core tries to connecto to the network
    PLC_STATE_DISCONNECTED, ///< PLC core is disconnected from the network
    PLC_STATE_DISCONNECTING ///< PLC core tries to disconnect from the network
} plc_state_t;

/**
 * PLC frequency band.
 */
typedef enum plc_band
{
    PLC_BAND_DEFAULT,   ///< Default band according to used PLC core.
    PLC_BAND_CENA,      ///< CENELEC A
    PLC_BAND_CENB,      ///< CENELEC B
    PLC_BAND_CENC,      ///< CENELEC C
    PLC_BAND_CENBC,     ///< CENELEC BC
    PLC_BAND_FCC,       ///< FCC
    PLC_BAND_CENHI,
    PLC_BAND_ARIB2,     ///< ARIB2
    PLC_BAND_FULL       ///< Full bandwidth
} plc_band_t;

/** PLC short address type defnition */
typedef uint16_t plc_addr_t;

/** PLC PAN ID type definition */
typedef uint16_t plc_panid_t;

/** PLC ASL parameter ID type definition */
typedef uint8_t plc_aslid_t;

/** PLC MAC parameter ID type definition */
typedef uint32_t plc_macid_t;

/** PLC parameter index type definition */
typedef uint16_t plc_index_t;

/**
 * @brief   PLC receive callback function type definition.
 * @param   data    Pointer to received data.
 * @param   bytes   Size of received data in bytes.
 */
typedef void (*plc_rx_callback_t)(const void* data, size_t bytes);

/**
 * @brief   Initialize PLC core.
 * @return  0 on success.
 */
int plc_init(void);

/**
 * @brief   Reset PLC core.
 * @return  0 on success.
 */
int plc_reset(void);

/**
 * @brief   Gets version string(s) of PLC core.
 * @param   version Pointer to buffer where to store vestion string(s).
 * @param   bytes   Size of the output buffer in bytes.
 * @return  Number of bytes stored in output buffer.
 */
int plc_get_version(char* version, size_t bytes);

/**
 * @brief   Request for PLC core to connect into the network.
 * @param   panId   PAN ID.
 * @param   role    PLC core role in the network.
 * @return  0 when request was successfully sent.
 */
int plc_connect_request(plc_panid_t panId, plc_role_t role);

/**
 * @brief   Request for PLC core to disconnect from the netwrok.
 * @return  0 when request was successfully sent.
 */
int plc_disconnect_request(void);

/**
 * @brief   Register data receive callback function.
 * @param   callback    Callback function pointer to register.
 * @return  0 when function was registered.
 */
int plc_set_rx_callback(plc_rx_callback_t callback);

/**
 * @brief   Deregister data receive callback function.
 * @param   callback    Callback function pointer to deregister.
 * @return  0 when function was deregistered.
 */
int plc_remove_rx_callback(plc_rx_callback_t callback);

/**
 * @brief   Gets PLC core mode.
 * @param   mode    Pointer to variable where to store mode of PLC core.
 * @return  0 when mode was retrieved.
 */
int plc_get_mode(plc_mode_t* mode);

/**
 * @brief   Sets PLC core mode.
 * @param   mode    Value of PLC core mode to set.
 * @return  0 when mode was set.
 */
int plc_set_mode(plc_mode_t mode);

/**
 * @brief   Gets role of the PLC core.
 * @param   role    Pointer to variable where to store role of PLC core.
 * @return  0 when role was retrieved.
 */
int plc_get_role(plc_role_t* role);

/**
 * @brief   Gets state of the PLC core.
 * @param   state   Pointer to variable where to store state of PLC core.
 * @return  0 when state was retrieved.
 */
int plc_get_state(plc_state_t* state);

/**
 * @brief   Gets IP address of the PLC core.
 * @param   address Pointer to structure where to store IP address.
 * @return  0 when address was retrieved.
 */
int plc_get_address(ip_addr_t* address);

/**
 * @brief   Gets IP address of the network coordinator.
 * @param   address Pointer to structure where to store IP address.
 * @return  0 when address was retrieved.
 */
int plc_get_coord_address(ip_addr_t* address);

/**
 * @brief   Gets IP address of the neighbour.
 * @param   index   Index of the neighbour in the table.
 * @param   address Pointer to structure where to store IP address.
 * @return  0 when address was retrieved.
 */
int plc_get_neighbor(plc_index_t index, ip_addr_t* address);

/**
 * @brief   Gets ASL parameter of the PLC core.
 * @param   id      ID of the ASL parameter to read.
 * @param   index   Index of item if ASL parameter is in form of table, otherwise 0.
 * @param   data    Pointer to buffer where to store parameter.
 * @param   size    Size of the output buffer in bytes.
 * @return  Number of bytes stored in output buffer.
 */
int plc_get_asl_param(plc_aslid_t id, plc_index_t index, void* data, size_t size);

/**
 * @brief   Sets ASL parameter of the PLC core.
 * @param   id      ID of the ASL parameter to set.
 * @param   index   Index of item if ASL parameter is in form of table, otherwise 0.
 * @param   data    Pointer to ASL parameter to set.
 * @param   size    Size of ASL parameter in bytes.
 * @return  0 when parameter was set.
 */
int plc_set_asl_param(plc_aslid_t id, plc_index_t index, const void* data, size_t size);

/**
 * @brief   Gets MAC parameter of the PLC core.
 * @param   id      ID of the MAC parameter to read.
 * @param   index   Index of item if MAC parameter is in form of table, otherwise 0.
 * @param   data    Pointer to buffer where to store parameter.
 * @param   size    Size of the output buffer in bytes.
 * @return  Number of bytes stored in output buffer.
 */
int plc_get_mac_param(plc_macid_t id, plc_index_t index, void* data, size_t size);

/**
 * @brief   Sets MAC parameter of the PLC core.
 * @param   id      ID of the MAC parameter to set.
 * @param   index   Index of item if MAC parameter is in form of table, otherwise 0.
 * @param   data    Pointer to MAC parameter to set.
 * @param   size    Size of ASL parameter in bytes.
 * @return  0 when parameter was set.
 */
int plc_set_mac_param(plc_macid_t id, plc_index_t index, const void* data, size_t size);

/**
 * @brief   Sets PHY parameters of the PLC core.
 * @param   minrssi Minimal RSSI for reception in multiples of 0.1 dBuV.
 * @param   band    Value of used band.
 * @return  0 when parameters were set.
 */
int plc_set_phy(int minrssi, plc_band_t band);

#if defined(__cplusplus)
}
#endif

#endif /* MODEMTEC_PLC_CORE_PLC_H_ */
