/**
 * @file    wd_tms320f28069.c
 * @author  Matej Turinsky
 * @date    27. 2. 2024
 * @brief   Watchdog driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_WD_DRIVER && defined(TMS320F28069)

#include <stddef.h>
#include <assert.h>
#include <stdarg.h>

#include "../../../../common/statuscodes.h"
#include "../../watchdog.h"
#include "wd_tms320f28069.h"


/** Watchdog registers - SCSR bit masks */
#define SCSR_WDINTS     (0x1 << 2)
#define SCSR_WDENINT    (0x1 << 1)
#define SCSR_WDOVERRIDE (0x1 << 0)

/** Watchdog registers - WDCR bit masks */
#define WDCR_WDFLAG     (0x1 << 7)
#define WDCR_WDDIS      (0x1 << 6)
#define WDCR_WDCHK      (0x5 << 3)  //> must be in each WDCR write!
#define WDCR_WDPS       (0x7 << 0)

/**
 * @brief   Initialization of the watchdog
 * @param   pDev    Pointer to the watchdog device handler
 * @return  Operation status code
 */
int wd_init(wd_dev_t* pDev)
{
    assert(pDev != NULL);

    // compute nearest higher possible prescaler
    uint16_t tmpPsc;
    for(tmpPsc = 0; tmpPsc < 7; ++tmpPsc)
    {
        if((1 << tmpPsc) >= pDev->prescaler)
        {
            break;
        }
    }
    tmpPsc += 1;

    EALLOW;
    SysCtrlRegs.SCSR = SCSR_WDINTS | SCSR_WDOVERRIDE;
    SysCtrlRegs.WDCR = WDCR_WDFLAG | WDCR_WDDIS | WDCR_WDCHK | (tmpPsc & WDCR_WDPS);
    EDIS;

    return IO_OK;
}

/**
 * @brief   IO control function for the watchdog
 * @param   pDev    Pointer to the watchdog device handler
 * @param   request Command for desired operation
 * @param   arg     Optional argument of the command
 * @return  Operation status code
 */
int wd_ioctl(wd_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    uint16_t tmpReg = SysCtrlRegs.WDCR | WDCR_WDCHK;

    EALLOW;
    switch(request)
    {
        case WD_IOC_START:
            SysCtrlRegs.WDCR = tmpReg & ~WDCR_WDDIS;
            break;

        case WD_IOC_STOP:
            SysCtrlRegs.WDCR = tmpReg | WDCR_WDDIS;
            break;

        case WD_IOC_RESET:
            SysCtrlRegs.WDKEY = 0x0055;
            SysCtrlRegs.WDKEY = 0x00AA;
            break;

        case WD_IOC_IS_RUN:
            *va_arg(arg, int*) = (tmpReg & WDCR_WDDIS) ? 0 : 1;
            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}


#endif /* USE_WD_DRIVER && defined(TMS320F28069) */
