/**
 * @file    uart_tms320f28069.c
 * @author  Matej Turinsky
 * @date    27. 2. 2024
 * @brief   UART driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_UART_DRIVER && defined(TMS320F28069)

#include <stddef.h>
#include <assert.h>
#include <stdarg.h>

#include "../../../../common/statuscodes.h"
#include "../../uart.h"
#include "uart_tms320f28069.h"


/**
 * Private function declaration
 */
static uint32_t private_getMuxValue(pUartPortCtrlReg_t pPort, const uint16_t pin);
static uint32_t private_getLSPCLKValue(void);
static void private_setParity(uart_dev_t* pDev, int parity);
static int private_getParity(uart_dev_t* pDev);
static void private_setStopbits(uart_dev_t* pDev, int stopbits);
static int private_getStopbits(uart_dev_t* pDev);
static void private_setBaudrate(uart_dev_t* pDev, uint32_t baudrate);
static uint32_t private_getBaudrate(uart_dev_t* pDev);
static void private_setDataBits(uart_dev_t* pDev, int databits);
static int private_getDataBits(uart_dev_t* pDev);

/**
 * @brief   UART general interrupt handler.
 * @param   pDev    Pointer to device handler.
 */
static void uart_irqHandler(uart_dev_t* pDev)
{
    assert(pDev != NULL);

    size_t headIndex = pDev->rxBufferHead;
    char receivedData;

    // Handler data receive
    if(pDev->pCtrlReg->SCIFFRX.bit.RXFFINT || pDev->pCtrlReg->SCIRXST.bit.RXRDY)
    {
        receivedData = (char)pDev->pCtrlReg->SCIRXBUF.bit.RXDT;
        pDev->pCtrlReg->SCIFFRX.bit.RXFFINTCLR = 1;

        if(pDev->pCtrlReg->SCIRXST.bit.RXERROR)
        {
            // error occurs when receiving
            pDev->pCtrlReg->SCICTL1.bit.SWRESET = 1;
        }
        else
        {
            // no error for received data
            headIndex = (headIndex + 1) & (pDev->rxBufferSize - 1);
            if(headIndex != pDev->rxBufferTail)
            {
                pDev->pRxBuffer[headIndex] = receivedData;
                pDev->rxBufferHead = headIndex;
            }
        }
    }

    PieCtrlRegs.PIEACK.bit.ACK9 = 1;    // acknowledge PIE that interrupt was been served
}

/**
 * SCIA peripheral
 */
#if !DISABLE_SCIA_ISR
static uart_dev_t* scia;
__interrupt void scia_interrupt(void)
{
    uart_irqHandler(scia);
}
#endif /* !DISABLE_SCIA_ISR */

/**
 * SCIB peripheral
 */
#if !DISABLE_SCIB_ISR
static uart_dev_t* scib;
__interrupt void scib_interrupt(void)
{
    uart_irqHandler(scib);
}
#endif /* !DISABLE_SCIB_ISR */

/**
 * @brief   Initialization of the UART
 * @param   pDev    Pointer to the UART device handler
 * @return  Operation status code
 */
int uart_init(uart_dev_t* pDev)
{
    assert(pDev != NULL);

    EALLOW;
    if((uint32_t)pDev->pCtrlReg == (uint32_t)UART_CTRL_REG_CAST(SciaRegs.SCICCR))
    {
        SysCtrlRegs.PCLKCR0.bit.SCIAENCLK = 1;
    #if !DISABLE_SCIA_ISR
        scia = pDev;
        PieCtrlRegs.PIEIER9.bit.INTx1 = 1;
        PieVectTable.SCIRXINTA = &scia_interrupt;
        IER |= M_INT9;
    #endif /* !DISABLE_SCIA_ISR */
        goto SKIP_SCI;
    }

    if((uint32_t)pDev->pCtrlReg == (uint32_t)UART_CTRL_REG_CAST(ScibRegs.SCICCR))
    {
        SysCtrlRegs.PCLKCR0.bit.SCIBENCLK = 1;
    #if !DISABLE_SCIB_ISR
        scib = pDev;
        PieCtrlRegs.PIEIER9.bit.INTx3 = 1;
        PieVectTable.SCIRXINTB = &scib_interrupt;
        IER |= M_INT9;
    #endif /* !DISABLE_SCIB_ISR */
        goto SKIP_SCI;
    }
    EDIS;
    return IO_ERROR;

SKIP_SCI:
    // Enable Rx/Tx
    if(pDev->pRxPort != NULL)
    {
        pDev->pCtrlReg->SCICTL1.bit.RXENA = 1;
    }
    if(pDev->pTxPort != NULL)
    {
        pDev->pCtrlReg->SCICTL1.bit.TXENA = 1;
    }

    // Initialize Rx/Tx FIFO and interrupts
    pDev->pCtrlReg->SCIFFTX.bit.SCIFFENA = 1;           // enable FIFO
    pDev->pCtrlReg->SCIFFTX.bit.SCIRST = 1;             // reset FIFO
    if(pDev->pRxPort != NULL)
    {
        pDev->pCtrlReg->SCIFFRX.bit.RXFIFORESET = 1;    // reset Rx FIFO head
        pDev->pCtrlReg->SCICTL1.bit.RXERRINTENA = 1;    // enable RxError interrupt
        pDev->pCtrlReg->SCICTL2.bit.RXBKINTENA = 1;     // enable Rx/BB interrupt
        pDev->pCtrlReg->SCIFFRX.bit.RXFFIENA = 1;       // enable Rx FIFO full interrupt
        pDev->pCtrlReg->SCIFFRX.bit.RXFFIL = 1;         // set FIFO length to 1-word
    }
    if(pDev->pTxPort != NULL)
    {
        pDev->pCtrlReg->SCIFFTX.bit.TXFIFOXRESET = 1;   // reset Tx FIFO head
    }
    pDev->pCtrlReg->SCIFFCT.all = 0x0000;               // disable auto baudrate and Tx delay

    // Configure Rx/Tx GPIO multiplexors
    pUartPortCtrlReg_t ports[] = {pDev->pRxPort, pDev->pTxPort};
    const uint16_t pins[] = {pDev->rxPin, pDev->txPin};

    int i; uint32_t mux; uint32_t tmpReg;
    for(i = 0; i < (sizeof(pins)/sizeof(pins[0])); i++)
    {
        if(ports[i] != NULL)
        {
            mux = private_getMuxValue(ports[i], pins[i]);
            if(pins[i] > 15)
            {
                tmpReg = ports[i]->GPAMUX2.all;
                tmpReg &= ~(0x03UL << (2 * (pins[i] - 16)));
                tmpReg |= mux << (2 * (pins[i] - 16));
                ports[i]->GPAMUX2.all = tmpReg;
            }
            else
            {
                tmpReg = ports[i]->GPAMUX1.all;
                tmpReg &= ~(0x03 << (2 * pins[i]));
                tmpReg |= mux << (2 * pins[i]);
                ports[i]->GPAMUX1.all = tmpReg;
            }
            ports[i]->GPAPUD.all |= (1U << pins[i]);
        }
    }

    // Configure UART to default state (9600-8N1)
    private_setBaudrate(pDev, 9600);
    private_setDataBits(pDev, 8);
    private_setParity(pDev, UART_PARITY_NONE);
    private_setStopbits(pDev, UART_STOP_BITS_1);

#if !DISABLE_SCIx_IN_DEBUG
    // Complete current Rx/Tx regardless to breakpoints
    pDev->pCtrlReg->SCIPRI.bit.FREE = 0;
    pDev->pCtrlReg->SCIPRI.bit.SOFT = 1;
#endif

    pDev->pCtrlReg->SCICTL1.bit.SWRESET = 1;    // re-enable SW reset

    EDIS;

    return IO_OK;
}

/**
 * @brief   Open UART device
 * @param   pDev    Pointer to the UART device handler
 * @return  Operation status code
 */
int uart_open(uart_dev_t* pDev)
{
    assert(pDev != NULL);

    if(pDev->pRxPort != NULL)
    {
        pDev->pCtrlReg->SCIFFRX.bit.RXFFINTCLR = 1;     // clear Rx FIFO flag
        pDev->rxBufferHead = 0;
        pDev->rxBufferTail = 0;
    }
    if(pDev->pTxPort != NULL)
    {
        pDev->pCtrlReg->SCIFFTX.bit.TXFFINTCLR = 1;     // clear Tx FIFO flag
    }

    return IO_OK;
}

/**
 * @brief   Close UART device
 * @param   pDev    Pointer to the UART device handler
 * @return  Operation status code
 */
int uart_close(uart_dev_t* pDev)
{
    assert(pDev != NULL);

    pDev->pCtrlReg->SCICTL1.bit.RXENA = 0;
    pDev->pCtrlReg->SCICTL1.bit.TXENA = 0;

    return IO_OK;
}

/**
 * @brief   IO control function for the UART
 * @param   pDev    Pointer to the UART device handler
 * @param   request Command for desired operation
 * @param   arg     Optional argument of the command
 * @return  Operation status code
 */
int uart_ioctl(uart_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    EALLOW;
    switch(request)
    {
        case UART_IOC_SET_BAUD:
            private_setBaudrate(pDev, (uint32_t)va_arg(arg, unsigned long));
            break;

        case UART_IOC_GET_BAUD:
            *va_arg(arg, uint32_t*) = private_getBaudrate(pDev);
            break;

        case UART_IOC_SET_DATA_BITS:
            private_setDataBits(pDev, va_arg(arg, int));
            break;

        case UART_IOC_GET_DATA_BITS:
            *va_arg(arg, int*) = private_getDataBits(pDev);
            break;

        case UART_IOC_SET_STOP_BITS:
            private_setStopbits(pDev, va_arg(arg, int));
            break;

        case UART_IOC_GET_STOP_BITS:
            *va_arg(arg, int*) = private_getStopbits(pDev);
            break;

        case UART_IOC_SET_PARITY:
            private_setParity(pDev, va_arg(arg, int));
            break;

        case UART_IOC_GET_PARITY:
            *va_arg(arg, int*) = private_getParity(pDev);
            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}

/**
 * @brief   Read data received by UART.
 * @param   pDev    Pointer to device handler.
 * @param   pBuffer Pointer to output buffer where to store data.
 * @param   bytes   Number of bytes to read.
 * @return  Number of bytes actually read.
 */
int uart_read(uart_dev_t* pDev, void* pBuffer, int bytes)
{
    assert(pDev != NULL);

    int bytesRead = 0;
    char* pReadBuffer = (char*)pBuffer;
    size_t tailIndex = pDev->rxBufferTail;

    while(bytesRead < bytes)
    {
        if(tailIndex == pDev->rxBufferHead)     // check if buffer is empty
        {
            // TODO - timeout

            break;
        }
        else
        {
            tailIndex = (tailIndex + 1) & (pDev->rxBufferSize - 1);

            *pReadBuffer++ = pDev->pRxBuffer[tailIndex];    // read data from buffer
            bytesRead++;
        }
    }
    pDev->rxBufferTail = tailIndex;

    return bytesRead;
}

/**
 * @brief   Send data over UART.
 * @param   pDev    Pointer to UART device handler.
 * @param   pBuffer Pointer to data to send.
 * @param   bytes   Number of bytes to send.
 * @return  Number of bytes actually sent.
 */
int uart_write(uart_dev_t* pDev, const void* pBuffer, int bytes)
{
    assert(pDev != NULL);

    int bytesWrote = 0;
    char* pTransmitBuffer = (char*)pBuffer;

    while(bytesWrote < bytes)
    {
        if(pDev->pCtrlReg->SCIFFTX.bit.TXFFST == 0)   // wait until data moves to shift register
        {
            pDev->pCtrlReg->SCITXBUF = (*pTransmitBuffer) & 0x00FF;
            pTransmitBuffer++;
            bytesWrote++;
        }
        else
        {
            // TODO - timeout
        }
    }
    while(pDev->pCtrlReg->SCICTL2.bit.TXEMPTY == 0);    // wait until Tx and shift registers are empty

    return bytesWrote;
}



static uint32_t private_getMuxValue(pUartPortCtrlReg_t pPort, const uint16_t pin)
{
                // GPIOx =  0  1  2  3  4  5  6  7  8  9  10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31
    uint32_t mux_gpioA[] = {0, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 2, 0, 2, 2, 0, 0, 2, 2, 0, 0, 3, 3, 0, 0, 0, 0, 1, 1, 0, 0};
                // GPIOx =  32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63
    uint32_t mux_gpioB[] = {0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0};

    if(pin < 32)
    {
        if((uint32_t)pPort == (uint32_t)UART_RX_PORT_REG_CAST(GpioCtrlRegs.GPACTRL))
        {
            return mux_gpioA[pin];
        }
        else if((uint32_t)pPort == (uint32_t)UART_RX_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL))
        {
            return mux_gpioB[pin];
        }
    }

    return 0UL;
}
static uint32_t private_getLSPCLKValue(void)
{
    uint32_t clk = 10000000;    // Internal OSC1/OSC2 frequency (default)

    if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 1 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 0)
    {
        clk = XCLKIN;   // predefined symbol for External OSC frequency
    }
    else if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 0 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 1)
    {
        clk = XTAL;     // predefined symbol for Crystal or Resonator frequency
    }

    // compute PLL output (SYSCLKOUT)
    if(SysCtrlRegs.PLLCR.bit.DIV != 0)
    {
        if(SysCtrlRegs.PLLSTS.bit.DIVSEL <= 1)
        {
            clk = clk >> 2;  // (clk / 4)
        }
        else if(SysCtrlRegs.PLLSTS.bit.DIVSEL == 2)
        {
            clk = clk >> 1;  // (clk / 2)
        }
        clk = clk * SysCtrlRegs.PLLCR.bit.DIV;
    }

    // compute Low speed peripheral clock
    if(SysCtrlRegs.LOSPCP.bit.LSPCLK != 0)
    {
        clk = clk / (SysCtrlRegs.LOSPCP.bit.LSPCLK << 1);   // (clk / 2*LSPCLK)
    }

    return clk;
}
static void private_setParity(uart_dev_t* pDev, int parity)
{
    pDev->pCtrlReg->SCICCR.bit.PARITYENA = parity ? 1 : 0;
    pDev->pCtrlReg->SCICCR.bit.PARITY = parity & 0x01;
}
static int private_getParity(uart_dev_t* pDev)
{
    if(pDev->pCtrlReg->SCICCR.bit.PARITYENA)
    {
        return pDev->pCtrlReg->SCICCR.bit.PARITY ? UART_PARITY_EVEN : UART_PARITY_ODD;
    }
    else
    {
        return UART_PARITY_NONE;
    }
}
static void private_setStopbits(uart_dev_t* pDev, int stopbits)
{
    pDev->pCtrlReg->SCICCR.bit.STOPBITS = stopbits ? UART_STOP_BITS_1 : UART_STOP_BITS_2;
}
static int private_getStopbits(uart_dev_t* pDev)
{
    return pDev->pCtrlReg->SCICCR.bit.STOPBITS ? UART_STOP_BITS_1 : UART_STOP_BITS_2;
}
static void private_setBaudrate(uart_dev_t* pDev, uint32_t baudrate)
{
    uint16_t baud = (uint16_t)((private_getLSPCLKValue() / (baudrate << 3)) - 1UL);

    pDev->pCtrlReg->SCILBAUD = baud & 0xFF;
    baud >>= 8;
    pDev->pCtrlReg->SCIHBAUD = baud & 0xFF;
}
static uint32_t private_getBaudrate(uart_dev_t* pDev)
{
    uint32_t baud = ((uint32_t)pDev->pCtrlReg->SCIHBAUD << 8) + (uint32_t)pDev->pCtrlReg->SCILBAUD;
    return (private_getLSPCLKValue() / ((baud + 1UL) << 3));
}
static void private_setDataBits(uart_dev_t* pDev, int databits)
{
    if(databits >= 1 && databits <= 8)
    {
        pDev->pCtrlReg->SCICCR.bit.SCICHAR = (uint16_t)(databits - 1);
    }
}
static int private_getDataBits(uart_dev_t* pDev)
{
    return (int)(pDev->pCtrlReg->SCICCR.bit.SCICHAR + 1);
}

#endif /* USE_UART_DRIVER && defined(TMS320F28069) */
