/**
 * @file    main.c
 * @author  Matej Turinsky
 * @date    16. 1. 2024
 * @brief   Main source code for SD11
 */

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>
#include <assert.h>
#include <nanomodbus_port.h>

#include "DSP28x_Project.h"
#include "host_config.h"
#include "cla_shared.h"
#include "devio.h"
#include "nanomodbus.h"
#include "nanomodbus_port.h"


/* Private typedef -----------------------------------------------------------*/
typedef union word32 {
    float real;
    uint32_t dword;
    uint16_t word[2];
} word32_t;

typedef struct arcStatus {
    volatile uint16_t* pArcCounter;
    float* pArcFilterOutputs;
} arcStatus_t;


/* Private define ------------------------------------------------------------*/
// GPIO portA defines
#define GPIOA_DIFF_CURR (1UL << GPIOA_PIN_DIFF_CURRENT)     // differential current sensor

// GPIO portB defines
#define GPIOB_LED_D9    (1UL << (34-32))                    // red on-board LED
#define GPIOB_LED_D10   (1UL << (39-32))                    // blue on-board LED

// Failed measurement codes
#define UPDATE_FAILED_VA            0
#define UPDATE_FAILED_ARCS          1
#define UPDATE_FAILED_DIFF_CURR     2
#define UPDATE_FAILED_IRRADIANCE    3
#define UPDATE_FAILED_PANEL_TEMP    4
#define UPDATE_FAILED_AMBIENT_TEMP  5
#define UPDATE_FAILED_MODBUS_POOL   6


/* Private macro -------------------------------------------------------------*/


/* Private variables ---------------------------------------------------------*/
coils_t modbus_coils;
discInputs_t modbus_discIn;
holdingRegs_t modbus_holdReg;
inputRegs_t modbus_inputReg;
arcStatus_t arcStatus;

char pUnitsBuffer[2];
word32_t pWord32Buffer[2];

volatile uint64_t tickCounter;
uint64_t nextPeriodTickValue;
uint16_t periodMissedCounter;

uint16_t i;
char sciA_rxBuffer[SCIA_BUFFER_SIZE];
char sciB_rxBuffer[SCIB_BUFFER_SIZE];


/* Private function prototypes -----------------------------------------------*/
static int tim_overflowCallback(void* pCounter);
static int adcCla_errorCallback(devio_t pSerial);
static int adcCla_convCallback(void* pArcStatus);
static void adcCla_initVariables(void);
static float tcn75a_convertTemperature(const uint16_t* const pBuffer);
static int sendBinaryToSerial(devio_t pSerial, int length, word32_t* pValues, char* pUnits);
static int sendErrorToSerial(devio_t pSerial, int code);


/* Start of user code --------------------------------------------------------*/

/**
 * @brief  The application entry point.
 * @return Non-zero value when error occurred
 */
void main(void)
{
    // Initialize core
#if PERFORM_DEVICE_CALL
    DisableDog();
    EALLOW;
    SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 1;
    (*Device_cal)();    // copies ADC and OCS manufactures calibration constants into proper registers.
    SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 0;
    EDIS;
#endif
#if USE_INT_OSC1 && !(USE_INT_OSC2 || USE_XTAL_OSC || USE_EXT_OCS)
    IntOsc1Sel();
#elif USE_INT_OSC2 && !(USE_INT_OSC1 || USE_XTAL_OSC || USE_EXT_OCS)
    IntOsc2Sel();
#elif USE_XTAL_OSC && !(USE_INT_OSC1 || USE_INT_OSC2 || USE_EXT_OCS)
    XtalOscSel();
#elif USE_EXT_OCS && !(USE_INT_OSC1 || USE_INT_OSC2 || USE_XTAL_OCS)
    ExtOscSel();
#else
#   error("Select one of the above oscillators..")
#endif
    InitPll(PLL_MULTIPLIER, PLL_DIVIDER);
#if USE_PROG_FLASH
    memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (size_t)&RamfuncsLoadSize);   // copy program from Flash to RAM
#endif

    // Configure Peripheral interrupt expansion
    InitPieCtrl();
    InitPieVectTable();

    // Application initialization
    assert(io_init() == IO_OK);

    // Initialize LaunchXL serial connection (115200-8N1)
    devio_t uart_serialPort = io_open("/per/sciA");
    assert(uart_serialPort != NULL);
    io_ioctl(uart_serialPort, UART_IOC_SET_BAUD, 115200UL);
#if USE_SERIAL_DEBUG
    char start[] = "[i] MT-SD11 starting";
    io_write(uart_serialPort, start, sizeof(start)/sizeof(start[0]));
#endif /* USE_SERIAL_DEBUG */

    // Initialize GPIOs
    devio_t gpio_portA = io_open("/per/gpioA");
    assert(gpio_portA != NULL);
    devio_t gpio_portB = io_open("/per/gpioB");
    assert(gpio_portB != NULL);

    // Configure on-board LEDs
    io_ioctl(gpio_portB, GPIO_IOC_SET_DIR, (GPIOB_LED_D9 | GPIOB_LED_D10));
    io_ioctl(gpio_portB, GPIO_IOC_SET_OUTHI, (GPIOB_LED_D9 | GPIOB_LED_D10));

    // Configure differential current input
    io_ioctl(gpio_portA, GPIO_IOC_GET_DIR, pWord32Buffer);
    io_ioctl(gpio_portA, GPIO_IOC_SET_DIR, ((pWord32Buffer[0].dword) & (~GPIOA_DIFF_CURR)));
    io_ioctl(gpio_portA, GPIO_IOC_SET_PULL, GPIOA_DIFF_CURR);

    // Initialize G3-PLC modem serial connection (115200-8N1)
    devio_t uart_plcModem = io_open("/per/sciB");
    assert(uart_plcModem != NULL);
    io_ioctl(uart_plcModem , UART_IOC_SET_BAUD, 115200UL);

    // Configure modbus server structures
    nmbs_platform_conf modbus_config = {
        .transport = NMBS_TRANSPORT_RTU,
        .read = modbus_read,
        .write = modbus_write,
        .arg = uart_serialPort
    };
    nmbs_callbacks modbus_callbacks = {
        .read_coils = modbus_readCoils,
        .read_discrete_inputs = modbus_readDiscInputs,
        .read_holding_registers = modbus_readHoldingRegsCallback,
        .read_input_registers = modbus_readInputRegsCallback,
        .write_multiple_coils = modbus_writeMultCoils,
        .write_multiple_registers = modbus_writeMultRegs
    };

    // Create modbus RTU server
    nmbs_t modbus_handler;
    assert(nmbs_server_create(&modbus_handler, NMBS_RTU_ADDRESS, &modbus_config, &modbus_callbacks) == NMBS_ERROR_NONE);
    nmbs_set_callbacks_arg(&modbus_handler, gpio_portA);

    // Configure millisecond tick timer
    devio_t tim_msTick = io_open("/per/epwm2");
    assert(tim_msTick != NULL);
    io_ioctl(tim_msTick, TIM_IOC_SET_FREQUENCY, 1000UL);
    io_callback_t callback;
    callback.function = tim_overflowCallback;
    callback.argument = (void*)(&tickCounter);
    io_ioctl(tim_msTick, TIM_IOC_SET_OF_CALLBACK, &callback);

    // Set modbus read/write timeouts
    nmbs_set_read_timeout(&modbus_handler, NMBS_READ_TIMEOUT);
    nmbs_set_byte_timeout(&modbus_handler, NMBS_BYTE_TIMEOUT);

    // Initialize modbus registers and coils
    memset(&modbus_discIn.registers, 0U, sizeof(modbus_discIn.registers)/sizeof(modbus_discIn.registers[0]));
    memset(&modbus_holdReg.registers, 0U, sizeof(modbus_holdReg.registers)/sizeof(modbus_holdReg.registers[0]));
    memset(&modbus_inputReg.registers, 0U, sizeof(modbus_inputReg.registers)/sizeof(modbus_inputReg.registers[0]));
    memset(&modbus_coils.registers, 0U, sizeof(modbus_coils.registers)/sizeof(modbus_coils.registers[0]));

    // Initialize I2C master (100000-7bit)
    devio_t i2c_master = io_open("/per/i2cA");
    assert(i2c_master != NULL);
    io_ioctl(i2c_master, I2C_IOC_SET_SPEED, 100000UL);

    // Configure temperature sensors
    char pCommand[] = {0x01, 0x60};
    int i2cAddresses[] = {I2C_ADDRESS_TEMPERATURE_PANEL, I2C_ADDRESS_TEMPERATURE_AMBIENT};
    for(i = 0; i < sizeof(i2cAddresses)/sizeof(i2cAddresses[0]); i++)
    {
        // Set 12bit resolution
        io_ioctl(i2c_master, I2C_IOC_SET_SLAVE_ADDR, i2cAddresses[i]);
        while(io_write(i2c_master, pCommand, 2) != 2)
        {
            DELAY_US(1000UL);
        }
    }
    pCommand[0] = 0x00;
    for(i = 0; i < sizeof(i2cAddresses)/sizeof(i2cAddresses[0]); i++)
    {
        // Reset sensor register pointer (temperature value)
        io_ioctl(i2c_master, I2C_IOC_SET_SLAVE_ADDR, i2cAddresses[i]);
        while(io_write(i2c_master, pCommand, 1) != 1)
        {
            DELAY_US(1000UL);
        }
    }
    pUnitsBuffer[0] = 'T';
    for(i = 0; i < sizeof(i2cAddresses)/sizeof(i2cAddresses[0]); i++)
    {
        // Read temperature values
        io_ioctl(i2c_master, I2C_IOC_SET_SLAVE_ADDR, i2cAddresses[i]);
        while(io_read(i2c_master, pWord32Buffer, 2) != 2)
        {
            DELAY_US(1000UL);
        }
        sendBinaryToSerial(uart_serialPort, 1, pWord32Buffer, pUnitsBuffer);
    }

    // Initialize ADC-CLA (current & voltage measurements)
    devio_t adcCla_VA = io_open("/per/adcVA");
    assert(adcCla_VA != NULL);
    io_ioctl(adcCla_VA, ADC_IOC_SET_SAMPLING_MODE, ADC_CLA_SAMPLING_MODE_SIMULTANEOUS);
    io_ioctl(adcCla_VA, ADC_IOC_SET_FREQUENCY, 500000UL);
    io_ioctl(adcCla_VA, ADC_IOC_SET_CHANNEL, ADC_CLA_VA_INPUT_CHANNEL);
    callback.function = adcCla_errorCallback;
    callback.argument = uart_serialPort;
    io_ioctl(adcCla_VA, ADC_IOC_SET_ERROR_CALLBACK, &callback);
    io_ioctl(adcCla_VA, ADC_IOC_CLEAR_FLAGS, ADC_FLAG_ERRORS);

    // Set ADC-CLA VA conversion callback argument to arc status
    arcStatus.pArcFilterOutputs = adcCla_arcFilterOutputs;
    arcStatus.pArcCounter = &modbus_holdReg.values.arcCounter;
    callback.function = adcCla_convCallback;
    callback.argument = &arcStatus;

    // Initialize ADC-CLA (solar irradiance measurements)
    devio_t adcCla_Irr = io_open("/per/adcIrr");
    assert(adcCla_Irr != NULL);
    io_ioctl(adcCla_Irr, ADC_IOC_SET_SAMPLING_MODE, ADC_CLA_SAMPLING_MODE_SEQUENTIAL);
    io_ioctl(adcCla_Irr, ADC_IOC_SET_FREQUENCY, 1000UL);
    io_ioctl(adcCla_Irr, ADC_IOC_SET_CHANNEL, ADC_CLA_IRR_INPUT_CHANNEL);
    callback.function = adcCla_errorCallback;
    callback.argument = uart_serialPort;
    io_ioctl(adcCla_Irr, ADC_IOC_SET_ERROR_CALLBACK, &callback);
    io_ioctl(adcCla_Irr, ADC_IOC_CLEAR_FLAGS, ADC_FLAG_ERRORS);

    // Initialize variables
    adcCla_initVariables();
    modbus_discIn.values.initialized = 1;
    modbus_discIn.values.running = 0;
#if USE_SERIAL_DEBUG
    char init[] = "[i] MT-SD11 successfully initialized";
    io_write(uart_serialPort, init, sizeof(init)/sizeof(init[0]));
#endif /* USE_SERIAL_DEBUG */

    // Start CPU watchdog timer
    devio_t watchdog_cpu = io_open("/per/wdCPU");
    assert(watchdog_cpu != NULL);
#ifndef DEBUG
    io_ioctl(watchdog_cpu, WD_IOC_START);
#endif /* DEBUG */

    // Enable global interrupt
    EnableInterrupts();
    ERTM;

    // Fill ADC-CLA delay line buffers
    while(io_read(adcCla_VA, pWord32Buffer, ADC_CLA_VA_IN_DELAY_LINE_LENGTH) != ADC_CLA_VA_IN_DELAY_LINE_LENGTH);
    pUnitsBuffer[ADC_CLA_VA_CURRENT_POS] = 'I';
    pUnitsBuffer[ADC_CLA_VA_VOLTAGE_POS] = 'U';
    sendBinaryToSerial(uart_serialPort, 2, pWord32Buffer, pUnitsBuffer);
    while(io_read(adcCla_Irr, pWord32Buffer, ADC_CLA_IRR_IN_DELAY_LINE_LENGTH) != ADC_CLA_IRR_IN_DELAY_LINE_LENGTH);
    pUnitsBuffer[0] = 'G';
    sendBinaryToSerial(uart_serialPort, 1, pWord32Buffer, pUnitsBuffer);

    // Start tick timer
    tickCounter = 0UL;
    nextPeriodTickValue = tickCounter + 1000UL;
    io_ioctl(tim_msTick, TIM_IOC_SET_ENABLED, 1);
    periodMissedCounter = 0U;

    // Super-loop
    for(;;)
    {
        io_ioctl(watchdog_cpu, WD_IOC_RESET);

        if(tickCounter > nextPeriodTickValue)
        {
            // Second passed routine
            nextPeriodTickValue = tickCounter + 1000UL;
            io_ioctl(gpio_portB, GPIO_IOC_SET_OUTTGL, GPIOB_LED_D10);
            modbus_holdReg.values.localTime++;
            modbus_discIn.values.running ^= 1;

            // Read differential current
            if(io_read(gpio_portA, pWord32Buffer, 4) == 4)
            {
                modbus_discIn.values.diffCurrent = (pWord32Buffer[0].dword & GPIOA_DIFF_CURR) ? 1U : 0U;
                modbus_discIn.values.diffCurrFailed = 0U;
            }
            else
            {
                modbus_discIn.values.diffCurrFailed = 1U;
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_DIFF_CURR);
            }

            // Read current/voltage measurements
            callback.function = NULL;
            io_ioctl(adcCla_VA, ADC_IOC_SET_CONV_CALLBACK, callback);
            if(io_read(adcCla_VA, pWord32Buffer, 1) == 1)
            {
                modbus_inputReg.values.stringCurrent = pWord32Buffer[ADC_CLA_VA_CURRENT_POS].real;
                modbus_inputReg.values.stringVoltage = pWord32Buffer[ADC_CLA_VA_VOLTAGE_POS].real;
                if(periodMissedCounter == 0U)
                {
                    modbus_inputReg.values.stringPower = modbus_inputReg.values.stringCurrent * modbus_inputReg.values.stringVoltage;
                    modbus_holdReg.values.stringEnergy += modbus_inputReg.values.stringPower;
                }
                else
                {
                    float avgStringPower = modbus_inputReg.values.stringPower;
                    modbus_inputReg.values.stringPower = modbus_inputReg.values.stringCurrent * modbus_inputReg.values.stringVoltage;
                    avgStringPower = (avgStringPower + modbus_inputReg.values.stringPower) / 2.0f;
                    modbus_holdReg.values.stringEnergy += (avgStringPower * (float)(periodMissedCounter + 1U));

                    periodMissedCounter = 0U;
                    (void)avgStringPower;
                }
                modbus_discIn.values.currFailed = 0U;
                modbus_discIn.values.voltFailed = 0U;
                modbus_discIn.values.powerFailed = 0U;
                modbus_discIn.values.energyFailed = 0U;
            }
            else
            {

                modbus_discIn.values.currFailed = 1U;
                modbus_discIn.values.voltFailed = 1U;
                modbus_discIn.values.powerFailed = 1U;
                modbus_discIn.values.energyFailed = 1U;
                periodMissedCounter++;
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_VA);
            }
            callback.function = adcCla_convCallback;
            io_ioctl(adcCla_VA, ADC_IOC_SET_CONV_CALLBACK, callback);

            // Read solar irradiance
            if(io_read(adcCla_Irr, pWord32Buffer, 1) == 1)
            {
                modbus_inputReg.values.solarIrradiance = pWord32Buffer[0].real;
                modbus_discIn.values.irradianceFailed = 0U;
            }
            else
            {
                modbus_discIn.values.irradianceFailed = 1U;
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_IRRADIANCE);
            }

            // Read panel temperature
            io_ioctl(i2c_master, I2C_IOC_SET_SLAVE_ADDR, I2C_ADDRESS_TEMPERATURE_PANEL);
            if(io_read(i2c_master, pWord32Buffer, 2) == 2)
            {
                modbus_inputReg.values.panelTemperature = tcn75a_convertTemperature(pWord32Buffer[0].word);
                modbus_discIn.values.panelTempFailed = 0U;
            }
            else
            {
                modbus_discIn.values.panelTempFailed = 1U;
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_PANEL_TEMP);
            }

            // Read ambient temperature
            io_ioctl(i2c_master, I2C_IOC_SET_SLAVE_ADDR, I2C_ADDRESS_TEMPERATURE_AMBIENT);
            if(io_read(i2c_master, pWord32Buffer, 2) == 2)
            {
                modbus_inputReg.values.ambientTemperature = tcn75a_convertTemperature(pWord32Buffer[0].word);
                modbus_discIn.values.ambientTempFailed = 0U;
            }
            else
            {
                modbus_discIn.values.ambientTempFailed = 1U;
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_AMBIENT_TEMP);
            }
        }
        else
        {
            // Read arc status (modbus arc register is updated in ADC-CLA callback)
            if(io_read(adcCla_VA, NULL, 500) != 500)
            {
                sendErrorToSerial(uart_serialPort, UPDATE_FAILED_ARCS);
            }
        }

        io_ioctl(watchdog_cpu, WD_IOC_RESET);
        if(nmbs_server_poll(&modbus_handler) != NMBS_ERROR_NONE)
        {
            sendErrorToSerial(uart_serialPort, UPDATE_FAILED_MODBUS_POOL);
        }
        else
        {
            io_ioctl(gpio_portB, GPIO_IOC_SET_OUTTGL, GPIOB_LED_D9);
        }
    }
}


/**
 * @brief   Millisecond timer overflow callback.
 * @param   pGPIO   pointer to GPIOB handler
 * @return  Non-zero value when message failed to send.
 */
static int tim_overflowCallback(void* pCounter)
{
    uint64_t* pTickCounter = (uint64_t*)pCounter;
    *pTickCounter += 1UL;

    return IO_OK;
}

/**
 * @brief   ADC-CLA error occurred callback.
 * @param   pSerial     pointer to serial communication handler
 * @return  Non-zero value when message failed to send.
 */
static int adcCla_errorCallback(devio_t pSerial)
{
#if USE_SERIAL_DEBUG
    char message[] = "[!] ADC-CLA overflowed\n";
    if(io_write(pSerial, message, sizeof(message)/sizeof(message[0])) != sizeof(message)/sizeof(message[0]))
    {
        return IO_ERROR;
    }
#endif /* USE_SERIAL_DEBUG */

    return IO_OK;
}

/**
 * @brief   ADC-CLA error occurred callback.
 * @param   pArcStatus      pointer to arc detection status structure
 * @return  Non-zero value when message failed to send.
 */
static int adcCla_convCallback(void* pArcStatus)
{
    volatile uint16_t* pArcCounter = ((arcStatus_t*)pArcStatus)->pArcCounter;
    float* arcFilterOutputs = ((arcStatus_t*)pArcStatus)->pArcFilterOutputs;
    uint16_t filterOvfCnt = 0U;

    uint16_t i;
    for(i = 0U; i < ADC_CLA_NUM_OF_ARC_FILTERS; i++)
    {
        if(arcFilterOutputs[i] > ADC_CLA_ARC_FILTER_AMPLITUDE_TRESHOLD)
        {
            filterOvfCnt++;
        }
    }

    if(filterOvfCnt >= ADC_CLA_NUM_OF_ARC_FILTERS_TO_DETECT)
    {
        *pArcCounter = (*pArcCounter) + 1U;
    }

    return IO_OK;
}

/**
 * @brief   Clears CLA task variables.
 * @return  None.
 */
static void adcCla_initVariables(void)
{
    uint16_t i;
    for(i = 0; i < ADC_CLA_VA_IN_DELAY_LINE_LENGTH; i++)
    {
        adcCla_VAInput[ADC_CLA_VA_CURRENT_POS][i] = 0;
        adcCla_VAInput[ADC_CLA_VA_VOLTAGE_POS][i] = 0;
    }
    for(i = 0; i < ADC_CLA_IRR_IN_DELAY_LINE_LENGTH; i++)
    {
        adcCla_irrInput[i] = 0;
    }

    adcCla_currentScale = ADC_CLA_INPUT_GAIN_CURRENT * (ADC_CLA_REF_VOLTAGE / 12.0f);
    adcCla_currentOffset = ADC_CLA_INPUT_OFFSET_CURRENT;
    adcCla_voltageScale = ADC_CLA_INPUT_GAIN_VOLTAGE * (ADC_CLA_REF_VOLTAGE / 12.0f);
    adcCla_voltageOffset = ADC_CLA_INPUT_OFFSET_VOLTAGE;

    adcCla_irrScale = ADC_CLA_INPUT_GAIN_IRRADIANCE * (ADC_CLA_REF_VOLTAGE / 12.0f);
    adcCla_irrOffset = ADC_CLA_INPUT_OFFSET_IRRADIANCE;
    adcCla_irrExp = ADC_CLA_CONV_POWER_IRRADIANCE;
    adcCla_irrMult = ADC_CLA_CONV_MULT_IRRADIANCE;

    memset(adcCla_arcFilterOutputs, 0.0f, sizeof(adcCla_arcFilterOutputs)/sizeof(adcCla_arcFilterOutputs[0]));
    uint16_t ii;
#ifdef ADC_CLA_ARC_FILTER_NUM_COEFFICIENT
#if (ADC_CLA_NUM_OF_ARC_FILTERS == 1U)
    float numCoef[ADC_CLA_LEN_OF_ARC_FILTERS] = ADC_CLA_ARC_FILTER_COEFFICIENT;
    for(i = 0; i < ADC_CLA_LEN_OF_ARC_FILTERS; i++)
    {
        adcCla_arcFilterNumCoefs[i] = numCoef[i];
    }
#else
    float numCoef[ADC_CLA_NUM_OF_ARC_FILTERS][ADC_CLA_LEN_OF_ARC_FILTERS] = ADC_CLA_ARC_FILTER_NUM_COEFFICIENT;
    for(i = 0; i < ADC_CLA_NUM_OF_ARC_FILTERS; i++)
    {
        for(ii = 0; ii < ADC_CLA_VA_IN_DELAY_LINE_LENGTH; ii++)
        {
            adcCla_arcFilterNumCoefs[i][ii] = numCoef[i][ii];
        }
    }
#endif /* (ADC_CLA_NUM_OF_ARC_FILTERS == 1U) */
    (void)numCoef;
#else
    memset(adcCla_arcFilterNumCoefs, 0.0f, sizeof(adcCla_arcFilterNumCoefs)/sizeof(adcCla_arcFilterNumCoefs[0]));
#endif /*ADC_CLA_ARC_FILTER_NUM_COEFFICIENT*/

#ifdef ADC_CLA_ARC_FILTER_DEN_COEFFICIENT
#if (ADC_CLA_NUM_OF_ARC_FILTERS == 1U)
    float denCoef[ADC_CLA_LEN_OF_ARC_FILTERS] = ADC_CLA_ARC_FILTER_COEFFICIENT;
    for(i = 0; i < ADC_CLA_LEN_OF_ARC_FILTERS; i++)
    {
        adcCla_arcFilterDenCoefs[i] = denCoef[i];
    }
#else
    float denCoef[ADC_CLA_NUM_OF_ARC_FILTERS][ADC_CLA_LEN_OF_ARC_FILTERS] = ADC_CLA_ARC_FILTER_DEN_COEFFICIENT;
    for(i = 0; i < ADC_CLA_NUM_OF_ARC_FILTERS; i++)
    {
        for(ii = 0; ii < ADC_CLA_LEN_OF_ARC_FILTERS; ii++)
        {
            adcCla_arcFilterDenCoefs[i][ii] = denCoef[i][ii];
        }
    }
#endif /* (ADC_CLA_NUM_OF_ARC_FILTERS == 1U) */
    (void)denCoef;
#else
    memset(adcCla_arcFilterDenCoefs, 0.0f, sizeof(adcCla_arcFilterDenCoefs)/sizeof(adcCla_arcFilterDenCoefs[0]));
#endif /*ADC_CLA_ARC_FILTER_DEN_COEFFICIENT*/
}

/**
 * @brief   Converts value from TCN75A temperature sensor.
 * @param   pValue     pointer to raw TCN75A value
 * @return  Temperature in degrees Celsius.
 */
static float tcn75a_convertTemperature(const uint16_t* const pValue)
{
    float temperature = 0.0f;

    temperature = (float)(pValue[0] & 0x7F);
    temperature *= (pValue[0] & 0x80) ? -1.0f : 1.0f;
    temperature += (float)((pValue[1] & 0xF0) >> 4) * 0.0625f;

    return temperature;
}

/**
 * @brief   Debug result print over serial in binary format.
 * @param   pSerial     pointer to serial communication handler
 * @param   length      length of pValues/pUnits buffers
 * @param   pValues     pointer to result buffer
 * @param   pUnits      pointer to result units buffer
 * @return  Non-zero value when message failed to send.
 */
static int sendBinaryToSerial(devio_t pSerial, int length, word32_t* pValues, char* pUnits)
{
#if USE_SERIAL_DEBUG
    int toSend = length;
    word32_t* pTmpValues = pValues;
    char* pTmpUnits = pUnits;

    while(toSend)
    {
        char message[] = "[>] X =                                 ";
        message[4] = *pTmpUnits;
        uint32_t number = pTmpValues->dword;

        uint16_t bit;
        for(bit = 2; bit < 34; bit++)
        {
            message[sizeof(message)/sizeof(message[0]) - bit] = (number & 1UL) ? '1' : '0';
            number >>= 1;
        }

        if(io_write(pSerial, message, sizeof(message)/sizeof(message[0])) != sizeof(message)/sizeof(message[0]))
        {
            return IO_ERROR;
        }

        toSend--;
        pTmpValues++;
        pTmpUnits++;
    }
#endif /* USE_SERIAL_DEBUG */

    return IO_OK;
}

/**
 * @brief   Debug modbus values update error log.
 * @param   pSerial     pointer to serial communication handler
 * @param   code        number of failed measurement
 * @return  Non-zero value when message failed to send.
 */
static int sendErrorToSerial(devio_t pSerial, int code)
{
#if USE_SERIAL_DEBUG
    char message[] = "[!] Error:         ";
    switch(code)
    {
        case UPDATE_FAILED_VA:
            message[12] = 'V';
            message[13] = '-';
            message[14] = 'A';
            message[15] = ' ';
            message[16] = 'v';
            message[17] = 'a';
            message[18] = 'l';
            message[19] = '.';
            break;

        case UPDATE_FAILED_ARCS:
            message[12] = 'A';
            message[13] = 'r';
            message[14] = 'c';
            message[15] = ' ';
            message[16] = 'd';
            message[17] = 'e';
            message[18] = 't';
            message[19] = '.';
            break;

        case UPDATE_FAILED_DIFF_CURR:
            message[12] = 'D';
            message[13] = 'i';
            message[14] = 'f';
            message[15] = 'f';
            message[16] = '.';
            message[17] = ' ';
            message[18] = 'c';
            message[19] = '.';
            break;

        case UPDATE_FAILED_IRRADIANCE:
            message[12] = 'S';
            message[13] = 'o';
            message[14] = 'l';
            message[15] = 'a';
            message[16] = 'r';
            message[17] = ' ';
            message[18] = 'I';
            message[19] = '.';
            break;

        case UPDATE_FAILED_PANEL_TEMP:
            message[12] = 'p';
            message[13] = 'a';
            message[14] = 'n';
            message[15] = 'e';
            message[16] = 'l';
            message[17] = ' ';
            message[18] = 't';
            message[19] = '.';
            break;

        case UPDATE_FAILED_AMBIENT_TEMP:
            message[12] = 'a';
            message[13] = 'm';
            message[14] = 'b';
            message[15] = 'i';
            message[16] = '.';
            message[17] = ' ';
            message[18] = 't';
            message[19] = '.';
            break;

        case UPDATE_FAILED_MODBUS_POOL:
            message[12] = 'm';
            message[13] = 'b';
            message[14] = '.';
            message[15] = ' ';
            message[16] = 'p';
            message[17] = 'o';
            message[18] = 'o';
            message[19] = 'l';
            break;

        default:
            return IO_ERROR;
    }

    if(io_write(pSerial, message, sizeof(message)/sizeof(message[0])) != sizeof(message)/sizeof(message[0]))
    {
        return IO_ERROR;
    }
#endif /* USE_SERIAL_DEBUG */

    return IO_OK;
}
