/**
 * @file    i2c_tms320f28069.c
 * @author  Matej Turinsky
 * @date    5. 4. 2024
 * @brief   I2C driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_I2C_DRIVER && defined(TMS320F28069)

#include <stddef.h>
#include <assert.h>
#include <stdarg.h>

#include "../../../../common/statuscodes.h"
#include "../../i2c.h"
#include "i2c_tms320f28069.h"


/**
 * Private function declaration
 */
static uint32_t private_getSYSCLKValue(void);
static uint32_t private_getMuxValue(pI2cPortCtrlReg_t pPort, const uint16_t pin);
static int private_setSlaveAddress(i2c_dev_t* pDev, const uint16_t slaveAddress);
static void private_setAddressMode(i2c_dev_t* pDev, const i2c_addr_mode_t addressMode);
static int private_setSpeed(i2c_dev_t* pDev, const uint32_t speed);
static int private_msgRx(i2c_dev_t* pDev, i2c_msg_t* const pMsg);
static int private_msgTx(i2c_dev_t* pDev, i2c_msg_t* const pMsg);

/** Simple delay macro */
#define _DELAY(loops) do{ uint32_t cycle = 0UL; while(cycle < loops) cycle++; } while(0)

/**
 * @brief   I2C general interrupt handler.
 * @param   pDev    Pointer to device handler.
 */
static void i2c_irqHandler(i2c_dev_t* pDev)
{
    assert(pDev != NULL);

    //TODO

    PieCtrlRegs.PIEACK.bit.ACK8 = 1;    // acknowledge PIE that interrupt was been served
}

/**
 * I2CA peripheral
 */
#if !DISABLE_I2CA_ISR
static i2c_dev_t* i2ca;
__interrupt void i2ca_interrupt(void)
{
    i2c_irqHandler(i2ca);
}
#endif /* !DISABLE_I2CA_ISR */

/**
 * @brief   Initialization of the I2C
 * @param   pDev    Pointer to the I2C device handler
 * @return  Operation status code
 */
int i2c_init(i2c_dev_t* pDev)
{
    assert(pDev != NULL);

    EALLOW;
    SysCtrlRegs.PCLKCR0.bit.I2CAENCLK = 1;
#if !DISABLE_I2CA_ISR
    i2ca = pDev;
    PieCtrlRegs.PIEIER8.bit.INTx1 = 1;
    PieVectTable.I2CINT1A = &i2ca_interrupt;    // general I2C interrupts
//    PieCtrlRegs.PIEIER8.bit.INTx2 = 1;
//    PieVectTable.I2CINT2A = &i2ca_interrupt;    // FIFO I2C interrupts
    IER |= M_INT8;
#endif /* !DISABLE_I2CA_ISR */

    pDev->pCtrlReg->I2CMDR.bit.IRS = 0;     // disable I2C during configuration

    pDev->pCtrlReg->I2CMDR.bit.TRX = 0;     // I2C starts in receiver mode
    pDev->pCtrlReg->I2CMDR.bit.FREE = 1;    // I2C runs in free mode - ignores breakpoints

    if(pDev->pSelfSlaveAddress != NULL)
    {
        // Initialize I2C as SLAVE device
        private_setAddressMode(pDev, I2C_ADDR_MODE_7_BIT);  // propagate self slave address (as 7-bit)
        //TODO
    }
    else
    {
        // Initialize I2C as MASTER device
        pDev->pCtrlReg->I2CMDR.bit.MST = 1;     // device is a master
    }

    pDev->pCtrlReg->I2CMDR.bit.IRS = 1;     // enable I2C

    // Configure SDA/SCL GPIO multiplexors
    pI2cPortCtrlReg_t ports[] = {pDev->pSdaPort, pDev->pSclPort};
    const uint16_t pins[] = {pDev->sdaPin, pDev->sclPin};

    int i; uint32_t mux; uint32_t tmpReg;
    for(i = 0; i < (sizeof(pins)/sizeof(pins[0])); i++)
    {
        if(ports[i] != NULL)
        {
            ports[i]->GPAPUD.all &= ~(1UL << pins[i]);  // enable pull-up resistors

            mux = private_getMuxValue(ports[i], pins[i]);
            if(pins[i] > 15)
            {
                tmpReg = ports[i]->GPAMUX2.all;
                tmpReg &= ~(0x03UL << (2 * (pins[i] - 16)));
                tmpReg |= mux << (2 * (pins[i] - 16));
                ports[i]->GPAMUX2.all = tmpReg;                             // GPIO mux to I2C use
                ports[i]->GPAQSEL2.all |= 0x03UL << (2 * (pins[i] - 16));   // GPIO pin to async mode
            }
            else
            {
                tmpReg = ports[i]->GPAMUX1.all;
                tmpReg &= ~(0x03 << (2 * pins[i]));
                tmpReg |= mux << (2 * pins[i]);
                ports[i]->GPAMUX1.all = tmpReg;                             // GPIO mux to I2C use
                ports[i]->GPAQSEL1.all |= 0x03UL << (2 * pins[i]);          // GPIO pin to async mode
            }
        }
    }
    EDIS;

    return IO_OK;
}

/**
 * @brief   IO control function for the I2C
 * @param   pDev    Pointer to the I2C device handler
 * @param   request Command for desired operation
 * @param   arg     Optional argument of the command
 * @return  Operation status code
 */
int i2c_ioctl(i2c_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    EALLOW;
    switch(request)
    {
        case I2C_IOC_SET_SLAVE_ADDR:
            retVal = private_setSlaveAddress(pDev, (uint16_t)va_arg(arg, unsigned int));
            break;

        case I2C_IOC_SET_ADDR_MODE:
            private_setAddressMode(pDev, (i2c_addr_mode_t)va_arg(arg, int));
            break;

        case I2C_IOC_SET_SPEED:
            retVal = private_setSpeed(pDev, (uint32_t)va_arg(arg, unsigned long));
            break;

        case I2C_IOC_MSGRX:
            retVal = private_msgRx(pDev, va_arg(arg, i2c_msg_t*));
            break;

        case I2C_IOC_MSGTX:
            retVal = private_msgTx(pDev, va_arg(arg, i2c_msg_t*));
            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}

/**
 * @brief   Read data received by I2C.
 * @param   pDev    Pointer to I2C device handler.
 * @param   pBuffer Pointer to output buffer where to store data.
 * @param   bytes   Number of bytes to read.
 * @return  Number of bytes actually read.
 */
int i2c_read(i2c_dev_t* pDev, void* pBuffer, int bytes)
{
    assert(pDev != NULL);

    int status = IO_OK;
    int bytesRead = 0;
    char* pReceiveBuffer = (char*)pBuffer;

    if(pDev->pCtrlReg->I2CSTR.bit.BB == 1)
    {
        return IO_ERR_ACCES;    // I2C bus is occupied
    }

    // Set receive mode
    pDev->pCtrlReg->I2CMDR.bit.TRX = 0;

    if(pDev->pSelfSlaveAddress == NULL)
    {
        // Configure as master
        pDev->pCtrlReg->I2CMDR.bit.MST = 1;

        // Set data counter
        pDev->pCtrlReg->I2CCNT = (uint16_t)bytes;
        pDev->pCtrlReg->I2CMDR.bit.STP = 1;     // generates Stop condition when data counter is zero
        pDev->pCtrlReg->I2CMDR.bit.STT = 1;     // generates Start condition

        // Read data
        while(bytesRead < bytes)
        {
            if(pDev->pCtrlReg->I2CSTR.bit.RRDY == 1)    // wait until data received
            {
                *pReceiveBuffer = pDev->pCtrlReg->I2CDRR & 0x00FF;
                pReceiveBuffer++;
                bytesRead++;

                if((bytesRead + 1U) == bytes)
                {
                    // Prepare NACK generation
                    pDev->pCtrlReg->I2CMDR.bit.NACKMOD = 1;
                }
                else if(bytesRead == bytes)
                {
                    if(pDev->pCtrlReg->I2CSTR.bit.NACKSNT == 1)
                    {
                        // Clear NACK generated flag
                        pDev->pCtrlReg->I2CSTR.bit.NACKSNT = 1;
                    }
                    else
                    {
                        // Did not generate NACK to slave
                        status = IO_ERROR;
                        break;
                    }
                }
                else
                {
                    if(pDev->pCtrlReg->I2CSTR.bit.NACK == 1)
                    {
                        // Did not get ACK from slave
                        status = IO_ERROR;
                        break;
                    }
                }
            }
            else
            {
                // TODO - timeout
            }
        }

        // Error occurs during transfer
        if(status != IO_OK)
        {
            // Reset I2C
            pDev->pCtrlReg->I2CMDR.bit.IRS = 0;
            _DELAY(1UL);
            pDev->pCtrlReg->I2CMDR.bit.IRS = 1;

            return status;
        }
        else
        {
            // Wait until Stop condition was generated
            while(pDev->pCtrlReg->I2CMDR.bit.STP == 1);
        }
    }
    else
    {
        // TODO - slave device
    }

    return bytesRead;
}

/**
 * @brief   Send data over I2C.
 * @param   pDev    Pointer to I2C device handler.
 * @param   pBuffer Pointer to data to send.
 * @param   bytes   Number of bytes to send.
 * @return  Number of bytes actually sent.
 */
int i2c_write(i2c_dev_t* pDev, const void* pBuffer, int bytes)
{
    assert(pDev != NULL);

    int status = IO_OK;
    int bytesWrote = 0;
    char* pTransmitBuffer = (char*)pBuffer;

    if(pDev->pCtrlReg->I2CSTR.bit.BB == 1)
    {
        return IO_ERR_ACCES;    // I2C bus is occupied
    }

    // Set transfer mode
    pDev->pCtrlReg->I2CMDR.bit.TRX = 1;

    if(pDev->pSelfSlaveAddress == NULL)
    {
        // Configure as master
        pDev->pCtrlReg->I2CMDR.bit.MST = 1;

        // Configure data counter
        pDev->pCtrlReg->I2CCNT = (uint16_t)bytes;
        pDev->pCtrlReg->I2CMDR.bit.STP = 1;     // generates Stop condition when data counter is zero
        pDev->pCtrlReg->I2CMDR.bit.STT = 1;     // generates Start condition and start data transfer

        // Send data
        while(bytesWrote < bytes)
        {
            if(pDev->pCtrlReg->I2CSTR.bit.XRDY == 1)    // wait until all data shifts
            {
                pDev->pCtrlReg->I2CDXR = (*pTransmitBuffer) & 0x00FF;
                pTransmitBuffer++;
                bytesWrote++;

                if(pDev->pCtrlReg->I2CSTR.bit.NACK == 1)
                {
                    // Did not get ACK from slave
                    status = IO_ERROR;
                    break;
                }
            }
            else
            {
                // TODO - timeout
            }
        }

        // Error occurs during transfer
        if(status != IO_OK)
        {
            // Reset I2C
            pDev->pCtrlReg->I2CMDR.bit.IRS = 0;
            _DELAY(1UL);
            pDev->pCtrlReg->I2CMDR.bit.IRS = 1;

            return status;
        }
        else
        {
            // Wait until Stop condition was generated
            while(pDev->pCtrlReg->I2CMDR.bit.STP == 1);
        }
    }
    else
    {
        // TODO - slave device
    }

    return bytesWrote;
}



static uint32_t private_getSYSCLKValue(void)
{
    uint32_t clk = 10000000;    // Internal OSC1/OSC2 frequency (default)

    if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 1 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 0)
    {
        clk = XCLKIN;   // predefined symbol for External OSC frequency
    }
    else if(SysCtrlRegs.CLKCTL.bit.XCLKINOFF == 0 && SysCtrlRegs.CLKCTL.bit.XTALOSCOFF == 1)
    {
        clk = XTAL;     // predefined symbol for Crystal or Resonator frequency
    }

    // compute PLL output (SYSCLKOUT)
    if(SysCtrlRegs.PLLCR.bit.DIV != 0)
    {
        if(SysCtrlRegs.PLLSTS.bit.DIVSEL <= 1)
        {
            clk = clk >> 2;  // (clk / 4)
        }
        else if(SysCtrlRegs.PLLSTS.bit.DIVSEL == 2)
        {
            clk = clk >> 1;  // (clk / 2)
        }
        clk = clk * SysCtrlRegs.PLLCR.bit.DIV;
    }

    return clk;
}

static uint32_t private_getMuxValue(pI2cPortCtrlReg_t pPort, const uint16_t pin)
{
    if((uint32_t)pPort == (uint32_t)I2C_SDA_PORT_REG_CAST(GpioCtrlRegs.GPACTRL))
    {
        if((pin == 28U) || (pin == 29U))
        {
            return 2UL;
        }
    }
    else if((uint32_t)pPort == (uint32_t)I2C_SDA_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL))
    {
        if((pin == 0U) || (pin == 1U))
        {
            return 1UL;
        }
    }
    return 0UL;
}

static int private_setSlaveAddress(i2c_dev_t* pDev, const uint16_t slaveAddress)
{
    if(pDev->pSelfSlaveAddress == NULL)
    {
        pDev->pCtrlReg->I2CMDR.bit.IRS = 0;     // disable I2C during configuration

        if(pDev->pCtrlReg->I2CMDR.bit.XA == 1)
        {
            pDev->pCtrlReg->I2CSAR = slaveAddress & 0x03FF; // 10-bit address
        }
        else
        {
            pDev->pCtrlReg->I2CSAR = slaveAddress & 0x007F; // 7-bit address
        }

        pDev->pCtrlReg->I2CMDR.bit.IRS = 1;     // enable I2C

        return IO_OK;
    }
    else
    {
        return IO_ERROR;
    }
}

static void private_setAddressMode(i2c_dev_t* pDev, const i2c_addr_mode_t addressMode)
{
    pDev->pCtrlReg->I2CMDR.bit.IRS = 0;     // disable I2C during configuration

    pDev->pCtrlReg->I2CMDR.bit.XA = (addressMode == I2C_ADDR_MODE_7_BIT) ? 0U : 1U;

    if(pDev->pSelfSlaveAddress != NULL)
    {
        if(pDev->pCtrlReg->I2CMDR.bit.XA == 1)
        {
            pDev->pCtrlReg->I2COAR = (*pDev->pSelfSlaveAddress) & 0x03FF; // 10-bit address
        }
        else
        {
            pDev->pCtrlReg->I2COAR = (*pDev->pSelfSlaveAddress) & 0x007F; // 7-bit address
        }
    }

    pDev->pCtrlReg->I2CMDR.bit.IRS = 1;     // enable I2C
}

static int private_setSpeed(i2c_dev_t* pDev, const uint32_t speed)
{
    if(pDev->pSelfSlaveAddress == NULL)
    {
        uint32_t clk = private_getSYSCLKValue();

        uint32_t prescaler = 0;
        while((clk / (prescaler + 1UL)) < 7000000UL || (clk / (prescaler + 1UL)) > 12000000UL)
        {
            prescaler++;
            if(prescaler == 256UL)
            {
                return IO_ERROR;
            }
        }

        uint16_t delay = 7U;
        delay -= (prescaler == 1) ? 1U : 0U;
        delay -= (prescaler > 1) ? 2U : 0U;

        float iccx = ((float)clk / ((speed << 1) * (prescaler + 1UL))) - (float)delay;
        if((iccx - (uint32_t)iccx) > 0.5f)
        {
            iccx++;
        }
        if(iccx > 65535.0f)
        {
            return IO_ERROR;
        }

        pDev->pCtrlReg->I2CMDR.bit.IRS = 0;     // disable I2C during configuration

        pDev->pCtrlReg->I2CPSC.all = (uint16_t)prescaler;
        pDev->pCtrlReg->I2CCLKH = (uint16_t)iccx;
        pDev->pCtrlReg->I2CCLKL = (uint16_t)iccx;

        pDev->pCtrlReg->I2CMDR.bit.IRS = 1;     // enable I2C

        return IO_OK;
    }
    else
    {
        return IO_ERROR;
    }
}

static int private_msgRx(i2c_dev_t* pDev, i2c_msg_t* const pMsg)
{
    // Send command part
    if((pMsg->pCmd != NULL) && (pMsg->cmdLen > 0))
    {
        if(i2c_write(pDev, pMsg->pCmd, pMsg->cmdLen) != pMsg->cmdLen)
        {
            return IO_ERROR;
        }
    }
    else
    {
        return IO_ERR_ARG;
    }

    // Receive data part
    if((pMsg->pData != NULL) && (pMsg->dataLen > 0))
    {
        if(i2c_read(pDev, pMsg->pData, pMsg->dataLen) != pMsg->dataLen)
        {
            return IO_ERROR;
        }
    }

    return IO_OK;
}

static int private_msgTx(i2c_dev_t* pDev, i2c_msg_t* const pMsg)
{
    int bytesTransmitted;
    char* pTxBuffer;

    // Set transfer mode
    pDev->pCtrlReg->I2CMDR.bit.TRX = 1;

    // Configure data counter
    pDev->pCtrlReg->I2CCNT = (uint16_t)(pMsg->cmdLen + pMsg->cmdLen);
    pDev->pCtrlReg->I2CMDR.bit.STP = 1;     // generates Stop condition when data counter is zero
    pDev->pCtrlReg->I2CMDR.bit.STT = 1;     // generates Start condition and start data transfer

    // Send command part
    bytesTransmitted = 0;
    pTxBuffer = pMsg->pCmd;
    if(pTxBuffer != NULL)
    {
        while(bytesTransmitted < pMsg->cmdLen)
        {
            if(pDev->pCtrlReg->I2CSTR.bit.XRDY == 1)    // wait until all data shifts
            {
                pDev->pCtrlReg->I2CDXR = (*pTxBuffer) & 0x00FF;
                bytesTransmitted++;
                pTxBuffer++;

                if(pDev->pCtrlReg->I2CSTR.bit.NACK == 1)
                {
                    // Did not get ACK from slave
                    return IO_ERROR;
                }
            }
        }
    }
    else
    {
        return IO_ERR_ARG;
    }

    // Send data part
    bytesTransmitted = 0;
    pTxBuffer = pMsg->pData;
    if(pTxBuffer != NULL)
    {
        while(bytesTransmitted < pMsg->dataLen)
        {
            if(pDev->pCtrlReg->I2CSTR.bit.XRDY == 1)    // wait until all data shifts
            {
                pDev->pCtrlReg->I2CDXR = (*pTxBuffer) & 0x00FF;
                bytesTransmitted++;
                pTxBuffer++;

                if(pDev->pCtrlReg->I2CSTR.bit.NACK == 1)
                {
                    // Did not get ACK from slave
                    return IO_ERROR;
                }
            }
        }
    }

    return IO_OK;
}

#endif /* USE_I2C_DRIVER && defined(TMS320F28069) */
