/**
 * @file    gpio_tms320f28069.c
 * @author  Matej Turinsky
 * @date    23. 2. 2024
 * @brief   GPIO driver for the TMS320F28069.
 */

#include "conf_driver.h"

#if USE_GPIO_DRIVER && defined(TMS320F28069)

#include <stdint.h>
#include <stddef.h>
#include <assert.h>
#include <stdarg.h>

#include "../../../../common/statuscodes.h"
#include "../../gpio.h"
#include "gpio_tms320f28069.h"


/**
 * @brief   Initialization of the GPIO
 * @param   pDev    Pointer to the GPIO device handler
 * @return  Operation status code
 */
int gpio_init(gpio_dev_t* pDev)
{
    assert(pDev != NULL);

    EALLOW;
    pDev->pCtrlReg->GPAPUD.all |= pDev->mask;    // disable pull-ups
    EDIS;

    return IO_OK;
}

/**
 * @brief   IO control function for the GPIO
 * @param   pDev    Pointer to the GPIO device handler
 * @param   request Command for desired operation
 * @param   arg     Optional argument of the command
 * @return  Operation status code
 */
int gpio_ioctl(gpio_dev_t* pDev, int request, va_list arg)
{
    assert(pDev != NULL);
    int retVal = IO_OK;

    EALLOW;
    switch(request)
    {
        case GPIO_IOC_SET_OUTHI:
            pDev->pDataReg->GPASET.all = (uint32_t)va_arg(arg, unsigned long) & pDev->mask;
            break;

        case GPIO_IOC_SET_OUTLO:
            pDev->pDataReg->GPACLEAR.all = (uint32_t)va_arg(arg, unsigned long) & pDev->mask;
            break;

        case GPIO_IOC_SET_OUTTGL:
            pDev->pDataReg->GPATOGGLE.all = (uint32_t)va_arg(arg, unsigned long) & pDev->mask;
            break;

        case GPIO_IOC_GET_DIR:
            *va_arg(arg, uint32_t*) = pDev->pCtrlReg->GPADIR.all & pDev-> mask;
            break;

        case GPIO_IOC_SET_DIR:
            pDev->pCtrlReg->GPADIR.all = (pDev->pCtrlReg->GPADIR.all & ~pDev->mask) | ((uint32_t)va_arg(arg, unsigned long) & pDev->mask);
            break;

        case GPIO_IOC_GET_PULL:
            *va_arg(arg, uint32_t*) = pDev->pCtrlReg->GPAPUD.all & pDev-> mask;
            break;

        case GPIO_IOC_SET_PULL:
            pDev->pCtrlReg->GPAPUD.all = (pDev->pCtrlReg->GPAPUD.all & ~pDev->mask) | (~(uint32_t)va_arg(arg, unsigned long) & pDev->mask);
            break;

        default:
            retVal = IO_ERR_REQ;
            break;
    }
    EDIS;

    return retVal;
}

/**
 * @brief   GPIO read function
 * @param   pDev    Pointer to the GPIO device handler
 * @param   pBuffer Pointer to the output variable
 * @param   bytes   number of bytes to read
 * @return  Number of bytes actually read
 */
int gpio_read(gpio_dev_t* pDev, void* pBuffer, int bytes)
{
    assert(pDev != NULL);
    int bytesRead;

    switch(bytes)
    {
        case 0:
            bytesRead = 0;
            break;

        case 1:
            *(uint16_t*)pBuffer = (uint16_t)(pDev->pDataReg->GPADAT.all & pDev->mask & 0x000000FF);
            bytesRead = 1;
            break;

        case 2:
            *(uint16_t*)pBuffer = (uint16_t)(pDev->pDataReg->GPADAT.all & pDev->mask & 0x0000FFFF);
            bytesRead = 2;
            break;

        case 3:
            *(uint32_t*)pBuffer = (uint32_t)(pDev->pDataReg->GPADAT.all & pDev->mask & 0x00FFFFFF);
            bytesRead = 3;
            break;

        default:
            *(uint32_t*)pBuffer = (uint32_t)(pDev->pDataReg->GPADAT.all & pDev->mask);
            bytesRead = 4;
            break;
    }

    return bytesRead;
}

/**
 * @brief   GPIO write function
 * @param   pDev    Pointer to the GPIO device handler
 * @param   pBuffer Pointer to the input variable
 * @param   bytes   number of bytes to write
 * @return  Number of bytes actually written
 */
int gpio_write(gpio_dev_t* pDev, void* pBuffer, int bytes)
{
    assert(pDev != NULL);
    int bytesWrite;
    uint32_t tmpVal = 0;

    switch(bytes)
    {
        case 0:
            bytesWrite = 0;
            break;

        case 1:
            tmpVal |= (uint32_t)(*(uint16_t*)pBuffer) & 0x000000FF;
            bytesWrite = 1;
            break;

        case 2:
            tmpVal |= (uint32_t)(*(uint16_t*)pBuffer) & 0x0000FFFF;
            bytesWrite = 2;
            break;

        case 3:
            tmpVal |= *(uint32_t*)pBuffer & 0x00FFFFFF;
            bytesWrite = 3;
            break;

        default:
            tmpVal |= *(uint32_t*)pBuffer;
            bytesWrite = 4;
            break;
    }

    EALLOW;
    pDev->pDataReg->GPADAT.all = (pDev->pDataReg->GPADAT.all & ~pDev->mask) | (tmpVal & pDev->mask);
    EDIS;

    return bytesWrite;
}


#endif /* USE_GPIO_DRIVER && defined(TMS320F28069) */
