/**
 * @file    conf_driver.h
 * @author  Matej Turinsky
 * @date    23.02.2024
 * @brief   Header configuration file for TMS320F28069 drivers
 */

#pragma once

#ifndef CONF_DRIVER_H_
#define CONF_DRIVER_H_

#include <stdint.h>

#include "cla_shared.h"


#define NASSERT 1   ///< use _nassert() instead of _assert() to lower .cio footprint

/** MT-SD11 mt-drivers configuration */
#define USE_GPIO_DRIVER     	1
#define USE_WD_DRIVER       	1
#define USE_UART_DRIVER     	1
#define USE_ADC_CLA_DRIVER      1
#define USE_TIM_DRIVER          1
#define USE_I2C_DRIVER          1

#define AVAILABLE_DEVICES                                                      	\
    DEVICE("/per/gpioA",    gpio,       0),     /* GPIO port A		        */  \
	DEVICE("/per/gpioB",    gpio,       1),     /* GPIO port B		        */  \
	DEVICE("/per/wdCPU",	wd,         0),     /* Independent watchdog 	*/  \
	DEVICE("/per/sciA",     uart,       0),     /* UART to serial port      */  \
    DEVICE("/per/sciB",     uart,       1),     /* UART to G3-PLC modem     */  \
    DEVICE("/per/adcVA",    adc_cla,    0),     /* ADC - voltage & current  */  \
    DEVICE("/per/adcIrr",   adc_cla,    1),     /* ADC - solar irradiance   */  \
    DEVICE("/per/epwm1",    tim,        0),     /* adcVA trigger timer      */  \
    DEVICE("/per/epwm2",    tim,        1),     /* adcIrr trigger timer     */  \
    DEVICE("/per/epwm3",    tim,        2),     /* Millisecond tick timer   */  \
    DEVICE("/per/i2cA",     i2c,        0)      /* I2C to sensors           */

/**
 * @note    GPIO instances
 */
#define AVAILABLE_GPIO_DEVICES                                  \
 /* GPIO_INSTANCE(_pCtrlReg, _pDataReg, _mask) */               \
    GPIO_INSTANCE(GPIO_CTRL_REG_CAST(GpioCtrlRegs.GPACTRL),     \
                  GPIO_DATA_REG_CAST(GpioDataRegs.GPADAT),      \
                  0xCFFFFFFF),                                  \
    GPIO_INSTANCE(GPIO_CTRL_REG_CAST(GpioCtrlRegs.GPBCTRL),     \
                  GPIO_DATA_REG_CAST(GpioDataRegs.GPBDAT),      \
                  0x07FC1CF0)

/**
 * @note    Watchdog instances
 */
#define AVAILABLE_WD_DEVICES    \
 /* WD_INSTANCE(_prescaler) */  \
    WD_INSTANCE(64U)

/**
 * @note    UART instances
 */
#define SCIA_BUFFER_SIZE    256
    extern char sciA_rxBuffer[SCIA_BUFFER_SIZE];
#define SCIB_BUFFER_SIZE    1024
    extern char sciB_rxBuffer[SCIB_BUFFER_SIZE];
#define AVAILABLE_UART_DEVICES                                                                      \
 /* UART_INSTANCE(_pUartReg_t, _pRxPort, _rxPin, _pTxPort, _txPin, _pRxBuffer, _rxBufferSize) */    \
    UART_INSTANCE(UART_CTRL_REG_CAST(SciaRegs),                                                     \
                  UART_RX_PORT_REG_CAST(GpioCtrlRegs.GPACTRL),                                      \
                  28U,                                                                              \
                  UART_TX_PORT_REG_CAST(GpioCtrlRegs.GPACTRL),                                      \
                  29U,                                                                              \
                  sciA_rxBuffer,                                                                    \
                  SCIA_BUFFER_SIZE),                                                                \
    UART_INSTANCE(UART_CTRL_REG_CAST(ScibRegs),                                                     \
                  UART_RX_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL),                                      \
                  9U,                                                                               \
                  UART_TX_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL),                                      \
                  8U,                                                                               \
                  sciB_rxBuffer,                                                                    \
                  SCIB_BUFFER_SIZE)

/**
 * @note    ADC instances
 */
#define AVAILABLE_ADC_CLA_DEVICES                                                                                           \
 /* ADC_CLA_INSTANCE(_ctrlReg, _dataReg, _claReg, _claChannel, _claTask, _claResult, _claResultSize, _timPath, _trigSrc) */ \
    ADC_CLA_INSTANCE(ADC_CLA_CTRL_REG_CAST(AdcRegs),                                                                        \
                     ADC_CLA_DATA_REG_CAST(AdcResult),                                                                      \
                     ADC_CLA_CORE_REG_CAST(Cla1Regs),                                                                       \
                     1U,                                                                                                    \
                     ADC_CLA_TASK_FUNC_CAST(adcCla_VAProcessTask),                                                          \
                     ADC_CLA_TASK_RESULT_CAST(adcCla_VAResult),                                                             \
                     sizeof(adcCla_VAResult),                                                                               \
                     "/per/epwm1",                                                                                          \
                     ADC_CLA_TRIG_TIM_EPWM_1),                                                                              \
     ADC_CLA_INSTANCE(ADC_CLA_CTRL_REG_CAST(AdcRegs),                                                                       \
                      ADC_CLA_DATA_REG_CAST(AdcResult),                                                                     \
                      ADC_CLA_CORE_REG_CAST(Cla1Regs),                                                                      \
                      2U,                                                                                                   \
                      ADC_CLA_TASK_FUNC_CAST(adcCla_irrProcessTask),                                                        \
                      ADC_CLA_TASK_RESULT_CAST(adcCla_irrResult),                                                           \
                      sizeof(adcCla_irrResult),                                                                             \
                      "/per/epwm2",                                                                                         \
                      ADC_CLA_TRIG_TIM_EPWM_2)

/**
 * @note    TIM instances
 */
#define AVAILABLE_TIM_DEVICES                       \
 /* TIM_INSTANCE(_ctrlReg) */                       \
    TIM_INSTANCE(TIM_CTRL_REG_CAST(EPwm1Regs)),     \
    TIM_INSTANCE(TIM_CTRL_REG_CAST(EPwm2Regs)),     \
    TIM_INSTANCE(TIM_CTRL_REG_CAST(EPwm3Regs))

/**
 * @note    I2C instances
 */
#define AVAILABLE_I2C_DEVICES                                                           \
 /* I2C_INSTANCE(_ctrlReg, _selfSlaveAddress, _sdaPort, _sdaPin, _sclPort, _sclPin) */  \
    I2C_INSTANCE(I2C_CTRL_REG_CAST(I2caRegs),                                           \
                 NULL,                                                                  \
                 I2C_SDA_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL),                           \
                 0U,                                                                    \
                 I2C_SDA_PORT_REG_CAST(GpioCtrlRegs.GPBCTRL),                           \
                 1U)

#endif /* CONF_DRIVER_H_ */
