/**
 * @file    adc_cla_tms320f28069.h
 * @author  Matej Turinsky
 * @date    5. 3. 2024
 * @brief   Header file for the TMS320F28069 ADC driver (with CLA Task-1 usage).
 */

#pragma once

#ifndef ADC_CLA_TMS320F28069_H_
#define ADC_CLA_TMS320F28069_H_

#include <stdint.h>
#include <stdbool.h>
#include "F2806x_Device.h"
#include "F2806x_Cla_defines.h"
#include "devio.h"

/**
 * ADC instance definition macro for driver configuration
 */
#define ADC_CLA_INSTANCE(_ctrlReg, _dataReg, _claReg, _claChannel, _claTask, _claResult, _claResultSize, _timPath, _trigSrc)    \
        {                                                                                                                       \
            .pCtrlReg = _ctrlReg,                                                                                               \
            .pDataReg = _dataReg,                                                                                               \
            .pClaReg = _claReg,                                                                                                 \
            .claChannel = _claChannel,                                                                                          \
            .pClaTask = _claTask,                                                                                               \
            .pClaTaskResult = _claResult,                                                                                       \
            .claTaskResultSize = _claResultSize,                                                                                \
            .timPath = _timPath,                                                                                                \
            .trigSrc = _trigSrc                                                                                                 \
        }

/**
 * ADC structure for the TMS320F28069.
 */
typedef struct ADC_REGS* const pAdcCtrlReg_t;
typedef struct ADC_RESULT_REGS* const pAdcDataReg_t;
typedef struct CLA_REGS* const pClaCtrlReg_t;
typedef __interrupt void (*const pClaTaskFunc_t)(void);

typedef struct {
    pAdcCtrlReg_t pCtrlReg;
    pAdcDataReg_t pDataReg;
    pClaCtrlReg_t pClaReg;
    const uint16_t claChannel;
    pClaTaskFunc_t pClaTask;
    void* const pClaTaskResult;
    const size_t claTaskResultSize;

    const char* const timPath;
    const int trigSrc;
    devio_t timDev;

    io_callback_t convCallback;
    io_callback_t errorCallback;
    volatile uint16_t flags;
    volatile bool claTaskDone;
} adc_cla_dev_t;

/**
 * ADC register to pointer casting macros (use these in 'conf_driver.h')
 */
#define ADC_CLA_CTRL_REG_CAST(reg)      (pAdcCtrlReg_t)(void*)(&reg)
#define ADC_CLA_DATA_REG_CAST(reg)      (pAdcDataReg_t)(void*)(&reg)
#define ADC_CLA_CORE_REG_CAST(reg)      (pClaCtrlReg_t)(void*)(&reg)
#define ADC_CLA_TASK_FUNC_CAST(func)    (pClaTaskFunc_t)(&func)
#define ADC_CLA_TASK_RESULT_CAST(val)   (void*)(&val)

/**
 * ADC trigger sources
 */
#define ADC_CLA_TRIG_SW_ONLY        (0x00)
#define ADC_CLA_TRIG_TIM_EPWM_1     (0x05)
#define ADC_CLA_TRIG_TIM_EPWM_2     (0x07)
#define ADC_CLA_TRIG_TIM_EPWM_3     (0x09)
#define ADC_CLA_TRIG_TIM_EPWM_4     (0x0B)
#define ADC_CLA_TRIG_TIM_EPWM_5     (0x0D)
#define ADC_CLA_TRIG_TIM_EPWM_6     (0x0F)
#define ADC_CLA_TRIG_TIM_EPWM_7     (0x11)
#define ADC_CLA_TRIG_TIM_EPWM_8     (0x13)

/**
 * CLA linker addresses
 */
extern uint16_t Cla1funcsLoadStart;
extern uint16_t Cla1funcsLoadEnd;
extern uint16_t Cla1funcsLoadSize;
extern uint16_t Cla1funcsRunStart;
extern uint16_t Cla1Prog_Start;

/**
 * CLA math tables addresses
 */
#if !DISABLE_CLA_MATH_TABLES
    extern uint16_t Cla1mathTablesLoadStart;
    extern uint16_t Cla1mathTablesLoadEnd;
    extern uint16_t Cla1mathTablesLoadSize;
    extern uint16_t Cla1mathTablesRunStart;
#endif


/**
 * CLA default task function declarations - tasks from 'adc_cla_tms320f28069_default_tasks.cla'
 */
__interrupt void AdcCla1Task1(void);
__interrupt void AdcCla1Task2(void);
__interrupt void AdcCla1Task3(void);
__interrupt void AdcCla1Task4(void);
__interrupt void AdcCla1Task5(void);
__interrupt void AdcCla1Task6(void);
__interrupt void AdcCla1Task7(void);
__interrupt void AdcCla1Task8(void);


#endif /* ADC_CLA_TMS320F28069_H_ */
