/**
 * @file    transfer.c
 * @author  Roman Mego, Matej Turinsky
 * @date    2. 2. 2024
 * @brief   Data transfer in transparent mode implementation.
 */

#include <string.h>

#include <lwip/err.h>
#include <lwip/udp.h>

#include "apps/transfer.h"

#define TRANSPARENT_PORT    23

struct transfer_state {
    struct udp_pcb *pcb;
    transfer_receive_call_t receive_call;
    void* receive_arg;
};

typedef struct transfer_state transfer_state_t;

static transfer_state_t state;

err_t transfer_send(const ip_addr_t* addr, const void* data, size_t bytes)
{
    err_t result;
    struct pbuf *p;

    p = pbuf_alloc(PBUF_TRANSPORT, bytes, PBUF_RAM);
    if (p == NULL)
    {
        return ERR_MEM;
    }

    pbuf_take(p, data, bytes);
    result = udp_sendto(state.pcb, p, addr, TRANSPARENT_PORT);
    pbuf_free(p);

    return result;
}

static void transfer_receive(void* arg, struct udp_pcb* pcb, struct pbuf* p, const ip_addr_t* addr, uint16_t port)
{
    state.receive_call(arg, addr, p->payload, p->len);
    pbuf_free(p);
}

err_t transfer_start(void)
{
    err_t result;

    result = udp_bind(state.pcb, IP_ANY_TYPE, TRANSPARENT_PORT);
    if (result == ERR_OK)
    {
        udp_recv(state.pcb, transfer_receive, state.receive_arg);
    }

    return result;
}

err_t transfer_stop(void)
{
    udp_disconnect(state.pcb);
    return ERR_OK;
}

err_t transfer_init(transfer_receive_call_t receive, void* arg)
{
    struct udp_pcb *pcb = udp_new_ip_type(IPADDR_TYPE_V6);
    if (pcb == NULL)
    {
        return ERR_MEM;
    }

    state.pcb = pcb;
    state.receive_call = receive;
    state.receive_arg = arg;

    return ERR_OK;
}
