/**
 * @file    plc.c
 * @author  Roman Mego, Matej Turinsky
 * @date    22. 2. 2024
 * @brief   MT-PLC-SEMx PLC core functions.
 */

#include <string.h>

#include <devio.h>
#include <FreeRTOS.h>
#include <event_groups.h>
#include <queue.h>

#include <lwip/tcpip.h>
#include <lwip/ip_addr.h>
#include <lwip/ip6.h>
#include <lwip/ip6_addr.h>

#include <fs.h>
#include <led.h>
#include <modemtec/twdog.h>
#include <sm2400/loader.h>

#include "modemtec/plc-core/plc.h"
#include "modemtec/plc-core/pool.h"
#include "modemtec/plc-core/if.h"
#include "modemtec/plc-core/mem.h"
#include "modemtec/plc-core/util.h"
#include "modemtec/plc-core/if/g3asl.h"
#include "modemtec/plc-core/if/g3phy.h"

#define PLC_DEVICE_PREFIX       "/dev/plc0"

static plc_core_t core;

static int plc_loader_path(char* path, size_t size)
{
    size_t slen;

    assert(path != NULL);

    strncpy(path, PLC_DEVICE_PREFIX, size);
    slen = strlen(path);
    strncpy(&path[slen], "-loader", size - slen);

    return 0;
}

static int plc_device_path(plc_mode_t mode, char* path, size_t size)
{
    int result;
    size_t slen;

    assert(path != NULL);

    strncpy(path, PLC_DEVICE_PREFIX, size);
    slen = strlen(path);

    switch (mode)
    {
        case PLC_MODE_G3_PHY:
            strncpy(&path[slen], "-phy", size - slen);
            result = 0;
            break;
        case PLC_MODE_G3_ASL:
            strncpy(&path[slen], "-asl", size - slen);
            result = 0;
            break;
        default:
            result = -1;
            break;
    }

    return result;
}

static void plc_rx_callback(const void* data, size_t bytes)
{
#if defined(PLC_NUM_RX_CALLBACK) && (PLC_NUM_RX_CALLBACK > 0)
    int n;

    for (n = 0; n < PLC_NUM_RX_CALLBACK; n++)
    {
        if (core.rxCallback[n] != NULL)
        {
            core.rxCallback[n](data, bytes);
        }
    }
#endif
}

err_t plc_if_ip6_input(struct pbuf* p, struct netif* netif)
{
    plc_rx_callback(p->payload, p->len);
    return ip6_input(p, netif);
}

err_t plc_if_link_input(struct pbuf* p, struct netif* netif)
{
    plc_rx_callback(p->payload, p->len);
    pbuf_free(p);
    return ERR_OK;
}

static int plc_connect(plc_panid_t panId, plc_role_t role)
{
    int result;

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->connect == NULL)
    {
        result = -2;
    }
    else if (core.call->connect(&core, panId, role) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

static int plc_disconnect(void)
{
    int result;

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->disconnect == NULL)
    {
        result = -2;
    }
    else if (core.call->disconnect(&core) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

static void plc_set_state(plc_state_t state)
{
    switch (state)
    {
        case PLC_STATE_DISCONNECTED:
            led_set_blink(LED_GREEN, LED_OFF, 1000, 1000);
            break;
        case PLC_STATE_CONNECTING:
            led_set_blink(LED_GREEN, LED_OFF, 200, 200);
            break;
        case PLC_STATE_CONNECTED:
            led_set_state(LED_GREEN);
            break;
        case PLC_STATE_DISCONNECTING:
            led_set_blink(LED_YELLOW, LED_OFF, 200, 200);
            break;
        default:
            led_set_blink(LED_RED, LED_OFF, 500, 500);
            break;
    }

    xQueueOverwrite(core.xQueueState, &state);
}

static void plc_connection_management(void* pvParameters)
{
    plc_connection_param_t param;

    while (true)
    {
        plc_mode_t mode;
        plc_state_t state;

        if (twdog_reset(TWDOG_TASK_ID_PLC_CONN_MNG) != 0)
        {
            vTaskDelay(pdMS_TO_TICKS(100));
            continue;
        }

        if ((plc_get_mode(&mode) != 0) ||
            (mode == PLC_MODE_NONE))
        {
            vTaskDelay(pdMS_TO_TICKS(100));
            continue;
        }

        if (plc_get_state(&state) != 0)
        {
            vTaskDelay(pdMS_TO_TICKS(100));
            continue;
        }

        switch (state)
        {
            case PLC_STATE_DISCONNECTED:
                if (xEventGroupWaitBits(core.xEventGroup, PLC_NETIF_CONNECT_REQUEST_FLAG, pdTRUE, pdTRUE, pdMS_TO_TICKS(100)) != PLC_NETIF_CONNECT_REQUEST_FLAG)
                {
                    continue;
                }

                if (xQueueReceive(core.xQueueConnectionParam, &param, 0) != pdTRUE)
                {
                    continue;
                }

                plc_set_state(PLC_STATE_CONNECTING);
                break;

            case PLC_STATE_CONNECTING:
                if (plc_connect(param.panId, param.role) != 0)
                {
                    plc_reset();
                    plc_set_state(PLC_STATE_DISCONNECTED);
                }
                else
                {
                    plc_set_state(PLC_STATE_CONNECTED);
                }

                break;

            case PLC_STATE_CONNECTED:
                if (xEventGroupWaitBits(core.xEventGroup, PLC_NETIF_DISCONNECT_REQUEST_FLAG, pdTRUE, pdTRUE, pdMS_TO_TICKS(100)) != PLC_NETIF_DISCONNECT_REQUEST_FLAG)
                {
                    continue;
                }

                plc_set_state(PLC_STATE_DISCONNECTING);
                break;

            case PLC_STATE_DISCONNECTING:
                plc_disconnect();
                plc_reset();
                plc_set_state(PLC_STATE_DISCONNECTED);
                break;

            default:
                break;
        }
    }
}

int plc_init(void)
{
    int result;
    const plc_mode_t initMode = PLC_MODE_NONE;
    const plc_state_t initState = PLC_STATE_NONE;

#if defined(PLC_NUM_RX_CALLBACK) && (PLC_NUM_RX_CALLBACK > 0)
    memset(core.rxCallback, 0, sizeof(core.rxCallback));
#endif

    /* Initialize memory pools */
    plc_pool_init();

    /* Initialize LED */
    led_init();
    led_set_state(LED_YELLOW);

#if configSUPPORT_STATIC_ALLOCATION
    if ((core.xEventGroup = xEventGroupCreateStatic(&core.xEventGroupBuffer)) == NULL)
#else
    if ((core.xEventGroup = xEventGroupCreate()) == NULL)
#endif
    {
        result = -1;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((core.xQueueMode = xQueueCreateStatic(1, sizeof(core.modeStorage), &core.modeStorage, &core.xQueueModeBuffer)) == NULL)
#else
    else if ((core.xQueueMode = xQueueCreate(1, sizeof(plc_mode_t))) == NULL)
#endif
    {
        result = -2;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((core.xQueueState = xQueueCreateStatic(1, sizeof(core.stateStorage), &core.stateStorage, &core.xQueueStateBuffer)) == NULL)
#else
    else if ((core.xQueueState = xQueueCreate(1, sizeof(plc_state_t))) == NULL)
#endif
    {
        result = -3;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((core.xQueueConnectionParam = xQueueCreateStatic(1, sizeof(core.connectionParamStorage), (void*)(&core.connectionParamStorage), &core.xQueueConnectionParamBuffer)) == NULL)
#else
    else if ((core.xQueueConnectionParam = xQueueCreate(1, sizeof(plc_connection_param_t))) == NULL)
#endif
    {
        result = -4;
    }
    else if (twdog_register(TWDOG_TASK_ID_PLC_CONN_MNG) != 0)
    {
        result = -5;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((core.xTaskConnectionManagement = xTaskCreateStatic(plc_connection_management, "PLC Conn. Mng.", PLC_CONNECTION_MANAGEMENT_STACK_SIZE, NULL, PLC_CONNECTION_MANAGEMENT_PRIORITY, core.xTaskConnectionManagementStack, &core.xTaskConnectionManagenentBuffer)) == NULL)
#else
    else if ((xTaskCreate(plc_connection_management, "PLC Conn. Mng.", PLC_CONNECTION_MANAGEMENT_STACK_SIZE, NULL, PLC_CONNECTION_MANAGEMENT_PRIORITY, &core.xTaskConnectionManagement)) != pdPASS)
#endif
    {
        result = -6;
    }
    else if (xQueueOverwrite(core.xQueueMode, &initMode) != pdPASS)
    {
        result = -7;
    }
    else if (xQueueOverwrite(core.xQueueState, &initState) != pdPASS)
    {
        result = -8;
    }
    else if (twdog_register(TWDOG_TASK_ID_LWIP) != 0)
    {
        result = -9;
    }
    else
    {
        result = 0;
    }

    tcpip_init(NULL, NULL);

    return result;
}

int plc_reset(void)
{
    plc_mode_t mode;

    if (plc_get_mode(&mode) != 0)
    {
        mode = PLC_MODE_DEFAULT;
    }

    return plc_set_mode(mode);
}

int plc_get_version(char* version, size_t bytes)
{
    int result;
    int len;

    assert(version != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->get_version == NULL)
    {
        result = -2;
    }
    else if ((len = core.call->get_version(&core, version, bytes)) < 0)
    {
        result = -3;
    }
    else
    {
        result = len;
    }

    return result;
}

int plc_get_mode(plc_mode_t* mode)
{
    int result;

    assert(mode != NULL);

    if (xQueuePeek(core.xQueueMode, mode, 0) != pdTRUE)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_set_mode(plc_mode_t mode)
{
    int retVal;
    plc_state_t state;
    char imagePath[32];
    char loaderPath[32];
    char plcPath[32];

    led_set_blink(LED_YELLOW, LED_OFF, 500, 500);

    if ((core.call != NULL) && (core.call->deinit != NULL))
    {
        core.call->deinit(&core);
    }

    if (core.sm2400 != NULL)
    {
        io_close(core.sm2400);
    }

    if (mode == PLC_MODE_NONE)
    {
        retVal = -1;
    }
    else if (plc_mem_image_path(mode, imagePath, sizeof(imagePath)) != 0)
    {
        retVal = -2;
    }
    else if (plc_loader_path(loaderPath, sizeof(loaderPath)) != 0)
    {
        retVal = -3;
    }
    else if (plc_device_path(mode, plcPath, sizeof(plcPath)) != 0)
    {
        retVal = -4;
    }
    /* Load PLC firmware */
    else if (sm2400_load_file(loaderPath, imagePath) != 0)
    {
        mode = PLC_MODE_NONE;
        retVal = -5;
    }
    /* Open PLC device */
    else if ((core.sm2400 = io_open(plcPath)) == NULL)
    {
        mode = PLC_MODE_NONE;
        retVal = -6;
    }
    else
    {
        switch (mode)
        {
            case PLC_MODE_G3_ASL:
            {
                core.call = plc_g3asl_call();
                break;
            }
            case PLC_MODE_G3_PHY:
            {
                core.call = plc_g3phy_call();
                break;
            }
            default:
                core.call = NULL;
                break;
        }

        if (core.call == NULL)
        {
            retVal = -7;
        }
        else if (core.call->init == NULL)
        {
            retVal = -8;
        }
        else if (core.call->init(&core) != 0)
        {
            retVal = -9;
        }
        else
        {
            retVal = 0;
        }
    }

    if (xQueueOverwrite(core.xQueueMode, &mode) != pdTRUE)
    {
        retVal = -10;
    }

    if (retVal == 0)
    {
        plc_set_state(PLC_STATE_DISCONNECTED);
    }
    else
    {
        plc_set_state(PLC_STATE_NONE);
    }

    return retVal;
}

int plc_get_role(plc_role_t* role)
{
    int result;

    assert(role != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->get_role == NULL)
    {
        result = -2;
    }
    else if (core.call->get_role(&core, role) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_connect_request(plc_panid_t panId, plc_role_t role)
{
    int result;
    plc_state_t state;
    plc_connection_param_t param = {
        .panId = panId,
        .role = role
    };

    if (plc_get_state(&state) != 0)
    {
        result = -1;
    }
    else if (state != PLC_STATE_DISCONNECTED)
    {
        result = -2;
    }
    else if (xQueueOverwrite(core.xQueueConnectionParam, &param) != pdTRUE)
    {
        result = -3;
    }
    else if ((xEventGroupSetBits(core.xEventGroup, PLC_NETIF_CONNECT_REQUEST_FLAG) & PLC_NETIF_CONNECT_REQUEST_FLAG) != PLC_NETIF_CONNECT_REQUEST_FLAG)
    {
        result = -4;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_disconnect_request(void)
{
    int result;
    plc_state_t state;

    if (plc_get_state(&state) != 0)
    {
        result = -1;
    }
    else if (state != PLC_STATE_CONNECTED)
    {
        result = -2;
    }
    else if ((xEventGroupSetBits(core.xEventGroup, PLC_NETIF_DISCONNECT_REQUEST_FLAG) & PLC_NETIF_DISCONNECT_REQUEST_FLAG) != PLC_NETIF_DISCONNECT_REQUEST_FLAG)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_set_rx_callback(plc_rx_callback_t callback)
{
    int result;

#if defined(PLC_NUM_RX_CALLBACK) && (PLC_NUM_RX_CALLBACK > 0)
    int n;

    for (n = 0; n < PLC_NUM_RX_CALLBACK; n++)
    {
        if (core.rxCallback[n] == NULL)
        {
            core.rxCallback[n] = callback;
            result = 0;
            break;
        }

        result = -1;
    }
#else
    result = -1;
#endif

    return result;
}

int plc_remove_rx_callback(plc_rx_callback_t callback)
{
    int result;

#if defined(PLC_NUM_RX_CALLBACK) && (PLC_NUM_RX_CALLBACK > 0)
    int n;

    for (n = 0; n < PLC_NUM_RX_CALLBACK; n++)
    {
        if (core.rxCallback[n] == callback)
        {
            core.rxCallback[n] = NULL;
            result = 0;
            break;
        }

        result = -1;
    }
#else
    result = -1;
#endif

    return result;
}

int plc_get_state(plc_state_t* state)
{
    int result;

    assert(state != NULL);

    if (xQueuePeek(core.xQueueState, state, 0) != pdTRUE)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_get_address(ip_addr_t* address)
{
    assert(address != NULL);
    ip_addr_copy_from_ip6(*address, core.netif.ip6_addr[0]);
    return 0;
}

int plc_get_coord_address(ip_addr_t* address)
{
    assert(address != NULL);
    ip_addr_copy(*address, core.coordAddr);
    return 0;
}

int plc_get_neighbor(plc_index_t index, ip_addr_t* address)
{
    int result;
    plc_addr_t short_address;
    ip_addr_t local;

    assert(address != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->get_neighbor(&core, index, &short_address) != 0)
    {
        result = -2;
    }
    else if (plc_get_address(&local) != 0)
    {
        result = -3;
    }
    else
    {
        plc_panid_t panid;

        panid = PLC_IP6_PANID(&local);
        PLC_IP6(address, panid, short_address);

        result = 0;
    }

    return result;
}

int plc_get_asl_param(plc_aslid_t id, plc_index_t index, void* data, size_t size)
{
    int result;
    int len;

    assert(data != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->get_asl_param == NULL)
    {
        result = -2;
    }
    else if ((len = core.call->get_asl_param(&core, id, index, data, size)) < 0)
    {
        result = -3;
    }
    else
    {
        result = len;
    }

    return result;
}

int plc_set_asl_param(plc_aslid_t id, plc_index_t index, const void* data, size_t size)
{
    int result;

    assert(data != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->set_asl_param == NULL)
    {
        result = -2;
    }
    else if (core.call->set_asl_param(&core, id, index, data, size) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_get_mac_param(plc_macid_t id, plc_index_t index, void* data, size_t size)
{
    int result;
    int len;

    assert(data != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->get_mac_param == NULL)
    {
        result = -2;
    }
    else if ((len = core.call->get_mac_param(&core, id, index, data, size)) < 0)
    {
        result = -3;
    }
    else
    {
        result = len;
    }

    return result;
}

int plc_set_mac_param(plc_macid_t id, plc_index_t index, const void* data, size_t size)
{
    int result;

    assert(data != NULL);

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->set_mac_param == NULL)
    {
        result = -2;
    }
    else if (core.call->set_mac_param(&core, id, index, data, size) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}

int plc_set_phy(int minrssi, plc_band_t band)
{
    int result;

    if (core.call == NULL)
    {
        result = -1;
    }
    else if (core.call->set_phy == NULL)
    {
        result = -2;
    }
    else if (core.call->set_phy(&core, minrssi, band) != 0)
    {
        result = -3;
    }
    else
    {
        result = 0;
    }

    return result;
}
