/**
 * @brief   conn-keep.c
 * @author  Roman Mego
 * @date    20. 2. 2024
 * @brief   Network connection keeper.
 */

#include <string.h>

#include <FreeRTOS.h>
#include <event_groups.h>
#include <timers.h>

#include <mbedtls/entropy.h>
#include <mbedtls/ctr_drbg.h>

#include <lwip/prot/ip6.h>

#include <modemtec/twdog.h>
#include <modemtec/plc-core/plc.h>

#include "conn-keep.h"
#include "apps/icmp-echo.h"

#define CONNKEEP_RX_FLAG                    (1 << 0)

#define CONNKEEP_IDLE_WAIT_COORD_MS         (5 * 1000)
#define CONNKEEP_IDLE_WAIT_DEVICE_MIN_MS    (30 * 1000)
#define CONNKEEP_IDLE_WAIT_DEVICE_MAX_MS    (90 * 1000)
#define CONNKEEP_CONNECTION_TIMEOUT_MS      (60 * 1000)

#define CONNKEEP_IDLE_WAIT_COORD_TICK       (CONNKEEP_IDLE_WAIT_COORD_MS / portTICK_PERIOD_MS)
#define CONNKEEP_IDLE_WAIT_DEVICE_MIN_TICK  (CONNKEEP_IDLE_WAIT_DEVICE_MIN_MS / portTICK_PERIOD_MS)
#define CONNKEEP_IDLE_WAIT_DEVICE_MAX_TICK  (CONNKEEP_IDLE_WAIT_DEVICE_MAX_MS / portTICK_PERIOD_MS)
#define CONNKEEP_CONNECTION_TIMEOUT_TICK    (CONNKEEP_CONNECTION_TIMEOUT_MS / portTICK_PERIOD_MS)

#define CONNKEEP_IDLE_WAIT_DEVICE_TICK      ((connkeep_random() % (CONNKEEP_IDLE_WAIT_DEVICE_MAX_TICK - CONNKEEP_IDLE_WAIT_DEVICE_MIN_TICK)) + CONNKEEP_IDLE_WAIT_DEVICE_MIN_TICK)

static mbedtls_ctr_drbg_context ctr_drbg;
static mbedtls_entropy_context entropy;

static TimerHandle_t xTimer;
#if configSUPPORT_STATIC_ALLOCATION
    static StaticTimer_t xTimerBuffer;
#endif

static EventGroupHandle_t xEventGroup;
#if configSUPPORT_STATIC_ALLOCATION
    static StaticEventGroup_t xEventGroupBuffer;
#endif

typedef enum {
    CONNKEEP_STATE_DISABLED,
    CONNKEEP_STATE_DISCONNECTED,
    CONNKEEP_STATE_CONNECTING,
    CONNKEEP_STATE_CONNECTED,
    CONNKEEP_STATE_CHECKING,
    CONNKEEP_STATE_DISCONNECTING,
} connkeep_state_t;

struct connkeep_handler
{
    plc_panid_t panid;
    plc_role_t role;
    TickType_t check_time;
    connkeep_state_t state;
    TickType_t wait;
    TickType_t last;
    size_t check_index;
};

typedef struct connkeep_handler connkeep_handler_t;

static connkeep_handler_t connkeep;

static int connkeep_random(void)
{
    int random;
    mbedtls_ctr_drbg_random(&ctr_drbg, (unsigned char*)(&random), sizeof(random));
    return random;
}

static void connkeep_rx_callback(const void* data, size_t bytes)
{
    ip_addr_t coord;
    ip_addr_t local;
    ip_addr_t remote;
    const struct ip6_hdr* hdr = data;

    ip_addr_copy_from_ip6_packed(remote, hdr->src);

    if ((plc_get_address(&local) == 0) &&
        (plc_get_coord_address(&coord) == 0))
    {
        if (ip_addr_cmp_zoneless(&local, &coord))
        {
            /* Coordinator */
            xEventGroupSetBits(xEventGroup, CONNKEEP_RX_FLAG);
        }
        else
        {
            /* Device */
            if (ip_addr_cmp_zoneless(&coord, &remote))
            {
                xEventGroupSetBits(xEventGroup, CONNKEEP_RX_FLAG);
            }
        }
    }
}

static void vTimerCallback(TimerHandle_t xTimer)
{
    plc_state_t plc_state;
    TickType_t now = xTaskGetTickCount();
    TickType_t diff = now - connkeep.last;

    if (plc_get_state(&plc_state) == 0)
    {
        switch (connkeep.state)
        {
            case CONNKEEP_STATE_DISCONNECTED:
                if (diff > connkeep.wait)
                {
                    plc_connect_request(connkeep.panid, connkeep.role);
                    connkeep.state = CONNKEEP_STATE_CONNECTING;
                    connkeep.last = now;
                }
                break;

            case CONNKEEP_STATE_CONNECTING:
                if (plc_state == PLC_STATE_CONNECTED)
                {
                    connkeep.state = CONNKEEP_STATE_CONNECTED;
                    connkeep.last = now;
                }
                else
                {
                    if (diff > CONNKEEP_CONNECTION_TIMEOUT_TICK)
                    {
                        plc_reset();
                        connkeep.state = CONNKEEP_STATE_DISCONNECTED;
                        connkeep.last = now;
                    }
                }
                break;

            case CONNKEEP_STATE_CONNECTED:
                if (plc_state != PLC_STATE_CONNECTED)
                {
                    plc_disconnect_request();
                    connkeep.state = CONNKEEP_STATE_DISCONNECTING;
                    connkeep.last = now;
                }
                else
                {
                    if ((xEventGroupWaitBits(xEventGroup, CONNKEEP_RX_FLAG, pdTRUE, pdTRUE, 0) & CONNKEEP_RX_FLAG) == CONNKEEP_RX_FLAG)
                    {
                        connkeep.last = now;
                    }
                    else
                    {
                        if (diff > connkeep.check_time)
                        {
                            connkeep.check_index = 0;
                            connkeep.state = CONNKEEP_STATE_CHECKING;
                            connkeep.last = now;
                        }
                    }
                }
                break;

            case CONNKEEP_STATE_CHECKING:
                if ((xEventGroupWaitBits(xEventGroup, CONNKEEP_RX_FLAG, pdTRUE, pdTRUE, 0) & CONNKEEP_RX_FLAG) == CONNKEEP_RX_FLAG)
                {
                    connkeep.state = CONNKEEP_STATE_CONNECTED;
                    connkeep.last = now;
                }
                else
                {
                    if (connkeep.role == PLC_ROLE_DEVICE)
                    {
                        if (connkeep.check_index == 0)
                        {
                            ip_addr_t remote;
                            plc_get_coord_address(&remote);
                            icmp_echo_request(&remote);
                            connkeep.check_index++;
                            connkeep.last = now;
                        }
                        else
                        {
                            if (diff > CONNKEEP_CONNECTION_TIMEOUT_TICK)
                            {
                                plc_disconnect_request();
                                connkeep.state = CONNKEEP_STATE_DISCONNECTING;
                                connkeep.last = now;
                            }
                        }
                    }
                    else
                    {
                        if (diff > CONNKEEP_CONNECTION_TIMEOUT_TICK)
                        {
                            ip_addr_t remote;

                            if (plc_get_neighbor(connkeep.check_index, &remote) == 0)
                            {
                                icmp_echo_request(&remote);
                                connkeep.check_index++;
                                connkeep.last = now;
                            }
                            else
                            {
                                plc_disconnect_request();
                                connkeep.state = CONNKEEP_STATE_DISCONNECTING;
                                connkeep.last = now;
                            }
                        }
                    }
                }
                break;

            case CONNKEEP_STATE_DISCONNECTING:
                if (plc_state == PLC_STATE_DISCONNECTED)
                {
                    connkeep.state = CONNKEEP_STATE_DISCONNECTED;
                    connkeep.last = now;
                }
                else
                {
                    if (diff > CONNKEEP_CONNECTION_TIMEOUT_TICK)
                    {
                        plc_reset();
                        connkeep.state = CONNKEEP_STATE_DISCONNECTED;
                        connkeep.last = now;
                    }
                }
                break;

            default:
                connkeep.state = (plc_state == PLC_STATE_CONNECTED) ? CONNKEEP_STATE_CONNECTED : CONNKEEP_STATE_DISCONNECTED;
                connkeep.last = now;
                break;
        }

        twdog_reset(TWDOG_TASK_ID_CONN_KEEP);
    }
}

int connkeep_init(void)
{
    int result;

    mbedtls_ctr_drbg_init(&ctr_drbg);
    mbedtls_entropy_init(&entropy);

    if (mbedtls_ctr_drbg_seed(&ctr_drbg, mbedtls_entropy_func, &entropy, NULL, 0) != 0)
    {
        result = -1;
    }
    else if (icmp_echo_init() != ERR_OK)
    {
        result = -2;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((xTimer = xTimerCreateStatic("Conn. keep", pdMS_TO_TICKS(1000), pdTRUE, NULL, vTimerCallback, &xTimerBuffer)) == NULL)
#else
    else if ((xTimer = xTimerCreate("Conn. keep", pdMS_TO_TICKS(1000), pdTRUE, NULL, vTimerCallback)) == NULL)
#endif
    {
        result = -3;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((xEventGroup = xEventGroupCreateStatic(&xEventGroupBuffer)) == NULL)
#else
    else if ((xEventGroup = xEventGroupCreate()) == NULL)
#endif
    {
        result = -4;
    }
    else
    {
        connkeep.state = CONNKEEP_STATE_DISABLED;
        result = 0;
    }

    return result;
}

int connkeep_start(const connkeep_settings_t* settings)
{
    int result;

    connkeep.panid = settings->panid;
    connkeep.role = settings->role;
    connkeep.check_time = settings->check_time * 1000 / portTICK_PERIOD_MS;
    connkeep.wait = (settings->role == PLC_ROLE_COORDINATOR) ? CONNKEEP_IDLE_WAIT_COORD_MS : CONNKEEP_IDLE_WAIT_DEVICE_TICK;
    connkeep.last =  xTaskGetTickCount();

    if (settings->check_time == 0)
    {
        result = 0;
    }
    else if (twdog_register(TWDOG_TASK_ID_CONN_KEEP) != 0)
    {
        result = -1;
    }
    else if (xTimerStart(xTimer, 0) != pdPASS)
    {
        result = -2;
    }
    else if (plc_set_rx_callback(connkeep_rx_callback) != 0)
    {
        result = -3;
    }
    else
    {

        result = 0;
    }

    return result;
}

int connkeep_stop(void)
{
    int result;

    if (connkeep.check_time == 0)
    {
        result = 0;
    }
    else if (xTimerStop(xTimer, 0) != pdPASS)
    {
        result = -1;
    }
    else if (plc_remove_rx_callback(connkeep_rx_callback) != 0)
    {
        result = -2;
    }
    else if (twdog_deregister(TWDOG_TASK_ID_CONN_KEEP) != 0)
    {
        result = -3;
    }
    else
    {
        connkeep.state = CONNKEEP_STATE_DISABLED;
        result = 0;
    }

    return result;
}
