/**
 * @brief   comm.c
 * @author  Roman Mego, Matej Turinsky
 * @date    14. 2. 2024
 * @brief   Transparent mode.
 */

#include <assert.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>

#include <FreeRTOS.h>
#include <queue.h>
#include <task.h>

#include <endian.h>

#include <modemtec/plc-core/plc.h>
#include <modemtec/plc-core/util.h>

#include "mt49s/autoconf.h"
#include "mt49s/comm.h"
#include "mt49s/commands.h"
#include "mt49s/sniffer.h"

#include <modemtec/msp.h>

#include "apps/transfer.h"
#include "apps/remote-server.h"
#include "apps/remote-client.h"

#if !defined(COMM_TASK_PRIORITY)
#   define COMM_TASK_PRIORITY           (configMAX_PRIORITIES - 3)
#endif

#if !defined(COMM_MODE_SWITCH_TIMEOUT)
#   define COMM_MODE_SWITCH_TIMEOUT     pdMS_TO_TICKS(60000)
#endif

static void comm_task(void* arg);

static int addr_to_ip(const plc_addr_t* addr, ip_addr_t* ip)
{
    int result;
    ip_addr_t local;

    if (plc_get_address(&local) != 0)
    {
        result = -1;
    }
    else
    {
        plc_panid_t panid = PLC_IP6_PANID(&local);
        PLC_IP6(ip, panid, *addr);
        result = 0;
    }

    return result;
}

static void transfer_plc_receive(void* arg, const ip_addr_t* addr, const void* data, size_t bytes)
{
    comm_mode_t mode;
    comm_handler_t* handler = arg;

    assert(arg != NULL);
    assert(data != NULL);

    if (comm_get_mode(handler, &mode) == 0)
    {
        if (mode == COMM_MODE_COMMAND)
        {
            uint8_t buffer[1500];
            plc_addr_t address = PLC_IP6_SHORTADDR(addr);
            mt49s_data_transfer_t* header = (mt49s_data_transfer_t*)buffer;
            void* payload = &buffer[sizeof(*header)];

            memcpy(payload, data, bytes);
            header->Address = htole16(address);

            MSP_Send(&handler->msp, MT49S_DATA_TRANSFER, 0, buffer, bytes + sizeof(*header));
        }
        else
        {
            io_write(handler->msp.commDevice, data, bytes);
        }
    }
}

static int remote_server_receive(void* arg, uint16_t* command, uint16_t* status, void* data, uint16_t* bytes)
{
    int result;
    mt49s_command_t* table = mt49s_remote_commands_table();
    size_t count = mt49s_remote_commands_count();
    size_t n;

    for (n = 0; n < count; n++, table++)
    {
        if (table->cmd != *command)
        {
            result = -1;
            continue;
        }

        *command |= MT49S_RSP_MASK;
        *status = table->call(arg, data, bytes);

        result = 0;
        break;
    }

    return result;
}

static void remote_client_receive(void* arg, const ip_addr_t* addr, uint16_t command, uint16_t status, void* data, uint16_t bytes)
{
    comm_mode_t mode;
    comm_handler_t* handler = arg;

    if ((comm_get_mode(handler, &mode) == 0) &&
        (mode == COMM_MODE_COMMAND))
    {
        uint8_t buffer[1500];
        plc_addr_t address = PLC_IP6_SHORTADDR(addr);
        mt49s_remote_command_t* header = (mt49s_remote_command_t*)buffer;
        void* payload = &buffer[sizeof(header)];

        memcpy(payload, data, bytes);
        header->RemoteAddress = htole16(address);
        header->Command = htole16(command);

        MSP_Send(&handler->msp, MT49S_REMOTE_CMD, status, buffer, bytes + sizeof(*header));
    }
}

static void sniffer_plc_receive(void* arg, const void* data, size_t bytes)
{
    comm_mode_t mode;
    comm_handler_t* handler = arg;

    if ((comm_get_mode(handler, &mode) == 0) &&
        (mode == COMM_MODE_COMMAND))
    {
        MSP_Send(&handler->msp, MT49S_DATA_G3_PHY, 0, data, bytes);
    }
}

static int comm_break_isr(void* arg)
{
    int result;
    comm_handler_t* handler = arg;
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;

    assert(arg != NULL);

    if (xEventGroupSetBitsFromISR(handler->xFlags, COMM_FLAG_BREAK_ISR, &xHigherPriorityTaskWoken) != pdPASS)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_init(comm_handler_t* handler)
{
    int result;
    devio_t uart;
    io_callback_t isr = {
        .function = comm_break_isr,
        .argument = handler
    };

    assert(handler != NULL);

    if (transfer_init(transfer_plc_receive, handler) != ERR_OK)
    {
        result = -1;
    }
    else if (remote_server_init(remote_server_receive, handler) != ERR_OK)
    {
        result = -2;
    }
    else if (remote_client_init(remote_client_receive, handler) != ERR_OK)
    {
        result = -3;
    }
    else if (connkeep_init() != 0)
    {
        result = -4;
    }
    else if (sniffer_init(sniffer_plc_receive, handler) != 0)
    {
        result = -5;
    }
    else if ((uart = io_open("/dev/uart0")) == NULL)
    {
        result = -6;
    }
    else if (MSP_Init(&handler->msp, uart, mt49s_commands_table(), mt49s_commands_count(), handler) != MSP_OK)
    {
        result = -7;
    }
    else if ((handler->xQueuePeer = xQueueCreateStatic(1, sizeof(plc_addr_t), handler->xQueuePeerStorage, &handler->xQueuePeerBuffer)) == NULL)
    {
        result = -8;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((handler->xQueueMode = xQueueCreateStatic(1, sizeof(comm_mode_t), handler->xQueueModeStorage, &handler->xQueueModeBuffer)) == NULL)
#else
    else if ((handler->xQueueMode = xQueueCreate(1, sizeof(comm_mode_t))) == NULL)
#endif
    {
        result = -9;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((handler->xFlags = xEventGroupCreateStatic(&handler->xFlagsBuffer)) == NULL)
#else
    else if ((handler->xFlags = xEventGroupCreate()) == NULL)
#endif
    {
        result = -10;
    }
#if configSUPPORT_STATIC_ALLOCATION
    else if ((handler->xTask = xTaskCreateStatic(comm_task, "COMM", COMM_STACK_SIZE, handler, COMM_TASK_PRIORITY, handler->xTaskStack, &handler->xTaskBuffer)) == NULL)
#else
    else if (xTaskCreate(comm_task, "COMM", COMM_STACK_SIZE, handler, COMM_TASK_PRIORITY, &handler->xTask) != pdPASS)
#endif
    {
        result = -11;
    }
    else if (io_ioctl(uart, UART_IOC_REGISTER_BREAK, &isr) != IO_OK)
    {
        result = -12;
    }
    else if (comm_set_mode(handler, COMM_MODE_DEFAULT) != 0)
    {
        result = -13;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_start(comm_handler_t* handler)
{
    int result;
    EventBits_t flags = xEventGroupSetBits(handler->xFlags, COMM_FLAG_RUN);

    if ((flags = xEventGroupSetBits(handler->xFlags, COMM_FLAG_RUN)) & COMM_FLAG_RUN != COMM_FLAG_RUN)
    {
        result = -1;
    }
    else if (transfer_start() != 0)
    {
        xEventGroupClearBits(handler->xFlags, COMM_FLAG_RUN);
        result = -2;
    }
    else if (remote_server_start() != 0)
    {
        xEventGroupClearBits(handler->xFlags, COMM_FLAG_RUN);
        result = -2;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_stop(comm_handler_t* handler)
{
    int result;

    if (transfer_stop() != 0)
    {
        result = -1;
    }
    else if (remote_server_stop() != 0)
    {
        result = -2;
    }
    else
    {
        xEventGroupClearBits(handler->xFlags, COMM_FLAG_RUN);
        result = 0;
    }

    return result;
}

int comm_get_mode(comm_handler_t* handler, comm_mode_t* mode)
{
    int result;

    assert(handler != NULL);
    assert(mode != NULL);

    if (xQueuePeek(handler->xQueueMode, mode, 0) != pdTRUE)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_set_mode(comm_handler_t* handler, comm_mode_t mode)
{
    int result;

    assert(handler != NULL);

    if ((mode != COMM_MODE_COMMAND) && (mode != COMM_MODE_TRANSPARENT))
    {
        result = -1;
    }
    else if ((mode == COMM_MODE_COMMAND) && (connkeep_stop() != 0))
    {
        result = -2;
    }
    else if ((mode == COMM_MODE_TRANSPARENT) && (connkeep_start(&handler->keep) != 0))
    {
        result = -3;
    }
    else if (xQueueOverwrite(handler->xQueueMode, &mode) != pdPASS)
    {
        result = -4;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_get_peer(comm_handler_t* handler, plc_addr_t* addr)
{
    int result;

    assert(handler != NULL);
    assert(addr != NULL);

    if (xQueuePeek(handler->xQueuePeer, addr, 0) != pdTRUE)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_set_peer(comm_handler_t* handler, plc_addr_t addr)
{
    int result;

    assert(handler != NULL);

    if (xQueueOverwrite(handler->xQueuePeer, &addr) != pdPASS)
    {
        result = -1;
    }
    else
    {
        result = 0;
    }

    return result;
}

int comm_get_peer_ip(comm_handler_t* handler, ip_addr_t* ip)
{
    int result;
    plc_addr_t addr;

    if (comm_get_peer(handler, &addr) != 0)
    {
        result = -1;
    }
    else if (addr_to_ip(&addr, ip) != 0)
    {
        result = -2;
    }
    else
    {
        result = 0;
    }

    return result;
}

static void comm_task(void* arg)
{
    comm_handler_t* handler = arg;
    bool modeSwitchActive = false;
    TickType_t modeSwitchTicks;
    msp_port_param_t portBackup;
    comm_mode_t modeBackup;

    assert(arg != NULL);

    while (true)
    {
        comm_mode_t mode;
        TickType_t nowTicks;

        while ((xEventGroupWaitBits(handler->xFlags, COMM_FLAG_RUN, pdFALSE, pdTRUE, portMAX_DELAY) & COMM_FLAG_RUN) == 0);

        nowTicks = xTaskGetTickCount();

        if ((xEventGroupGetBits(handler->xFlags) & COMM_FLAG_BREAK_ISR) == COMM_FLAG_BREAK_ISR)
        {
            xEventGroupClearBits(handler->xFlags, COMM_FLAG_BREAK_ISR);

            do
            {
                if (MSP_GetPortParam(&handler->msp, &portBackup) != true)
                {
                    break;
                }

                if (comm_get_mode(handler, &modeBackup) != 0)
                {
                    break;
                }

                if (MSP_SetPortDefault(&handler->msp) != true)
                {
                    break;
                }

                if (comm_set_mode(handler, COMM_MODE_COMMAND) != 0)
                {
                    modeSwitchActive = false;
                }
                else
                {
                    modeSwitchTicks = nowTicks;
                    modeSwitchActive = true;
                }
            } while (false);
        }

        if (modeSwitchActive && ((nowTicks - modeSwitchTicks) > COMM_MODE_SWITCH_TIMEOUT))
        {
            modeSwitchActive = false;
            MSP_SetPortParam(&handler->msp, &portBackup);
            comm_set_mode(handler, modeBackup);
        }

        if (comm_get_mode(handler, &mode) != 0)
        {
            vTaskDelay(100);
        }

        if (mode == COMM_MODE_COMMAND)
        {
            if (MSP_Handle(&handler->msp) == MSP_OK)
            {
                modeSwitchActive = false;
            }
        }
        else if (mode == COMM_MODE_TRANSPARENT)
        {
            int bytes;

            if ((bytes = io_read(handler->msp.commDevice, handler->msp.buff, sizeof(handler->msp.buff))) > 0)
            {
                ip_addr_t peer_ip;
                ip_addr_t local_ip;

                if ((plc_get_address(&local_ip) == 0) &&
                    (comm_get_peer_ip(handler, &peer_ip) == 0) &&
                    !ip_addr_cmp_zoneless(&local_ip, &peer_ip))
                {
                    transfer_send(&peer_ip, handler->msp.buff, bytes);
                }
            }
        }
        else
        {
            vTaskDelay(100);
        }
    }
}
