/**
 * @file    modemtec/plc-core/util.h
 * @author  Roman Mego
 * @date    23. 2. 2024
 * @brief   MT-PLC-SEMx PLC core utilities.
 */

#pragma once

#ifndef MODEMTEC_PLC_CORE_UTIL_H_
#define MODEMTEC_PLC_CORE_UTIL_H_

#if defined(__cplusplus)
extern "C" {
#endif

#include <lwip/ip_addr.h>

/**
 * @brief   Initialize IP address structure from PAN ID and short address.
 * @param   panid       PAN ID of the PLC network.
 * @param   shortaddr   Short address of the PLC device.
 * @return  IP address structure.
 */
#define PLC_IP6_INIT(panid, shortaddr)                                      \
        IPADDR6_INIT_HOST(                                                  \
            (((shortaddr) & 0x8000U) == 0) ? 0xFE800000UL : 0xFF020000UL,   \
            0UL,                                                            \
            ((panid) << 16) + 0x000000FFUL,                                 \
            0xFE000000UL + (shortaddr & 0xFFFFU)                            \
        )

/**
 * @brief   Sets IP address according to PAN ID and short address.
 * @param   ipaddr  Pointer to IP address structure to set.
 * @param   panid       PAN ID of the PLC network.
 * @param   shortaddr   Short address of the PLC device.
 */
#define PLC_IP6(ipaddr, panid, shortaddr)                                   \
        IP_ADDR6_HOST(                                                      \
            (ipaddr),                                                       \
            (((shortaddr) & 0x8000U) == 0) ? 0xFE800000UL : 0xFF020000UL,   \
            0UL,                                                            \
            ((panid) << 16) + 0x000000FFUL,                                 \
            0xFE000000UL + (shortaddr & 0xFFFFU)                            \
        )

/**
 * @brief   Gets PAN ID from IP address.
 * @param   ipaddr  Pointer to IP address structure.
 * @return  PLC network PAN ID.
 */
#if LWIP_IPV4 && LWIP_IPV6
#   define PLC_IP6_PANID(ipaddr)        ((PP_NTOHL((ipaddr)->u_addr.ip6.addr[2]) & 0xFFFF0000UL) >> 16)
#elif LWIP_IPV4
#   define PLC_IP6_PANID(ipaddr)        (0U)
#else
#   define PLC_IP6_PANID(ipaddr)        ((PP_NTOHL((ipaddr)->addr[2]) & 0xFFFF0000UL) >> 16)
#endif

/**
 * @brief   Gets short address from IP address.
 * @param   ipaddr  Pointer to IP address structure.
 * @return  Short address of the device.
 */
#if LWIP_IPV4 && LWIP_IPV6
#   define PLC_IP6_SHORTADDR(ipaddr)    (PP_NTOHL((ipaddr)->u_addr.ip6.addr[3]) & 0x0000FFFFUL)
#elif LWIP_IPV4
#   define PLC_IP6_SHORTADDR(ipaddr)    (0U)
#else
#   define PLC_IP6_SHORTADDR(ipaddr)    (PP_NTOHL((ipaddr)->addr[3]) & 0x0000FFFFUL)
#endif

#if defined(__cplusplus)
}
#endif

#endif /* MODEMTEC_PLC_CORE_UTIL_H_ */
