/**
 * @file    modemtec/plc-core/core.h
 * @author  Roman Mego
 * @date    23. 2. 2024
 * @brief   MT-PLC-SEMx PLC core handler definitions.
 */

#pragma once

#ifndef MODEMTEC_PLC_CORE_CORE_H_
#define MODEMTEC_PLC_CORE_CORE_H_

#ifndef MODEMTEC_PLC_CORE_PLC_H_
#   error "Do not include <modemtec/plc-core/core.h> in application."
#endif

#include <stddef.h>
#include <stdint.h>

#include <FreeRTOS.h>
#include <event_groups.h>
#include <task.h>
#include <queue.h>

#include <lwip/netif.h>
#include <devio.h>

#if !defined(PLC_NETIF_STACK_SIZE)
    /** Stack size for internal tasks of PLC network interface */
#   define PLC_NETIF_STACK_SIZE                 (configMINIMAL_STACK_SIZE)
#endif

#if !defined(PLC_NETIF_TASK_PRIORITY)
    /** Priority of internal tasks for PLC network interface */
#   define PLC_NETIF_TASK_PRIORITY              (configMAX_PRIORITIES - 2)
#endif

#if !defined(PLC_CONNECTION_MANAGEMENT_STACK_SIZE)
    /** Stack size for PLC connection management task */
#   define PLC_CONNECTION_MANAGEMENT_STACK_SIZE 2048
#endif

#if !defined(PLC_CONNECTION_MANAGEMENT_PRIORITY)
    /** Priority of PLC connection management task */
#   define PLC_CONNECTION_MANAGEMENT_PRIORITY   (configMAX_PRIORITIES - 2)
#endif

#define PLC_NETIF_CONNECT_REQUEST_FLAG          (1 << 0)    ///< Network interface connect request flag
#define PLC_NETIF_DISCONNECT_REQUEST_FLAG       (1 << 1)    ///< Network interface disconnect request flag
#define PLC_NETIF_REMOVE_FLAG                   (1 << 2)    ///< Remove network interface flag

/** Connection request parameters structure */
struct plc_connection_param
{
    uint16_t panId;     ///< PLC network PAN ID
    plc_role_t role;    ///< PLC device role
};

/** Type definition for connection request parameters */
typedef struct plc_connection_param plc_connection_param_t;

/** PLC core handler structure */
typedef struct plc_core
{
    devio_t sm2400;                     ///< PLC device handler
    struct netif netif;                 ///< IP stack network device
    const struct plc_call* call;        ///< Pointer to mode-dependent function calls

    ip_addr_t coordAddr;                ///< IP address of network coordinator

#if defined(PLC_NUM_RX_CALLBACK) && (PLC_NUM_RX_CALLBACK > 0)
    plc_rx_callback_t rxCallback[PLC_NUM_RX_CALLBACK];  ///< Data receive callback for application
#endif

    QueueHandle_t xQueueMode;           ///< Queue handler for storing PLC core mode
#if configSUPPORT_STATIC_ALLOCATION
    StaticQueue_t xQueueModeBuffer;     ///< Statically allocated queue structure for PLC core mode
    plc_mode_t modeStorage;             ///< Statically allocated storage memory for PLC core mode
#endif /* configSUPPORT_STATIC_ALLOCATION */

    QueueHandle_t xQueueState;          ///< Queue handler for storing PLC core state
#if configSUPPORT_STATIC_ALLOCATION
    StaticQueue_t xQueueStateBuffer;    ///< Statically allocated queue structure for PLC core state
    plc_state_t stateStorage;           ///< Statically allocated storage memory for PLC core state
#endif /* configSUPPORT_STATIC_ALLOCATION */

    QueueHandle_t xQueueConnectionParam;            ///< Queue handler for storing PLC core connection request parameter
#if configSUPPORT_STATIC_ALLOCATION
    StaticQueue_t xQueueConnectionParamBuffer;      ///< Statically allocated queue structure for PLC core connection request parameter
    plc_connection_param_t connectionParamStorage;  ///< Statically allocated storage memory for PLC core request parameter
#endif /* configSUPPORT_STATIC_ALLOCATION */

    TaskHandle_t xTaskConnectionManagement;         ///< Task handler for PLC core connection management
#if configSUPPORT_STATIC_ALLOCATION
    StaticTask_t xTaskConnectionManagenentBuffer;   ///< Statically allocated task structure for PLC core connection management
    StackType_t xTaskConnectionManagementStack[PLC_CONNECTION_MANAGEMENT_STACK_SIZE];   ///< Statically allocated heap for PLC core connection management
#endif /* configSUPPORT_STATIC_ALLOCATION */

    EventGroupHandle_t xEventGroup;                 ///< Event group handler for PLC core flags
#if configSUPPORT_STATIC_ALLOCATION
    StaticEventGroup_t xEventGroupBuffer;           ///< Statically allocated event group structure for PLC core flags
#endif

    TaskHandle_t xTaskNetif;                            ///< Task handler for PLC core network interface
#if configSUPPORT_STATIC_ALLOCATION
    StaticTask_t xTaskNetifBuffer;                      ///< Statically allocated task structure for PLC core network interface
    StackType_t xTaskNerifStack[PLC_NETIF_STACK_SIZE];  ///< Statically allocated heap for PLC core network interface
#endif
} plc_core_t;

#endif /* MODEMTEC_PLC_CORE_CORE_H_ */
