/**
 * @brief   comm.h
 * @author  Roman Mego, Matej Turinsky
 * @date    13. 2. 2024
 * @brief   Transparent mode definitions.
 */

#pragma once

#ifndef INCLUDE_COMM_H_
#define INCLUDE_COMM_H_

#if defined(__cplusplus)
extern "C" {
#endif

#include <FreeRTOS.h>
#include <event_groups.h>
#include <queue.h>
#include <task.h>

#include <modemtec/plc-core/plc.h>
#include <modemtec/msp.h>

#include "conn-keep.h"

#if !defined(COMM_STACK_SIZE)
#   define COMM_STACK_SIZE  512
#endif

typedef enum {
    COMM_MODE_COMMAND,
    COMM_MODE_TRANSPARENT
} comm_mode_t;

#define COMM_MODE_DEFAULT       COMM_MODE_COMMAND

#define COMM_FLAG_RUN           (1 << 0)
#define COMM_FLAG_BREAK_ISR     (1 << 1)

struct comm_handler
{
    msp_handler_t msp;
    connkeep_settings_t keep;

    QueueHandle_t xQueuePeer;
#if configSUPPORT_STATIC_ALLOCATION
    StaticQueue_t xQueuePeerBuffer;
    uint8_t xQueuePeerStorage[sizeof(plc_addr_t)];
#endif

    TaskHandle_t xTask;
#if configSUPPORT_STATIC_ALLOCATION
    StaticTask_t xTaskBuffer;
    StackType_t xTaskStack[COMM_STACK_SIZE];
#endif

    EventGroupHandle_t xFlags;
#if configSUPPORT_STATIC_ALLOCATION
    StaticEventGroup_t xFlagsBuffer;
#endif

    QueueHandle_t xQueueMode;
#if configSUPPORT_STATIC_ALLOCATION
    StaticQueue_t xQueueModeBuffer;
    uint8_t xQueueModeStorage[sizeof(comm_mode_t)];
#endif
};

typedef struct comm_handler comm_handler_t;

int comm_init(comm_handler_t* handler);

int comm_start(comm_handler_t* handler);

int comm_stop(comm_handler_t* handler);

int comm_get_mode(comm_handler_t* handler, comm_mode_t* mode);

int comm_set_mode(comm_handler_t* handler, comm_mode_t mode);

int comm_get_peer(comm_handler_t* handler, plc_addr_t* addr);

int comm_set_peer(comm_handler_t* handler, plc_addr_t addr);

#if defined(__cplusplus)
}
#endif

#endif /* INCLUDE_COMM_H_ */
