/* USER CODE BEGIN Header */
/**
  ******************************************************************************
  * @file           : main.c
  * @brief          : Main program body
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2023 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/******************	Nastaveni adresy jednotky, nazvu snimace a merenych jednotek ******************	*/
#define ADRESA_JEDNOTKY 2				// (0 - 255)
#define TYP_SNIMACE SHT30
#define VELICINA_1 TEMPERATURE_C
#define VELICINA_2 HUMIDITY_PER

// pro pridani noveho nazvu snimace a merenych jednotek musi byt tyto nazvy pridany do vyctu nazvu na radku 53 a 64
// (tyto vycty musi byt ve vsech jednotkach (master i slave) stejne, i se stejnym poradim




/* USER CODE END Header */
/* Includes ------------------------------------------------------------------*/
#include "main.h"				//include hlavickoveho souboru hlavniho programu

/* Private includes ----------------------------------------------------------*/
/* USER CODE BEGIN Includes */
#include "platform.h"			//include knihoven pro vyvojovou desku
#include "radio.h"				//include knihovny pro ovladani radioveho modulu
#include "stdio.h"
#include "stdbool.h"
#include "sys_app.h"
#include "stm32_systime.h"		//nevyuzito (generovano CubeMX automaticky)
#include "stm32_timer.h"		//include funkci pro spravu casu
#include "radio_driver.h"		//utility pro managenent casu
/* USER CODE END Includes */

/* Private typedef -----------------------------------------------------------*/
/* USER CODE BEGIN PTD */

/* Vycet typu merenych jednotek
   Lze pridat libovolny pocet merenych jednotek
   (tento vycet musi byt ve vsech jednotkach datalogeru shodny) */
typedef enum {
    TEMPERATURE_C,
	HUMIDITY_PER,
    PRESSURE_kPa,

    NumberOfUnitName // automaticky urci pocet zadanych jednotek
} UnitName_t;

/* Vycet typu snimace
   Lze pridat libovolny pocet nazvu snimacu
   (tento vycet musi byt ve vsech jednotkach datalogeru shodny) */
typedef enum {
    SHT30,

    NumberOfSensorName // automaticky urci pocet zadanych snimacu
} SensorName_t;

/* Moznosti nastaveni dosahu komunikace */
enum RangeSetting {
	Range750 = 5,
	Range600 = 4,
	Range500 = 3,
	Range300 = 2,
	Range200 = 1,
	Range100 = 0
};

/* Stavy stavoveho automatu */
typedef enum{
  STATE_WAIT,
  STATE_SEND,
  STATE_RXDONE,
  STATE_TXDONE,
  STATE_RXTIMEOUT,
  STATE_TXTIMEOUT,
  STATE_RXERROR,
} State_t;

/* Vycet typu zprav posilanych z master jednotky do slave jednotek */
/* (lze pridat dalsi typy zprav)*/
enum MsgType {NONE, TimeSynch, MeasureRequest, ReceiveSettings, Status_MeasRequest};		// typ zpravy

/* Struktura Slave jednotky */
typedef struct{
	uint8_t ID;					// adresa slave
	uint32_t MeasurePeriod;		// perioda mereni slavu [ms]
	uint8_t CurRange;			// aktualne nastaveny dosah komunikace, parametry modulace
	uint8_t NewRange;			// novy dosah komunikace, ktery ma byt nastaven (parametry modulace)
	uint8_t BatteryLvl;			// uroven nabiti baterie slave [%]
} SlaveInit_t;

/* Struktura systemoveho casu */
typedef struct{
	uint8_t Year;			// rok 2 cifry
	uint8_t	Month;
	uint8_t Day;
	uint8_t Hours;
	uint8_t Minutes;
	uint8_t Seconds;
	uint16_t Milliseconds;
} SystemTime_t;

/* Struktura pro ulozeni informaci o prijate zprave pres LoRa*/
typedef struct{
	uint16_t Size;				// delka prijate zpravy
	int16_t RSSI;				// sila signalu prijate zpravy [dBm]
	int16_t SNR;				// odstup signal sum [dB]
}RxMsgInfo_t;

/* USER CODE END PTD */

/* Private define ------------------------------------------------------------*/
/* USER CODE BEGIN PD */

/* USER CODE END PD */

/* Private macro -------------------------------------------------------------*/
/* USER CODE BEGIN PM */


/* USER CODE END PM */

/* Private variables ---------------------------------------------------------*/
ADC_HandleTypeDef hadc;			// struktura ADC
I2C_HandleTypeDef hi2c3;		// struktura I2C
RTC_HandleTypeDef hrtc;			// struktura RTC
SUBGHZ_HandleTypeDef hsubghz;	// struktura SUBGHZ pro radio

/* USER CODE BEGIN PV */
RadioEvents_t RadioEvents;		// struktura pro nastaveni Callback funkci pro radiovy modul

State_t State = STATE_WAIT;		// State - nasledujici stav stavoveho automatu (WAIT vychozi stav automatu)

SlaveInit_t Slave = {.MeasurePeriod = 1000, .ID = ADRESA_JEDNOTKY};		// inicializace nastaveni slave jednotky

RxMsgInfo_t RxMsgInfo = {0};		// informace o posleni prijate zprave pres LoRa radio
SystemTime_t SystemTime = {0};		// struktura systemoveho casu
SystemTime_t TimeStamp = {0};		// casova znacka mereni dat ze snimace (casova znacka 1. mereni)

float MeasuredValueBuff[60];		// buffer pro ukladani namerenych hodnot (60 hodnot - 30 mereni (mereno po 2 float hodnotach)
uint8_t MeasuredValCnt = 0;			// pocet namerenych hodnot v MeasuredValueBuff (ukazatel na volnou pozici v bufferu)


uint8_t RxBuffer[255];			// buffer pro ulozeni prijate zpravy pres LoRa radio
uint8_t TxBuffer[255];			// buffer pro odeslani zpravy pres LoRa radio
uint8_t TxSize = 0;				// delka zpravy pro odeslani v TxBuffer pres Lora radio (v bytech)

uint32_t RTC_CurTicks;			// casova znacka RTC (hodnota RTC tiku) pro ktere je platny systemovy cas ve strukture SystemTime

uint32_t RXstartTimeStamp = 0;	// casova znacka, kdy ma dojit k zapnutí prijimaciho rezimu LoRa Radia

RTC_AlarmTypeDef sAlarmB = {0};	// struktura RTC alarmu B pro casovani mereni ze snimace
RTC_TimeTypeDef sTime = {0};	// struktura pro nacteni hodnoty casovace RTC
RTC_DateTypeDef sDate = {0};	// struktura pro nacteni RTC datumu (nutno cist i datum z RTC)


/* USER CODE END PV */

/* Private function prototypes -----------------------------------------------*/
void SystemClock_Config(void);
static void MX_GPIO_Init(void);
static void MX_I2C3_Init(void);
static void MX_ADC_Init(void);
/* USER CODE BEGIN PFP */

/* Inicializace Callback funci Radioveho modulu */
static void Radio_Init(void);
/* Funkce pro vychozi nastaveni parametru radioveho modulu */
static void Radio_InitRxTxParams(void);
/* Funkce pro prenastaveni modulacnich parametru radioveho modulu podle nastaveni komuikace
   @param Range Nastaveni komunikace*/
static void Radio_SetRxTxParams(uint8_t Range);
/* Obsluha preruseni, vola se pri uspesnem odeslani zpravy radiovym modulem */
static void OnTxDone(void);
/* Obsluha preruseni, vola se pri uspesnem prijeti zpravy radiovym modulem
   @param payload Ukazatel na prijatou zpravu
   @param size Delka prijate zpravy v bytech
   @param rssi Rssi posledni prijate zpravy (sila prijateho singalu v dBm)
   @param LoraSnr_FskCfo SNR posledni prijate zpravy (SRN odstup signal sum) */
static void OnRxDone(uint8_t *payload, uint16_t size, int16_t rssi, int8_t LoraSnr_FskCfo);
/* Obsluha preruseni, vola se pokud se radiovemu modulu nepodarilo zpravu uspesne odeslat v nastavenem intervalu 10s */
static void OnTxTimeout(void);
/* Obsluha preruseni, vola se pokud v nastavenem intervalu prijimani zpravy nebyla prijata zadna zprava radiovym modeulem */
static void OnRxTimeout(void);
/* Obsluha preruseni, vola se pokud byla radiovym modulem prijata poskozena zprava (chyba v hlavicce, chybny CRC kod) */
static void OnRxError(void);

/* Funkce pro zjisteni poctu dni v mesici
   @param Month Cislo mesice
   @param Year Rok
   @return Pocet dni v mesici*/
static uint8_t DaysInMonth(uint8_t Month, uint8_t Year);
/* Funkce pro aktualizaci systemoveho casu SystemTime podle RTC  */
static void SysTimeUpdate(void);
/* Funkce pro pricteni casu ke strukture a jeho konverze (zarovnani)
   @param Time Ukazatel na casovou strukturu, ke ktere ma byt cas pricten
   @param AddTimeMS Cas v milisekundach, ktery ma byt pricten k casove strukture*/
static void AddTime(SystemTime_t *Time, uint32_t AddTimeMS);
/* Funkce pro mereni externim snimacem*/
static void Sensor_measure(void);
/* Funkce pro mereni stavu baterie slave jednotky*/
static void Battery_measure(void);
/* Funkce pro vypocet casu, kdy neni mozne odeslat dalsi zpravu
   @param Range Nastaveni dosahu (pro jake parametry modulace ma byt cas vypocten)
   @param size Delka odeslane zpravy v bytech
   @param per DutyCycle v procentech (vyuziva se 100 a 99, 100 pro vypocet pred odeslanim zpravy, 99 po odeslani zpravy)
   @return Cas v milisekundach, za ktery je mozne odeslat dalsi zpravu*/
uint32_t CalcDutyCycle(uint8_t Range, uint8_t size, uint8_t per);

/* USER CODE END PFP */

/* Private user code ---------------------------------------------------------*/
/* USER CODE BEGIN 0 */

/* USER CODE END 0 */

/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
  /* USER CODE BEGIN 1 */

  /* USER CODE END 1 */

  /* MCU Configuration--------------------------------------------------------*/

  /* Reset of all peripherals, Initializes the Flash interface and the Systick. */
  HAL_Init();

  /* USER CODE BEGIN Init */

  /* USER CODE END Init */

  /* Configure the system clock */
  SystemClock_Config();

  /* USER CODE BEGIN SysInit */

  /* USER CODE END SysInit */

  /* Initialize all configured peripherals */
  MX_GPIO_Init();
  MX_I2C3_Init();
  MX_ADC_Init();
  /* USER CODE BEGIN 2 */
  /* Ensure that MSI is wake-up system clock */
  __HAL_RCC_WAKEUPSTOP_CLK_CONFIG(RCC_STOP_WAKEUPCLOCK_MSI);

  UTIL_TIMER_Init();			// inicializace soft. timeru a RTC
  Radio_Init();					// nastaveni callback funkci pro preruseni od radioveho modulu
  Radio_InitRxTxParams();		// nastaveni vychozich parametru modulace radioveho modulu
  Radio.Standby();				// prepnuti radia do pohotovostniho rezimu

  __HAL_RTC_WRITEPROTECTION_DISABLE(&hrtc);
  hrtc.Instance->CALR = 0b000000000000000001000000111001010;		//nataveni kalibracniho registru RTC
  __HAL_RTC_WRITEPROTECTION_ENABLE(&hrtc);

  /* USER CODE END 2 */

  /* Infinite loop */
  /* USER CODE BEGIN WHILE */

  /* Hlavni smycka celeho programu jednotky (implementace stavoveho automatu) */
  while (1)
  {

	  switch(State)						// stavovy automat (vyber stavu)
	  {
	  case STATE_SEND:					// stav pro odeslani nastavene zpravy v TxBuffer (prechod do tohoto stavu ze stavu TXtimeout nebo RXdone)
		  Radio.Send(TxBuffer,TxSize);	// odeslani obsahu TxBuffer pres radiovy modul
		  State = STATE_WAIT;			// nastaveni nasledujiciho stavu na WAIT
		  break;

	  case STATE_TXDONE:				// stav pro uspani radioveho modulu, pripadne prenastaveni dosahu komunikace (tento stav je zavolan pri uspesnem odeslani zpravy pres radiovy modul)
			RXstartTimeStamp =  UTIL_TIMER_GetCurrentTime() + CalcDutyCycle(Slave.CurRange, TxSize, 98); 			// vypocet casu kdy ma byt radiovy modul prepnut do prijimaciho rezimu ( prepnuti az je mozne odeslat novou zpravu)
			 if (Slave.CurRange != Slave.NewRange)		// pokud byla posledni prijata zprava zprava nastaveni 									 //	aktualni cas + doba kdy neni mozne vysilat (neni mozne vysilat 99 nasobek doby prenosu zpravy, ale je nastaveno
			 {											// a aktualni nasteveni dosahu neni shodne s novym nastavenim dosahu			 		 // o 1 % mene aby stihlo nabehnout radio ze sleep modu)
				 Radio_SetRxTxParams(Slave.NewRange);	// je po odeslani potvrzeni prijmu nastaveni (se starym nastavenim dosahu) prenastaven dosah komunikace
				 Slave.CurRange = Slave.NewRange;		// ulozeni aktualniho pouzivaneho dosahu komunikace
			 }
			Radio.Sleep();				// prepnuti radioveho modulu do rezimu Sleep
			State = STATE_WAIT;			// nastaveni nasledujiciho stavu na WAIT
			break;

	  case STATE_TXTIMEOUT:				// stav pro opetovne odeslani zpravy pres radiovy modul (tento stav je zavolan pri neuspesnem odeslani zpravy pres radiovy modul)
		  State = STATE_SEND;			// nastaveni nasledujiciho stavu na SEND pro dalsi pokus o odeslani zpravy
		  break;

	  case STATE_RXERROR:				// stavy pro opetovne prepnuti radia do prijimaciho rezimu (jsou volany v pripade prijeti chybne zpravy (chyba v hlavicce, nebo spatne CRC ve zprave) nebo pri vyprseni casoveho limitu pro prijeti zpravy (nebo znaku ve zprave))
	  case STATE_RXTIMEOUT:				// castecne nactena zprava ani poskozena zprava neni zpracovanava (je zahozena a ceka se na prijem dalsi zpravy)
		  Radio.Rx(0);					// prepnuti radioveho modulu do prijimaciho rezimu (bez omezeni doby poslechu)
		  State = STATE_WAIT;			// nastaveni nasledujiciho stavu na WAIT
		  break;

	  case STATE_RXDONE:					// stav pro zpracovani prijate zpravy pres radiovy modul (je volan pokud je radiovym modulem uspesne prijata jakakoliv zprava)
			if (Slave.ID != RxBuffer[0]){	// kontrola shody adresy jednotky a adresy v prijate zprave (zda je zprava pro tohoto slava, pokud neni tak cekani na prijeti dalsi zpravy)
				Radio.Rx(0);				// opetovne prepnuti radioveho modulu do prijimaciho rezimu
				State = STATE_WAIT;			// nastaveni nasledujiciho stavu na WAIT
				break;
			}

			switch(RxBuffer[1])				// vyber zpracovani zpravy na zaklade prijateho typu zpravy
			{

			case TimeSynch:					// prijeti zpravy casove synchronizace
				RTC_CurTicks = UTIL_TimerDriver.GetTimerValue();	// nova casova znacka RTC (ulozeni aktualni hodnoty RTC casovace, pro kterou je platny cas v SystemTime)
				memcpy(&SystemTime, &RxBuffer[2], 6);				// ulozeni 1. casti casove znacky ze zpravy (rok, mesic, den, hodina, minuta, sekunda)
				memcpy(&SystemTime.Milliseconds, &RxBuffer[8],2);	// ulozeni 2.casti casove znacky ze zpravy (milisekundy), (zvlast kvuli problemu se zarovnanim struktury)
				memcpy(&TxBuffer,&Slave.ID,sizeof(Slave.ID));		// vlozeni ID (adresy jednotky) do prenosovehu bufferu pro odeslani potvrzeni prijeti zpravy casove synch. master jednotce
				TxSize = sizeof(Slave.ID);							// nastaveni delky odesilane zpravy na 1 byte
				break;

			case MeasureRequest:			// prijeti zpravy pozadavku na zaslani namerenych dat
			case Status_MeasRequest:		// prijeti zpravy pozadavku na zaslani namerenych dat se stavem baterie
				uint8_t i;					// pozice pro zapis do prenosoveho bufferu TxBuffer
				memcpy(&TxBuffer,&Slave.ID,sizeof(Slave.ID));		// vlozeni ID (adresy jednotky) do bufferu k odeslani
				if(RxBuffer[1] == Status_MeasRequest)				// prijata zprava pozadavku na zaslani namerenych dat se stavem baterie
				{
					i = 1;											// posunuti pozice v TxBufferu o 1 (posunuti o vlozeny stav baterie)
					Battery_measure();								// provedeni mereni stavu baterie jednotky
					TxBuffer[1] = Slave.BatteryLvl;					// vlozeni stavu baterie jednotky do prenosoveho bufferu
					TxSize = 2;										// delka zpravy 2 byty (ID + stav baterie)
				}
				else												// prijata zprava pozadavku na zaslani namerenych dat (bez stavu baterie)
				{
					i = 0;											// zadny posun pozice v prenosovem bufferu
					TxSize = 1;										// delka zpravy 1 byte (pouze ID)
				}

				if (MeasuredValCnt != 0)	// pokud nejsou namerene zadne data (MeasuredValCnt == 0, neni do prenosoveho bufferu vlozena zadna dalsi informace) je poslana prazdna zprava (pouze s ID, pripadne stavem baterie)
				{							// tato situace by nemela nastat (master jednotka si hlida jestli uz ma slave neco namereno), ale kdyby nahodou tato situace nastala je potreba odeslat alespon potvrzeni o prijeti zpravy
					memcpy(&TxBuffer[i+1],&TimeStamp,6);					// vlozeni casove znacky mereni do prenosoveo bufferu (casova znacka mereni 1. vzorku dat v bufferu) (rok, mesic, den, hodina, minuta, sekunda)
					memcpy(&TxBuffer[i+7], &TimeStamp.Milliseconds,2);		// vlozeni milisekund casove znacky do prenosoveho bufferu (zvlast kvuli problemu se zarovnanim struktury)
					memcpy(&TxBuffer[i+9], &MeasuredValueBuff, MeasuredValCnt * sizeof(float));	 // vlozeni vsech namerenych hodnot do prenosoveho bufferu
					TxSize += 8 + MeasuredValCnt * sizeof(float);			// nastaveni delky zpravy pro odeslani (predchozi nastavena delka 1 nebo 2 byty + 8 bytu (casova znacka) + pocet namerenych hodnot, ktere byly vlozeny do bufferu)
					MeasuredValCnt = 0;			// vyprazdneni bufferu pro namerene hodnoty
				}
				break;

			case ReceiveSettings:			// prijeti zpravy nastaveni
				MeasuredValCnt = 0;								//vymzani namerenych dat ze snimace
				HAL_RTC_DeactivateAlarm(&hrtc, RTC_ALARM_B);	//deaktivace RTC alarmu B pro casovani mereni ze snimace

				memcpy(&Slave.NewRange, &RxBuffer[2],sizeof(Slave.NewRange));			// ulozeni noveho dosahu komunikace z prijate zpravy nastaveni (samotne prepnuti nastaveni dosahu je provedeno az po odeslani odpovedi na prijem zpravy nastaveni)
				memcpy(&Slave.MeasurePeriod, &RxBuffer[3],sizeof(Slave.MeasurePeriod));	// ulozeni nove periody mereni z prijate zpravy nastaveni

				Battery_measure();					// provedeni mereni stavu baterie slave jednotky
				TxBuffer[0] = Slave.ID;				// vlozeni ID (adresy jednotky) do prenosoveho bufferu pro odeslani
				TxBuffer[1] = Slave.BatteryLvl;		// vlozeni stavu baterie do prenosoveho bufferu
				TxBuffer[2] = TYP_SNIMACE;			// vlozeni typu pouzivaneho snimace
				TxBuffer[3] = VELICINA_1;			// vlozeni merene veliciny 1 (jednotka merene veliciny 1)
				TxBuffer[4] = VELICINA_2;			// vlozeni merene veliciny 2 (jednotka merene veliciny 2)
				TxSize = 5;							// nastaveni delky zpravy pro deslani na 5 bytu

				HAL_RTCEx_AlarmBEventCallback(&hrtc);	// nove nastaveni RTC alarmu B pro casovani mereni a provedeni mereni dat ze snimace
				break;

			}
			State = STATE_SEND;		// nastaveni nasledujiciho stavu na SEND pro odeslani prave nastavene zpravy
			break;


	  case STATE_WAIT:				// stav pro vyckani nez ma byt provedena nejaka akce (preruseni od RTC alarmu nebo od radioveho modulu)
		  if (SUBGRF_GetOperatingMode() ==  MODE_STDBY_RC || SUBGRF_GetOperatingMode() ==  MODE_SLEEP) 	// pokud se radiovy modul nachazi v pohotovostnim rezimu nebo rezimu Sleep (tedy neni aktivni)
			  if (RXstartTimeStamp <  UTIL_TIMER_GetCurrentTime()) 	// radio je po uplynuti doby kdy neni mozne odeslat novou zpravu (nastaveno ve stavu TxDone) prepnuto do prijimaciho rezimu
			  {
				  Radio.Rx(0);	// prepnuti radioveho modulu do prijimaciho rezimu (bez omezeni doby poslechu)
			  }
		  break;
	  }

    /* USER CODE END WHILE */

    /* USER CODE BEGIN 3 */
  }
  /* USER CODE END 3 */
}

/**
  * @brief System Clock Configuration
  * @retval None
  */
void SystemClock_Config(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /** Configure LSE Drive Capability
  */
  HAL_PWR_EnableBkUpAccess();
  __HAL_RCC_LSEDRIVE_CONFIG(RCC_LSEDRIVE_LOW);

  /** Configure the main internal regulator output voltage
  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE2);

  /** Initializes the CPU, AHB and APB buses clocks
  */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI|RCC_OSCILLATORTYPE_LSE
                              |RCC_OSCILLATORTYPE_MSI;
  RCC_OscInitStruct.LSEState = RCC_LSE_ON;
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;
  RCC_OscInitStruct.MSIState = RCC_MSI_ON;
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.MSICalibrationValue = RCC_MSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.MSIClockRange = RCC_MSIRANGE_5;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_NONE;
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure the SYSCLKSource, HCLK, PCLK1 and PCLK2 clocks dividers
  */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK3|RCC_CLOCKTYPE_HCLK
                              |RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_PCLK1
                              |RCC_CLOCKTYPE_PCLK2;
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_MSI;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.AHBCLK3Divider = RCC_SYSCLK_DIV1;

  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    Error_Handler();
  }
}

/**
  * @brief ADC Initialization Function
  * @param None
  * @retval None
  */
static void MX_ADC_Init(void)
{

  /* USER CODE BEGIN ADC_Init 0 */

  /* USER CODE END ADC_Init 0 */
  /* USER CODE BEGIN ADC_Init 1 */

  /* USER CODE END ADC_Init 1 */

  /** Configure the global features of the ADC (Clock, Resolution, Data Alignment and number of conversion)
  */
  hadc.Instance = ADC;
  hadc.Init.ClockPrescaler = ADC_CLOCK_ASYNC_DIV2;
  hadc.Init.Resolution = ADC_RESOLUTION_12B;
  hadc.Init.DataAlign = ADC_DATAALIGN_RIGHT;
  hadc.Init.ScanConvMode = ADC_SCAN_ENABLE;
  hadc.Init.EOCSelection = ADC_EOC_SINGLE_CONV;
  hadc.Init.LowPowerAutoWait = DISABLE;
  hadc.Init.LowPowerAutoPowerOff = DISABLE;
  hadc.Init.ContinuousConvMode = DISABLE;
  hadc.Init.NbrOfConversion = 1;
  hadc.Init.DiscontinuousConvMode = DISABLE;
  hadc.Init.ExternalTrigConv = ADC_SOFTWARE_START;
  hadc.Init.ExternalTrigConvEdge = ADC_EXTERNALTRIGCONVEDGE_NONE;
  hadc.Init.DMAContinuousRequests = DISABLE;
  hadc.Init.Overrun = ADC_OVR_DATA_PRESERVED;
  hadc.Init.SamplingTimeCommon1 = ADC_SAMPLETIME_39CYCLES_5;
  hadc.Init.SamplingTimeCommon2 = ADC_SAMPLETIME_39CYCLES_5;
  hadc.Init.OversamplingMode = DISABLE;
  hadc.Init.TriggerFrequencyMode = ADC_TRIGGER_FREQ_HIGH;
  if (HAL_ADC_Init(&hadc) != HAL_OK)
  {
    Error_Handler();
  }

  /* USER CODE BEGIN ADC_Init 2 */

  /* USER CODE END ADC_Init 2 */

}

/**
  * @brief I2C3 Initialization Function
  * @param None
  * @retval None
  */
static void MX_I2C3_Init(void)
{

  /* USER CODE BEGIN I2C3_Init 0 */

  /* USER CODE END I2C3_Init 0 */

  /* USER CODE BEGIN I2C3_Init 1 */

  /* USER CODE END I2C3_Init 1 */
  hi2c3.Instance = I2C3;
  hi2c3.Init.Timing = 0x00000103;
  hi2c3.Init.OwnAddress1 = 0;
  hi2c3.Init.AddressingMode = I2C_ADDRESSINGMODE_7BIT;
  hi2c3.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;
  hi2c3.Init.OwnAddress2 = 0;
  hi2c3.Init.OwnAddress2Masks = I2C_OA2_NOMASK;
  hi2c3.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;
  hi2c3.Init.NoStretchMode = I2C_NOSTRETCH_DISABLE;
  if (HAL_I2C_Init(&hi2c3) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure Analogue filter
  */
  if (HAL_I2CEx_ConfigAnalogFilter(&hi2c3, I2C_ANALOGFILTER_ENABLE) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure Digital filter
  */
  if (HAL_I2CEx_ConfigDigitalFilter(&hi2c3, 0) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN I2C3_Init 2 */

  /* USER CODE END I2C3_Init 2 */

}

/**
  * @brief RTC Initialization Function
  * @param None
  * @retval None
  */
void MX_RTC_Init(void)
{

  /* USER CODE BEGIN RTC_Init 0 */

  /* USER CODE END RTC_Init 0 */

  RTC_AlarmTypeDef sAlarm = {0};

  /* USER CODE BEGIN RTC_Init 1 */

  /* USER CODE END RTC_Init 1 */

  /** Initialize RTC Only
  */
  hrtc.Instance = RTC;
  hrtc.Init.AsynchPrediv = RTC_PREDIV_A;
  hrtc.Init.OutPut = RTC_OUTPUT_DISABLE;
  hrtc.Init.OutPutRemap = RTC_OUTPUT_REMAP_NONE;
  hrtc.Init.OutPutPolarity = RTC_OUTPUT_POLARITY_HIGH;
  hrtc.Init.OutPutType = RTC_OUTPUT_TYPE_OPENDRAIN;
  hrtc.Init.OutPutPullUp = RTC_OUTPUT_PULLUP_NONE;
  hrtc.Init.BinMode = RTC_BINARY_ONLY;
  if (HAL_RTC_Init(&hrtc) != HAL_OK)
  {
    Error_Handler();
  }

  /* USER CODE BEGIN Check_RTC_BKUP */

  /* USER CODE END Check_RTC_BKUP */

  /** Initialize RTC and set the Time and Date
  */
  if (HAL_RTCEx_SetSSRU_IT(&hrtc) != HAL_OK)
  {
    Error_Handler();
  }

  /** Enable the Alarm A
  */
  sAlarm.BinaryAutoClr = RTC_ALARMSUBSECONDBIN_AUTOCLR_NO;
  sAlarm.AlarmTime.SubSeconds = 0x0;
  sAlarm.AlarmMask = RTC_ALARMMASK_NONE;
  sAlarm.AlarmSubSecondMask = RTC_ALARMSUBSECONDBINMASK_NONE;
  sAlarm.Alarm = RTC_ALARM_A;
  if (HAL_RTC_SetAlarm_IT(&hrtc, &sAlarm, 0) != HAL_OK)
  {
    Error_Handler();
  }

  /** Enable the Alarm B
  */
  sAlarm.Alarm = RTC_ALARM_B;
  if (HAL_RTC_SetAlarm_IT(&hrtc, &sAlarm, 0) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN RTC_Init 2 */

  /* USER CODE END RTC_Init 2 */

}

/**
  * @brief SUBGHZ Initialization Function
  * @param None
  * @retval None
  */
void MX_SUBGHZ_Init(void)
{

  /* USER CODE BEGIN SUBGHZ_Init 0 */

  /* USER CODE END SUBGHZ_Init 0 */

  /* USER CODE BEGIN SUBGHZ_Init 1 */

  /* USER CODE END SUBGHZ_Init 1 */
  hsubghz.Init.BaudratePrescaler = SUBGHZSPI_BAUDRATEPRESCALER_4;
  if (HAL_SUBGHZ_Init(&hsubghz) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN SUBGHZ_Init 2 */

  /* USER CODE END SUBGHZ_Init 2 */

}

/**
  * @brief GPIO Initialization Function
  * @param None
  * @retval None
  */
static void MX_GPIO_Init(void)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};
/* USER CODE BEGIN MX_GPIO_Init_1 */
/* USER CODE END MX_GPIO_Init_1 */

  /* GPIO Ports Clock Enable */
  __HAL_RCC_GPIOA_CLK_ENABLE();
  __HAL_RCC_GPIOB_CLK_ENABLE();
  __HAL_RCC_GPIOC_CLK_ENABLE();
  __HAL_RCC_GPIOH_CLK_ENABLE();

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(LED1_GPIO_Port, LED1_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOC, FE_CTRL3_Pin|FE_CTRL2_Pin|FE_CTRL1_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pins : PA12 PA15 PA11 PA10
                           PA0 PA9 PA6 PA1
                           PA3 PA2 PA7 PA4
                           PA5 PA8 */
  GPIO_InitStruct.Pin = GPIO_PIN_12|GPIO_PIN_15|GPIO_PIN_11|GPIO_PIN_10
                          |GPIO_PIN_0|GPIO_PIN_9|GPIO_PIN_6|GPIO_PIN_1
                          |GPIO_PIN_3|GPIO_PIN_2|GPIO_PIN_7|GPIO_PIN_4
                          |GPIO_PIN_5|GPIO_PIN_8;
  GPIO_InitStruct.Mode = GPIO_MODE_ANALOG;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /*Configure GPIO pin : LED1_Pin */
  GPIO_InitStruct.Pin = LED1_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
  HAL_GPIO_Init(LED1_GPIO_Port, &GPIO_InitStruct);

  /*Configure GPIO pins : PB3 PB4 PB7 PB9
                           PB14 PB5 PB8 PB2
                           PB6 PB12 PB1 PB0
                           PB11 */
  GPIO_InitStruct.Pin = GPIO_PIN_3|GPIO_PIN_4|GPIO_PIN_7|GPIO_PIN_9
                          |GPIO_PIN_14|GPIO_PIN_5|GPIO_PIN_8|GPIO_PIN_2
                          |GPIO_PIN_6|GPIO_PIN_12|GPIO_PIN_1|GPIO_PIN_0
                          |GPIO_PIN_11;
  GPIO_InitStruct.Mode = GPIO_MODE_ANALOG;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

  /*Configure GPIO pins : PC13 PC2 PC0 PC6 */
  GPIO_InitStruct.Pin = GPIO_PIN_13|GPIO_PIN_2|GPIO_PIN_0|GPIO_PIN_6;
  GPIO_InitStruct.Mode = GPIO_MODE_ANALOG;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pins : FE_CTRL3_Pin FE_CTRL2_Pin FE_CTRL1_Pin */
  GPIO_InitStruct.Pin = FE_CTRL3_Pin|FE_CTRL2_Pin|FE_CTRL1_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pin : PH3 */
  GPIO_InitStruct.Pin = GPIO_PIN_3;
  GPIO_InitStruct.Mode = GPIO_MODE_ANALOG;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOH, &GPIO_InitStruct);

/* USER CODE BEGIN MX_GPIO_Init_2 */
/* USER CODE END MX_GPIO_Init_2 */
}

/* USER CODE BEGIN 4 */

/* Inicializace Callback funci Radioveho modulu */
static void Radio_Init(void){					// ktere funkce maji byt volany pri preruseni od radioveho modulu
	  RadioEvents.TxDone = OnTxDone;
	  RadioEvents.RxDone = OnRxDone;
	  RadioEvents.TxTimeout = OnTxTimeout;
	  RadioEvents.RxTimeout = OnRxTimeout;
	  RadioEvents.RxError = OnRxError;
	  Radio.Init(&RadioEvents);					// ulozeni do driveru radioveho modulu
}

/* Funkce pro vychozi nastaveni parametru radioveho modulu */
static void Radio_InitRxTxParams(void){
	  Radio.SetModem(MODEM_LORA);						// konfigurace radioveho modulu na LoRa modulaci
	  Radio.SetChannel(868000000);						// nastaveni nosne frekvence
	  Radio.SetMaxPayloadLength(MODEM_LORA, 255);		// nastaveni maximalni delky zpravy

	  Radio.SetRxConfig(MODEM_LORA, BW250, SF7, CR4_5, 0, PREAM_LEN8,
			  	  	  	  SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0, CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni vychozich parametru modulace pro prijem signalu

	  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW250, SF7, CR4_5,
			  	  	  	  PREAM_LEN8, NO_FIX_PAYLOAD_LEN, CRCon, 0, 0, IQ_NORMAL, 100000 );			// nastaveni vychozich parametru modulace pro vysilani signalu

}

/* Funkce pro prenastaveni modulacnich parametru radioveho modulu podle nastaveni komuikace
   @param Range Nastaveni komunikace */
static void Radio_SetRxTxParams(uint8_t Range){
	switch(Range)									// vyber sady modulacnich parametru podle nastaveni komunikace
	{
	case Range100:
		  Radio.SetRxConfig(MODEM_LORA, BW250, SF7, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW250, SF7, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range200:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF8, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF8, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range300:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF9, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF9, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range500:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF10, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF10, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range600:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF11, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF11, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range750:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF12, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF12, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 100000 ); // nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;
	}
}

/* Obsluha preruseni, vola se pri uspesnem odeslani zpravy radiovym modulem */
static void OnTxDone(void)
{
  /* USER CODE BEGIN OnTxDone */
	State = STATE_TXDONE;		// nasledujici stav TXdone (pro prepnuti do radia do Sleep modu)
  /* USER CODE END OnTxDone */
}

/* Obsluha preruseni, vola se pri uspesnem prijeti zpravy radiovym modulem
   @param payload Ukazatel na prijatou zpravu
   @param size Delka prijate zpravy v bytech
   @param rssi Rssi posledni prijate zpravy (sila prijateho singalu v dBm)
   @param LoraSnr_FskCfo SNR posledni prijate zpravy (SRN odstup signal sum) */
static void OnRxDone(uint8_t *payload, uint16_t size, int16_t rssi, int8_t LoraSnr_FskCfo)
{
  /* USER CODE BEGIN OnRxDone */

	//char Buffer[size];
	memset(RxBuffer,0,255);				// vycisteni bufferu pro prijatou zpravu (aby nedoslo k pouziti spatnych dat)
	memcpy(RxBuffer,payload, size );	// kopie prijate zpravy
	RxMsgInfo.Size = size;				// ulozeni delky prijate zpravy v bytech
	RxMsgInfo.RSSI = rssi;				// ulozeni sily prijateho signalu v dBm
	RxMsgInfo.SNR = LoraSnr_FskCfo;		// ulozeni odstupu signal sum v dB
	State = STATE_RXDONE;				// nasledujici stav RXdone (pro zpracovani prijate vzpavy)

  /* USER CODE END OnRxDone */
}

/* Obsluha preruseni, vola se pokud se radiovemu modulu nepodarilo zpravu uspesne odeslat v nastavenem intervalu 10s */
static void OnTxTimeout(void)
{
  /* USER CODE BEGIN OnTxTimeout */
	State = STATE_TXTIMEOUT;			// nasledujici stav TXtimeout (pro opetovne odelsani zpravy)
  /* USER CODE END OnTxTimeout */
}

/* Obsluha preruseni, vola se pokud v nastavenem intervalu prijimani zpravy nebyla prijata zadna zprava radiovym modeulem */
static void OnRxTimeout(void)
{
  /* USER CODE BEGIN OnRxTimeout */
	State = STATE_RXTIMEOUT;			// nasledujici stav RXtimeout
  /* USER CODE END OnRxTimeout */
}

/* Obsluha preruseni, vola se pokud byla radiovym modulem prijata poskozena zprava (chyba v hlavicce, chybny CRC kod) */
static void OnRxError(void)
{
  /* USER CODE BEGIN OnRxError */
	State = STATE_RXERROR;  			// nasledujici stav RXerror
  /* USER CODE END OnRxError */
}


/*Funkce pro aktualizaci systemoveho casu SystemTime podle RTC  */
static void SysTimeUpdate(void)
{	/*Funkce aktualizuje cas ve strukture SystemTime podle ubehleho casu RTC od posledni aktualizace*/

	uint32_t RTC_tmp = UTIL_TimerDriver.GetTimerValue();	// ulozeni aktualni hodnoty casovace RTC (tiky)
	uint32_t TicksElapsed = RTC_tmp - RTC_CurTicks;			// pocet tiku od posledni aktualizace
	RTC_CurTicks = RTC_tmp;									// ulozeni kdy byla provedena posledni aktualizace

	//krystal 32768 impulsu za sekundu
	//pouzita preddelicka 32 (RTC_PREDIV_A + 1)
	//pocet nactenych impulsu do RTC citace je 32768/32 = 1024 impulsu za sekundu
	//za milisekundu je do RTC citace nacteno 1024/1000 = 1,024 impulsu

	//Vypocet ubehleho casu v milisekundach
	uint32_t MsElapsed = TicksElapsed / ((32768 / (RTC_PREDIV_A + 1)) / 1000.0);

	AddTime(&SystemTime, MsElapsed);	//pricteni ubehleho casu ke strukture SystemTime
}

/* Funkce pro pricteni casu ke strukture a jeho konverze (zarovnani)
   @param Time Ukazatel na casovou strukturu, ke ktere ma byt cas pricten
   @param AddTimeMS Cas v milisekundach, ktery ma byt pricten k casove strukture */
static void AddTime(SystemTime_t *Time, uint32_t AddTimeMS)
{
	AddTimeMS += Time->Milliseconds;		//soucet milisekund
	while(AddTimeMS >= 1000)				//pro kazdych 1000 ms je odecteno 1000 ms a provedena konverze casu
	{
		AddTimeMS -= 1000;

	   ++Time->Seconds;
	   Time->Seconds %= 60;

	   if (Time->Seconds == 0)
	   {
		   ++Time->Minutes;
		   Time->Minutes %= 60;

		   if (Time->Minutes == 0)
		   {
			   ++Time->Hours;
			   Time->Hours %= 24;

			   if (Time->Hours == 0)
			   {
				   ++Time->Day;

				   if (Time->Day > DaysInMonth(Time->Month, Time->Year))	// kontrola poctu dni v mesici
				   {
					   Time->Day = 1;
					   ++Time->Month;

					   if (Time->Month == 13)
					   {
						   Time->Month = 1;
						   ++Time->Year;

						   if (Time->Year == 100)							// u roku uchovany pouze jednotky a desitky
							   Time->Year = 0;
					   }
				   }
			   }
		   }
	   }
   }

   Time->Milliseconds = AddTimeMS;		// ulozeni zbylych milisekund
}

/* Funkce pro zjisteni poctu dni v mesici
   @param Month Cislo mesice
   @param Year Rok
   @return Pocet dni v mesici*/
static uint8_t DaysInMonth(uint8_t Month, uint8_t Year)
{
   uint8_t result;
   switch (Month)
   {
   case 1:  // leden
   case 3:  // brezen
   case 5:  // kveten
   case 7:  // cevenec
   case 8:  // srpen
   case 10: // rijen
   case 12: // prosinec
       result = 31;
       break;

   case 4:  // duben
   case 6:  // cerven
   case 9:  // zari
   case 11: // listopad
       result = 30;
       break;

   case 2: 	// unor				// zjisteni prestupneho roku
       if (Year % 4 == 0)		// staci pouze delitelnost 4, pokud se pocita 0-99
       {
    	   result = 29; 		// prestupny rok
       	   break;
       }
       else
       {
    	   result = 28;
       	   break;
       }
   }
   return result;
}

/* Funkce pro vypocet casu, kdy neni mozne odeslat dalsi zpravu
   @param Range Nastaveni dosahu (pro jake parametry modulace ma byt cas vypocten)
   @param size Delka odeslane zpravy v bytech
   @param per DutyCycle v procentech kdy neni mozne vysilat (vyuziva se 100 a 99, 100 pro vypocet pred odeslanim zpravy, 99 po odeslani zpravy)
   @return Cas v milisekundach, za ktery je mozne odeslat dalsi zpravu*/
uint32_t CalcDutyCycle(uint8_t Range, uint8_t size, uint8_t per)
{
	uint32_t result;

	switch(Range)		//vyber podle dosahu komunikace
	{
	case Range100:
		 result = per * Radio.TimeOnAir(MODEM_LORA, BW250, SF7, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		 break;			//TimeOnAir vraci dobu prenosu celeho paketu v ms
	case Range200:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF8, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range300:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF9, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range500:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF10, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range600:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF11, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range750:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF12, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	}

	return result;
}

/* Obsluha preruseni od RTC alarmu B
   Slouzi pro nove nastaveni alarmu B a provedeni mereni ze snimace*/
void HAL_RTCEx_AlarmBEventCallback(RTC_HandleTypeDef *hrtc)
{
	  //nove nastevni alarmu B
	  HAL_RTC_GetTime(hrtc, &sTime, RTC_FORMAT_BIN);			//nacteni casu z RTC
	  HAL_RTC_GetDate(hrtc, &sDate, RTC_FORMAT_BIN);			//datum neni pouzivat v binarnim rezimu RTC (ale potreba volat tuto funkci po GetTime protoze RTC stoji dokud neni nacten i datum)

	  sAlarmB.BinaryAutoClr = RTC_ALARMSUBSECONDBIN_AUTOCLR_NO;	//nenunovat alarm pri preruseni
	  sAlarmB.AlarmTime.SubSeconds = sTime.SubSeconds - ((uint32_t)((((uint64_t) Slave.MeasurePeriod) * 1024) / 1000));	  //hodnota pri ktere vznikce preruseni , + prevod z milisekund na tiky
	  sAlarmB.AlarmMask = RTC_ALARMMASK_NONE;																			  //aktualni hodnota citace - perioda mereni (v tikach) (odcitani protoze casovat RTC cita do 0)
	  sAlarmB.AlarmSubSecondMask = RTC_ALARMSUBSECONDBINMASK_NONE;	//zadne masky (alarm bude spusten presne pri hodnote v subseconds)
	  sAlarmB.Alarm = RTC_ALARM_B;									//nastaveni typu alarmu na B
	  if (HAL_RTC_SetAlarm_IT(hrtc, &sAlarmB, 0) != HAL_OK)			//nasteveni vzniku preruseni od noveho nastaveni Alarmu B
	  {
	    Error_Handler();
	  }

	  Sensor_measure();		// mereni dat snimacem
}


/* Funkce pro mereni externim snimacem
   Slouzi pro ziskani surove hodnoty z externiho snimace, prepocet na fyz. hodnotu a ulozeni do pameti */
static void Sensor_measure(void){
	  if (MeasuredValCnt != 60)				// mereni pouze pokud jiz neni namereno 60 hodnot (30 mereni) (zcela zaplnen prenosovy buffer slave jednotky), nezle ve zprave odeslat vice dat
	  	{
			if (MeasuredValCnt == 0)		// pokud probiha 1. mereni dat (po odeslani predchozich dat do master jednotky)
			{
				SysTimeUpdate();			// aktualizace systemoveho casu
				TimeStamp = SystemTime;		// ulozeni casove znacky 1. mereni
			}
														/** Je pouzit snimac SHT30 **/
			bool SensorReceiveSuccess = false;			// nastaveni neuspesneho prijmu dat ze snimace
			uint8_t SensorAdress = 0x44 << 1u;			// I2C adresa snimace (7 MSB adresa, LSB 0 write)
			uint8_t  SensorCommand[2] = {0x2c, 0x06};	// 16 bit prikaz pro snimac na provedeni jednoho mereni
			uint8_t RawSensorData[6];					// pro nacteni surovych dat ze snimace

			HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, GPIO_PIN_SET);	// rozsviti se dioda LED1 pri mereni

			while(SensorReceiveSuccess == false)		// opakovane mereni pro pridat chybneho crc v datech od snimace
			{
				HAL_I2C_Master_Transmit(&hi2c3, SensorAdress, SensorCommand, sizeof(SensorCommand), HAL_MAX_DELAY);	// odeslani commandu senzoru pres i2c
				HAL_I2C_Master_Receive(&hi2c3, SensorAdress, RawSensorData, sizeof(RawSensorData), HAL_MAX_DELAY);	// prijeti surovych dat pres i2c ze snimace
														// snimac vrati celkem 6 bytu, 2 byty teplota, 1 byt CRC pro teplotu, 2 byty vlhkost, 1 byt CRC pro vlhkost
				//vypocet crc prijatych dat
				uint8_t TempCRC = 0xff;					// vychozi hodnota CRC pro vypocet
				uint8_t HumCRC = 0xff;
				for (size_t i = 0; i < 2; i++)			// 2 opakovani (pro 1. a 2. byte teploty a vlhkosti)
				{
					TempCRC = TempCRC ^ RawSensorData[i];	// xor s 1. (resp 2.) bytem teploty
					HumCRC = HumCRC ^ RawSensorData[i+3];	// xor s 1. (resp 2.) bytem vlhkosti

					for (size_t j = 0; j < 8; j++)			// 8 opakovani (jeden pruchod pro kazdy bit crc)
					{
						if ((TempCRC & 0b10000000) == 0)	// pokud je nejvyssi (MSB) bit roven 0
							TempCRC = TempCRC << 1;			// je proveden bitovy posuv vlevo
						else
							TempCRC = (TempCRC << 1) ^ 0x31;	// pokud je nejvyssi bit roven 1, je proveden bitovy posuv vlebo a xor s crc polynomen (0x31)

						if ((HumCRC & 0b10000000) == 0)		// stejny postup pro vlhkost
							HumCRC = HumCRC << 1;
						else
							HumCRC = (HumCRC << 1) ^ 0x31;
					}
				}
				if ((TempCRC == RawSensorData[2]) && (HumCRC == RawSensorData[5]))	//pokud je vypocitane CRC shodne s prijatym CRC od snimace jsou data validni
					SensorReceiveSuccess = true;			// oznaceni dat ze snimace jako validni
			}

			HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, GPIO_PIN_RESET);	// po dokonceni mereni je zhasnuta dioda LED1


			uint16_t temperature_raw = ((uint16_t)RawSensorData[0] << 8) | RawSensorData[1];	// prevedeni 2 bytu teploty na jedno 16 bit cislo
			uint16_t humidity_raw = ((uint16_t)RawSensorData[3] << 8) | RawSensorData[4];		// prevedeni 2 bytu vlhkoti na jedno 16 bit cislo

			//ulozeni namerenych dat ze snimace do bufferu
			MeasuredValueBuff[MeasuredValCnt++] = -45.0 + 175.0 * (float)temperature_raw / 65535.0;		// vypocet float teploty ze surovych dat (rovnice z datasheetu pro snimac SHT30)
			MeasuredValueBuff[MeasuredValCnt++] = 100.0 * (float)humidity_raw / 65535.0;				// vypocet float vlhkosti ze surovych dat (rovnice z datasheetu pro snimac SHT30)
	  	}
}

/* Funkce pro mereni stavu baterie slave jednotky
   Slouzi pro mereni napeti na beterii pres napetovy delic pomoci ADC a prevodu napeti na stav baterie v procentech */
static void Battery_measure(void)
{
	  ADC_ChannelConfTypeDef sConfig = {0};	// pro konfiguraci ADC kanalu na kterem ma byt provedeno mereni
	  uint32_t VrefADCval;					// namerena hodnota ADC vnitrniho referencniho napeti
	  uint32_t BatADCval;					// namerena hodnota ADC na napetovem delici beterie
	  double Vref;							// velikost rereferencniho napeti AD prevodniku ve voltech
	  double Vbat;							// velikost napeti na baterii ve voltech

	  sConfig.Channel = ADC_CHANNEL_VREFINT;				// nastaveni kanalu na kterem se ma provest mereni (mereni vnitrniho referecniho napeti)
	  sConfig.Rank = ADC_REGULAR_RANK_1;					// nastveni nejvyssi priority (nema vliv - mereni pouze jednoho kanalu)
	  sConfig.SamplingTime = ADC_SAMPLINGTIME_COMMON_1;		// nastaveni doby vzorkovani podle skupiny 1 (39,5 taktovaciho cyklu ADC) (taktovaci frekvence ADC 16 MHz)
	  if (HAL_ADC_ConfigChannel(&hadc, &sConfig) != HAL_OK)	// ulozeni konfigurace kanalu ADC
	  {
	    Error_Handler();
	  }

	  HAL_ADCEx_Calibration_Start(&hadc);					// automaticka kalibrace ADC (melo by byt provadeno pred kazdym merenim)
	  HAL_ADC_Start(&hadc);									// zapnuti mereni pomoci ADC
	  HAL_ADC_PollForConversion(&hadc, HAL_MAX_DELAY);		// cekani na dokonceni prevodu ADC
	  VrefADCval = HAL_ADC_GetValue(&hadc);					// nacteni namerene hodnoty referencniho napeti z ADC
	  HAL_ADC_Stop(&hadc);									// zastaveni prevodu ADC

	  sConfig.Channel = ADC_CHANNEL_0;						// nastaveni kanalu na kterem se ma provest mereni (ADC0)
	  sConfig.Rank = ADC_REGULAR_RANK_1;					// nastveni nejvyssi priority (nema vliv - mereni pouze jednoho kanalu)
	  sConfig.SamplingTime = ADC_SAMPLINGTIME_COMMON_2;		// nastaveni doby vzorkovani podle skupiny 2 (39,5 taktovaciho cyklu ADC) (taktovaci frekvence ADC 16 MHz)
	  if (HAL_ADC_ConfigChannel(&hadc, &sConfig) != HAL_OK)	// ulozeni konfigurace kanalu ADC
	  {
	    Error_Handler();
	  }

	  HAL_ADCEx_Calibration_Start(&hadc);					// automaticka kalibrace ADC (melo by byt provadeno pred kazdym merenim)
	  HAL_ADC_Start(&hadc);									// zapnuti mereni pomoci ADC
	  HAL_ADC_PollForConversion(&hadc, HAL_MAX_DELAY);		// cekani na dokonceni prevodu ADC
	  BatADCval = HAL_ADC_GetValue(&hadc);					// nacteni namerene hodnoty na delici napeti baterie z ADC
	  HAL_ADC_Stop(&hadc);									// zastaveni prevodu ADC

	  uint16_t CalibVal = *(uint16_t *)(0x1FFF75AA);		// nacteni kalibrovane hodnoty od vyrobce z pameti FLASH pro vypocet referencniho napeti AD prevodniku
	  Vref = 3.3 * (CalibVal)/(double)VrefADCval;			// vypocet referecniho napeti AD prevodniku ve voltech (3,3 (ref. napeti pri mereni vyrobcem), CalibVal (kalibrovana hodnota od vyrobce), VrefADCval (namerena hodnota vnitrni reference)
	  Vbat = (Vref/4096.0)*BatADCval * 2.0;					// vypocet napeti baterie ve voltech (Vref (referecni hodnota AD prevodniku), 4096 (pocet kvantizacnich urovni AD prevodniku - 12 bit),
	  	  	  	  	  	  	  	  	  	  	  	  	  	  	// BatADCval (namerena hodnota AD prevodnikem), *2.0 (kvuli delici napeti na baterii s pomerem 1:2)
	  Slave.BatteryLvl = (uint8_t)((Vbat-2.6)/(4.2-2.6)*100);	// prepocet napeti baterie na procenta	( 2,6 V = 0 %		4,2 V = 100 %)
}

/* USER CODE END 4 */

/**
  * @brief  This function is executed in case of error occurrence.
  * @retval None
  */
void Error_Handler(void)
{
  /* USER CODE BEGIN Error_Handler_Debug */
  /* User can add his own implementation to report the HAL error return state */
  __disable_irq();
  while (1)
  {

  }
  /* USER CODE END Error_Handler_Debug */
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* USER CODE BEGIN 6 */
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* USER CODE END 6 */
}
#endif /* USE_FULL_ASSERT */
