/* USER CODE BEGIN Header */
/**
  ******************************************************************************
  * @file           : main.c
  * @brief          : Main program body
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2023 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/******************	Nastaveni nazvu snimace a merenych jednotek ******************	*/
#define TYP_SNIMACE SHT30
#define VELICINA_1 TEMPERATURE_C
#define VELICINA_2 HUMIDITY_PER

// pro pridani noveho nazvu snimace a merenych jednotek musi byt tyto nazvy pridany do vyctu nazvu na radku 57 a 68
// (tyto vycty musi byt ve vsech jednotkach (master i slave) stejne, i se stejnym poradim
// po pridani noveho nazvu snimace nebo merene jednotky je take nutne zadat na radku 293 nebo 299
// format v jakem maji byt nazvy zapisovany do hlavicky souboru pri jejich ukladani



/* USER CODE END Header */
/* Includes ------------------------------------------------------------------*/
#include "main.h"			//include hlavickoveho souboru hlavniho programu
#include "ff.h"				//include FATFS knihovny
#include "ff_gen_drv.h"		//include FATFS driveru
#include "user_diskio.h" 	//include driveru pro SD kartu

/* Private includes ----------------------------------------------------------*/
/* USER CODE BEGIN Includes */
#include "platform.h"		//include knihoven pro vyvojovou desku
#include "radio.h"			//include knihovny pro ovladani radioveho modulu
#include "stdio.h"
#include "stdbool.h"
#include "sys_app.h"		//nevyuzito (generovano CubeMX automaticky)
#include "stm32_systime.h"	//include funkci pro spravu casu
#include "stm32_timer.h"	//utility pro managenent casu


/* USER CODE END Includes */

/* Private typedef -----------------------------------------------------------*/
/* USER CODE BEGIN PTD */

/* Vycet typu merenych jednotek
   Lze pridat libovolny pocet merenych jednotek
   (tento vycet musi byt ve vsech jednotkach datalogeru shodny) */
typedef enum {
    TEMPERATURE_C,
	HUMIDITY_PER,
    PRESSURE_kPa,

    NumberOfUnitName // automaticky urci pocet zadanych jednotek
} UnitName_t;

/* Vycet typu snimace
   Lze pridat libovolny pocet nazvu snimacu
   (tento vycet musi byt ve vsech jednotkach datalogeru shodny) */
typedef enum {
    SHT30,

    NumberOfSensorName // automaticky urci pocet zadanych snimacu
} SensorName_t;

/* Vycet typu zprav posilanych z master jednotky do slave jednotek */
/* (lze pridat dalsi typy zprav)*/
enum MsgType {
	NONE = 0,
	TimeSynch = 1,
	MeasureRequest = 2,
	SendSettings = 3,
	Status_MeasRequest = 4
};

/* Moznosti nastaveni dosahu komunikace */
enum RangeSetting {
	Range750 = 5,
	Range600 = 4,
	Range500 = 3,
	Range300 = 2,
	Range200 = 1,
	Range100 = 0
};

/* Struktura Slave jednotky */
typedef struct{
	uint8_t ID;						// adresa slave
	bool Enable;					// povoleni komunikace se slavem
	uint32_t MeasurePeriod;			// perioda mereni slavu [ms]
	uint8_t Msg;					// zprava, ktera ma byt slave jednotce odeslana
	uint32_t DCWaitTime;			// cas po jehoz uplynuti muze slave poslat dalsi zpravu
	uint32_t DCTimeStamp;			// casova znacka prijmu posleni zpravy od slava
	bool SettingChange;				// indikace zmeny nastaveni slave
	bool TimeSynch;					// indikace zda ma byt provedena cas. synch.
	uint8_t CurRange;				// aktualne nastaveny dosah komunikace, parametry modulace
	uint32_t LastReadTimeStamp;		// casova znacka posledniho vycteni dat z jednotky
	uint32_t LastStatusTimeStamp;	// casova znacka posledni zpravy se statusem
	uint32_t LastTimeSynchTimeStap;	// casova znacka posledni casove synchronizace
	int16_t RSSI;					// sila prijateho signalu [dBm] (posledni prijate zpravy)
	uint8_t BatteryLvl;				// stav nabiti baterie (v %)
	uint8_t SensorType;				// typ snimace
	uint8_t UnitName1;				// jednotka 1
	uint8_t UnitName2;				// jednotka 2
} SlaveInit_t;

/* Struktura Master jednotky */
typedef struct{
	bool EnableMeasure;				// povoleni mereni master jednotkou
	uint32_t MeasurePeriod;			// perioda mereni master jednotky [ms]
	uint8_t SettedRange;			// nastaveny dosah komunikace uzivatelem
	bool SettedPeriodChange;		// indikace zmeny nastaveni periody mereni master jednotky
	uint8_t TimeSynchInterval;		// interval casove synchronizace slave jednotek
	uint8_t SensorType;				// typ snimace
	uint8_t UnitName1;				// jednotka 1
	uint8_t UnitName2;				// jednotka 2
} MasterInit_t;

/* Stavy stavoveho automatu */
typedef enum{
  STATE_WAIT,
  STATE_SEND,
  STATE_RXDONE,
  STATE_TXDONE,
  STATE_RXTIMEOUT,
  STATE_TXTIMEOUT,
  STATE_RXERROR,
  STATE_START_MASTER_MEASURE,
  STATE_STOP_MASTER_MEASURE
} State_t;

/* Struktura systemoveho casu */
typedef struct{
	uint8_t Year;			// rok 2 cifry
	uint8_t	Month;
	uint8_t Day;
	uint8_t Hours;
	uint8_t Minutes;
	uint8_t Seconds;
	uint16_t Milliseconds;
} SystemTime_t;

/* Struktura pro ulozeni informaci o prijate zprave pres LoRa*/
typedef struct{
	uint16_t Size;				// delka prijate zpravy
	int16_t RSSI;				// sila signalu prijate zpravy [dBm]
	int16_t SNR;				// odstup signal sum [dB]
}RxMsgInfo_t;

/* Struktura pro docasne ulozeni dat, ktere maji byt zapsany na SD kartu*/
typedef struct{
	uint8_t ID;				// adresa jednotky, ktere data nalezi
	char Data[1500];		// data pro ulozeni na SD kartu
	bool UnitType;			// typ jednotky ktere data nazeli (Master nebo Slave)
	bool Unstored;			// indikace, ze data jeste nebyla ulozena na SD kartu
	int DataSize;			// velikost dat v poli Data
}DataToSD;

/* Struktura pro ulozeni informaci o kapacite SD karty*/
typedef struct{
	DWORD TotalKiB;			// celkova kapacita SD karty [KiB]
	DWORD FreeKiB;			// volna kapacita SD karty [KiB]
	float UsePER;			// vyuziti SD karty [%]
	DWORD UseKiB;			// vyuzita kapacita SD karty [KiB]
}UsageSD;

/* Struktura pro praci s modulem DCF77*/
typedef struct{
	uint8_t Data[59];			// buffer pro nacitani signalu DCF77
	uint8_t	DataCnt;			// pozice v bufferu (pozice volneho mista)
	bool DataValid;				// indikace	prijeti validnich dat
	bool MinuteStart;			// indikace zacatku minuty
	bool PinState;				// aktualni stav datoveho pinu DCF modulu (pin PC13)
	bool PrevPinState;			// predchozi stav datoveho pinu DCF modulu (pin PC13)
	uint32_t EdgeTS;			// casova znacka detekovane hrany (ms)
	uint32_t RaiseEdgeTS;		// casova znacka nastupne hrany (ms)
	uint32_t PrevRaiseEdgeTS;	// casova znacka predchozi nastupne hrany (ms)
	uint32_t FallEdgeTS;		// casova znacka sestupne hrany (ms)
	bool IsPulse;				// indikace pulsu (probiha zrovna pulz?)
	uint8_t Minute;				// minuty
	uint8_t Hour;				// hodiny
	uint8_t Day;				// den v mesici
	uint8_t Month;				// mesic
	uint8_t Year;				// rok
	uint8_t PrevMinute;			// minuty z predchozi casove znacky
	uint8_t PrevHour;			// hodiny z predchozi casove znacky
	uint8_t PrevDay;			// den v mesici z predchozi casove znacky
	uint8_t PrevMonth;			// mesic z predchozi casove znacky
	uint8_t PrevYear;			// rok z predchozi casove znacky
}DCF_t;
/* USER CODE END PTD */

/* Private define ------------------------------------------------------------*/
/* USER CODE BEGIN PD */

/* USER CODE END PD */

/* Private macro -------------------------------------------------------------*/
/* USER CODE BEGIN PM */

/* USER CODE END PM */

/* Private variables ---------------------------------------------------------*/
I2C_HandleTypeDef hi2c3;			// struktura I2C
RTC_HandleTypeDef hrtc;				// struktura RTC
SPI_HandleTypeDef hspi2;			// struktura SPI
SUBGHZ_HandleTypeDef hsubghz;		// struktura SUBGHZ pro radio
UART_HandleTypeDef huart1;			// struktura UART1
UART_HandleTypeDef huart2;			// struktura UART2 (pouzivano pro debug)

/* USER CODE BEGIN PV */
RadioEvents_t RadioEvents;			// struktura pro nastaveni Callback funkci pro radiovy modul

State_t State = STATE_WAIT;			// State - nasledujici stav stavoveho automatu (WAIT vychozi stav automatu)

MasterInit_t Master = {.SensorType = TYP_SNIMACE, .UnitName1 = VELICINA_1, .UnitName2 = VELICINA_2 };	 // inicializace nastaveni master jednotky (ostatni hodnoty na 0)
float MasterTemperature;			// namerena teplota master jednotkou
float MasterHumidity;		 		// namerena vlhkost master jednotkou

SlaveInit_t Slave[10] = {0}; 		// inicializace nastaveni slave jednotek (pocet slavu 10)

DataToSD SDunstoredData[11] = {0};	// pole struktur pro docasne ulozeni dat, pred jejich hromadnem zapisu na SD kartu (pocet prvku = pocet slave + master (10+1)

bool UnstoredData = false;			// indikace zda jsou k dispozici data (v SDunstoredData) pro zapsani na SD kartu
bool DataTransSDtoPC = false;		// indikace zda ma byt zahajen nebo probiha prenos dat z SD karty do PC

uint8_t SlaveCounter = 0;			// pocitadlo pro prochazeni slavu (urcuje ktery slave bude aktualne obslouzen)
uint32_t MasterDCWaitTime = 0;		// cas po ktery nemuze master odelat novou zpravu
uint32_t MasterDCTimeStamp = 0;		// casova znakca odeslani posledni zpavy mastrem (cas od kdy se pocita MasterDCWaitTime)

uint32_t MinPeriod_1vz_ms[6] = {	// perioda mereni slave od ktere probiha vycitani dat z jednotky nezavisle na zaplneni prenosoveho bufferu jednotky
		3000,		// pro range 0
	   10000,		// pro range 1
	   17000,		// pro range 2
	   33000,		// pro range 3
	   66000,		// pro range 4
	  132000		// pro range 5
};

uint32_t TimeSychDelayCorection_ms[6] = {  // korekcni casy pro casovou synch. jednotek pro jednotlive dosahy komunikace
		  39,		// pro range 0		   // korekce prodlevy pri synchronizaci casu jednotek (zpozdeni pri zpracovani (master/slave) a prenosu zpravy)
		  92,		// pro range 1
		 166,		// pro range 2
		 313,		// pro range 3
		 610,		// pro range 4
		1040		// pro range 5
};

RxMsgInfo_t RxMsgInfo = {0};			// informace o posleni prijate zprave pres LoRa radio
SystemTime_t SystemTime = {0};			// struktura systemoveho casu
SystemTime_t MeasureTimeStamp = {0};	// casova znacka mereni dat ze snimace

uint8_t RxBuffer[255];		// buffer pro ulozeni prijate zpravy pres LoRa radio
uint8_t TxBuffer[255];		// buffer pro odeslani zpravy pres LoRa radio
uint8_t TxSize = 0;			// delka zpravy pro odeslani v TxBuffer pres Lora radio (v bytech)

char UART_RxBuffer[400]; 	// buffer pro nacitani prichozich znaku pres UART
uint8_t pos = 0;		 	// pozice pro ulozeni dalsiho znaku do UART_RxBuffer
uint8_t znak;		     	// posledny nacteny znak pres UART
char UART_TxBuffer[400]; 	// buffer pro odeslani zpravy pres UART

FATFS SD_FatFs; 		 	// Objekt souboroveho systemu SD karty
char SDdiskPath[4]; 		// cesta k diskove jednotce (SD karte), (Drive path)

uint32_t RTC_CurTicks;		// casova znacka RTC (hodnota RTC tiku) pro ktere je platny systemovy cas ve strukture SystemTime
float MsRest = 0;			// uchovava desetinnou cast milisekund (informace o kolik je cas SystemTime zpozden vlivem zaokrouhleni na cele milisekundy)

bool RxConfirm = false;				// TRUE v pripade ze slave na odeslanou zpravu odpovedel, FALSE v pripade neprijeti odpovedi
uint8_t TransnRepCnt = 0;			// pocitadlo pokusu o spojeni se slave jednotkou

bool AlarmB_Running = false;		// indikace zda je RTC alarm B v provozu (probiha casovani mereni ze snimace master jednotkou)

UsageSD SDUsage = {0};				// informace o SD karte
bool PCconnected = false;			// indikace aktivniho pripojeni k PC pres USB (v PC spusten program k dataloggeru)

uint32_t StatusToPCTimestamp = 0;	// casova znacka posledniho odeslani zpravy Status do PC (odeslani info o signalu, baterii slave a info o SD karte do PC)
uint32_t StatusStoreTimestamp = 0;	// casova znacka kdy ma byt provedeno dalsi ulozeni Statusu na SD kartu (ulozeni informace o nastaveni, sile signalu, bateriich a SD karte na SD kartu)
uint32_t LastDCFsynchTimestamp = 0;	// casova znacka posledni provedene cas. synch. z modulu DFC77

DCF_t DCF = {0};					// inicializace strktury pro DCF77 modul

/* Format zapisu merenych jednotek do souboru
   Udava v jakem formatu budou merene jednotky zapsany do hlavicky v souboru
   (poradi zapisu musi odpovidat poradi v UnitName_t na radku 53) */
const char *UnitType[NumberOfUnitName] = {
    "Temperature[°C]",
    "Humidity[%]",
	"Pressure[kPa]",
};

/* Format zapisu nazvu snimace do souboru
   Udava v jakem formatu bude nazev snimace zapsan do hlavicky v souboru
   (poradi zapisu musi odpovidat poradi v SensorName_t na radku 69) */
const char *SensorType[NumberOfSensorName] = {
    "SHT30",
};



/* USER CODE END PV */

/* Private function prototypes -----------------------------------------------*/
void SystemClock_Config(void);
static void MX_GPIO_Init(void);
static void MX_USART2_UART_Init(void);
static void MX_I2C3_Init(void);
static void MX_SPI2_Init(void);
static void MX_USART1_UART_Init(void);
/* USER CODE BEGIN PFP */

/*Inicializace Callback funci Radioveho modulu*/
static void Radio_Init(void);
/*Funkce pro vychozi nastaveni parametru radioveho modulu*/
static void Radio_InitRxTxParams(void);
/* Funkce pro prenastaveni modulacnich parametru radioveho modulu podle nastaveni komuikace
   @param Range Nastaveni komunikace*/
static void Radio_SetRxTxParams(uint8_t Range);
/*Obsluha preruseni, vola se pri uspesnem odeslani zpravy radiovym modulem */
static void OnTxDone(void);
/* Obsluha preruseni, vola se pri uspesnem prijeti zpravy radiovym modulem
   @param payload Ukazatel na prijatou zpravu
   @param size Delka prijate zpravy v bytech
   @param rssi Rssi posledni prijate zpravy (sila prijateho singalu v dBm)
   @param LoraSnr_FskCfo SNR posledni prijate zpravy (SRN odstup signal sum) */
static void OnRxDone(uint8_t *payload, uint16_t size, int16_t rssi, int8_t LoraSnr_FskCfo);
/*Obsluha preruseni, vola se pokud se radiovemu modulu nepodarilo zpravu uspesne odeslat v nastavenem intervalu 10s*/
static void OnTxTimeout(void);
/*Obsluha preruseni, vola se pokud v nastavenem intervalu prijimani zpravy nebyla prijata zadna zprava radiovym modeulem*/
static void OnRxTimeout(void);
/*Obsluha preruseni, vola se pokud byla radiovym modulem prijata poskozena zprava (chyba v hlavicce, chybny CRC kod)*/
static void OnRxError(void);

static void State_SEND_Subroutine(void);					//podprogram pro stav SEND
static void State_TXdone_Subroutine(void);					//podprogram pro stav TXdone
static void State_TXtimeout_Subroutine(void);				//podprogram pro stav TXtimeout
static void State_RXerr_Subroutine(void);					//podprogram pro stav RXtimout a RXerror
static void State_RXdone_Subroutine(void);					//podprogram pro stav RXdone
static void State_Start_measure_Subroutine(void);			//podprogram pro stav Start_measure
static void State_Stop_measure_Subroutine(void);			//podprogram pro stav Stop_measure
static void State_WAIT_Subroutine(void);					//podprogram pro stav Wait

/* Funkce pro zjisteni poctu dni v mesici
   @param Month Cislo mesice
   @param Year Rok
   @return Pocet dni v mesici*/
static uint8_t DaysInMonth(uint8_t Month, uint8_t Year);
/* Funkce pro aktualizaci systemoveho casu SystemTime podle RTC  */
static void SysTimeUpdate(void);
/* Funkce pro zpracovani prichozi zpravy pres UART
   @param buffer Ukazatel na prijatou zpravu pres UART*/
static void UartReadProcess(char *buffer);
/* Funkce pro ulozeni nastaveni dataloggeru do pameti FLASH*/
static void StoreToFLASH(void);
/* Funkce pro nacteni ulozeneho nastavení dataloggeru z pameti FLASH*/
static void LoadFromFLASH(void);
/* Funkce pro preposlani obsahu SD karty pres rozharani UART do PC*/
static void LoadFromSDtoPC(void);
/* Funkce pro mereni externim snimacem*/
static void Sensor_measure(void);
/* Funkce pro pricteni casu ke strukture a jeho konverze (zarovnani)
   @param Time Ukazatel na casovou strukturu, ke ktere ma byt cas pricten
   @param AddTimeMS Cas v milisekundach, ktery ma byt pricten k casove strukture*/
static void AddTime(SystemTime_t *Time, uint32_t AddTimeMS);
/*Funkce pro ulozeni dat na SD kartu, dojde k zapisu obsahu SDunstoredData na SD kartu*/
static void StoreToSDcard(void);
/* Callback funkce pro odeslani bloku dat pres rozhrani UART do PC (volano funkci f_forward)
   @param data Ukazatel na blok dat
   @param size Velikost bloku dat
   @return UINT Pocet odeslanych bytu dat*/
static UINT Out_SD_Stream(const BYTE* data, UINT size);
/* Funkce pro vymazani ulozenych dat na SD karte*/
static void DeleteSDcard(void);
/* Funkce pro udeslani zpravy Status pres UART do PC*/
static void SendStatusMsg(void);
/* Inicializace SD karty*/
static bool SD_Init(void);
/* Funkce pro vypocet celkove kapacity a zaplneni SD karty
   @return true, pokud vypocet byl uspesny, false v pripadě chyby*/
static bool SD_CardInfo(void);
/* Funkce pro zapsani Statusu na SD kartu*/
static void StoreStatusToSD(void);
/* Funkce pro resetovani nastaveni dataloggeru*/
static void RessetSettings(void);
/* Funkce pro zpracovani signalu DCF77*/
static void DCF_Process(void);
/* Funkce pro vypocet casu, kdy neni mozne odeslat dalsi zpravu
   @param Range Nastaveni dosahu (pro jake parametry modulace ma byt cas vypocten)
   @param size Delka odeslane zpravy v bytech
   @param per DutyCycle v procentech (vyuziva se 100 a 99, 100 pro vypocet pred odeslanim zpravy, 99 po odeslani zpravy)
   @return Cas v milisekundach, za ktery je mozne odeslat dalsi zpravu*/
uint32_t CalcDutyCycle(uint8_t Range, uint8_t size, uint8_t per);

/* USER CODE END PFP */

/* Private user code ---------------------------------------------------------*/
/* USER CODE BEGIN 0 */

/*Predefinovani weak funkce _write pro moznost pouziti printf pro odeslani dat pres UART
  @param file Identifikator vystupu (nevyuzito)
  @param buf Ukazatel na data
  @param size Delka dat v bytech
  @return Pocet odeslanych bytu*/
int _write(int file, char *buf, int size){
		HAL_UART_Transmit(&huart1, (uint8_t*)buf, size, 100);	//odeslani dat pres UART
	return size;
}

/* USER CODE END 0 */

/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
  /* USER CODE BEGIN 1 */

  /* USER CODE END 1 */

  /* MCU Configuration--------------------------------------------------------*/

  /* Reset of all peripherals, Initializes the Flash interface and the Systick. */
  HAL_Init();

  /* USER CODE BEGIN Init */

  /* USER CODE END Init */

  /* Configure the system clock */
  SystemClock_Config();

  /* USER CODE BEGIN SysInit */

  /* USER CODE END SysInit */

  /* Initialize all configured peripherals */
  MX_GPIO_Init();
  MX_USART2_UART_Init();
  MX_I2C3_Init();
  MX_SPI2_Init();
  MX_USART1_UART_Init();
  /* USER CODE BEGIN 2 */

  /* Ensure that MSI is wake-up system clock */
  __HAL_RCC_WAKEUPSTOP_CLK_CONFIG(RCC_STOP_WAKEUPCLOCK_MSI);

  UTIL_TIMER_Init();  			// inicializace soft. timeru a RTC
  Radio_Init();					// nastaveni callback funkci pro preruseni od radioveho modulu
  Radio_InitRxTxParams();		// nastaveni vychozich parametru modulace radioveho modulu

  RTC_CurTicks = UTIL_TimerDriver.GetTimerValue();				// Ulozeni hodnoty casovace RTC (pocet tiku)
  StatusToPCTimestamp = UTIL_TIMER_GetCurrentTime();			// Ulozeni casove znacky RTC v milisekundach
  StatusStoreTimestamp = UTIL_TIMER_GetCurrentTime() + 300000;	// Ulozeni casove znacky RTC v milisekundach + 5 minut (prvni ulozeni statusu az po 5 min po startu, aby byly uz nacteny vsechny informace ze Slavu)

  HAL_UART_Receive_IT(&huart1, &znak, 1);	// Nastaveni vzniku preruseni od UART po prijeti jednoho znaku

  //inicializace SD karty
  if(!SD_Init())
	  {
	  printf("SD init Error\r\n");
	  Error_Handler();				//v pripade neuspesne inicializace SD karty nelze ukladat data -> porucha
	  }

  LoadFromFLASH();		//nacteni aktualniho nastaveni dataloggeru z pameti FLASH

  HAL_GPIO_WritePin(GPIOA, GPIO_PIN_15, GPIO_PIN_RESET);	//aktivace modulu DCF77

  __HAL_RTC_WRITEPROTECTION_DISABLE(&hrtc);
  hrtc.Instance->CALR = 0b000000000000000001000000111010010;	//nataveni kalibracniho registru RTC
  __HAL_RTC_WRITEPROTECTION_ENABLE(&hrtc);

  /* USER CODE END 2 */

  /* Infinite loop */
  /* USER CODE BEGIN WHILE */

  /* Hlavni smycka celeho programu loggovaci jednotky (implementace stavoveho automatu) */
  while (1)
  {

	  switch (State)			//stavovy automat (vyber stavu)
	  {

	  case STATE_SEND:			//stav pro odeslani zpravy senzoricke jednotce, prechod do stavu WAIT
		  State_SEND_Subroutine();
		  break;

	  case STATE_TXDONE:		//pri uspesnem odeslani zpravy, prepnuti radia do prijimaciho rezimu, prechod do stavu WAIT
		  State_TXdone_Subroutine();
		  break;

	  case STATE_TXTIMEOUT:		//pri neuspesnem odeslani zpravy, prechod do stavu SEND
		  State_TXtimeout_Subroutine();
		  break;

	  case STATE_RXERROR:		//pri neuspesnem prijeti zpravy, prechod do stavu WAIT
	  case STATE_RXTIMEOUT:
		  State_RXerr_Subroutine();
		  break;

	  case STATE_RXDONE:		//pri uspesnem prijeti zpravy, kontrola a zpracovani zpravy, prechod do stavu WAIT
		  State_RXdone_Subroutine();
		  break;

	  case STATE_START_MASTER_MEASURE:		//spusteni periodickeho mereni loggovaci jednotky, prechod do stavu WAIT
		  State_Start_measure_Subroutine();
		  break;

	  case STATE_STOP_MASTER_MEASURE:		//deaktivace periodickeho mereni loggovaci jednotky, prechod do stavu WAIT
		  State_Stop_measure_Subroutine();
		  break;

	  case STATE_WAIT:			//cekani loggovaci jednotky, nez muze byt provedena nejaka akce
		  State_WAIT_Subroutine();
		  break;
	  }

    /* USER CODE END WHILE */

    /* USER CODE BEGIN 3 */
  }
  /* USER CODE END 3 */
}

/**
  * @brief System Clock Configuration
  * @retval None
  */
void SystemClock_Config(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /** Configure LSE Drive Capability
  */
  HAL_PWR_EnableBkUpAccess();
  __HAL_RCC_LSEDRIVE_CONFIG(RCC_LSEDRIVE_LOW);

  /** Configure the main internal regulator output voltage
  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

  /** Initializes the CPU, AHB and APB buses clocks
  */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI|RCC_OSCILLATORTYPE_LSE
                              |RCC_OSCILLATORTYPE_MSI;
  RCC_OscInitStruct.LSEState = RCC_LSE_ON;
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;
  RCC_OscInitStruct.MSIState = RCC_MSI_ON;
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.MSICalibrationValue = RCC_MSICALIBRATION_DEFAULT;
  RCC_OscInitStruct.MSIClockRange = RCC_MSIRANGE_11;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_NONE;
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure the SYSCLKSource, HCLK, PCLK1 and PCLK2 clocks dividers
  */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK3|RCC_CLOCKTYPE_HCLK
                              |RCC_CLOCKTYPE_SYSCLK|RCC_CLOCKTYPE_PCLK1
                              |RCC_CLOCKTYPE_PCLK2;
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_MSI;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.AHBCLK3Divider = RCC_SYSCLK_DIV1;

  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_2) != HAL_OK)
  {
    Error_Handler();
  }
}

/**
  * @brief I2C3 Initialization Function
  * @param None
  * @retval None
  */
static void MX_I2C3_Init(void)
{

  /* USER CODE BEGIN I2C3_Init 0 */

  /* USER CODE END I2C3_Init 0 */

  /* USER CODE BEGIN I2C3_Init 1 */

  /* USER CODE END I2C3_Init 1 */
  hi2c3.Instance = I2C3;
  hi2c3.Init.Timing = 0x20303E5D;
  hi2c3.Init.OwnAddress1 = 0;
  hi2c3.Init.AddressingMode = I2C_ADDRESSINGMODE_7BIT;
  hi2c3.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;
  hi2c3.Init.OwnAddress2 = 0;
  hi2c3.Init.OwnAddress2Masks = I2C_OA2_NOMASK;
  hi2c3.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;
  hi2c3.Init.NoStretchMode = I2C_NOSTRETCH_DISABLE;
  if (HAL_I2C_Init(&hi2c3) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure Analogue filter
  */
  if (HAL_I2CEx_ConfigAnalogFilter(&hi2c3, I2C_ANALOGFILTER_ENABLE) != HAL_OK)
  {
    Error_Handler();
  }

  /** Configure Digital filter
  */
  if (HAL_I2CEx_ConfigDigitalFilter(&hi2c3, 0) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN I2C3_Init 2 */

  /* USER CODE END I2C3_Init 2 */

}

/**
  * @brief RTC Initialization Function
  * @param None
  * @retval None
  */
void MX_RTC_Init(void)
{

  /* USER CODE BEGIN RTC_Init 0 */

  /* USER CODE END RTC_Init 0 */

  RTC_AlarmTypeDef sAlarm = {0};

  /* USER CODE BEGIN RTC_Init 1 */

  /* USER CODE END RTC_Init 1 */

  /** Initialize RTC Only
  */
  hrtc.Instance = RTC;
  hrtc.Init.AsynchPrediv = RTC_PREDIV_A;
  hrtc.Init.OutPut = RTC_OUTPUT_DISABLE;
  hrtc.Init.OutPutRemap = RTC_OUTPUT_REMAP_NONE;
  hrtc.Init.OutPutPolarity = RTC_OUTPUT_POLARITY_HIGH;
  hrtc.Init.OutPutType = RTC_OUTPUT_TYPE_OPENDRAIN;
  hrtc.Init.OutPutPullUp = RTC_OUTPUT_PULLUP_NONE;
  hrtc.Init.BinMode = RTC_BINARY_ONLY;
  if (HAL_RTC_Init(&hrtc) != HAL_OK)
  {
    Error_Handler();
  }

  /* USER CODE BEGIN Check_RTC_BKUP */

  /* USER CODE END Check_RTC_BKUP */

  /** Initialize RTC and set the Time and Date
  */
  if (HAL_RTCEx_SetSSRU_IT(&hrtc) != HAL_OK)
  {
    Error_Handler();
  }

  /** Enable the Alarm A
  */
  sAlarm.BinaryAutoClr = RTC_ALARMSUBSECONDBIN_AUTOCLR_NO;
  sAlarm.AlarmTime.SubSeconds = 0x0;
  sAlarm.AlarmMask = RTC_ALARMMASK_NONE;
  sAlarm.AlarmSubSecondMask = RTC_ALARMSUBSECONDBINMASK_NONE;
  sAlarm.Alarm = RTC_ALARM_A;
  if (HAL_RTC_SetAlarm_IT(&hrtc, &sAlarm, 0) != HAL_OK)
  {
    Error_Handler();
  }

  /** Enable the Alarm B
  */
  sAlarm.Alarm = RTC_ALARM_B;
  if (HAL_RTC_SetAlarm_IT(&hrtc, &sAlarm, 0) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN RTC_Init 2 */

  /* USER CODE END RTC_Init 2 */

}

/**
  * @brief SPI2 Initialization Function
  * @param None
  * @retval None
  */
static void MX_SPI2_Init(void)
{

  /* USER CODE BEGIN SPI2_Init 0 */

  /* USER CODE END SPI2_Init 0 */

  /* USER CODE BEGIN SPI2_Init 1 */

  /* USER CODE END SPI2_Init 1 */
  /* SPI2 parameter configuration*/
  hspi2.Instance = SPI2;
  hspi2.Init.Mode = SPI_MODE_MASTER;
  hspi2.Init.Direction = SPI_DIRECTION_2LINES;
  hspi2.Init.DataSize = SPI_DATASIZE_8BIT;
  hspi2.Init.CLKPolarity = SPI_POLARITY_LOW;
  hspi2.Init.CLKPhase = SPI_PHASE_1EDGE;
  hspi2.Init.NSS = SPI_NSS_SOFT;
  hspi2.Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_64;
  hspi2.Init.FirstBit = SPI_FIRSTBIT_MSB;
  hspi2.Init.TIMode = SPI_TIMODE_DISABLE;
  hspi2.Init.CRCCalculation = SPI_CRCCALCULATION_DISABLE;
  hspi2.Init.CRCPolynomial = 7;
  hspi2.Init.CRCLength = SPI_CRC_LENGTH_DATASIZE;
  hspi2.Init.NSSPMode = SPI_NSS_PULSE_ENABLE;
  if (HAL_SPI_Init(&hspi2) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN SPI2_Init 2 */

  /* USER CODE END SPI2_Init 2 */

}

/**
  * @brief SUBGHZ Initialization Function
  * @param None
  * @retval None
  */
void MX_SUBGHZ_Init(void)
{

  /* USER CODE BEGIN SUBGHZ_Init 0 */

  /* USER CODE END SUBGHZ_Init 0 */

  /* USER CODE BEGIN SUBGHZ_Init 1 */

  /* USER CODE END SUBGHZ_Init 1 */
  hsubghz.Init.BaudratePrescaler = SUBGHZSPI_BAUDRATEPRESCALER_4;
  if (HAL_SUBGHZ_Init(&hsubghz) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN SUBGHZ_Init 2 */

  /* USER CODE END SUBGHZ_Init 2 */

}

/**
  * @brief USART1 Initialization Function
  * @param None
  * @retval None
  */
static void MX_USART1_UART_Init(void)
{

  /* USER CODE BEGIN USART1_Init 0 */

  /* USER CODE END USART1_Init 0 */

  /* USER CODE BEGIN USART1_Init 1 */

  /* USER CODE END USART1_Init 1 */
  huart1.Instance = USART1;
  huart1.Init.BaudRate = 460800;
  huart1.Init.WordLength = UART_WORDLENGTH_8B;
  huart1.Init.StopBits = UART_STOPBITS_1;
  huart1.Init.Parity = UART_PARITY_NONE;
  huart1.Init.Mode = UART_MODE_TX_RX;
  huart1.Init.HwFlowCtl = UART_HWCONTROL_NONE;
  huart1.Init.OverSampling = UART_OVERSAMPLING_16;
  huart1.Init.OneBitSampling = UART_ONE_BIT_SAMPLE_DISABLE;
  huart1.Init.ClockPrescaler = UART_PRESCALER_DIV1;
  huart1.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
  if (HAL_UART_Init(&huart1) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_SetTxFifoThreshold(&huart1, UART_TXFIFO_THRESHOLD_1_8) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_SetRxFifoThreshold(&huart1, UART_RXFIFO_THRESHOLD_1_8) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_DisableFifoMode(&huart1) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN USART1_Init 2 */

  /* USER CODE END USART1_Init 2 */

}

/**
  * @brief USART2 Initialization Function
  * @param None
  * @retval None
  */
static void MX_USART2_UART_Init(void)
{

  /* USER CODE BEGIN USART2_Init 0 */

  /* USER CODE END USART2_Init 0 */

  /* USER CODE BEGIN USART2_Init 1 */

  /* USER CODE END USART2_Init 1 */
  huart2.Instance = USART2;
  huart2.Init.BaudRate = 460800;
  huart2.Init.WordLength = UART_WORDLENGTH_8B;
  huart2.Init.StopBits = UART_STOPBITS_1;
  huart2.Init.Parity = UART_PARITY_NONE;
  huart2.Init.Mode = UART_MODE_TX_RX;
  huart2.Init.HwFlowCtl = UART_HWCONTROL_NONE;
  huart2.Init.OverSampling = UART_OVERSAMPLING_16;
  huart2.Init.OneBitSampling = UART_ONE_BIT_SAMPLE_DISABLE;
  huart2.Init.ClockPrescaler = UART_PRESCALER_DIV1;
  huart2.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
  if (HAL_UART_Init(&huart2) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_SetTxFifoThreshold(&huart2, UART_TXFIFO_THRESHOLD_1_8) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_SetRxFifoThreshold(&huart2, UART_RXFIFO_THRESHOLD_1_8) != HAL_OK)
  {
    Error_Handler();
  }
  if (HAL_UARTEx_DisableFifoMode(&huart2) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN USART2_Init 2 */

  /* USER CODE END USART2_Init 2 */

}

/**
  * @brief GPIO Initialization Function
  * @param None
  * @retval None
  */
static void MX_GPIO_Init(void)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};
/* USER CODE BEGIN MX_GPIO_Init_1 */
/* USER CODE END MX_GPIO_Init_1 */

  /* GPIO Ports Clock Enable */
  __HAL_RCC_GPIOA_CLK_ENABLE();
  __HAL_RCC_GPIOB_CLK_ENABLE();
  __HAL_RCC_GPIOC_CLK_ENABLE();

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOA, GPIO_PIN_15, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOB, LED1_Pin|SPI2_CS_Pin|LED2_Pin|LED3_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOC, FE_CTRL3_Pin|FE_CTRL2_Pin|FE_CTRL1_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pin : PA15 */
  GPIO_InitStruct.Pin = GPIO_PIN_15;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /*Configure GPIO pins : LED1_Pin LED2_Pin LED3_Pin */
  GPIO_InitStruct.Pin = LED1_Pin|LED2_Pin|LED3_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
  HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

  /*Configure GPIO pin : SPI2_CS_Pin */
  GPIO_InitStruct.Pin = SPI2_CS_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_PULLUP;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(SPI2_CS_GPIO_Port, &GPIO_InitStruct);

  /*Configure GPIO pin : PC13 */
  GPIO_InitStruct.Pin = GPIO_PIN_13;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pins : FE_CTRL3_Pin FE_CTRL2_Pin FE_CTRL1_Pin */
  GPIO_InitStruct.Pin = FE_CTRL3_Pin|FE_CTRL2_Pin|FE_CTRL1_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pins : B1_Pin B2_Pin */
  GPIO_InitStruct.Pin = B1_Pin|B2_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_IT_FALLING;
  GPIO_InitStruct.Pull = GPIO_PULLUP;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /*Configure GPIO pin : B3_Pin */
  GPIO_InitStruct.Pin = B3_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_IT_FALLING;
  GPIO_InitStruct.Pull = GPIO_PULLUP;
  HAL_GPIO_Init(B3_GPIO_Port, &GPIO_InitStruct);

/* USER CODE BEGIN MX_GPIO_Init_2 */
/* USER CODE END MX_GPIO_Init_2 */
}

/* USER CODE BEGIN 4 */



//************ Zacatek podprogramu stavoveho automatu **********************

/*Podprogram pro stav SEND
 * Slouzi pro vyber a odeslani zpravy slave jednotce*/
static void State_SEND_Subroutine(void)
{
	  if (SlaveCounter == sizeof(Slave) / sizeof(Slave[0])) //kontrola konce pole Slave (zda byly projeti vsechny slave jednotky)
	  {
		  SlaveCounter = 0;			//resetuje slave counter
		  RxConfirm = false;		//resetuje potrvzeni prijeti zpravy
		  State = STATE_WAIT;		//prejde do stavu WAIT
		  return;
	  }

	  if (!Slave[SlaveCounter].Enable)		//kontrola jestli je povoleno vycitani slave jednotky
	  {
		  ++SlaveCounter;					//pokud ne tak jde na dalsiho slave
		  return;
	  }


	  if (  ((TransnRepCnt == 3) || RxConfirm))	//kontrola 3 pokusu o spojeni se slave, nebo pokud prislo potrvrzení od slava
	  {
		  if(TransnRepCnt == 3)					//pokud 3 po sobe jdouci pokusy o spojení selzou
			  Slave[SlaveCounter].RSSI = 0;		//vypadek spojeni
		  TransnRepCnt = 0;						//resetovani pocitadla pokusu o spojeni
		  Slave[SlaveCounter].Msg = NONE;		//resetovani nastavene zpravy pro odeslani
		  RxConfirm = false;					//resetovani potvrzeni od slava
		  ++SlaveCounter;						//prechod na dalsi slave jednotku
		  return;
	  }

	  //kontrola zda muze slave poslat dalsi zpravu (dutycycle time), uplynul cas, kdy slave nemuze vysilat
	  if (UTIL_TIMER_GetElapsedTime(Slave[SlaveCounter].DCTimeStamp) < Slave[SlaveCounter].DCWaitTime)
	  {
		  ++SlaveCounter;						//pokud ne, tak jde na dalsiho slave
		  return;
	  }


	  //kontrola, zda ma byt provedena casova synchronizace slave jednotky podle nastaveni uzivatele
	  if (UTIL_TIMER_GetElapsedTime(Slave[SlaveCounter].LastTimeSynchTimeStap) >= (Master.TimeSynchInterval * 3600000))	//pokud cas od posledni cas. synch je >= nastavenemu intervalu cas. synch.
	  {		//nema smysl provadet cas. synch. napriklad kazdou hodinu pokud jednotka meri jednou za den
		  if ((UTIL_TIMER_GetElapsedTime(Slave[SlaveCounter].LastReadTimeStamp) + 120000) > Slave[SlaveCounter].MeasurePeriod) //synchronizace jednotky se tedy provede az nejdrive 2 minuty pred samotnym merenim dat jednotkou
		  {																													   //na intervaly mereni pod 2 minuty nema vliv (pro tyto intervaly se cas. synch. provede ihned)
			  Slave[SlaveCounter].TimeSynch = true;				//indikace, ze ma byt pro tuto jednotku provedena casova synchronizace
		  }
	  }


	  if (Slave[SlaveCounter].Msg == NONE)						//vybez typu zpravy pro odeslani slave
	  {															//priorita (1 nejvyssi, 4 nejnizsi)
		  if (Slave[SlaveCounter].TimeSynch)					//my byt provedena casova synchronizace
			  Slave[SlaveCounter].Msg = TimeSynch;				//nastaveni zpravy casove synchronizace (priorita 1)
		  else if (Slave[SlaveCounter].SettingChange)			//ma byt provedena zmena nastaveni slave
			  Slave[SlaveCounter].Msg = SendSettings;			//nastaveni zpravy nastaveni (priorita 2)
		  else if ((Slave[SlaveCounter].LastStatusTimeStamp + 3600000) <= UTIL_TIMER_GetCurrentTime()) //uplynula hodina od posleni informace o bateii slave (jednou za hodinu),  vycist data + status
			  Slave[SlaveCounter].Msg = Status_MeasRequest;		//nastaveni zpravy Status + namerena data (priorita 3)
		  else
			  Slave[SlaveCounter].Msg = MeasureRequest;			//nastaveni zpravy vycteni dat (priorita 4)
	  }

	  //kontrola zaplneni prenosoveho bufferu slave jednotky
	  if ((Slave[SlaveCounter].Msg == MeasureRequest) || (Slave[SlaveCounter].Msg == Status_MeasRequest))  //zaplneni se kontroluje pouze pro zpravy s vyctem dat
	  {
		  if (Slave[SlaveCounter].MeasurePeriod < MinPeriod_1vz_ms[Slave[SlaveCounter].CurRange])	//ANO pokud prenos probiha zavisle na zaplneni prenosovehu bufferu
		  {
			  if ((Slave[SlaveCounter].MeasurePeriod * 16) > UTIL_TIMER_GetElapsedTime(Slave[SlaveCounter].LastReadTimeStamp)) //je zaplnena alespon polovina prenosoveho bufferu? (probehlo uz alespon 16 mereni od posledniho vycteni dat?)
			  {										//pokud NE
				  Slave[SlaveCounter].Msg = NONE;	//resetuje nastavenou zpravu (protoze se nema provest)
				  ++SlaveCounter;					//prechod na dalsiho slave
				  return;
			  }
		  }
		  else if(Slave[SlaveCounter].MeasurePeriod > UTIL_TIMER_GetElapsedTime(Slave[SlaveCounter].LastReadTimeStamp))	//pokud prenos probiha nezavisle na zaplneni prenosoveho bufferu slave, tak musi byt nameren alespon jeden vzorek dat
		  {										//pokud neni nameren jeste zadny vzorek dat tak nevyzadat data (slave jeste nema nic co by mohl poslat)
			  Slave[SlaveCounter].Msg = NONE;	//resetuje nastavenou zpravu (protoze se nema provest)
			  ++SlaveCounter;					//prechod na dalsiho slave
			  return;
		  }
	  }


	  switch(Slave[SlaveCounter].Msg)	//sestaveni zpravy pro odeslani podle predchoziho vyberu typu zpravy
	  {
	  case TimeSynch:					//sestaveni zpravy casove synchroniazce
		  printf("------ Posílám Synchonizaci pro SlaveID: %u ------\r\n",Slave[SlaveCounter].ID);		//pro debug, uzivateli jinak skryto
		  TxBuffer[0] = Slave[SlaveCounter].ID;		//vlozeni ID (adresy) senzoricke jednotky
		  TxBuffer[1] = (uint8_t)TimeSynch;			//vlozeni typu zpravy
		  SysTimeUpdate();							//aktualizace systemoveho casu
		  SystemTime_t TimeTmp = SystemTime;		//porizeni kopie aktualniho systemoveho casu
		  AddTime(&TimeTmp, TimeSychDelayCorection_ms[Slave[SlaveCounter].CurRange]);	//(pricteni korekcniho casu) korekce zpozdeni casove znacky pri zpracovani a prenosu zpravy
		  memcpy(&TxBuffer[2],&TimeTmp, 6);			//vlozeni 1. casti finalni casove znacky (rok, mesic, den, hodina, minuta, sekunda)
		  memcpy(&TxBuffer[8],&TimeTmp.Milliseconds, 2); //vlozeni 2.casti finalni casove znacky (milisekundy), (zvlast kvuli problemu se zarovnanim struktury)
		  TxSize = 10;								//nastaveni delky odesilane zpravy
		  break;

	  case MeasureRequest:				//sestaveni zpravy pozadavku na vycet namerenych dat
		  printf("------ Posílám Request pro SlaveID: %u ------\r\n",Slave[SlaveCounter].ID);		//pro debug, uzivateli jinak skryto
		  TxBuffer[0] = Slave[SlaveCounter].ID;		//vlozeni ID (adresy) senzoricke jednotky
		  TxBuffer[1] = (uint8_t)MeasureRequest;	//vlozeni typu zpravy
		  TxSize = 2;								//nastaveni delky odesilane zpravy
		  break;

	  case Status_MeasRequest:			//sestaveni zpravy pozadavku na vycet namerenych dat + zaslani status (stav baterie)
		  printf("------ Posílám Stat_MeasRequest pro SlaveID: %u ------\r\n",Slave[SlaveCounter].ID);	//pro debug, uzivateli jinak skryto
		  TxBuffer[0] = Slave[SlaveCounter].ID;			//vlozeni ID (adresy) senzoricke jednotky
		  TxBuffer[1] = (uint8_t)Status_MeasRequest;	//vlozeni typu zpravy
		  TxSize = 2;									//nastaveni delky odesilane zpravy
		  break;

	  case SendSettings:				//sestaveni zpravy nastaveni slave jednotky
		  printf("------ Posílám Settings pro SlaveID: %u ------\r\n",Slave[SlaveCounter].ID);		//pro debug, uzivateli jinak skryto
		  TxBuffer[0] = Slave[SlaveCounter].ID;		//vlozeni ID (adresy) senzoricke jednotky
		  TxBuffer[1] = (uint8_t)SendSettings;		//vlozeni typu zpravy
		  TxBuffer[2] = Master.SettedRange;			//vlozeni noveho nastaveni dosahu (parametru modulace)
		  memcpy(&TxBuffer[3],&Slave[SlaveCounter].MeasurePeriod,4);	//vlozeni nove periody mereni
		  TxSize = 7;								//nastaveni delky odesilane zpravy
		  break;
	  }

	  ++TransnRepCnt;			//vzyseni pocitadla pokusu o spojeni
	  RxConfirm = false;		//deaktivuje potvrzeni prijeti zpravy od slave
	  Radio_SetRxTxParams(Slave[SlaveCounter].CurRange);	//nastavi pozadovane modulace, kterou aktualne slave pouziva
	  Radio.Send(TxBuffer,TxSize);							//odeslani sestavene zpravy pres radiovy modul
	  MasterDCTimeStamp = UTIL_TIMER_GetCurrentTime();		//porizeni casove znacky odeslani zpravy
	  MasterDCWaitTime = CalcDutyCycle(Slave[SlaveCounter].CurRange, TxSize, 100);		//vypocet doby cekani, nez muze byt odeslana dalsi zprava
	  printf("TIMEWAIT: %lu\r\n",CalcDutyCycle(Slave[SlaveCounter].CurRange, TxSize, 100));		//pro debug, uzivateli jinak skryto
	  State	= STATE_WAIT;		//prechod do stavu WAIT
	  return;
}

/*Podprogram pro stav TXdone*/
static void State_TXdone_Subroutine(void)
{
	  Radio.Rx(3000);		//prepnuti radia do prijimaciho rezimu po dobu 3 sekund
	  State = STATE_WAIT;	//prechod do stavu WAIT
	  return;
}
/*Podprogram pro stav TXtimeout*/
static void State_TXtimeout_Subroutine(void)
{
	  State = STATE_SEND;	//prechod do stavu SEND (opetovny pokus o odeslani zpravy)
	  return;
}

/*Podprogram pro stav RXerror a RXtimeout*/
static void State_RXerr_Subroutine(void)
{
	  printf("RXtimeout nebo RXerror\r\n");		//pro debug, uzivateli jinak skryto
	  Slave[SlaveCounter].RSSI = 0;				//vypadek spojeni (reset sily signalu jednotky)
	  State = STATE_WAIT;						//prechod do stavu WAIT
	  return;
}

/*Podprogram pro stav RXdone*/
static void State_RXdone_Subroutine(void)
{
	if (Slave[SlaveCounter].ID == RxBuffer[0])		//kontrola zda zpravu poslal spravny slave (kontrola ID (adresy) v prijate zprave)
	{
		Slave[SlaveCounter].DCTimeStamp = UTIL_TIMER_GetCurrentTime();			//porizeni casove znacky prijeti zpravy
		Slave[SlaveCounter].DCWaitTime = CalcDutyCycle(Slave[SlaveCounter].CurRange, RxMsgInfo.Size, 99);		//vypocet doby cekani, kdy slave nemuze poslat dalsi zpravu (dutycycle)
		Slave[SlaveCounter].RSSI = RxMsgInfo.RSSI;	//ulozeni sily prijateho signalu
		printf("Slave %u wait: %lu\r\n",Slave[SlaveCounter].ID, Slave[SlaveCounter].DCWaitTime); //pouze pro debug, pro uzivalete skryto
		RxConfirm = true;							//potvrzeni prijeti zpravy
		switch(Slave[SlaveCounter].Msg)				//zpracovani prijate zpravy podle typu odeslaneho pozadavku
		{
		case TimeSynch:								//prijeti potvrzeni casove synchronizace
			Slave[SlaveCounter].TimeSynch = false;	//deaktivace pozadavku na casovou synchronizaci slava
			Slave[SlaveCounter].LastTimeSynchTimeStap = UTIL_TIMER_GetCurrentTime(); //porizeni casove znacky provedeni posledni casove synchronizace
			break;
		case SendSettings:							//prijeti odpovedi na zpavu nastaveni
			Slave[SlaveCounter].LastReadTimeStamp = UTIL_TIMER_GetCurrentTime();	//porizeni casove znacky prijeti posledniho vyctu dat (protoze pri zmene nastaveni se mazou namerana data ve slave)
			Slave[SlaveCounter].SettingChange = false;			//deaktivace pozadavku nastaveni slava
			Slave[SlaveCounter].CurRange = Master.SettedRange;	//ulozeni aktualne pouzivane modulace pro tohoto slava
			Slave[SlaveCounter].BatteryLvl = RxBuffer[1];		//ulozeni stavu baterie
			Slave[SlaveCounter].SensorType = RxBuffer[2];		//ulozeni typu snimace
			Slave[SlaveCounter].UnitName1 = RxBuffer[3];		//ulozeni jednotky 1
			Slave[SlaveCounter].UnitName2 = RxBuffer[4];		//ulozeni jednotky 2
			Slave[SlaveCounter].LastStatusTimeStamp = Slave[SlaveCounter].LastReadTimeStamp;	//casova znacka posleni zpravy se statusem (stavem baterie)
			StoreToFLASH();										//ulozeni nastaveni na FLASH (ulozeni nove konfigurace slava)
			break;
		case MeasureRequest:		//prijeti namerenych dat
		case Status_MeasRequest:	//prijeti namerenych dat + status (stav baterie)
			Slave[SlaveCounter].LastReadTimeStamp = UTIL_TIMER_GetCurrentTime();	//porizeni casove znacky prijeti posledniho vyctu dat ze slave jednotky
			uint8_t i;						//promenna pro posunuti na zacatek namerenych dat v bufferu
			SystemTime_t TimeStamp_tmp;
			if(Slave[SlaveCounter].Msg == Status_MeasRequest)		//zprava obsahuje status (info o baterii)
			{
				Slave[SlaveCounter].BatteryLvl = RxBuffer[1];		//ulozeni stavu baterie
				i = 10;												//posunuti o velikost adresy, stavu baterie a casove znacky v bufferu na zacatek namerenych dat dat
				memcpy(&TimeStamp_tmp, RxBuffer+2, sizeof(SystemTime_t));	//ulozeni casove znacky ze zpravy (posun v prijate zprave o 2 (ID,baterie)
				Slave[SlaveCounter].LastStatusTimeStamp = Slave[SlaveCounter].LastReadTimeStamp;	//casova znacka posleni zpravy se statusem (stavem baterie)
			}
			else				//pro MeasureRequest (zprava neobsahuje status (info o baterii))
			{
				i = 9;			//posunuti o velikost adresy a casove znacky v bufferu na zacatek namerenych dat
				memcpy(&TimeStamp_tmp, RxBuffer+1, sizeof(SystemTime_t));	//ulozeni casove znacky ze zpravy (posun v prijate zprave o 1 (ID)
			}

			if (((Slave[SlaveCounter].Msg == MeasureRequest) && (RxMsgInfo.Size != 1)) || ((Slave[SlaveCounter].Msg == Status_MeasRequest) && (RxMsgInfo.Size != 2)))   //pokud slave odpovedel, ale nemel namerena zadna data, je preskoceno ulozeni dat
			{																																							//k vyctu dat by nemelo dojit, pokud jeste nebylo nic namereno, presto jsem toto podminku radeji uvedl
				//vytvoreni zpravy pro zapsani na SD kartu (formatovani prijatych dat)
				float TempMsg, HumMsg;
				memcpy(&TempMsg, &RxBuffer[i],sizeof(float));
				memcpy(&HumMsg, &RxBuffer[i+4],sizeof(float)); //nacteni prvniho mereni ve zprave (tomuto mereni odpovida casova znacka ve zprave)
				SDunstoredData[SlaveCounter].DataSize += sprintf(SDunstoredData[SlaveCounter].Data + SDunstoredData[SlaveCounter].DataSize, "%02u.%02u.20%02u/%02u:%02u:%02u.%03u\t%f\t%f\r\n", TimeStamp_tmp.Day, TimeStamp_tmp.Month, TimeStamp_tmp.Year,
															TimeStamp_tmp.Hours, TimeStamp_tmp.Minutes, TimeStamp_tmp.Seconds, TimeStamp_tmp.Milliseconds, TempMsg, HumMsg);	//formatovani dat pro zapis do souboru

				for (i += 8; i < RxMsgInfo.Size ; i += 8)	//dopocet casovych znacek pro dalsi namerene vzorky ve zprave podle periody mereni (je provenodo pro kazdy namereny vzorek ve zprave)
				{
					AddTime(&TimeStamp_tmp, Slave[SlaveCounter].MeasurePeriod);	 //pricteni periody mereni k casove znacne
					memcpy(&TempMsg, &RxBuffer[i],sizeof(float));
					memcpy(&HumMsg, &RxBuffer[i+4],sizeof(float));				//nacteni dalsiho namereneho vzorku ze zpravy
					SDunstoredData[SlaveCounter].DataSize += sprintf(SDunstoredData[SlaveCounter].Data + SDunstoredData[SlaveCounter].DataSize, "%02u.%02u.20%02u/%02u:%02u:%02u.%03u\t%f\t%f\r\n", TimeStamp_tmp.Day, TimeStamp_tmp.Month, TimeStamp_tmp.Year,
																TimeStamp_tmp.Hours, TimeStamp_tmp.Minutes, TimeStamp_tmp.Seconds, TimeStamp_tmp.Milliseconds, TempMsg, HumMsg); //formatovani dat pro zapis do souboru
				}

				//nastaveni informaci o neulozenych datech na SD katru (neprobiha zde zadny zapis na SD kartu aby nebylo na SD kartu pristupovano prilis casto)
				SDunstoredData[SlaveCounter].UnitType = SlaveUnit;	//typ jednotky (data patri senzoricke jednotce)
				SDunstoredData[SlaveCounter].Unstored = true;		//tyto data jeste nebyly ulozeny na SD kartu
				SDunstoredData[SlaveCounter].ID = RxBuffer[0];		//ID jednotky, ktere tyto data patri
				UnstoredData = true;								//indikace, ze jsou k dispozici data pro zapis na SD kartu (jsou nejake neulozene data)

				//odeslani nove prijatych dat do PC
				if((PCconnected == true) && (DataTransSDtoPC == false)) //pouze pokud je PC pripojeno a neprobiha prenos obsahu SD karty do PC
				{
					char TemrChar = '\0';	//definice ukoncovaciho znaku (terminator char)
					sprintf(UART_TxBuffer,"Value%u,%u,%s,%s,%s\n", SlaveUnit, RxBuffer[0], UnitType[Slave[SlaveCounter].UnitName1], UnitType[Slave[SlaveCounter].UnitName2], SensorType[Slave[SlaveCounter].SensorType]);	//nejprve se vytvori prikaz pro PC nesouci typ zpravy (Value), typ jednotky,ID, val1,val2,typ snimace
					HAL_UART_Transmit(&huart1, (uint8_t*)UART_TxBuffer, strlen(UART_TxBuffer), HAL_MAX_DELAY);		//odeslani prikazu do PC (informace o jednoce poslany zduvodu potreby vytvoreni hlavicky souboru v PC pro ulozeni dat, pokud soubor pro jednotku v PC jeste neexistuje)
					HAL_UART_Transmit(&huart1, (uint8_t*)SDunstoredData[SlaveCounter].Data, SDunstoredData[SlaveCounter].DataSize, HAL_MAX_DELAY);	//odeslani nove prijatych dat do PC
					HAL_UART_Transmit(&huart1, (uint8_t*)&TemrChar, 1, HAL_MAX_DELAY); 				//odeslani terminator char pro ukonceni prenosu dat do PC
				}
			}
		}
	}
	else	//prijeti neocekavane zpravy (od jine jednotky nebo od uplne jineho zarizeni)
	{
		printf("Přijata neočekávaná zpráva!!!!\r\n");		//pouze pro debug

		//zjisteni zda zpravu poslal jiny slave, pripadne vypocet a nastaveni casu (dutycycle),kdy nemuze dany slave vysilat
		for (uint8_t i = 0; i < sizeof(Slave) / sizeof(Slave[0]); i++)	//pruchod vsech slave jednotek
		{
			if (Slave[i].ID == RxBuffer[0])								//kontrola shody ID (adresy)
			{
				Slave[i].DCTimeStamp = UTIL_TIMER_GetCurrentTime();		//porizeni casove znacky prijeti zpravy
				Slave[i].DCWaitTime = CalcDutyCycle(Slave[i].CurRange, RxMsgInfo.Size, 99);	//nastaveni casu kdy nemuze dany slave vysilat
			}
		}
	}

	State = STATE_WAIT;		//prechod do stavu WAIT
	return;
}

/*Podprogram pro stav Start_measure*/
static void State_Start_measure_Subroutine(void)
{
	HAL_RTC_DeactivateAlarm(&hrtc, RTC_ALARM_B);	//deaktivuje RTC alarm B pro casovani mereni mastrem

	HAL_RTCEx_AlarmBEventCallback(&hrtc);			//nastaveni casovace mereni a mereni ze snimace

	Master.SettedPeriodChange = false;				//deaktivace indikace zmeny nastaveni periody mereni mastra
	AlarmB_Running = true;							//indikace aktivniho mereni
	State = STATE_WAIT;								//prechod do stavu WAIT
	return;
}

/*Podprogram pro stav Stop_measure*/
static void State_Stop_measure_Subroutine(void)
{
	HAL_RTC_DeactivateAlarm(&hrtc, RTC_ALARM_B);	//deaktivuje RTC alarm B pro casovani mereni mastrem
	AlarmB_Running = false;							//mereni neni aktivni
	State = STATE_WAIT;								//prechod do stavu WAIT
	return;
}

/*Podprogram pro stav WAIT*/
static void State_WAIT_Subroutine(void)
{
	  if (UTIL_TIMER_GetElapsedTime(MasterDCTimeStamp) >= MasterDCWaitTime)			//ubehl cas, kdy nemuze master odelsat novou zpravu
	  {
		  State = STATE_SEND;														//nastaveni nasledujiciho stavu na SEND
	  }

	  if (Master.EnableMeasure && (!AlarmB_Running || Master.SettedPeriodChange))	//pokud je povoleno mereni mastrem a mereni neni aktivni nebo doslo k prenastaveni periody mereni mastra
		  State = STATE_START_MASTER_MEASURE;										//nasledujici stav Start_measure pro aktivaci/prenastaveni mereni mastra (vyssi priorita nez stav SEND)

	  if (!Master.EnableMeasure && AlarmB_Running)									//pokud je zakazano mereni mastrem a mereni je aktivni
	  	  State = STATE_STOP_MASTER_MEASURE;										//nasledujici stav Stop_measure pro deaktivaci mereni mastra (vyssi priorita nez stav Start_measure)

	  if ((UnstoredData == true) && (DataTransSDtoPC == false))						//pokud jsou k dispozici nejake neulozena data na SD kartu a neprobiha prenost dat z SD karty do PC
		  StoreToSDcard();															//je proveden zapis neulozenych dat na SD kartu

	  if((StatusToPCTimestamp + 10000 < UTIL_TIMER_GetCurrentTime()) && (PCconnected == true))	//jednou za 10 sekund je aktualizovan status v PC pokud je PC pripojen
		  SendStatusMsg();															//odeslani statusu do PC (info o SD karte, slave sila prijateho signalu a baterie)

	  if (((StatusStoreTimestamp) < UTIL_TIMER_GetCurrentTime()) && (DataTransSDtoPC == false))	//jednou za hodinu je ulozen aktualni status na SD kartu (pripadne az po dokonceni prenosu dat z SD karty do PC)
		  StoreStatusToSD();														//zapis statusu na SD kartu (info o zaplneni SD karty, nastaveni, sile signalu a bateriich)

	  if ((DataTransSDtoPC == true) && (PCconnected == true))						//pokud uzivatel zahajil stazeni dat z SD karty a je pripojen pocitac
		  LoadFromSDtoPC();															//odeslani obsahu SD karty do PC

	  if (((LastDCFsynchTimestamp + 3600000) <= UTIL_TIMER_GetCurrentTime()) || LastDCFsynchTimestamp == 0)		//pokud ubehla hodina od posledni casove synchronizace z modulu DCF77 je zahajena synchronizace z modulu DCF77
	  {																											//nebo pokud jeste nebyla provedena casova synchronizace z DCF77 po restartovani jednotky je zahajena synchronizace z modulu DCF77
		  DCF_Process();				//zpracovani signalu na vystupu modulu DCF77
	  }


	  return;
}

//************ Konec podprogramu stavoveho automatu **********************









/*Inicializace SD karty*/
static bool SD_Init(void)
{
	  if(FATFS_LinkDriver(&USER_Driver, SDdiskPath) != FR_OK) return false;		//propojeni FatFs knihovny s ovladacem SD karty (user_diskio)
	  else
	  {
		  if((f_mount(&SD_FatFs, (TCHAR const*)SDdiskPath, 0)) != FR_OK) return false;	//nacteni souboroveho systemu SD karty (pripojeni na SD kartu)
		  else
		  {
			  if(SD_CardInfo() != true) return false;		//zisk informaci o SD karte

			 f_mount(NULL, "", 0);		//odpojeni od souboroveho systemu SD karty
		  }
	  }
	  return true;
}

/* Inicializace Callback funci Radioveho modulu */
static void Radio_Init(void){						// ktere funkce maji byt volany pri preruseni od radioveho modulu
	  RadioEvents.TxDone = OnTxDone;
	  RadioEvents.RxDone = OnRxDone;
	  RadioEvents.TxTimeout = OnTxTimeout;
	  RadioEvents.RxTimeout = OnRxTimeout;
	  RadioEvents.RxError = OnRxError;
	  Radio.Init(&RadioEvents);						// ulozeni do driveru radioveho modulu
}

/* Funkce pro vychozi nastaveni parametru radioveho modulu */
static void Radio_InitRxTxParams(void){
	  Radio.SetModem(MODEM_LORA);									// konfigurace radioveho modulu na LoRa modulaci
	  Radio.SetChannel(868000000);									// nastaveni nosne frekvence
	  Radio.SetMaxPayloadLength(MODEM_LORA, 255);					// nastaveni maximalni delky zpravy

	  Radio.SetRxConfig(MODEM_LORA, BW250, SF7, CR4_5, 0, PREAM_LEN8,
			  	  	  	  SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0, CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni vychozich parametru modulace pro prijem signalu

	  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW250, SF7, CR4_5,
			  	  	  	  PREAM_LEN8, NO_FIX_PAYLOAD_LEN, CRCon, 0, 0, IQ_NORMAL, 10000 );			// nastaveni vychozich parametru modulace pro vysilani signalu
}

/* Funkce pro prenastaveni modulacnich parametru radioveho modulu podle nastaveni komuikace
   @param Range Nastaveni komunikace */
static void Radio_SetRxTxParams(uint8_t Range){
	switch(Range)									// vyber sady modulacnich parametru podle nastaveni komunikace
	{
	case Range100:
		  Radio.SetRxConfig(MODEM_LORA, BW250, SF7, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	// nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW250, SF7, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 );	// nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range200:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF8, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	//nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF8, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 ); //nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range300:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF9, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	//nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF9, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 ); //nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range500:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF10, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	//nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF10, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 ); //nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range600:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF11, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	//nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF11, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 ); //nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;

	case Range750:
		  Radio.SetRxConfig(MODEM_LORA, BW125, SF12, CR4_5, 0, PREAM_LEN8, SYMBOL_TIMEOUT5, NO_FIX_PAYLOAD_LEN, 0,
				  CRCon, 0, 0, IQ_NORMAL, true);	//nastaveni novych modulacnich parametru radia pro prijem signalu
		  Radio.SetTxConfig( MODEM_LORA, TX_PRW14_dbm, 0, BW125, SF12, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN,
				  CRCon, 0, 0, IQ_NORMAL, 10000 ); //nastaveni novych modulacnich parametru radia pro vysilani signalu
		  break;
	}
}

/* Obsluha preruseni, vola se pri uspesnem odeslani zpravy radiovym modulem */
static void OnTxDone(void)
{
  /* USER CODE BEGIN OnTxDone */
	State = STATE_TXDONE; 				// nasledujici stav TXdone (pro prepnuti do radia do prijimaciho rezimu)
  /* USER CODE END OnTxDone */
}


/* Obsluha preruseni, vola se pri uspesnem prijeti zpravy radiovym modulem
   @param payload Ukazatel na prijatou zpravu
   @param size Delka prijate zpravy v bytech
   @param rssi Rssi posledni prijate zpravy (sila prijateho singalu v dBm)
   @param LoraSnr_FskCfo SNR posledni prijate zpravy (SRN odstup signal sum) */
static void OnRxDone(uint8_t *payload, uint16_t size, int16_t rssi, int8_t LoraSnr_FskCfo)
{
  /* USER CODE BEGIN OnRxDone */
	memset(RxBuffer,0,255);				// vycisteni bufferu pro prijatou zpravu (aby nedoslo k pouziti spatnych dat)
	memcpy(RxBuffer,payload, size );	// kopie prijate zpravy
	RxMsgInfo.Size = size;				// ulozeni delky prijate zpravy v bytech
	RxMsgInfo.RSSI = rssi;				// ulozeni sily prijateho signalu v dBm
	RxMsgInfo.SNR = LoraSnr_FskCfo;		// ulozeni odstupu signal sum v dB
	State = STATE_RXDONE;				// nasledujici stav RXdone (pro zpracovani prijate vzpavy)
  /* USER CODE END OnRxDone */
}

/* Obsluha preruseni, vola se pokud se radiovemu modulu nepodarilo zpravu uspesne odeslat v nastavenem intervalu 10s */
static void OnTxTimeout(void)
{
  /* USER CODE BEGIN OnTxTimeout */
	State = STATE_TXTIMEOUT;			// nasledujici stav TXtimeout (pro opetovne odelsani zpravy)
  /* USER CODE END OnTxTimeout */
}

/* Obsluha preruseni, vola se pokud v nastavenem intervalu prijimani zpravy nebyla prijata zadna zprava radiovym modeulem */
static void OnRxTimeout(void)
{
  /* USER CODE BEGIN OnRxTimeout */
	State = STATE_RXTIMEOUT;			// nasledujici stav RXtimeout
  /* USER CODE END OnRxTimeout */
}

/* Obsluha preruseni, vola se pokud byla radiovym modulem prijata poskozena zprava (chyba v hlavicce, chybny CRC kod) */
static void OnRxError(void)
{
  /* USER CODE BEGIN OnRxError */
	State = STATE_RXERROR;			// nasledujici stav RXerror
  /* USER CODE END OnRxError */
}


/*Funkce pro aktualizaci systemoveho casu SystemTime podle RTC  */
static void SysTimeUpdate(void)
{	/*Funkce aktualizuje cas ve strukture SystemTime podle ubehleho casu RTC od posledni aktualizace*/

	uint32_t RTC_tmp = UTIL_TimerDriver.GetTimerValue();	// ulozeni aktualni hodnoty casovace RTC (tiky)
	uint32_t TicksElapsed = RTC_tmp - RTC_CurTicks;			// pocet tiku od posledni aktualizace
	RTC_CurTicks = RTC_tmp;									// ulozeni kdy byla provedena posledni aktualizace

	// krystal 32768 impulsu za sekundu
	// pouzita preddelicka 32 (RTC_PREDIV_A + 1)
	// pocet nactenych impulsu do RTC citace je 32768/32 = 1024 impulsu za sekundu
	// za milisekundu je do RTC citace nacteno 1024/1000 = 1,024 impulsu

	// Vypocet ubehleho casu v milisekundach
	float MsElapsed = TicksElapsed / ((32768 / (RTC_PREDIV_A + 1)) / 1000.0);

	// Vypocet desetinne casti ubehleho casu
	// protoze cas je uchovavan s rozlisenim na milisekundy nelze ulozit desetinnou cast MsElapsed
	// desetinna cast se pricte k celkovemu zpozdeni
	// celkove zpozdeni MsRest uchovana microsekundy (z duvodu potlaceni zaokrouhleni na cele milisekundy)
	MsRest +=  MsElapsed - (uint32_t)MsElapsed;

	if (MsRest >= 1)	// pokud celkove zpozdeni presahne 1 ms
	{
		--MsRest;
		++MsElapsed;	// je prictena 1 ms
	}

	AddTime(&SystemTime, (uint32_t)MsElapsed);	// pricteni ubehleho casu ke strukture SystemTime
}

/* Funkce pro pricteni casu ke strukture a jeho konverze (zarovnani)
   @param Time Ukazatel na casovou strukturu, ke ktere ma byt cas pricten
   @param AddTimeMS Cas v milisekundach, ktery ma byt pricten k casove strukture */
static void AddTime(SystemTime_t *Time, uint32_t AddTimeMS)
{
	AddTimeMS += Time->Milliseconds;		//soucet milisekund
	while(AddTimeMS >= 1000)				//pro kazdych 1000 ms je odecteno 1000 ms a provedena konverze casu
	{
		AddTimeMS -= 1000;

	   ++Time->Seconds;
	   Time->Seconds %= 60;

	   if (Time->Seconds == 0)
	   {
		   ++Time->Minutes;
		   Time->Minutes %= 60;

		   if (Time->Minutes == 0)
		   {
			   ++Time->Hours;
			   Time->Hours %= 24;

			   if (Time->Hours == 0)
			   {
				   ++Time->Day;

				   if (Time->Day > DaysInMonth(Time->Month, Time->Year))	// kontrola poctu dni v mesici
				   {
					   Time->Day = 1;
					   ++Time->Month;

					   if (Time->Month == 13)
					   {
						   Time->Month = 1;
						   ++Time->Year;

						   if (Time->Year == 100)							// u roku uchovany pouze jednotky a desitky
							   Time->Year = 0;
					   }
				   }
			   }
		   }
	   }
   }

   Time->Milliseconds = AddTimeMS;		// ulozeni zbylych milisekund
}

/* Funkce pro zjisteni poctu dni v mesici
   @param Month Cislo mesice
   @param Year Rok
   @return Pocet dni v mesici*/
static uint8_t DaysInMonth(uint8_t Month, uint8_t Year)
{
   uint8_t result;
   switch (Month)
   {
   case 1:  // leden
   case 3:  // brezen
   case 5:  // kveten
   case 7:  // cevenec
   case 8:  // srpen
   case 10: // rijen
   case 12: // prosinec
       result = 31;
       break;

   case 4:  // duben
   case 6:  // cerven
   case 9:  // zari
   case 11: // listopad
       result = 30;
       break;

   case 2: 	// unor				// zjisteni prestupneho roku
       if (Year % 4 == 0)		// staci pouze delitelnost 4, pokud se pocita 0-99
       {
    	   result = 29; 		// prestupny rok
       	   break;
       }
       else
       {
    	   result = 28;
       	   break;
       }
   }
   return result;
}

/* Obsluha presuseni od UART
   Kazdy nacteny znak pres UART vyvola preruseni
   Slouzi pro nacteni cele zpravy pres UART*/
void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
	if (huart->Instance == USART1)				// preruseni vyvolal UART1
	{
		UART_RxBuffer[pos] = znak;				// ulozeni nacteneho znaku do bufferu
		++pos;									// zvyseni ukazatele na volne misto v bufferu
		if (znak == '\r')						// kontrola nacteni ukoncovaciho znaku (term. char)
		{
			UartReadProcess(UART_RxBuffer);		// zpracovani nactene zpravy
			pos = 0;							// reset pozice v bufferu
		}
		HAL_UART_Receive_IT(&huart1, &znak, 1);	// povoleni vzniku preruseni od UART1 po nacteni 1 znaku
	}
}

/* Funkce pro zpracovani prichozi zpravy pres UART
   @param buffer Ukazatel na prijatou zpravu pres UART*/
static void UartReadProcess(char *buffer)
{
	// zprava load = odeslani nastaveni do PC
	if (strncmp(UART_RxBuffer, "load", 4) == 0)
	{
		sprintf(UART_TxBuffer, "load%u,%u,%lu,%u,", Master.SettedRange, Master.EnableMeasure, Master.MeasurePeriod, Master.TimeSynchInterval);	// vlozeni identifikatoru zpravy load a nastaveni mastra do bufferu

		for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)	// prochazeni vsech slave jednotek
		{
			sprintf(UART_TxBuffer + strlen(UART_TxBuffer),"%u,%u,%lu,", Slave[i].Enable, Slave[i].ID, Slave[i].MeasurePeriod);	// vlozeni nastaveni slave do bufferu
		}
		sprintf(UART_TxBuffer + strlen(UART_TxBuffer),"\n");								// vlozeni ukoncovaciho znaku (term. char)
		HAL_UART_Transmit(&huart1, (uint8_t*)UART_TxBuffer, strlen(UART_TxBuffer),100);		// odeslani dat pres UART1
		printf("LOAD\n");																	// potvrzeni pro PC o provedeni odeslani nastaveni
	}

	// zprava store = ulozeni prijateho nastaveni z PC
	else if (strncmp(UART_RxBuffer, "store", strlen("store")) == 0)
	{
		int sum_offset = strlen("store");		// posunuti v prijate zprave o pocet znaku store
		int offset_n, tmp_Enable, tmp_MasterEn;
		uint8_t tmp_ID;
		uint32_t tmp_Period;
		uint8_t tmp_MasterPrevRange = Master.SettedRange;		// ulozeni predchoziho nastaveneho dosahu
		uint32_t tmp_MasterPverPeriod = Master.MeasurePeriod;	// ulozeni predchozi periody mereni master jednotky

		sscanf(UART_RxBuffer + sum_offset, "%hhu,%u,%lu,%hhu,%n", &Master.SettedRange, &tmp_MasterEn, &Master.MeasurePeriod, &Master.TimeSynchInterval, &offset_n);	// ulozeni nastaveni master jednotky ze zpravy
		Master.EnableMeasure = tmp_MasterEn;					// ulozeni povoleni mereni master jednotkou (zvlast protoze sscanf nepodporuje bool hodnoty)
		sum_offset += offset_n;									// posunuti v bufferu o pocet nactenych znaku
		if (Master.MeasurePeriod != tmp_MasterPverPeriod)		// doslo ke zmene periody mereni master jednotky?
			Master.SettedPeriodChange = true;					// indikace zmeny periody mereni master jednotky

		for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)	// prochazeni vsech slave jednotek
		{
			if (Master.SettedRange != tmp_MasterPrevRange) 	 // pokud doslo k prenastaveni dosahu komunikace, je potreba prenastavit i slave jednotky
			{
				Slave[i].TimeSynch = true;				  	 // indikace provedeni cas. synch. (pri zmene nastaveni slave jednotky se nejprve provede casova synchronizace)
				Slave[i].SettingChange = true;			 	 // indikace zmeny nastaveni jednotky
			}

			sscanf(UART_RxBuffer + sum_offset, "%u,%hhu,%lu,%n", &tmp_Enable, &tmp_ID, &tmp_Period, &offset_n);			// nacteni nastaveni slave jednotky ze zpravy
			sum_offset += offset_n;							 // posunuti v bufferu o pocet nactenych znaku
			if ( (Slave[i].Enable != tmp_Enable) || (Slave[i].ID != tmp_ID) || (Slave[i].MeasurePeriod != tmp_Period))	// pokud pro tuto slave jednotku doslo ke zmene nastaveni
			{
				Slave[i].TimeSynch = true;					// indikace provedeni cas. synch. (pri zmene nastaveni slave jednotky se nejprve provede casova synchronizace)
				Slave[i].SettingChange = true;				// indikace zmeny nastaveni slave jednotky
				Slave[i].ID = tmp_ID;						// ulozeni ID (adresy) slave jednotky
				Slave[i].Enable = tmp_Enable;				// ulozeni povoleni (aktivace) slave jednotky
				Slave[i].MeasurePeriod = tmp_Period;		// ulozeni periody mereni slave jednotky

				if (Slave[i].Enable == false)				// pri deaktivaci jednotky
				{
					Slave[i].CurRange = 0;					// resetuje se nastaveny dosah pro tuto slave jednotku (protoze pri zapnuti senzoricke jednotky je vychozi range 0)(pri deaktivaci se predpoklada jeji vypnuti)
					Slave[SlaveCounter].Msg = NONE;			// resetuje se nastavená zprava pro jednotku
					Slave[i].RSSI = Slave[i].BatteryLvl = 0;	//resetuje se sila prijateho signalu od jednotky a stav baterie teto jednotky
				}
			}
		}
		StoreToFLASH();			// ulozeni noveho nastaveni do pameti FLASH
		printf("STORE\n");		// potvrzeni pro PC o prijeti nastaveni
	}

	// zprava synch = synchronizace casu z PC
	else if (strncmp(UART_RxBuffer, "synch", strlen("synch")) == 0)
	{
		uint8_t offset = strlen("synch");					// posunuti v prijate zprave o pocet znaku synch
		sscanf(UART_RxBuffer + offset, "%hhu,%hhu,%hhu,%hhu,%hhu,%hhu,%hu",&SystemTime.Year, &SystemTime.Month, &SystemTime.Day,
										&SystemTime.Hours, &SystemTime.Minutes, &SystemTime.Seconds, &SystemTime.Milliseconds);		// nacteni a ulozeni casove znacky ze zpravy do SystemTime
		RTC_CurTicks = UTIL_TimerDriver.GetTimerValue();	// nova casova znacka RTC (ulozeni aktualni hodnoty RTC casovace, pro kterou je platny cas v SystemTime)
		printf("SYNCH\n");									// potvrzeni pro PC o provedeni cas. synch z PC
	}

	// zprava SDcard = preposlani dat z SD karty do PC
	else if (strncmp(UART_RxBuffer, "SDcard", strlen("Sdcard")) == 0)
	{
		DataTransSDtoPC = true;		// indikace ze ma by prenesen obsah SD karty do PC, samotny prenos bude zahajen ve stavu WAIT
	}								// prenos neni zahajen ihned, protoze zrovna muze probihat zapis/cteni z SD karty

	// zprava Delete = vymazani obsahu SD karty
	else if (strncmp(UART_RxBuffer, "Delete", strlen("Delete")) == 0)
	{
		DeleteSDcard();				// mazani dat na SD karte
		printf("Delete\n");			// potvrzeni pro PC o provedeni vymyzani SD karty
	}

	// zprava Connect = master jednotka pripojena k PC
	else if (strncmp(UART_RxBuffer, "Connect", strlen("Connect")) == 0)
	{
		PCconnected = true;			// indikace pripojeneho PC k master jednotce (respektive spusteneho programu v PC a pripojeni k master jednotce)
		HAL_GPIO_WritePin(LED2_GPIO_PORT, LED2_PIN, GPIO_PIN_SET);		// rozsviceni LED2, indikuje PC pripojen
	}

	// zprava Discon = master jednotka odpojena od PC
	else if (strncmp(UART_RxBuffer, "Discon", strlen("Discon")) == 0)
	{
		PCconnected = false;											// indikace odpojeni od PC (respektive neni na PC spusten program, ktery by naslouchal komunikaci master jednotky)
		HAL_GPIO_WritePin(LED2_GPIO_PORT, LED2_PIN, GPIO_PIN_RESET);	// zhasnuti LED2, indikuje PC odpojen
	}

	// zprava Reset = resetovani uzivatelskeho nastaveni (je resetovano nastaveni v master jednotce)
	else if (strncmp(UART_RxBuffer, "Reset", strlen("Reset")) == 0)
	{
		RessetSettings();			// resetovani nastaveni
		printf("Reset\n");			// potvrzeni pro PC o provedeni reset
	}
}

/* Funkce pro vypocet casu, kdy neni mozne odeslat dalsi zpravu
   @param Range Nastaveni dosahu (pro jake parametry modulace ma byt cas vypocten)
   @param size Delka odeslane zpravy v bytech
   @param per DutyCycle v procentech kdy neni mozne vysilat (vyuziva se 100 a 99, 100 pro vypocet pred odeslanim zpravy, 99 po odeslani zpravy)
   @return Cas v milisekundach, za ktery je mozne odeslat dalsi zpravu*/
uint32_t CalcDutyCycle(uint8_t Range, uint8_t size, uint8_t per)
{
	uint32_t result;

	switch(Range)		//vyber podle dosahu komunikace
	{
	case Range100:
		 result = per * Radio.TimeOnAir(MODEM_LORA, BW250, SF7, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		 break;			//TimeOnAir vraci dobu prenosu celeho paketu v ms
	case Range200:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF8, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range300:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF9, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range500:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF10, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range600:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF11, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	case Range750:
		result = per * Radio.TimeOnAir(MODEM_LORA, BW125, SF12, CR4_5, PREAM_LEN8, NO_FIX_PAYLOAD_LEN, size, CRCon);
		break;
	}

	return result;
}

/* Obsluha preruseni od RTC alarmu B
   Slouzi pro nove nastaveni alarmu B a provedeni mereni ze snimace */
void HAL_RTCEx_AlarmBEventCallback(RTC_HandleTypeDef *hrtc)
{
	  //nove nastevni alarmu B
	  RTC_AlarmTypeDef sAlarmB = {0};					//pro nastaveni noveho RTC alarm B pro casovani mereni snimacem
	  RTC_TimeTypeDef sTime = {0};						//pro nacteni casu z RTC
      RTC_DateTypeDef sDate = {0};						//pro nacteni datumu z RTC
	  HAL_RTC_GetTime(hrtc, &sTime, RTC_FORMAT_BIN);	//nacteni casu z RTC
	  HAL_RTC_GetDate(hrtc, &sDate, RTC_FORMAT_BIN);	//datum neni pouzivat v binarnim rezimu RTC (ale potreba volat tuto funkci po GetTime protoze RTC stoji dokud neni nacten i datum)
	  sAlarmB.BinaryAutoClr = RTC_ALARMSUBSECONDBIN_AUTOCLR_NO;		//nenunovat alarm pri preruseni
	  sAlarmB.AlarmTime.SubSeconds = sTime.SubSeconds - ((uint32_t)((((uint64_t)Master.MeasurePeriod) * 1024) / 1000));	  //hodnota pri ktere vznikce preruseni , + prevod z milisekund na tiky
	  sAlarmB.AlarmMask = RTC_ALARMMASK_NONE;																			  //aktualni hodnota citace - perioda mereni (v tikach) (odcitani protoze casovat RTC cita do 0)
	  sAlarmB.AlarmSubSecondMask = RTC_ALARMSUBSECONDBINMASK_NONE;	//zadne masky (alarm bude spusten presne pri hodnote v subseconds)
	  sAlarmB.Alarm = RTC_ALARM_B;									//nastaveni typu alarmu na B
	  if (HAL_RTC_SetAlarm_IT(hrtc, &sAlarmB, 0) != HAL_OK)			//nasteveni vzniku preruseni od noveho nastaveni Alarmu B
	  {
	    Error_Handler();
	  }

	  Sensor_measure();		// mereni dat snimacem
}

/* Funkce pro mereni externim snimacem
   Slouzi pro ziskani surove hodnoty z externiho snimace, prepocet na fyz. hodnotu a ulozeni do pameti */
static void Sensor_measure(void)
{
	SysTimeUpdate();							//aktualizace systemoveho casu
	MeasureTimeStamp = SystemTime;				//ulozeni casove znacky mereni
												/** Je pouzit snimac SHT30 **/
	bool SensorReceiveSuccess = false;			//nastaveni neuspesneho prijmu dat ze snimace
	uint8_t SensorAdress = 0x44 << 1u;			//I2C adresa snimace (7 MSB adresa, LSB 0 write)
	uint8_t  SensorCommand[2] = {0x2c, 0x06};	//16 bit prikaz pro snimac na provedeni jednoho mereni
	uint8_t RawSensorData[6];					//pro nacteni surovych dat ze snimace

	HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, GPIO_PIN_SET);	//rozsviti se dioda LED1 pri mereni

	while(SensorReceiveSuccess == false)		//opakovane mereni pro pridat chybneho crc v datech od snimace
	{
		HAL_I2C_Master_Transmit(&hi2c3, SensorAdress, SensorCommand, sizeof(SensorCommand), HAL_MAX_DELAY);	//odeslani commandu senzoru pres i2c
		HAL_I2C_Master_Receive(&hi2c3, SensorAdress, RawSensorData, sizeof(RawSensorData), HAL_MAX_DELAY);	//prijeti surovych dat pres i2c ze snimace
												//snimac vrati celkem 6 bytu, 2 byty teplota, 1 byt CRC pro teplotu, 2 byty vlhkost, 1 byt CRC pro vlhkost
		//vypocet crc prijatych dat
		uint8_t TempCRC = 0xff;					//vychozi hodnota CRC pro vypocet
		uint8_t HumCRC = 0xff;
		for (size_t i = 0; i < 2; i++)			//2 opakovani (pro 1. a 2. byte teploty a vlhkosti)
		{
			TempCRC = TempCRC ^ RawSensorData[i];	//xor s 1. (resp 2.) bytem teploty
			HumCRC = HumCRC ^ RawSensorData[i+3];	//xor s 1. (resp 2.) bytem vlhkosti

			for (size_t j = 0; j < 8; j++)			//8 opakovani (jeden pruchod pro kazdy bit crc)
			{
				if ((TempCRC & 0b10000000) == 0)	//pokud je nejvyssi (MSB) bit roven 0
					TempCRC = TempCRC << 1;			//je proveden bitovy posuv vlevo
				else
					TempCRC = (TempCRC << 1) ^ 0x31;	//pokud je nejvyssi bit roven 1, je proveden bitovy posuv vlebo a xor s crc polynomen (0x31)

				if ((HumCRC & 0b10000000) == 0)		//stejny postup pro vlhkost
					HumCRC = HumCRC << 1;
				else
					HumCRC = (HumCRC << 1) ^ 0x31;
			}
		}
		if ((TempCRC == RawSensorData[2]) && (HumCRC == RawSensorData[5])) //pokud je vypocitane CRC shodne s prijatym CRC od snimace jsou data validni
			SensorReceiveSuccess = true;			//oznaceni dat ze snimace jako validni
	}

	HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, GPIO_PIN_RESET);	//po dokonceni mereni je zhasnuta dioda LED1

	uint16_t temperature_raw = ((uint16_t)RawSensorData[0] << 8) | RawSensorData[1];	//prevedeni 2 bytu teploty na jedno 16 bit cislo
	uint16_t humidity_raw = ((uint16_t)RawSensorData[3] << 8) | RawSensorData[4];		//prevedeni 2 bytu vlhkoti na jedno 16 bit cislo

	MasterTemperature = -45.0 + 175.0 * (float)temperature_raw / 65535.0;		//vypocet float teploty ze surovych dat (rovnice z datasheetu pro snimac SHT30)
	MasterHumidity = 100.0 * (float)humidity_raw / 65535.0;						//vypocet float vlhkosti ze surovych dat (rovnice z datasheetu pro snimac SHT30)

	//formatovani a pripraveni dat pro ulozeni (samotne ulozeni je provedeno ve stavu WAIT), zapis na SD neni proveden ihned, protoze muze zrovna probihat cteni/zapis z SD karty
	SDunstoredData[10].DataSize += sprintf(SDunstoredData[10].Data + SDunstoredData[10].DataSize, "%02u.%02u.20%02u/%02u:%02u:%02u.%03u\t%f\t%f\r\n", MeasureTimeStamp.Day, MeasureTimeStamp.Month, MeasureTimeStamp.Year,
											MeasureTimeStamp.Hours, MeasureTimeStamp.Minutes, MeasureTimeStamp.Seconds, MeasureTimeStamp.Milliseconds, MasterTemperature, MasterHumidity);
	SDunstoredData[10].UnitType = MasterUnit;	//informace, ze data parti master jednotce
	SDunstoredData[10].Unstored = true;			//tyto data jeste nejsou ulozena na SD karte
	UnstoredData = true;						//indikace, ze jsou k dispozici neulozena data na SD kartu

	if((PCconnected == true) && (DataTransSDtoPC == false))	//pokud je pripojen PC a neprobiha prenos obsahu SD karty jsou prave namerena data odeslany do PC
	{
		char TemrChar = '\0';								//terminator char
		sprintf(UART_TxBuffer,"Value%u,%u,%s,%s,%s\n", MasterUnit, 0, UnitType[Master.UnitName1], UnitType[Master.UnitName2], SensorType[Master.SensorType]);		//vytvoreni hlavicky dat pro odeslani
		HAL_UART_Transmit(&huart1, (uint8_t*)UART_TxBuffer, strlen(UART_TxBuffer), HAL_MAX_DELAY);					//odeslani prikazu, typu jednotky, merene jednotky do PC
		HAL_UART_Transmit(&huart1, (uint8_t*)SDunstoredData[10].Data, SDunstoredData[10].DataSize, HAL_MAX_DELAY);	//odeslani namerenych dat do PC
		HAL_UART_Transmit(&huart1, (uint8_t*)&TemrChar, 1, HAL_MAX_DELAY); 											//odeslani terminator char
	}
}

/* Funkce pro ulozeni nastaveni dataloggeru do pameti FLASH*/
static void StoreToFLASH(void)
{
	HAL_FLASH_Unlock();										//otevreni pristupu do pameti flash

	FLASH_EraseInitTypeDef EraseInitStruct;
	EraseInitStruct.TypeErase = FLASH_TYPEERASE_PAGES;		//mazani po strankach (1 stranka 2kB)
	EraseInitStruct.Page = 127;								//pouziva se pouze stranka 127
	EraseInitStruct.NbPages = 1;							//pouziva se pouze 1 stranka

	uint64_t Data;											//data pro zapis na FLASH
	uint32_t PageError;
	HAL_FLASHEx_Erase(&EraseInitStruct, &PageError);		//pred zapisem je potraba pamet FLASH vymazat (vymaze se pouze pouzivana stranka)

	Data = ((((((uint64_t)Master.EnableMeasure << 32) | Master.MeasurePeriod) << 8 ) | Master.SettedRange) << 8 ) | Master.TimeSynchInterval;	//poskladani nastaveni mater jednotky do bloku double-word (64 bit)
	HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, FlashFirstAdress, Data);	//zapis nastaveni mastra do pameti flash (data jsou poskladany do bloku po 64 bitech, protoze zapis na flash probiha po 64 bitech)

	for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)			//prochazeni pro kazdou slave jednotku
	{
		Data = ((((((((uint64_t)Slave[i].ID << 8) | Slave[i].Enable) << 32 ) | Slave[i].MeasurePeriod) << 8) | Slave[i].CurRange) << 8) | Slave[i].SensorType;		//poskladani nastaveni slave jednotky do bloku double-word (64 bit)
		HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, FlashFirstAdress - (2*i + 1) * 8, Data);		//ulozeni prvni casti (64bit) nastaveni slave jednotky
		Data = ((uint64_t)Slave[i].UnitName1 << 8) | Slave[i].UnitName2;								//poskladani druheho bluku dat nastaveni slave jednotky (co se nevlezlo do prvnich 64 bit)
		HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, FlashFirstAdress - (2*i + 2) * 8, Data);		//ulozeni druhe casti (64bit) nastaveni slave jednotky
	}
	HAL_FLASH_Lock();		//uzavreni pristupu k flash
}

/*Funkce pro nacteni ulozeneho nastavení dataloggeru z pameti FLASH*/
static void LoadFromFLASH(void)
{
	uint64_t Data = *(uint64_t *)FlashFirstAdress;					//nacteni bloku 64 bit z prvni adresy (nastaveni master jednotky)
	Master.TimeSynchInterval = (uint8_t)Data;						//postupne rozdeleni bloku dat na jednotliva nastaveni
	Master.SettedRange = (uint8_t)(Data >> 8);
	Master.MeasurePeriod = (uint32_t)(Data >> 16);
	Master.EnableMeasure = (uint8_t)(Data >> 48);

	for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i) //prochazeni pro kazdou slave jednotku
	{
		Data = *(uint64_t *)(FlashFirstAdress - (2*i + 1) * 8);		//nacteni dalsiho bloku dat z pameti flash (posun v adrese o 2i protoze se v kazdem pruchodu nacitaji 2 bloky dat a + 1 je posud o nacteni bloku dat pro mastra pred zapocetim smycky, *8 pocet bytu double word)
		Slave[i].SensorType = (uint8_t)Data;						//postupne rozdeleni bloku dat na jednotliva nastaveni
		Slave[i].CurRange = (uint8_t)(Data >> 8);
		Slave[i].MeasurePeriod = (uint32_t)(Data >> 16);
		Slave[i].Enable = (uint8_t)(Data >> 48);
		Slave[i].ID = (uint8_t)(Data >> 56);

		Data = *(uint64_t *)(FlashFirstAdress - (2*i + 2) * 8);		//nasteni dalsiho bloku dat z pameti flash (posun v pameti jeste o predchozi nacteny blok dat)
		Slave[i].UnitName2 = (uint8_t)Data;							//postupne rozdeleni bloku dat na jednotliva nastaveni
		Slave[i].UnitName1 = (uint8_t)(Data >> 8);
	}
}

/*Funkce pro ulozeni dat na SD kartu, dojde k zapisu obsahu SDunstoredData na SD kartu*/
static void StoreToSDcard(void){
	  char FileName[25];	//nazev souboru
	  FIL FileObject; 		//Objekt souboru
	  FRESULT FileResult; 	//Vysledek po operaci se souborem
	  uint wb; 				//pro zavolani f_write (pocet zapsanych bytu do souboru)

	for (uint8_t i = 0; i < (sizeof(SDunstoredData) / sizeof(SDunstoredData[0])); ++i)	//prochazeni vsech prvku SDunstoredData (struktura pro ulozeni dat)
	{
		if(SDunstoredData[i].Unstored == false)				//hledani neulozenych dat
			continue;										//pokud prvek neobsahuje neulozena data prechod na dalsi prvek ve strukture

		if (SDunstoredData[i].UnitType == MasterUnit)		//vyber nazvu souboru, podle typu jednotky, ktera data ulozila
			sprintf(FileName, "Log_Unit.txt");
		else
			sprintf(FileName, "Sensor_Unit_%02u.txt", SDunstoredData[i].ID);	//do nazvu souboru pro slave jednotku je doplneno ID jednotky, ktera data ulozila

		if((FileResult = f_mount(&SD_FatFs, (TCHAR const*)SDdiskPath, 0)) == FR_OK)		//nacteni souboroveho systemu SD karty
		{
			FileResult = f_open(&FileObject, FileName, FA_OPEN_EXISTING  | FA_WRITE );  //otevreni souboru na SD karte pro zapis

			if(FileResult == FR_NO_FILE)	//soubor nenalezen (neexistuje)
			{
				char FileHeader[100]; 							//pro vytvoreni hlavicky dat
				if(SDunstoredData[i].UnitType == MasterUnit)	//vytvoreni hlavicky dat pro soubor podle typu jednotky
					sprintf(FileHeader, "DATE/TIME\t\t%s\t%s\tSensor: %s\r\n", UnitType[Master.UnitName1], UnitType[Master.UnitName2], SensorType[Master.SensorType] );
				else
					sprintf(FileHeader, "DATE/TIME\t\t%s\t%s\tSensor: %s\r\n",UnitType[Slave[i].UnitName1], UnitType[Slave[i].UnitName2], SensorType[Slave[i].SensorType] );
				FileResult = f_open(&FileObject, FileName, FA_CREATE_NEW  | FA_WRITE);		//vytvoreni noveho souboru na SD karte
				if(FileResult == FR_OK)														//pokud byl soubor uspesne vytvoren
					FileResult = f_write(&FileObject, FileHeader, strlen(FileHeader), &wb);	//zapise se hlavicka dat do souboru
				printf("Vytvoril jsem hlavicku souboru %s\r\n", FileName);					//pouze pro debug, uzivateli skryto
			}

			if(FileResult == FR_OK)		//soubor nalazen (existuje) (pridadne byl v predchozim IFu vytvoren)
			{
				if((FileResult = f_lseek(&FileObject, f_size(&FileObject)) != FR_OK)) break;	//presun ukazaltele na konec souboru
				if((FileResult = f_write(&FileObject, SDunstoredData[i].Data, SDunstoredData[i].DataSize, &wb) != FR_OK)) break; //zapis neulozenych dat do souboru na SD karte
				f_close(&FileObject);																//uzavreni souboru
				printf("%s zapsan, ulozeno: %i bytu\r\n", FileName, SDunstoredData[i].DataSize);	//pouze pro debug, uzivateli skryto
				SDunstoredData[i].Unstored = false;													//oznaceni dat jako ulozena
				SDunstoredData[i].DataSize = 0;														//resetovani velikosti (bytu) neulozenych dat
			}

			if(FileResult != FR_OK)
				printf("Zapis do souboru selhal, %s Error (%i)\r\n", FileName, FileResult);			//informace pokud selhal zapis do souboru
		}
		else
			printf("Mount %s Error (%i)\r\n", FileName, FileResult);								//informace pokud se nelze pripojit k SD karte

		SD_CardInfo();			//vypocet noveho zaplneni SD karty

		f_mount(NULL, "", 0);	//odpojeni od souboroveho systemu SD karty
	}

	UnstoredData = false;		//indikace ze nejsou zadna data, ktera by nebyla ulozena na SD karte
}

/*Funkce pro preposlani obsahu SD karty pres rozharani UART do PC*/
static void LoadFromSDtoPC(void)
{
	  FIL FileObject; 			//Objekt souboru
	  FRESULT FileResult; 		//Vysledek po operaci se souborem
	  uint ReadByte; 			//pro zavolani f_write (pocet zapsanych bytu do souboru)
	  FILINFO FileInfo;     	//informace o souboru
	  DIR Directory;   			//Objekt adresare (slozky), slozky nejsou na SD karte vyuzivany, ale je to potrebna promena pro hledani souboru
	  char FileList[12][30];	//pro ulozeni nazvu vsech souboru na karte (az 10 slave + master + status)
	  uint8_t FileCnt = 0;		//pocitadlo nalezenych souboru na SD karte

	  printf("SDcard\n");		//zprava pro pocitac (zahajeni prenosu obsahu SD karty)
	  if((FileResult = f_mount(&SD_FatFs, (TCHAR const*)SDdiskPath, 0)) == FR_OK)		//nacteni souboroveho systemu SD karty (pripojeni na SD kartu)
	  {
		  FileResult =  f_findfirst(&Directory, &FileInfo, (TCHAR const*)SDdiskPath, "*.txt");   //hledani prvniho souboru na SD karte s priponou .txt

		  while (FileResult == FR_OK && FileInfo.fname[0]) {  							//pokud hledani skoncilo uspesne a byl nalazen soubor
			  memcpy(&FileList[FileCnt], &FileInfo.fname, 1+strlen(FileInfo.fname));	//ulozeni nazvu nalezeneho souboru (delka nazvu + 1 aby byl nazev ulozen i s ukoncovacim znakem "\0" )
			  ++FileCnt;																//zvyseni pocitadla nalezenych souboru
			  FileResult = f_findnext(&Directory, &FileInfo);   						//hledani dalsiho souboru
		  }

		  f_closedir(&Directory);			//uzavreni slozky (slozky nejsou na SD karte pouzivany)
		  char uart[30];					//pro odeslani informace do PC kolik souboru bude preposilano
		  sprintf(uart, "%u", FileCnt);		//vlozeni poctu nalezenych souboru
		  HAL_UART_Transmit(&huart1, (uint8_t*)uart, strlen(uart)+1, HAL_MAX_DELAY); //odeslani do PC informace o poctu souboru, ktere budou odeslany do PC

		  for(uint8_t i = 0; i < FileCnt; ++i)		//opakovani pro kazdy nalezeny soubor na SD karte
		  {
			  HAL_UART_Transmit(&huart1, (uint8_t*)FileList[i], strlen(FileList[i])+1, HAL_MAX_DELAY); //odeslani nazvu souboru do PC
			  if((FileResult = f_open(&FileObject, FileList[i], FA_READ)) == FR_OK)		//otevreni souboru pro cteni
			  {
					while (FileResult == FR_OK && !f_eof(&FileObject))					//opakovani dokud neni dosazeno konce souboru nebo nedoslo k chybe
					{
						FileResult =  f_forward(&FileObject, Out_SD_Stream, 512, &ReadByte);	//funkce f_forward precte ze souboru az 512 bytu a preda je funkci Out_SD_Stream pro odeslani do PC
					}																			//cteni po 1 sektoru (512 bytu)
					f_close(&FileObject);	//uzavreni souboru po jeho precteni
					uart[0] = '\0';			//vlozeni ukoncovaciho znaku (terminator char.)
					HAL_UART_Transmit(&huart1, (uint8_t*)uart, 1, HAL_MAX_DELAY); //odeslani ukoncovaciho znaku do PC (konec prenosu dat tohoto souboru)
			  }
			  else
				  printf("Open %s Error (%i)\r\n", FileList[i], FileResult);	//informace o neuspesnem otevreni souboru
		  }
	  }
	  else
		  printf("Mount Error (%i)\r\n", FileResult);			//nelze se pripojit na SD kartu

	f_mount(NULL, "", 0);					//odpojeni od souboroveho systemu SD karty

	DataTransSDtoPC = false;				//indikace, konec probihajiciho prenosu dat z SD karty do PC
}

/* Callback funkce pro odeslani bloku dat pres rozhrani UART do PC (volano funkci f_forward)
   @param data Ukazatel na blok dat
   @param size Velikost bloku dat
   @return UINT Pocet odeslanych bytu dat*/
UINT Out_SD_Stream(const BYTE* data, UINT size)
{
	if (size == 0)	return 1;								//funkce f_forward neprve kontroluje zaneprazdnenos sbernice poslanim prazdnych dat (ocekava vraceni 1 jinak nezahaji cteni)
	HAL_UART_Transmit(&huart1, data, size, HAL_MAX_DELAY);	//odelsani bloku dat pres UART do PC
    return size; 											//pocet prenesenych bytu
}

/*Funkce pro vymazani ulozenych dat na SD karte
  Maze pouze .txt soubory, jine datalogger nepouziva*/
static void DeleteSDcard(void)
{
	FRESULT FileResult; 	//Vysledek po operaci se souborem
	FILINFO FileInfo;   	//informace o souboru
	DIR Directory;  		//Objekt adresare (slozky), slozky nejsou na SD karte vyuzivany, ale je to potrebna promena pro hledani souboru
	char FileList[11][30];	//pro ulozeni nazvu vsech souboru na karte (az 10 slave + master + status)
	uint8_t FileCnt = 0;	//pocitadlo nalezenych souboru na SD karte

	if((FileResult = f_mount(&SD_FatFs, (TCHAR const*)SDdiskPath, 0)) == FR_OK)				//nacteni souboroveho systemu SD karty (pripojeni na SD kartu)
	{
		FileResult =  f_findfirst(&Directory, &FileInfo, (TCHAR const*)SDdiskPath, "*.txt");   //hledani prvniho souboru na SD karte s priponou .txt

		while (FileResult == FR_OK && FileInfo.fname[0]) { 								//pokud hledani skoncilo uspesne a byl nalazen soubor
			memcpy(&FileList[FileCnt], &FileInfo.fname, 1+strlen(FileInfo.fname));		//ulozeni nazvu nalezeneho souboru (delka nazvu + 1 aby byl nazev ulozen i s ukoncovacim znakem "\0" )
			++FileCnt;																	//zvyseni pocitadla nalezenych souboru
			FileResult = f_findnext(&Directory, &FileInfo);   							//hledani dalsiho souboru
		}

		f_closedir(&Directory);							//uzavreni slozky (slozky nejsou na SD karte pouzivany)

		for(uint8_t i = 0; i < FileCnt; ++i)			//pro kazdy nalezeny soubor
		{
			if((FileResult = f_unlink(FileList[i])) != FR_OK)		//odstraneni souboru ze souboroveho systemu SD karty (oznaceni clusteru jako free)
				printf("Unlink %s Error (%i)\r\n", FileList[i], FileResult);	//informace pokud se nepovede odstraneni souboru
		}

		SD_CardInfo();			//vypocet noveho zaplneni SD karty
	}
	else
		printf("Mount Error (%i)\r\n", FileResult);		//informace pokud se nelze pripojit na SD kartu

	f_mount(NULL, "", 0);		//odpojeni od souboroveho systemu SD karty
}

/*Funkce pro udeslani zpravy Status pres UART do PC
  Volano ze stavu WAIT jednou za 10s pokud je PC pripojen
  Zprava Status obsahuje info o zaplneni SD karty, sile prijateho signalu od slave jednotek a stav baterie jednotek*/
static void SendStatusMsg(void)
{
	StatusToPCTimestamp = UTIL_TIMER_GetCurrentTime();		//casova znacka posledniho odeslani zpravy Status do PC
	char StatusMsg[200];									//pro vytvoreni zpravy Status

	//sestaveni 1. casti zpravy Status (Status identifikator zpravy pro PC, dale info o zaplneni SD karty
	sprintf(StatusMsg, "Status%lu,%lu,%lu,%.02f,", SDUsage.TotalKiB, SDUsage.FreeKiB, SDUsage.UseKiB, SDUsage.UsePER);

	for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)	//provedeni pro vsechny slave jednotky
	{
		if((Slave[i].Enable == true) && (Slave[i].RSSI < 0))			//pokud je povoleno vycitani slave jednotky a neni vypadek spojeni s jednotkou (pri vypadku spojeni RSSI = 0)
			sprintf(StatusMsg + strlen(StatusMsg),"%i,%u,", Slave[i].RSSI, Slave[i].BatteryLvl);	//vlozeni sily signalu posleni prijate zpravy od slave jednotky a stavu baterie
		else															//pokud neni vycitani slave jednotky povoleno nebo je vypadek spojeni se slave jednotkou nejsou hodnoty sily signálu a baterie validni
			sprintf(StatusMsg + strlen(StatusMsg),"%s,", "NaN,NaN");	//proto je vlozeno NaN, reprezentujici nedostupne data
	}
	sprintf(StatusMsg + strlen(StatusMsg),"\n");						//vlozeni ukoncovaciho znaku
	HAL_UART_Transmit(&huart1, (uint8_t*)StatusMsg, strlen(StatusMsg),HAL_MAX_DELAY);	//odeslani zpravy Status pres UART do PC
}

/* Funkce pro vypocet celkove kapacity a zaplneni SD karty
   @return true, pokud vypocet byl uspesny, false v pripadě chyby*/
static bool SD_CardInfo(void)
{
	  FATFS *SD_GetFree;		//objekt souboroveho systemu
	  DWORD FreeClusters;		//pocet volnych clusteru
	  if((f_getfree((TCHAR const*)SDdiskPath, &FreeClusters, &SD_GetFree)) != FR_OK) return false;	//funkce f_getfree nacte informace o souborovem systemu SD karty do SD_GetFree a do FreeClusters vlozi pocet volnych clusteru
	  else
	  {
			SDUsage.FreeKiB = FreeClusters * SD_GetFree->csize / 2;						//vypocet volneho mista na SD karte (pocet volnych clusteru * velikost clusteru (pocet sektoru v clusteru), (/2 pro prevod na kiB, protoze velikost sektoru 512 B / 1024 = 0.5)
			SDUsage.TotalKiB = SD_GetFree->csize * (SD_GetFree->n_fatent -2) / 2;		//vypocet celkove kapacity SD karty (velikost clusteru (pocet sektoru v clusteru) * celkovy pocet clusteru), (/2 opet pro prevod na kiB)
			SDUsage.UseKiB = SDUsage.TotalKiB - SDUsage.FreeKiB;						//vypocet vyuziteho mista na SD karte (celkova  kapacita - volna kapacita)
			SDUsage.UsePER = 100.0 * (SDUsage.UseKiB / ((float)SDUsage.TotalKiB));		//vypocet procentualniho vyuziti SD karty
	  }
	  return true;
}

/*Funkce pro zapsani Statusu na SD kartu
  Volano ze stavu WAIT jednou za hodinu
  Status obsahuje cas zapsani na SD kartu, zaplneni SD karty a nastaveni vsech jednotek*/
static void StoreStatusToSD(void)
{
	  StatusStoreTimestamp = UTIL_TIMER_GetCurrentTime() + 3600000;	//ulozeni casove znacky, kdy ma byt provedeno ulozeni dalsiho Statusu na SD kartu
	  SysTimeUpdate();								//aktualizace systemoveho casu v SystemTime
	  SystemTime_t StoreTimeStamp = SystemTime;		//ulozeni casove znacky zapsani dat do souboru na SD karte

	  char FileName[11] = "Status.txt\0";			//nazev souboru, kde je status ukladan
	  FIL FileObject; 								//Objekt souboru
	  FRESULT FileResult; 							//Vysledek po operaci se souborem
	  uint wb; 										//pro zavolani f_write (pocet zapsanych bytu do souboru)

	  char WriteData[3000];							//data pro zapsani na SD kartu
	  uint32_t DataSize;							//velikost dat pro zapsani na SD kartu v bytech

	  //formatovani dat pro zapis (vlozeni casu zapsani do souboru, zaplneni SD karty, hlavicky pro data a nastaveni master jednotky)
	  DataSize = sprintf(WriteData, "Date/Time: %02u.%02u.20%02u/%02u:%02u:%02u.%03u\r\nSD card memory usage: %.02f %%\r\n", StoreTimeStamp.Day, StoreTimeStamp.Month, StoreTimeStamp.Year,
			  StoreTimeStamp.Hours, StoreTimeStamp.Minutes, StoreTimeStamp.Seconds, StoreTimeStamp.Milliseconds, SDUsage.UsePER);
	  DataSize += sprintf(WriteData + DataSize, "UnitType	Unit ID		Measure En.	Measure T [ms]	Sig. str [dBm]	Battery [%%]	Sensor		Measure Unit:1		Measure Unit:2\r\n");
	  DataSize += sprintf(WriteData + DataSize, "Log_Unit\txx\t\t%s\t\t%lu\t\txx\t\txx\t\t%s\t\t%s\t\t%s\r\n", Master.EnableMeasure?"true":"false", Master.MeasurePeriod, SensorType[Master.SensorType],
			  UnitType[Master.UnitName1], UnitType[Master.UnitName2]);

	  for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)	//pro kazdou slave jednotku
	  {
		  if(Slave[i].Enable == true)		//pokud je povoleno vycitani slave jednotky je vlozeno nastaveni slave jednotky, sila signalu a stav baterie
			  DataSize += sprintf(WriteData + DataSize, "Sensor_Unit\t%u\t\t%s\t\t%lu\t\t%d\t\t%u\t\t%s\t\t%s\t\t%s\r\n", Slave[i].ID, Slave[i].Enable?"true":"false", Slave[i].MeasurePeriod, Slave[i].RSSI, Slave[i].BatteryLvl, SensorType[Slave[i].SensorType], UnitType[Slave[i].UnitName1], UnitType[Slave[i].UnitName2]);
		  else								//pokud neni povoleno vycitani slave jednotky neni nastaveni slave jednotky dostupne (vlozeni prazdnych hodnot)
			  DataSize += sprintf(WriteData + DataSize, "Sensor_Unit\t--\t\t%s\t\t--\t\t--\t\t--\t\t--\t\t--\t\t\t--\r\n", Slave[i].Enable?"true":"false");
	  }
	  DataSize += sprintf(WriteData + DataSize, "\n\n");	//vlozeni 2 radkove mezery

	  if((FileResult = f_mount(&SD_FatFs, (TCHAR const*)SDdiskPath, 0)) == FR_OK)					//nacteni souboroveho systemu SD karty (pripojeni k SD karte)
	  {
		  if((FileResult = f_open(&FileObject, FileName, FA_OPEN_APPEND  | FA_WRITE )) == FR_OK) 	//otevreni/vytvoreni souboru pro zapis
		  {
				FileResult = f_write(&FileObject, WriteData, DataSize, &wb) != FR_OK;				//zapis sestavenych dat do souboru
				f_close(&FileObject);																//uzavreni souboru
		  }
	  }

	  if(FileResult != FR_OK)
		  printf("Zapis do souboru selhal, Statts Error (%i)\r\n", FileResult);						//informace pokud zapis do souboru selhal

	  SD_CardInfo();				//vypocet noveho zaplneni SD karty
	  f_mount(NULL, "", 0);			//odpojeni od souboroveho systemu SD karty
}

/*Funkce pro resetovani nastaveni dataloggeru
  Vymazani uzivatelskeho nastaveni*/
static void RessetSettings(void)
{
	memset(&Master, 0, sizeof(MasterInit_t));		// Vynulovani vsech prvku struktury Master
	Master.SensorType = TYP_SNIMACE;				// Opetovne nastavení typu snimace (po resetu ma furt stejny snimac)
	Master.UnitName1 = VELICINA_1;					// veliciny 1
	Master.UnitName2 = VELICINA_2;					// veliciny 2
	for (uint8_t i = 0; i < (sizeof(Slave) / sizeof(Slave[0])); ++i)	//prochazeni vsech slave jednotek
	{
		memset(&Slave[i], 0, sizeof(SlaveInit_t));		// Vynulovani vsech prvku struktury Slave
	}
	StoreToFLASH();									//ulozeni nastaveni na FLASH
}


/*Funkce pro zpracovani signalu DCF77
  Volano pri kazdem pruchodu stavem WAIT
  Signal DCF obsahuje informaci o nadchazejici minute, kazdou minutu je poslano 59 bitu*/
static void DCF_Process(void)
{
	DCF.PinState = HAL_GPIO_ReadPin(GPIOC, GPIO_PIN_13);			//nacteni log. urovne na vystupu moducu DCF77
	HAL_GPIO_WritePin(LED3_GPIO_PORT, LED3_PIN, DCF.PinState);		//rozsviceni LED3 podle vystupu modulu DCF77

	if (DCF.PrevPinState != DCF.PinState)				//(detekce hrany) pokud doslo ke zmene urovne signalu od predchoziho pruchodu
	{
		DCF.PrevPinState = DCF.PinState;				//ulozeni noveho stavu pinu (vystupu modulu DCF77)
		DCF.EdgeTS = UTIL_TIMER_GetCurrentTime();		//ulozeni casove znacky hrany

		if ((DCF.EdgeTS - DCF.PrevRaiseEdgeTS) < 900)	//pulsy prijate do 900 ms od predchozi nastupne hrany jsou neplatne (sum)
			return;

		if ((DCF.EdgeTS - DCF.RaiseEdgeTS) < 80)		//pulzy prijate do 80 ms od posledni nastupne hrany nejsou platne (sum)
			return;										//prilis kratky puls = neplatny


		if (DCF.PinState == true)						//pokud je pin ve vysoke urovni = detekce nastupne hrany
		{
			if (DCF.IsPulse == false)					//pokud jeste nebyl detekovat pulz na singalu DCF (mozne ruseni/zaknity behem pulsu signalu DCF)
			{
				DCF.RaiseEdgeTS = DCF.EdgeTS;			//ulozeni casove znacky hrany jako cas nastupne hrany pulzu
				DCF.IsPulse = true;						//indikace ze na signalu DCF probiha zrovna pulz
			}
		}
		else											//pokud je pin v nizke urovni = detekce sestupne hrany
		{
			if (DCF.IsPulse == true)					//pokud predchazel puls pred sestupnou hranou
			{
				DCF.IsPulse = false;					//deaktivace indikovaneho pulsu
				DCF.FallEdgeTS = DCF.EdgeTS;			//ulozeni casove znakcy hrany jako cas sestupne hrany

				if ((DCF.RaiseEdgeTS - DCF.PrevRaiseEdgeTS) > 1800)		//pokud je cas mezi predchozi nastupnou hrano a posledni nastupnou hranou vetsi nez 1800 ms odpovida to zacatku minuty
				{														//dlouha pomlka (vynechany bit) odpovida zacatku nove minuty
					if (DCF.DataValid == true)							//pokud jsou nactena data z predchozi minuty validni jsou pouzita jako novy systemovy cas master jednotky
					{													//signal DCF posila kazdou minutu casovou informaci o nadchazejici minute
						SysTimeUpdate();
						RTC_CurTicks = UTIL_TimerDriver.GetTimerValue();	//nova casova znacka RTC pro kterou jsou data v SystemTime platna
						LastDCFsynchTimestamp = UTIL_TIMER_GetCurrentTime(); //casova znacka posledni provedene casove synchronizace z modulu DCF

						SystemTime.Year = DCF.Year;
						SystemTime.Month = DCF.Month;
						SystemTime.Day = DCF.Day;							//ulozeni noveho systemoveho casu master jednotky
						SystemTime.Hours = DCF.Hour;
						SystemTime.Minutes = DCF.Minute;
						SystemTime.Seconds = SystemTime.Milliseconds = 0;
					}
					DCF.DataCnt = 0;				//vynulovani pocitadla nactenych bitu signalu behem minuty
					DCF.DataValid = false;			//oznaceni dat jako nevalidni
					DCF.MinuteStart = true;			//detekce zacatku minuty
					DCF.Minute = DCF.Hour = DCF.Day = DCF.Month = DCF.Year = 0;		//vynulovani predchozi nactene casove informace
					printf("Minute start\r\n");		//pouze pro debug
				}

				DCF.PrevRaiseEdgeTS = DCF.RaiseEdgeTS;	//aktualizace casove znacky predchozi nastupne hrany podle posledni nastupne hrany

			}

			if (DCF.MinuteStart == false)				//pokud nebyl detekovan zacatek minuty, neukladat nacitana data ze signalu (neni znam zacatek dat)
				return;

			if ((DCF.FallEdgeTS - DCF.RaiseEdgeTS) < 140)	//dekodovani logicke urovne z delky nasteneho pulzu
				DCF.Data[DCF.DataCnt] = 0;					//puls < 140 ms = log 0
			else
				DCF.Data[DCF.DataCnt] = 1;					//puls >= 140 ms = log 1
			DCF.DataCnt++;									//zvyseni pocitadla nactenych bitu signalu

			if (DCF.DataCnt == 59)			//vsech 59 bitu nacteno
			{
				DCF.MinuteStart = false;	//konec nacitani

				  //****kontroda dat****
				  DCF.DataValid = true;		//oznaceni dat jako validni (pukud nejaka kontrola neprojde budou pozdeji oznaceny jako nevalidni)

				  if ((DCF.Data[0] != 0) && (DCF.Data[20] != 1))	//kontrola znamych bitu v signalu
					  DCF.DataValid = false;						//1. bit vzdy 0,  21. bit vzdy 1

				  //kontrola parity minut
				  uint8_t BitSum = 0;
				  for (uint8_t i = 0; i < 7; i++)					//suma vsech minutovych bitu
				  {
					  BitSum += DCF.Data[21 + i];
				  }
				  if(BitSum % 2 != DCF.Data[28])					//kontrola sude parity
					  DCF.DataValid = false;						//pokud zbytek po deleni 2 neodpovida paritnimu bitu = data neplatna

				  //kontrola parity hodin
				  BitSum = 0;
				  for (uint8_t i = 0; i < 6; i++)					//stejny postup jako pro kontrolu minut
				  {
					  BitSum += DCF.Data[29 + i];
				  }
				  if(BitSum % 2 != DCF.Data[35])					//suda parita
					  DCF.DataValid = false;

				  //kontrola parity datumu
				  BitSum = 0;
				  for (uint8_t i = 0; i <22; i++)					//stejny postup jako pro kontrolu minut
				  {
					  BitSum += DCF.Data[36 + i];
				  }
				  if(BitSum % 2 != DCF.Data[58])					//suda parita
					  DCF.DataValid = false;


				  printf("Data valid: %s", (DCF.DataValid ? "true":"false"));	//pouze pro debug
				  if (DCF.DataValid == false)						//pokud nactena data nejsou validni nepokracovat
					  return;

				  //****prevod dat na cas***
				  DCF.Minute += DCF.Data[27] ? 40 : 0;				//prevod bitu na minuty
				  DCF.Minute += DCF.Data[26] ? 20 : 0;
				  DCF.Minute += DCF.Data[25] ? 10 : 0;
				  DCF.Minute += DCF.Data[24] ?  8 : 0;
				  DCF.Minute += DCF.Data[23] ?  4 : 0;
				  DCF.Minute += DCF.Data[22] ?  2 : 0;
				  DCF.Minute += DCF.Data[21] ?  1 : 0;

				  DCF.Hour += DCF.Data[34] ? 20 : 0;				//prevod bitu na hodiny
				  DCF.Hour += DCF.Data[33] ? 10 : 0;
				  DCF.Hour += DCF.Data[32] ?  8 : 0;
				  DCF.Hour += DCF.Data[31] ?  4 : 0;
				  DCF.Hour += DCF.Data[30] ?  2 : 0;
				  DCF.Hour += DCF.Data[29] ?  1 : 0;

				  DCF.Day += DCF.Data[41] ? 20 : 0;					//prevod bitu na dny
				  DCF.Day += DCF.Data[40] ? 10 : 0;
				  DCF.Day += DCF.Data[39] ?  8 : 0;
				  DCF.Day += DCF.Data[38] ?  4 : 0;
				  DCF.Day += DCF.Data[37] ?  2 : 0;
				  DCF.Day += DCF.Data[36] ?  1 : 0;

				  DCF.Month += DCF.Data[49] ? 10 : 0;				//prevod bitu na mesic
				  DCF.Month += DCF.Data[48] ?  8 : 0;
				  DCF.Month += DCF.Data[47] ?  4 : 0;
				  DCF.Month += DCF.Data[46] ?  2 : 0;
				  DCF.Month += DCF.Data[45] ?  1 : 0;

				  DCF.Year += DCF.Data[57] ? 80 : 0;				//prevod bitu na rok
				  DCF.Year += DCF.Data[56] ? 40 : 0;
				  DCF.Year += DCF.Data[55] ? 20 : 0;
				  DCF.Year += DCF.Data[54] ? 10 : 0;
				  DCF.Year += DCF.Data[53] ?  8 : 0;
				  DCF.Year += DCF.Data[52] ?  4 : 0;
				  DCF.Year += DCF.Data[51] ?  2 : 0;
				  DCF.Year += DCF.Data[50] ?  1 : 0;

				  printf("\n%u.%u.%u %u:%u\r\n",DCF.Day, DCF.Month, DCF.Year, DCF.Hour, DCF.Minute); //pouze pro debug

				  // kontrola zda je nactena casova znacka z DCF shodna z predchozi nactenou znackou DCF + 1 minuta
				  // cas z modulu DCF je oznacen jako validni pouze, pokud jsou dobre nacteny 2 po sobe jdouci casove informace z DCF modulu
				  DCF.DataValid = false;									//oznaceni dat jako neplatna
				  if (DCF.Year == DCF.PrevYear)								//musi se shodovat rok
					  if (DCF.Month == DCF.PrevMonth)						//mesic
						  if (DCF.Day == DCF.PrevDay)						//i den
							  if ((DCF.Hour == DCF.PrevHour) || DCF.Hour == ((DCF.PrevHour == 23) ? 0 : DCF.PrevHour + 1))	//musi se zhodovat dodiny (nebo nanejvis + 1)
								  if (DCF.Minute == ((DCF.PrevMinute == 59) ? 0 : DCF.PrevMinute + 1))						//minuty musi byt o 1 vetsi
								  {
									  printf("Pouzitelny cas DCF77: ");														//pouze pro debug
									  printf("%u.%u.%u %u:%u\r\n",DCF.Day, DCF.Month, DCF.Year, DCF.Hour, DCF.Minute);		//pouze pro debug
									  DCF.DataValid = true;					//oznaceni dat jako validni
									  	  	  	  	  	  	  	  	  	  	//data nezle pouzit hned jsou platna az se zacatkem nasledujici minuty
								  }

				  //ulozeni nacteneho casu z modulu DCF jako predchozi nacteny cas z modulu
				  DCF.PrevYear = DCF.Year;
				  DCF.PrevMonth = DCF.Month;
				  DCF.PrevDay = DCF.Day;
				  DCF.PrevHour = DCF.Hour;
				  DCF.PrevMinute = DCF.Minute;
			}
		}
	}
}




/* USER CODE END 4 */

/**
  * @brief  This function is executed in case of error occurrence.
  * @retval None
  */
void Error_Handler(void)
{
  /* USER CODE BEGIN Error_Handler_Debug */
  /* User can add his own implementation to report the HAL error return state */
  __disable_irq();
  while (1)
  {
  }
  /* USER CODE END Error_Handler_Debug */
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* USER CODE BEGIN 6 */
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* USER CODE END 6 */
}
#endif /* USE_FULL_ASSERT */
