import { Box, Button, useTheme } from "@mui/material";
import { tokens } from "../../theme";
import Header from "../../components/Header";
import AddIcon from '@mui/icons-material/Add';
import { useState, useContext, useEffect } from "react";
import { Modal } from "../../components/Modal";
import { ModalContext } from '../../context/ModalContext';
import canAddNewGraph from "../../functions/canWeAddNewGraph";
import { useIsMounting } from "../../hooks/useIsMounting";
import GraphDataSet from "../../components/GraphDataSet";
import Graph from "../../components/Graph";
import EditGraph from "../../components/EditGraph";
import { BAR_PROPS, BULLET_PROPS, LINE_PROPS, PIE_PROPS } from "../../constants/graphProps";
import DeleteIcon from '@mui/icons-material/Delete';

const MAX_WIDTH = 6
const MAX_HEIGHT = 3
const LS_DASHBOARD_NAME = 'local_storage_dashboard_graphs'

const Dashboard = () => {
  const theme = useTheme();
  const colors = tokens(theme.palette.mode);

  const isMounting = useIsMounting();

  const [colsToShow, setColsToShow] = useState(1);
  const [rowsToShow, setRowsToShow] = useState(1);
  const [errorMessage, setErrorMessage] = useState();
  const [graphList, setGraphList] = useState(JSON.parse(localStorage.getItem(LS_DASHBOARD_NAME)) || []);
  const [radioGraphValue, setRadioGraphValue] = useState('bar');
  const [radioDataValue, setRadioDataValue] = useState('result');

  const [modalType, setModal] = useState("")
  const {modal, open: openModal, close: closeModal} = useContext(ModalContext)
  const [graphOnEditing, setGraphOnEditing] = useState()


  const handleRadioGraphChange = (event) => {
    setRadioGraphValue(event.target.value);
  };

  const handleRadioDataChange = (event) => {
    setRadioDataValue(event.target.value);
  };

  const onAddBtnClick = (event) => {
    setErrorMessage(null);

    if (!canAddNewGraph(colsToShow, rowsToShow, MAX_WIDTH, MAX_HEIGHT, graphList)) {
      setErrorMessage("There is no available space for this graph.")
      return
    }

    const graphProps = radioGraphValue === "bar"
    ? BAR_PROPS
    : radioGraphValue === "line"
    ? LINE_PROPS
    : radioGraphValue === "pie"
    ? PIE_PROPS
    : BULLET_PROPS

    setGraphList(prev => prev.concat({
      colors: colors,
      type: radioGraphValue,
      dataType: radioDataValue,
      cols: colsToShow,
      rows: rowsToShow,
      graphProps: graphProps,
    }));

    closeModal()
  };

  const removeGraphFromList = (index) => {
    setGraphList(prev => prev.filter((_, idx) =>
      idx !== index
    ))

    setGraphList(prev => prev.sort((a, b) => b.rows - a.rows))
  }

  const onChangeGraphProps = (index, newProps) => {
    setGraphList(prev => prev.map((graph, idx) => {
        if (idx === index) {
          return {...graph, graphProps: {...graph.graphProps, ...newProps}}
        }

        return graph
      }
    ))

    setGraphList(prev => prev.sort((a, b) => b.rows - a.rows))
  }

  // Setting local storage graph list if we change graphListState
  useEffect(() => {
    if (!isMounting) {
      if (graphList.length === 0) {
        localStorage.removeItem(LS_DASHBOARD_NAME)
        return
      }

      localStorage.setItem(LS_DASHBOARD_NAME, JSON.stringify(graphList))
    }
  }, [graphList])

  return (
    <>
    <Box m="10px">
      {/* HEADER */}
      <Box display="flex" justifyContent="space-between" alignItems="center">
        <Header title="DASHBOARD"/>

        <Box>
          <Button
            sx={{
              backgroundColor: colors.blueAccent[700],
              color: colors.grey[100],
              fontSize: "14px",
              fontWeight: "bold",
              padding: "5px 20px",
            }}
            onClick={() => {
              setModal("addGraph")
              setColsToShow(1)
              setRowsToShow(1)
              setRadioGraphValue("bar")
              setRadioDataValue("result")
              setErrorMessage(null);
              openModal()
            }}
          >
            <AddIcon sx={{ mr: "10px" }} />
            Add graph
          </Button>
        </Box>
      </Box>

      {/* GRID & CHARTS */}
      <Box
        display="grid"
        gridTemplateColumns="repeat(12, 1fr)"
        gridTemplateRows="repeat(3, 1fr)"
        gridAutoRows="100px"
        gap="20px"
      >
        {!graphList.length &&
          <Box gridColumn="span 12"> No graphs added </Box>
        }
        {graphList.map((graph, idx) =>
          <>
            <Box
              gridColumn={`span ${(graph.cols * 2)}`}
              gridRow={`span ${graph.rows}`}
              backgroundColor={colors.primary[400]}
              key={idx + 'graph' + graph.type + graph.dataType}
              position="relative"
            >
              <Box sx={{padding: "20px"}}>
                <span
                    style={{cursor: "pointer", color: "inherit", textDecoration: "inherit"}}
                    onClick={() => {
                      setModal("editGraph")
                      setGraphOnEditing(idx)
                      openModal()
                    }}
                  >
                  Edit
                </span>
                <span
                  style={{cursor: "pointer", float: "right"}}
                  onClick={() => {removeGraphFromList(idx)}}
                >
                  <DeleteIcon/>
                </span>
              </Box>
              <Graph
                colors={graph.colors}
                type={graph.type}
                dataType={graph.dataType}
                rows={graph.rows}
                graphProps={graph.graphProps}
              />
            </Box>

            {/* Edit graph modal */}
            {modal && modalType === "editGraph" && graphOnEditing === idx &&
            <Modal title="Graph parameters" width="80%" left="3%" onClose={closeModal}>
                <div>
                    <EditGraph
                      type={graph.type}
                      graph={graph}
                      onChangeGraphProps={
                        (newProps) => {onChangeGraphProps(idx, newProps)}
                      }
                      graphProps={graph.graphProps}
                    />
                </div>
            </Modal>
            }
          </>
        )}
        </Box>
      </Box>

    {/* Add graph modal */}
    {modal && modalType === "addGraph" &&
      <Modal
        title="Select type of graph"
        width="50%"
        left="15%"
        onClose={closeModal}
      >
        <GraphDataSet
          radioGraphValue={radioGraphValue}
          handleRadioGraphChange={handleRadioGraphChange}
          radioDataValue={radioDataValue}
          handleRadioDataChange={handleRadioDataChange}
          colsToShow={colsToShow}
          setColsToShow={setColsToShow}
          rowsToShow={rowsToShow}
          setRowsToShow={setRowsToShow}
          errorMessage={errorMessage}
        />

        <div style={{display: "flex", justifyContent: "right", marginRight: "50px"}}>
          {radioGraphValue && radioDataValue
          ?
          <Button
            sx={{
              backgroundColor: colors.blueAccent[700],
              color: colors.grey[100],
              fontSize: "16px",
              fontWeight: "bold",
              padding: "5px 20px",
            }}
            onClick={() => {
              onAddBtnClick()
            }}
          >
            Create
          </Button>
          :
          <Button
            sx={{
              backgroundColor: colors.blueAccent[700],
              color: colors.grey[100],
              fontSize: "16px",
              fontWeight: "bold",
              padding: "5px 20px",
            }}
            onClick={() => {
              onAddBtnClick()
            }}
            disabled
          >
            Create
          </Button>
          }
        </div>
      </Modal>
    }
  </>
  );
};

export default Dashboard;