/**
 * operators.js
 * 
 * Contains JS classes, that provide abstraction for making rules (especially 
 * operator rules)
 * 
 * @author Vojtěch Dvořák
 */


const OPERATOR_LEX_PREC = 0;


/**
 * Precedence object template, that holds information about level of precedence
 * and associativity of a rule
 * @param {Int} level_ Level of the precedence (higher integer means higher 
 * precedence)
 * @param {Function} func_ Precedence function of the tree-sitter library 
 * (prec.left by default)
 */
class Precedence {
    constructor(level_, func_ = prec.left) {
        this.level = level_;
        this.func = func_;
    }
}



/**
 * @class Operator object template containing all important information 
 * about given operator (name of node, operator sign...)
 * @param {String} name_ Name of node (representing operator) in syntax tree
 * @param {Array} pattern_ Sequence of tokens/symbols their sequence represents
 * rule for operator, simple tokens can be passed as strings, symbols should be
 * passed as a string that starts with '$' (so symbols should look like:
 *  $'symbol_name')
 * @param {Precedence} precedence_ Precedence of operator
 * @param {Array} fields_ Array of field names for symbols in the rule of 
 * operator (null by default - any field is not defined)
 * @note If you need to pass token, that starts with '$'just escape it with
 * '\\' (one backslash is not enough cause JavaScript escape sequences)
 * @important pattern_ and fields_ must have same length!
 */
class Operator {
    constructor(name_, pattern_, precedence_, fields_ = null) {
        this.name = name_;
        this.pattern = pattern_;
        this.fields = fields_;
        this.seq_arr = Operator.make_seq_arr(pattern_, fields_);
        this.precedence = precedence_;
    }


    /**
     * Creates array that can be easily passed to tree-sitter seq function
     * @param {Array} pattern Pattern array given by user (see Operator object)
     * @param {Array} fields Array with field names for symbols (can be null if
     * if there is no need of fields)
     * @returns Array with sequence of tree-sitter symbols
     */
    static make_seq_arr(pattern, fields = null) {
        let seq_arr = []; //< Output array

        for(let index in pattern) {
            let cur = pattern[index], tb_pushed;
            if(cur.startsWith('$')) { //< Detect special symbol character
                tb_pushed = {
                    type: "SYMBOL", 
                    name: cur.substr(1), //< Remove $ from the beginning
                };
            }
            else {
                tb_pushed = token(
                    prec(
                        OPERATOR_LEX_PREC, 
                        cur.startsWith('\\') ? cur.substr(1) : cur, //< Remove \ from the beginning
                    )
                );
            }

            if(fields != null) { //< Wrapping symbol to the field object (if field is given)
                if(pattern.length == fields.length && fields[index] !== null) {
                    tb_pushed = field(fields[index], tb_pushed);
                }
            }

            seq_arr.push(tb_pushed); //< Add new element to the end of the output array
        }

        return seq_arr;
    }


    /**
     * Get sequence rule of defined operator
     * @param {Precedence} prec_ Object with precedence level and precedence
     * function for returned rule
     * @param  {...Symbol|Token} seq_elements Variable amount of symbols and 
     * tokens (strings/regexes), that make up rule (in sequence sorted base 
     * on the  order of arguments of this function)  
     * @returns Sequence rules of symbols and tokens given as parameters
     */
    make_seq_rule() {
        return this.precedence.func(this.precedence.level, seq(
                ...this.seq_arr
            )
        );
    }
}


/**
 * @class Specialized class for binary operators, that have operands both with
 * the same type
 * @param {String} name_ Name of node, that represents operator in syntax tree
 * @param {String} op_sign_ Token that represents operator in parsed string
 * @param {String} left_op_ Symbol/Token that represents left operand of operator
 * @param {String} right_op_ Symbol/Token that represents right operand of operator
 * @param {Precedence} precedence_ Precedence of operator
 */
class BinaryOperator extends Operator {
    constructor(name_, op_sign_, left_op_, right_op_, precedence_) {
        let fields = ["lop", null, "rop"]; //< Field names for typical binary operator (in form <operand> <operator> <operand>)
        
        super(name_, [left_op_, op_sign_, right_op_], precedence_, fields);
    }
}


/**
 * @class Specialized class for binary operators, that have operands both with
 * the same type
 * @param {String} name_ Name of node, that represents operator in syntax tree
 * @param {String} operator_sign_ Token that represents operator in parsed string
 * @param {String} operand_ Symbol/Token that represents both operands of operator
 * @param {Precedence} precedence_ Precedence of operator
 */
class BinaryOperatorSym extends BinaryOperator {
    constructor(name_, op_sign_, operands_, precedence_) {
        let fields = ["lop", null, "rop"]; //< Field names for typical binary operator (in form <operand> <operator> <operand>)
        
        super(name_, op_sign_, operands_, operands_, precedence_, fields);
    }
}


/**
 * @class Specialized Operator prototype, that abstracts unary operator from
 * general operators by adding default field pattern typical for
 * unary operators (<operator> 'op')
 * @param {String} name_ Name of node, that represents operator in syntax tree
 * @param {String} operator_sign_ Token that represents operator in parsed string
 * @param {String} operand_ Symbol/Token that represents operand of operator
 * @param {Precedence} precedence_ Precedence of operator
 */
class UnaryOperator extends Operator {
    constructor(name_, operator_sign_, operand_, precedence_) {
        let fields = [null, "op"]; //< Field names for typical unary operator (in form <operator> <operand>)

        super(name_, [operator_sign_, operand_], precedence_, fields);
    }
}

module.exports.OPERATOR_LEX_PREC = OPERATOR_LEX_PREC;
module.exports.Precedence = Precedence;
module.exports.Operator = Operator;
module.exports.BinaryOperator = BinaryOperator
module.exports.UnaryOperator = UnaryOperator;
module.exports.BinaryOperatorSym = BinaryOperatorSym;
