/**
 * grammar.js
 * 
 * Contains YARA grammar in format required by tree-sitter
 * 
 * @author Vojtěch Dvořák
 */


const op = require("./operators");


/**
 * Lexical precedences
 */
const LEX_PREC = {
    ID : 0,
    COMMENT : 0,
    OPERATOR : op.OPERATOR_LEX_PREC,
}


/**
 * 20 - not ~
 * 10 - alternatives |
 */

/**
 * Operators that may appear in the definition of hex_strings
 */
const HEX_STR_OP = {
    //                          rule name     sign       operands        precedence object
    NOT : new op.UnaryOperator("hex_str_not", "~", "$_hex_str_byte", new op.Precedence(20, prec.right)),
    ALT : new op.BinaryOperatorSym("hex_str_alt", "|", "$_hex_str_nested_sequence", new op.Precedence(10, prec.left)),
}


/**
 * Precedence of operators in regexes
 */
 const REGEXP_OP = {
    CAT : new op.Operator("regexp_cat", ["$_regexp_seq", "$_regexp_seq"], new op.Precedence(20)),
    ALT : new op.Operator("regexp_alt", ["$_regexp_seq", "|", "$_regexp_seq"], new op.Precedence(10)),
}

/**
 * Precedence of rule for "wrapping" symbol _expression (all types of 
 * expressions can be derived from it)
 */
const EXPR_PREC_LEVEL = 0;


/**
 * Reference value of dynamic precedence level of _regexp_class_element (it
 * is used in case of conflicts with other rules @see yara_gramar.conflicts)
 */
const REGEXP_CLS_EL_DYNAM_LEVEL = 0;

/**
 * Reference value of dynamic precedence level of _regexp_seq (it
 * is used in case of conflicts with other rules @see yara_gramar.conflicts)
 */
const REGEXP_SEQ_EL_DYNAM_LEVEL = 0;

/**
 * 160 - array access [], stucture access
 * 150 - unary minus -, bitwise not ~
 * 140 - multiplication *, division \, remainder %
 * 130 - addition +, subtraction -
 * 120 - left shift <<, right shift >>
 * 110 - bitwise and &
 * 100 - bitwise xor ^
 * 90 - bitwise or |
 * 80 - greater than >, greater than or equal >=, less than <, less than or equal >=
 * 70 - equal ==, not equal !=, contains, icontains, starts, istarts, endswith, iendswith, iequals, matches
 * 60 - defined
 * 50 - at, in
 * 40 - not
 * 30 - and
 * 20 - or
 */


/**
 * Operators for accessing variables, members of structures or elements of arrays...
 */
const VARIABLE_OP = {
    //                      rule name                       pattern                       precedence object       fields
    ARR : new op.Operator("array_access", ["$_variable", "[", "$_expression", "]"], new op.Precedence(160), ["array", null, "key", null]),
    STRUCT : new op.Operator("structure_access", ["$_variable", ".", "$_variable"], new op.Precedence(160), ["struct", null, "member"]),
}


/**
 * All available operators that can be substituted for primary_expression (to avoid repetetive rules)
 */
const PRIMARY_EXPRESSION_OP = {
    //                         rule name    sign      operand(s)                  precedence object          
    UMIN : new op.UnaryOperator("unary_minus", "-", "$_primary_expression", new op.Precedence(150, prec.right)),
    BNOT : new op.UnaryOperator("bitwise_not", "~", "$_primary_expression", new op.Precedence(150, prec.right)),

    MUL : new op.BinaryOperatorSym("multiplication", "*", "$_primary_expression", new op.Precedence(140)),
    DIV : new op.BinaryOperatorSym("division", "\\\\", "$_primary_expression", new op.Precedence(140)),
    ADD : new op.BinaryOperatorSym("addition", "+", "$_primary_expression", new op.Precedence(130)),
    SUB : new op.BinaryOperatorSym("subtraction", "-", "$_primary_expression", new op.Precedence(130)),

    LSH : new op.BinaryOperatorSym("left_shift", "<<", "$_primary_expression", new op.Precedence(120)),
    RSH : new op.BinaryOperatorSym("right_shift", ">>", "$_primary_expression", new op.Precedence(120)),
    BAND : new op.BinaryOperatorSym("bitwise_and", "&", "$_primary_expression", new op.Precedence(110)),
    BXOR : new op.BinaryOperatorSym("bitwise_xor", "^", "$_primary_expression", new op.Precedence(100)),
    BOR : new op.BinaryOperatorSym("bitwise_or", "|", "$_primary_expression", new op.Precedence(90)),
}


/**
 * Precedence of remainer operator
 * Note: Rule for remainder is defined explicitly, because there is similar 
 * rule for percent quantifier, so possible conflicts between these two symbols
 * can be better resolved
 */
const REM_EXPR_PREC = new op.Precedence(140);


/**
 * Precedence of "in" operator - it can be used in combination with
 * string count or with string identifier
 */
const IN_EXPR_PREC = new op.Precedence(50);

/**
 * All operators that can represent expression symbol
 */
 const EXPRESSION_OP = {
    //                             rule name     op_sign      operand(s)          precedence object            
    GTH : new op.BinaryOperatorSym("greater_than", ">", "$_primary_expression", new op.Precedence(80)),
    GTE : new op.BinaryOperatorSym("greater_than_or_equal", ">=", "$_primary_expression", new op.Precedence(80)),
    LTH : new op.BinaryOperatorSym("less_than", "<", "$_primary_expression", new op.Precedence(80)),
    LTE : new op.BinaryOperatorSym("less_than_or_equal", "<=", "$_primary_expression", new op.Precedence(80)),

    EQ : new op.BinaryOperatorSym("equal", "==", "$_primary_expression", new op.Precedence(70)),
    NEQ : new op.BinaryOperatorSym("not_equal", "!=", "$_primary_expression", new op.Precedence(70)),

    CON : new op.BinaryOperatorSym("contains", "contains", "$_primary_expression", new op.Precedence(70)),
    ICON : new op.BinaryOperatorSym("icontains", "icontains", "$_primary_expression", new op.Precedence(70)),
    STARTS : new op.BinaryOperatorSym("startswith", "startswith", "$_primary_expression", new op.Precedence(70)),
    ISTARTS : new op.BinaryOperatorSym("istartswith", "istartswith", "$_primary_expression", new op.Precedence(70)),
    END : new op.BinaryOperatorSym("endswith", "endswith", "$_primary_expression", new op.Precedence(70)),
    IEND : new op.BinaryOperatorSym("iendswith", "iendswith", "$_primary_expression", new op.Precedence(70)),
    IEQ : new op.BinaryOperatorSym("iequals", "iequals", "$_primary_expression", new op.Precedence(70)),
    MAT : new op.BinaryOperator("matches", "matches", "$_primary_expression", "$regexp", new op.Precedence(70)),

    DEF : new op.UnaryOperator("defined", "defined", "$_expression", new op.Precedence(60, prec.right)),

    AT: new op.BinaryOperator("at", "at", "$string_identifier", "$_primary_expression", new op.Precedence(50)),
}


/**
 * Operators for boolean expressions
 */
const BOOLEAN_EXPRESSION_OP = {
    //                     rule name  op_sign    operand(s)         precedence object
    NOT : new op.UnaryOperator("not", "not", "$_expression", new op.Precedence(40, prec.right)),
    AND : new op.BinaryOperatorSym("and", "and", "$_expression", new op.Precedence(30)),
    OR : new op.BinaryOperatorSym("or", "or", "$_expression", new op.Precedence(20)),
}


let yara_grammar = grammar({
    name: "yara",

    //Following tokens may appear everywhere:
    extras: $ => [ $.comment, /\s/ ],

    word: $ => $.identifier,

    inline: $ => [
        $._hex_str_start_element, 
        $._hex_str_inner_element,
    ],

    supertypes: $ => [
        $._expression,
        $._primary_expression,
        $._boolean_expression,
        $._string_set_element,
        $._rule_set_element,
    ],

    conflicts: $ => [
        [$._regexp_class_element, $.regexp_class_range], //Example: [a-] X [a-a]
        [$._regexp_seq, $.regexp_rep] //Example: a{1,a} X a{1,1}
    ],

    rules: {

        //Non-terminals

        //Starting rule
        yara_file: $ => repeat(
            choice(
                $.include,
                $.import,
                $.rule,
            )
        ),


        include: $ => seq(
            token(prec(LEX_PREC.OPERATOR + 1, "include")), //< There is collision with 'in' operator (with lex precedence 2)
            field("path", $.string_literal),
        ),
        

        import: $ => seq(
            "import",
            field("module", $.string_literal),
        ),


        rule: $ => seq(
            field("head", $.rule_head),
            field("body", $.rule_body),
        ),

        
        rule_head: $ => seq(
            field("mods", optional($.rule_mod_list)),
            "rule",
            $._rule_id,
            optional($._rule_tags),
        ),

        rule_mod_list: $ => repeat1(
            $.rule_modifier
        ),

        _rule_tags: $ => seq(
            ":",
            field("tags", $.tag_list),
        ),

        tag_list: $ => repeat1(
            $.tag
        ),

        rule_body: $ => seq(
            "{",
            optional($._meta_body),
            choice(
                seq(
                    optional($._internal_var_body),
                    optional($._strings_body),
                ),
                seq(
                    optional($._strings_body),
                    optional($._internal_var_body),
                ),
            ),
            $._condition_body,
            "}",
        ),
        

        //---------------------------------Metas-------------------------------

        _meta_body: $ => seq(
            "meta",
            ":",
            field("metas", $.meta_list),
        ),

        meta_list: $ => repeat1(
            $.meta,
        ),

        meta: $ => seq(
            $._meta_id,
            "=",
            field("value", $._meta_value),
        ),

        _meta_value: $ => choice(
            $.string_literal,
            $.bool_literal,
            $.int_literal,
        ),

        //------------------------------Variables------------------------------
        //Avast feature

        _internal_var_body: $ => seq(
            "variables",
            ":",
            field("variables", $.internal_var_list),
        ),

        internal_var_list: $ => repeat1(
            $.internal_var
        ),

        internal_var: $ => seq(
            $._internal_var_id,
            "=",
            field("value", $._internal_var_value),
        ),

        _internal_var_value: $ => choice(
            $._expression,
        ),

        //-------------------------------Strings-------------------------------

        _strings_body: $ => seq(
            "strings",
            ":",
            field("strings", $.strings_list),
        ),  

        strings_list: $ => repeat1(
            $.string,
        ),

        string: $ => seq(
            field("id", $.string_identifier),
            "=",
            field("value", $._string_value),
        ),

        _string_value: $ => seq(
            choice(
                $.string_literal,
                $.hex_str,
                $.regexp,
            ),
            optional(field("mods", $.str_mod_list)),
        ),

        str_mod_list: $ => repeat1(
            $.str_mod
        ),

        str_mod: $ => choice(
            "nocase",
            "base64",
            "base64wide",
            "wide", 
            "ascii",
            "xor",
            "fullword",
            "private",
            seq(
                "base64", "(", field("alphabet", $.string_literal), ")"
            ),
            seq(
                "xor", "(", field("range", $.byte_range), ")"
            ),
        ),

        byte_range: $ => choice(
            seq(
                field("lower", $.uint_literal), 
                "-", 
                field("upper", $.uint_literal),
            ),
            field("key", $.uint_literal),
        ),

        //---------------------------Hex strings-------------------------------

        hex_str: $ => seq(
            "{",
            $.hex_str_sequence,
            "}",
        ),

        hex_str_sequence: $ => choice(
            $._hex_str_start_element,
            seq(
                $._hex_str_start_element, 
                optional($._hex_str_inner_sequence), 
                $._hex_str_start_element
            ),
        ),

        _hex_str_start_element: $ => choice(
            $._hex_str_byte,
            $._hex_str_wildcard,
            $.hex_str_parentheses,
            ...get_operators(HEX_STR_OP, "NOT"),
        ),

        _hex_str_inner_element: $ => choice(
            $._hex_str_start_element,
            $.hex_str_jump,
        ),

        _hex_str_inner_sequence: $ => choice(
            $._hex_str_inner_element,
            seq(
                $._hex_str_inner_sequence, 
                $._hex_str_inner_element,
            ),
        ),

        hex_str_parentheses: $ => seq(
            "(",
            $._hex_str_nested_sequence,
            ")",
        ),

        _hex_str_nested_sequence: $ => choice(
            $.hex_str_sequence,
            ...get_operators(HEX_STR_OP, "ALT"),
        ),

        hex_str_jump: $ => seq(
            "[",
            choice(
                seq(
                    optional(field("lower", $.hex_str_jump_number)),
                    "-",
                    optional(field("upper", $.hex_str_jump_number)),
                ),
                optional(field("count", $.hex_str_jump_number)),
            ),
            "]",
        ),

        //--------------------------Regular Expressions------------------------

        regexp: $ => seq(
            "/",
            field("body", $.regexp_body),
            "/",
            optional(
                field("flags", $.regexp_flags),
            ),
        ),

        regexp_body: $ => $._regexp_seq,

        _regexp_seq: $ => prec.dynamic(REGEXP_SEQ_EL_DYNAM_LEVEL,
            choice(
                $._regexp_single,
                $.regexp_rep,
                ...get_operators(REGEXP_OP),
            ),
        ),

        regexp_rep: $ => prec.dynamic(REGEXP_SEQ_EL_DYNAM_LEVEL + 1, //< If it is possible consider string as regexp_rep
            seq(
                field("base", $._regexp_single),
                field("quant", $.regexp_rep_quantifier),
            ),
        ),

        _regexp_single: $ => choice(
            $.regexp_char,
            $.regexp_char_beg,
            $.regexp_char_eof,
            $.regexp_char_any,
            $.regexp_esc_seq,
            $.regexp_class,
            $.regexp_group,
        ),

        regexp_class: $ => seq(
            "[",
            optional(
                field("neg", $.regexp_class_neg)
            ),
            $._regexp_class_element_list,
            "]",
        ),

        _regexp_class_element: $ => prec.dynamic(REGEXP_CLS_EL_DYNAM_LEVEL,
            choice(
                $.regexp_class_range,
                $.regexp_esc_seq,
                $.regexp_class_char,
                $.hyphen,
            )
        ),

        _regexp_class_element_list: $ => choice(
            $._regexp_class_element,
            seq(
                $._regexp_class_element_list,
                $._regexp_class_element,
            ),
        ),

        regexp_class_range: $ => prec.dynamic(REGEXP_CLS_EL_DYNAM_LEVEL + 1, 
            seq(
                field("from", 
                    choice(
                        $.regexp_class_char,
                        $.regexp_esc_seq,
                        $.hyphen    
                    ),
                ),
                $.hyphen,
                field("to", 
                    choice(
                        $.regexp_class_char,
                        $.regexp_esc_seq,
                        $.hyphen    
                    ),
                ),
            )
        ),

        regexp_group: $ => seq(
            "(",
            $._regexp_seq,
            ")",
        ),

        regexp_rep_quantifier: $ => seq(
            choice(
                /\*/,
                /\+/,
                /\?/,
                seq(
                    $._regexp_lcb,
                    choice(
                        field("count", $.regexp_rep_quantifier_number),
                        seq(
                            field("lower", $.regexp_rep_quantifier_number),
                            ",",
                            field("upper", $.regexp_rep_quantifier_number), 
                        ),
                        seq(
                            field("lower", $.regexp_rep_quantifier_number),
                            ","
                        ),
                        seq(
                            ",",
                            field("upper", $.regexp_rep_quantifier_number),
                        ),
                    ),
                    $._regexp_rcb,
                ),
            ),
            optional(
                field("greedy", $.regexp_rep_quantifier_greedy),
            ),
        ),

        regexp_rep_quantifier_number: $ => $.dec_uint_literal_value_imm,

        regexp_rep_quantifier_greedy: _ => "?",

        //------------------------------Conditions-----------------------------

        _condition_body: $ => seq(
            "condition",
            ":",
            field("condition", $._expression),
        ),

        _expression: $ => choice( //< 'Main' expression
            $.string_identifier,
            $.parentheses,
            $.for,
            $.for_int,
            $.of,
            $._boolean_expression,
            $._primary_expression,
            ...get_operators(EXPRESSION_OP),
        ),
        
        parentheses: $ => seq(
            "(",
            $._expression,
            ")",
        ),

        for: $ => seq(
            "for",
            field("quant", $._quantifier),
            "of",
            field("set",
                choice(
                    $.string_set,
                    $.them,
                )
            ),
            ":",
            "(",
            field("expression", $._expression),
            ")",
        ),

        _quantifier: $ => choice(
            $._primary_expression,
            $.percent_quantifier,
            $.all,
            $.any,
            $.none,
        ),

        percent_quantifier: $ => seq(
            field("value", $._primary_expression),
            "%",
        ),

        string_set: $ => seq(
            "(",
            $._string_set_element,
            optional(
                repeat1(
                    seq(
                        ",",
                        $._string_set_element,
                    )
                )
            ),
            ")",
        ),

        _string_set_element: $ => choice(
            $.string_identifier,
            $.string_wildcard,
        ),

        string_wildcard: $ => seq(
            field("id", $.string_identifier),
            token.immediate("*"),
        ),

        for_int: $ => seq(
            "for",
            field("quant", $._quantifier),
            field("var_list", $.var_list),
            "in",
            field("iterable", 
                choice(
                    $.range,
                    $.enum,
                    $._variable,
                )
            ),
            ":",
            "(",
            field("expression", $._expression),
            ")",
        ),

        var_list: $ => seq(
            $.identifier,
            optional(
                repeat1(
                    seq(
                        ",",
                        $.identifier,
                    )
                ),
            ),
        ),

        range: $ => seq(
            "(",
            field("from", $._primary_expression),
            "..",
            field("to", $._primary_expression),
            ")",
        ),

        enum: $ => seq(
            "(",
            $._primary_expression,
            optional(
                repeat1(
                    seq(
                        ",",
                        $._primary_expression,
                    )
                ),
            ),
            ")",
        ),

        of: $ => seq(
            field("quant", $._quantifier),
            "of",
            choice(
                seq( //< Variant with string references
                    field("set",
                        choice(
                            $.string_set,
                            $.them,
                        )
                    ),
                    optional(
                        seq(
                            "in",
                            field("range", $.range),
                        ),
                    ),
                ),
                seq( //< Variant with rule references
                    field("set", $.rule_set),
                    optional(
                        seq(
                            "in",
                            field("range", $.range),
                        ),
                    ),
                ),
                field("set", $.expression_enum) //< Avast specific symbol
            ),
        ),

        rule_set: $ => seq(
            "(",
            $._rule_set_element,
            optional(
                repeat1(
                    seq(
                        ",",
                        $._rule_set_element,
                    )
                )
            ),
            ")",
        ),

        _rule_set_element: $ => choice(
            $.identifier,
            $.rule_wildcard,
        ),

        rule_wildcard: $ => seq(
            $.identifier,
            token.immediate("*"),
        ),

        //Avast specific symbol
        expression_enum: $ => seq(
            "[",
            $._expression,
            optional(
                repeat1(
                    seq(
                        ",",
                        $._expression,
                    )
                )
            ),
            "]",
        ),

        _boolean_expression: $ => choice(
            $.bool_literal,
            ...get_operators(BOOLEAN_EXPRESSION_OP),
        ),

        _primary_expression: $ => choice(
            $.string_literal,     
            $.uint_literal,
            $.float_literal,
            $.size_literal,

            $.regexp,

            $.string_count,
            $.string_match_length,
            $.string_offset,
            $.indexed_match_length, //< Indexed version of string_match_length and string_offset tokens
            $.indexed_offset,

            $.primary_parentheses,

            $.function_call,
            
            $._variable,

            $.in,
            $.remainder,
            ...get_operators(PRIMARY_EXPRESSION_OP),
        ),

        size_literal : $ => seq(
            field("value", $.dec_uint_literal_value),
            field("unit", $.filesize_unit)
        ),

        in: $ => IN_EXPR_PREC.func(IN_EXPR_PREC.level, 
            seq(
                field("str", 
                    choice(
                        $.string_identifier,
                        $.string_count,
                    )
                ),
                "in",
                field("range", $.range),
            )
        ),

        remainder: $ => REM_EXPR_PREC.func(REM_EXPR_PREC.level, 
            seq(
                field("lop", $._primary_expression),
                "%",
                field("rop", $._primary_expression),
            )
        ),

        indexed_match_length: $ => seq(
            field("str",  $.string_match_length),
            "[",
            field("index", $._expression),
            "]",
        ),

        indexed_offset: $ => seq(
            field("str", $.string_offset),
            "[",
            field("index", $._expression),
            "]",
        ),

        primary_parentheses: $ => prec(EXPR_PREC_LEVEL + 1, seq(
                "(",
                $._primary_expression,
                ")",
            ),
        ),

        function_call: $ => seq(
            field("id", $._variable),
            field("args", $.argument_list),
        ),

        _variable: $ => choice(
            $.identifier,
            ...get_operators(VARIABLE_OP), //< All variable operators from VARIABLE_OP
        ),

        argument_list: $ => seq(
            "(",
            optional(
                seq(
                    $._primary_expression,
                    optional(
                        repeat1(
                            seq(
                                ",",
                                $._primary_expression,
                            )
                        )
                    )
                )
            ),
            ")",
        ),


        //--------------------------------Other--------------------------------

        _rule_id: $ => field("id", $.identifier),

        tag: $ => field("id", $.identifier),

        _meta_id: $ => field("id", $.identifier),

        _internal_var_id: $ => field("id", $.identifier),

        string_literal: $ => seq(
            "\"",
            optional(field("str", $.string_literal_str)),
            "\"",
        ),

        string_literal_str: $ => repeat1(
            choice(
                token.immediate(prec(LEX_PREC.COMMENT + 1, /[^\"\\\r\n]/)),
                $.string_esc_seq,
            )
        ),


        //Terminals

        rule_modifier: _ => token(
            choice(
                "private",
                "global",
            )
        ),

        bool_literal: _ => token(
            choice(
                "true",
                "false",
            )
        ),

        them: _ => token("them"),
        
        all: _ => token("all"),

        any: _ => token("any"),

        none: _ => token("none"),

        filesize_unit: _ => token.immediate(/KB|MB/),

        int_literal: $ => seq( //< Integer value for meta
            optional("-"),
            $._uint_literal_value,
        ),

        uint_literal: $ => $._uint_literal_value,

        _uint_literal_value: $ => choice(
            $.dec_uint_literal_value,
            $.hex_uint_literal_value,   
            $.oct_uint_literal_value,
        ),

        dec_uint_literal_value: _ => /\d+/,

        dec_uint_literal_value_imm: _ => token.immediate(/\d+/),

        hex_uint_literal_value: _ => /0x[A-Fa-f\d]+/,

        oct_uint_literal_value: _ => /0o[0-7]+/,

        float_literal: _ => /\d+\.\d+/,

        // Strings can be also anonymous - then string name is just $
        string_identifier: _ => /\$\w*/,

        string_count: _ => /#\w*/,

        string_offset: _ => /@\w*/,

        string_match_length: _ => /![A-Za-z\d]*/,

        string_esc_seq: _ => /(\\[^x])|(\\x[A-Fa-f\d]{2})/,

        _hex_str_byte: _ => /[A-Fa-f\d]{2}|\?[A-Fa-f\d]|[A-Fa-f\d]\?/, //< Byte with at least one concrete nibble

        _hex_str_wildcard: _ => /\?\?/, //< Full masked byte with ? sign

        hex_str_jump_number: _ => /\d+/,

        /**
         * Lexical precedence here is workaround (regexp flags interferes with
         * word token (identifier) and token.immediate does not work without 
         * it in this case)
         */
        regexp_flags: _ => token.immediate(prec(LEX_PREC.ID - 1, /i|s|is/)),

        regexp_esc_seq: _ => token(
            choice(
                /\\./,
                /\\x[A-Fa-f\d]{2}/,
            ),
        ),

        regexp_char: $ => choice(
            $._regexp_lcb,
            $._regexp_rcb,
            token.immediate(/[^\\()*+?\[\/|^$.{}]/),
        ),

        _regexp_lcb: _ => token.immediate("{"),

        _regexp_rcb: _ => token.immediate("}"),

        regexp_class_char: _ => token.immediate(/[^\]\\-]/),

        hyphen: _ => token.immediate("-"),

        regexp_class_neg: _ => token.immediate("^"),

        regexp_char_beg: _ => "^",

        regexp_char_eof: _ => "$",

        regexp_char_any: _ => ".",

        identifier: _ => token(prec(LEX_PREC.ID, /[A-Za-z_]\w*/)),

        comment: _ => token(
            prec(LEX_PREC.COMMENT,
                choice(
                    /\/\/[^\r\n]*/,
                    seq(
                        "/*", 
                        prec(LEX_PREC.COMMENT - 1, /(.|[\r\n])*/), 
                        "*/",
                    )
                )
            )
        ),
    },
});


add_condition_operator_rules( //< Add definition of all used symbols
    yara_grammar,
    HEX_STR_OP,
    REGEXP_OP,
    VARIABLE_OP,
    PRIMARY_EXPRESSION_OP,
    EXPRESSION_OP,
    BOOLEAN_EXPRESSION_OP,
);

module.exports = yara_grammar;


/**
 * Adds condition operator rules to given grammar
 * @param {Grammar} grammar Target grammar object (from tree-sitter lib) 
 * @param {...any} source_objects Source object(s) (= sets of operators) 
 * that should be added into target grammar
 */
function add_condition_operator_rules(grammar, ...source_objects) {

    for(let source_obj_key in source_objects) {

        for(let operator_key in source_objects[source_obj_key]) { //< Iterate through all given keys
            if(!source_objects[source_obj_key].hasOwnProperty(operator_key)) {
                continue;
            }

            let op_object = source_objects[source_obj_key][operator_key];

            grammar.rules[op_object.name] = op_object.make_seq_rule(); //< Creates tree-sitter sequence from array stored in Operator object
        }
    }
}


/**
 * Converts operators described in source_obj by Operator prototype to 
 * tree-sitter symbols
 * @param {*} source_obj Object containing multiple Operator objects 
 * @param  {...String} op_keys Keys from source_obj, that should be 
 * converted to symbol object (if it is not specified, all symbols from
 * source_obj are converted)
 * @returns Array with chosen symbols, that are described in source_obj
 * @note pay attention to the casing of key strings in the op_keys array
 */
function get_operators(source_obj, ...op_keys) {
    let sym_arr = [];

    let key_arr = Object.keys(source_obj);
    if(op_keys.length) {
        key_arr = op_keys;
    }

    key_arr.forEach((key) => { //< Iterate through all given keys
        if(!source_obj.hasOwnProperty(key)) {
            return;
        }

        sym_arr.push({type: "SYMBOL", name: source_obj[key].name});
    });

    return sym_arr;
}

