/**
 * yaramod_yarasource.cpp - Unit tests focused on basic parsing and reparsing 
 * of the simplest yara rules. Interfaces of Yaramod, YaraSource and partly 
 * YaraFile classes are tested in these unit tests. 
 * 
 * @author Vojtěch Dvořák 
 */

#include "gtest/gtest.h"
#include "yaramod.h"

#include "common.cpp"


class BasicParsingFromString : public ::testing::Test {
    protected:
        std::string yara = R"(
rule simple {
    condition:
        true
}
        )";

        BasicParsingFromString() {
        }

        ~BasicParsingFromString() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(BasicParsingFromString, ParseDoesNotThrow) {
    std::shared_ptr<YaraSource> yara_src;

    EXPECT_NO_THROW(yara_src = y->parseString(yara));
}


TEST_F(BasicParsingFromString, YaraSourceContainsYaraFile) {
    std::shared_ptr<YaraSource> yara_src = y->parseString(yara);

    EXPECT_EQ(yara_src->getFile({}), yara_src->getEntryFile());
}


TEST_F(BasicParsingFromString, YaraFileContainsRule) {
    std::shared_ptr<YaraSource> yara_src = y->parseString(yara);

    auto yara_entry_file = yara_src->getEntryFile();

    Rule *rule;

    EXPECT_NO_THROW(rule = yara_entry_file->getLocalRule(0));
    EXPECT_EQ(rule->getId(), "simple");
}




class BasicParsingFromFile : public ::testing::Test {
    protected:
        const char *tmp_filename = "_parsing_test_.tmp";

        std::string yara = R"(
rule simple {
    condition:
        true
}
        )";

        BasicParsingFromFile() {
        }

        ~BasicParsingFromFile() {
        }

        void SetUp() override {
            writeStringToFile(tmp_filename, yara);

            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
            remove(tmp_filename);
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(BasicParsingFromFile, ParseDoesNotThrow) {
    std::shared_ptr<YaraSource> yara_src;

    EXPECT_NO_THROW(yara_src = y->parseFile(tmp_filename));
}


TEST_F(BasicParsingFromFile, YaraSourceContainsYaraFile) {
    std::shared_ptr<YaraSource> yara_src = y->parseFile(tmp_filename);

    EXPECT_EQ(yara_src->getFile(tmp_filename), yara_src->getEntryFile());
}


TEST_F(BasicParsingFromFile, YaraFileContainsRule) {
    std::shared_ptr<YaraSource> yara_src = y->parseFile(tmp_filename);

    auto yara_entry_file = yara_src->getEntryFile();

    Rule *rule;

    EXPECT_NO_THROW(rule = yara_entry_file->getLocalRule(0));
    EXPECT_EQ(rule->getId(), "simple");
}




class ParsingFromFileWithIncludes : public ::testing::Test {
    protected:
        const char *tmp1_filename = "_parsing_test1_.tmp";

        std::string yara1 = R"(
include "_parsing_test2_.tmp"

rule simple {
    condition:
        true
}
        )";

        const char *tmp2_filename = "_parsing_test2_.tmp";

        std::string yara2 = R"(
rule simple2 {
    condition:
        true
}
        )";

        ParsingFromFileWithIncludes() {
        }

        ~ParsingFromFileWithIncludes() {
        }

        void SetUp() override {
            writeStringToFile(tmp1_filename, yara1);
            writeStringToFile(tmp2_filename, yara2);

            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
            remove(tmp1_filename);
            remove(tmp2_filename);
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(ParsingFromFileWithIncludes, YaraSourceIsCorrect) {
    std::shared_ptr<YaraSource> yara_src;

    EXPECT_NO_THROW(yara_src = y->parseFile(tmp1_filename));

    EXPECT_EQ(yara_src->getEntryFile(), yara_src->getFile(tmp1_filename));
}


TEST_F(ParsingFromFileWithIncludes, IncludedYaraFileContainsRule) {
    std::shared_ptr<YaraSource> yara_src = y->parseFile(tmp1_filename);
    auto included_file = yara_src->getFile(tmp2_filename);

    Rule *rule;
    
    EXPECT_NO_THROW(rule = included_file->getLocalRule(0));
    EXPECT_EQ(rule->getId(), "simple2");
}


class ParsingOfEmptyString : public ::testing::Test {
    protected:
        std::string yara = R"()";

        ParsingOfEmptyString() {
        }

        ~ParsingOfEmptyString() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(ParsingOfEmptyString, ParseDoesNotThrow) {
    std::shared_ptr<YaraSource> yara_src;

    EXPECT_NO_THROW(yara_src = y->parseString(yara));
}


class BasicReparsing : public ::testing::Test {
    protected:
        std::string yara1 = "";

        std::string yara2 = "rule simplest\n{\ncondition:\ntrue\n}";

        std::string yara3 = "rule abc\n{\ncondition:\ntrue\n}";

        std::string yara4 = "";

        BasicReparsing() {
        }

        ~BasicReparsing() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(BasicReparsing, ReparseDoesNotThrow) {
    YaraSourcePtr yara_src = y->parseString(yara1);

    yara_src->edit(0, 33, 0);

    EXPECT_NO_THROW(y->parseString(yara2, yara_src));
}


TEST_F(BasicReparsing, EntryFileContainsUpdatedContent) {
    YaraSourcePtr yara_src = y->parseString(yara1);
    yara_src->edit(0, 33, 0);
    y->parseString(yara2, yara_src);

    Rule *rule;
    EXPECT_NO_THROW(rule = yara_src->getEntryFile()->getLocalRules().dataByOffset().find(0)->second); ///< "0" is offset not index 

    EXPECT_EQ(rule->getId(), "simplest");
    EXPECT_EQ(yara_src->getTextFormatted().str(), "rule simplest {\n\tcondition:\n\t\ttrue\n}\n\n");

}


TEST_F(BasicReparsing, EntryFileContainsUpdatedContentAfterTwoEdits) {
    YaraSourcePtr yara_src = y->parseString(yara1);
    yara_src->edit(0, 33, 0);
    y->parseString(yara2, yara_src);

    yara_src->edit(5, 3, 8);
    y->parseString(yara3, yara_src);

    Rule *rule;
    EXPECT_NO_THROW(rule = yara_src->getEntryFile()->getLocalRules().dataByOffset().find(0)->second);

    EXPECT_EQ(rule->getId(), "abc"); 
}


TEST_F(BasicReparsing, TwoEditsSpecifiedByText) {
    Rule *rule;
    YaraSourcePtr yara_src = y->parseString(yara1);
    yara_src->edit({{0, 0}, {4, 0}}, "rule simplest\n{\ncondition:\ntrue\n}");
    y->parseString(yara2, yara_src);

    EXPECT_NO_THROW(rule = yara_src->getEntryFile()->getLocalRules().dataByOffset().find(0)->second);
    EXPECT_EQ(rule->getId(), "simplest"); 

    yara_src->edit({{0, 5}, {0, 13}}, "abc");
    y->parseString(yara3, yara_src);

    EXPECT_NO_THROW(rule = yara_src->getEntryFile()->getLocalRules().dataByOffset().find(0)->second);
    EXPECT_EQ(rule->getId(), "abc"); 
}


TEST_F(BasicReparsing, ThreeEditsSpecifiedByTextWithoutUpdate) {
    std::shared_ptr<Rule> rule;
    YaraSourcePtr yara_src = y->parseString(yara1);
    yara_src->edit({{0, 0}, {0, 0}}, "rule simplest\n{\ncondition:\ntrue\n}");
    yara_src->edit({{0, 5}, {0, 13}}, "abc");
    yara_src->edit({{0, 0}, {4, 1}}, {});
    y->parseString(yara4, yara_src);

    EXPECT_EQ(yara_src->getEntryFile()->getRules().size(), 0);
}


int main(int argc, char **argv) {
    ::testing::InitGoogleTest(&argc, argv);
    return RUN_ALL_TESTS();
}
