/**
 * local_symbol_management.cpp - Unit tests focused on management of local 
 * symbols - metas, variables, strings...
 * 
 * @author Vojtěch Dvořák 
 */

#include "gtest/gtest.h"
#include "yaramod.h"

#include "common.cpp"


class LiteralManagement : public ::testing::Test {
    protected:
        LiteralManagement() {
        }

        ~LiteralManagement() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Literal literal;
};

TEST_F(LiteralManagement, UsingLiteral) {
    literal.set("test");

    EXPECT_TRUE(literal.isType<std::string>());
    EXPECT_EQ(literal.get<std::string>(), "test");

    literal.set(123);

    EXPECT_TRUE(literal.isType<int64_t>());
    EXPECT_EQ(literal.get<int64_t>(), 123);

    EXPECT_FALSE(literal.isType<std::string>());
};


class MetasBasics : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a : b {
                meta:
                    author = "author"
                    is_test = true
                    level = 10
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a : b {
                meta:
                    int = 16
                    neg_int = -123
                    hex_int = 0x10
                    oct_int = 0o42
                    neg_hex_int = -0x11
                    neg_oct_int = -0o11
                condition:
                    true
            }
        )";

        std::string yara3 = R"(
            rule a : b {
                meta:
                    neg_int = -          123
                    neg_hex_int = - 0x11
                    neg_oct_int = -         0o11
                condition:
                    true
            }
        )";

        MetasBasics() {
        }

        ~MetasBasics() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(MetasBasics, ParsingMetas) {
    std::shared_ptr<YaraSource> y_src;

    EXPECT_NO_THROW(y_src = y->parseString(yara1););  

    EXPECT_EQ(y_src->getEntryFile()->getLocalRule("a")->getMetas().size(), 3);
}

TEST_F(MetasBasics, MetasHaveCorrectLiteral) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.at("author").at(0)->getValue()->get<std::string>(), "author");
    EXPECT_TRUE(metas.at("is_test").at(0)->getValue()->get<bool>());
    EXPECT_EQ(metas.at("level").at(0)->getValue()->get<int64_t>(), 10);
}

TEST_F(MetasBasics, VariousIntegerLiteralTypes) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.at("int").at(0)->getValue()->get<int64_t>(), 16);
    EXPECT_EQ(metas.at("neg_int").at(0)->getValue()->get<int64_t>(), -123);
    EXPECT_EQ(metas.at("hex_int").at(0)->getValue()->get<int64_t>(), 16);
    EXPECT_EQ(metas.at("oct_int").at(0)->getValue()->get<int64_t>(), 34);

    EXPECT_EQ(metas.at("neg_hex_int").at(0)->getValue()->get<int64_t>(), -17);
    EXPECT_EQ(metas.at("neg_oct_int").at(0)->getValue()->get<int64_t>(), -9);
}

TEST_F(MetasBasics, WeirdFormattingOfIntegerLiterals) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara3);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.at("neg_int").at(0)->getValue()->get<int64_t>(), -123);
    EXPECT_EQ(metas.at("neg_hex_int").at(0)->getValue()->get<int64_t>(), -17);
    EXPECT_EQ(metas.at("neg_oct_int").at(0)->getValue()->get<int64_t>(), -9);
}


class MetasErrors : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a : b {
                meta:
                    a = "b\ad"
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a : b {
                meta:
                    int_ok = 9223372036854775807
                    int = 9223372036854775808
                condition:
                    true
            }
        )";

        MetasErrors() {
        }

        ~MetasErrors() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(MetasErrors, UnknownEscapeSequence) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
}

TEST_F(MetasErrors, IntegerOverflow) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.size(), 2);
    EXPECT_TRUE(metas.at("int").at(0)->getValue()->isUndefined());
    EXPECT_FALSE(metas.at("int_ok").at(0)->getValue()->isUndefined());
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


class Metas : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                    a = "two"
                    c = "three"
                condition:
                    true
            }
        )";


        Metas() {
        }

        ~Metas() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(Metas, MultipleMetasWithSameId) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &metas = y_src->getEntryFile()->getLocalRule("a")->getMetas();
    EXPECT_EQ(metas.size(), 3);
    EXPECT_EQ(metas.at("a").size(), 2);
    EXPECT_EQ(metas.at("a")[0]->getValue()->get<std::string>(), "one");
    EXPECT_EQ(metas.at("a")[1]->getValue()->get<std::string>(), "two");
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
}


class Strings : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = "abc"
                    $b = { AA BB [5] CC }
                    $c = /abc/i
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule b {
                strings:
                    $a = "abc" base64
                    $b = "def" xor(1-3)
                    $c = "ghi" base64("0123456789012345678901234567890123456789012345678901234567890123")
                    $d = "abc" xor(3) nocase
                condition:
                    false

            }
        )";


        Strings() {
        }

        ~Strings() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(Strings, RuleContainsCorrectStrings) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_EQ(strings.size(), 3);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);

    EXPECT_EQ(strings.at("a")->getContent(), "abc");
    EXPECT_EQ(strings.at("b")->getContent(), "AA BB [5] CC");
    EXPECT_EQ(strings.at("c")->getContent(), "/abc/i");

    EXPECT_ANY_THROW(y_src->getEntryFile()->getLocalRule("a")->getStringById("d"));

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}

TEST_F(Strings, StringContainsCorrectMods) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &string_a = y_src->getEntryFile()->getLocalRule("b")->getStringById("a");
    const auto &string_b = y_src->getEntryFile()->getLocalRule("b")->getStringById("b");
    const auto &string_c = y_src->getEntryFile()->getLocalRule("b")->getStringById("c");
    const auto &string_d = y_src->getEntryFile()->getLocalRule("b")->getStringById("d");
    EXPECT_TRUE(string_a->getModifiers().has(StringModifier::Type::Base64));

    auto mod_b = string_b->getModifiers().get(StringModifier::Type::Xor);
    EXPECT_EQ(std::get<StringModifier::Range>(mod_b->getArg()).first, 1);
    EXPECT_EQ(std::get<StringModifier::Range>(mod_b->getArg()).second, 3);

    auto mod_c = string_c->getModifiers().get(StringModifier::Type::Base64);

    const std::array<char, 64> alphabet = std::array<char, 64>{
        '0','1','2','3','4','5','6','7','8','9','0','1','2','3','4','5','6','7','8','9',
        '0','1','2','3','4','5','6','7','8','9','0','1','2','3','4','5','6','7','8','9',
        '0','1','2','3','4','5','6','7','8','9','0','1','2','3','4','5','6','7','8','9',
        '0','1','2','3'
    };

    EXPECT_EQ(std::get<StringModifier::Alphabet>(mod_c->getArg()), alphabet);

    auto mod_d = string_d->getModifiers().get(StringModifier::Type::Xor);
    EXPECT_EQ(std::get<StringModifier::Key>(mod_d->getArg()), 3);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


class StringErrors : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = { AA BB [5-4] CC }
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = "ab\kc"
                condition:
                    true
            }
        )";

        std::string yara3 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /aůb/
                condition:
                    true
            }
        )";

        std::string yara4 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /a[b-a]bc/
                condition:
                    true
            }
        )";

        std::string yara5 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /a[\r-\n]bc{4,3}/
                condition:
                    true
            }
        )";


        std::string yara6 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $string = /a/
                    $string = "a"
                condition:
                    true
            }
        )";

        StringErrors() {
        }

        ~StringErrors() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(StringErrors, HexStrInvalidJumpRange) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);

    // Semantic error should be at offset 132
    const auto &semantic_err_data = y_src->getEntryFile()->getSemanticErrors().data();
    EXPECT_NE(semantic_err_data.find(132),semantic_err_data.end());
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


TEST_F(StringErrors, PlainStringWithInvalidEscSeq) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
}


TEST_F(StringErrors, RegexpWithNonAsciiChar) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara3);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
}


TEST_F(StringErrors, RegexpWithInvalidClassRange) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara4);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


TEST_F(StringErrors, RegexpInvalidClassRangeAndQuantifier) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara5);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 2);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


TEST_F(StringErrors, DuplicitStringIdentifiers) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara6);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}



class RegexRepeatInteval : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /a{2,1}/
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /ů{1,1}/
                condition:
                    true
            }
        )";

        std::string yara3 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = /a{,54235425423532452345}/
                condition:
                    true
            }
        )";

        RegexRepeatInteval() {
        }

        ~RegexRepeatInteval() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(RegexRepeatInteval, BadRepeatInterval) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_NO_THROW(strings.at("a"));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


TEST_F(RegexRepeatInteval, NonAsciiCharAsBase) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_NO_THROW(strings.at("a"));
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
}


TEST_F(RegexRepeatInteval, TooLargeRepeatInterval) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara3);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_NO_THROW(strings.at("a"));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


class StringModifierErrors : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = { AA BB CC } base64
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = "abc" xor(4-1)
                condition:
                    true
            }
        )";

        std::string yara3 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = "a" base64("1")
                condition:
                    true
            }
        )";

        StringModifierErrors() {
        }

        ~StringModifierErrors() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(StringModifierErrors, StringHasInvalidModifier) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_EQ(strings.at("a")->getModifiers().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}

TEST_F(StringModifierErrors, XorHasInvalidRange) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_TRUE(strings.at("a")->getModifiers().has(StringModifier::Type::Xor));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}

TEST_F(StringModifierErrors, AlphabetHasInvalidNumberOfCharacters) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara3);

    const auto &strings = y_src->getEntryFile()->getLocalRule("a")->getStrings();
    EXPECT_TRUE(strings.at("a")->getModifiers().has(StringModifier::Type::Base64));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


class Variables : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = "123"
                variables:
                    c = "a"
                condition:
                    true
            }
        )";

        std::string yara2 = R"(
            rule a {
                meta:
                    a = "one"
                variables:
                    d = 123
                    c = true or false
                strings:
                    $a = { AA BB CC }
                condition:
                    true
            }
        )";


        Variables() {
        }

        ~Variables() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(Variables, SingleVariable) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    const auto &vars = y_src->getEntryFile()->getLocalRule("a")->getVars();
    EXPECT_NO_THROW(vars.at("c"));
}

TEST_F(Variables, MultipleVariablesBeforeStrings) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara2);

    const auto &vars = y_src->getEntryFile()->getLocalRule("a")->getVars();
    EXPECT_NO_THROW(vars.at("c"));
    EXPECT_NO_THROW(vars.at("d"));
}


class VariableErrors : public ::testing::Test {
    protected:
        std::string yara1 = R"(
            rule a {
                meta:
                    a = "one"
                strings:
                    $a = { AA BB CC }
                variables:
                    c = "a"
                    c = "g"
                condition:
                    true
            }
        )";


        VariableErrors() {
        }

        ~VariableErrors() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(VariableErrors, DuplicitVarId) {
    std::shared_ptr<YaraSource> y_src;

    y_src = y->parseString(yara1);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


int main(int argc, char **argv) {
    ::testing::InitGoogleTest(&argc, argv);
    return RUN_ALL_TESTS();
}
