/**
 * global_symbol_management.cpp - Unit tests focused on basic parsing and 
 * reparsing of the simplest yara rules. Interfaces of Yaramod, YaraSource 
 * and partly YaraFile classes are tested in these unit tests. 
 * 
 * @author Vojtěch Dvořák 
 */

#include "gtest/gtest.h"
#include "yaramod.h"

#include "common.cpp"


class RuleManagement : public ::testing::Test {
    protected:
        std::string yara = R"(
rule z : abc {
    condition:
        true
}

rule b : def {
    condition:
        true
}

rule c :abc{
    condition:
        true
}

rule d : def abc {
    condition:
        true
}
        )";

        RuleManagement() {
        }

        ~RuleManagement() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();

            y_src = y->parseString(yara);

            entry_file = y_src->getEntryFile();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        YaraFilePtr entry_file;
        std::shared_ptr<YaraSource> y_src;
};


TEST_F(RuleManagement, RuleTabContainsAllRules) {
    const SymTab<Rule> &rules = entry_file->getLocalRules();
    EXPECT_EQ(rules.size(), 4);

    EXPECT_EQ(entry_file->getLocalRule(0)->getId(), "z");
    EXPECT_EQ(entry_file->getLocalRule(1)->getId(), "b");
    EXPECT_EQ(entry_file->getLocalRule(3)->getId(), "d");
}

TEST_F(RuleManagement, RulesHaveCorrectTags) {
    EXPECT_TRUE(entry_file->getLocalRule(0)->hasTag("abc"));
    EXPECT_TRUE(entry_file->getLocalRule(1)->hasTag("def"));
    EXPECT_TRUE(entry_file->getLocalRule(3)->hasTag("def"));
    EXPECT_EQ(entry_file->getLocalRule(3)->getTags().size(), 2);
}

TEST_F(RuleManagement, RulesHaveCorrectOffset) {
    const SymTab<Rule> &rules = entry_file->getLocalRules();

    EXPECT_NE(rules.dataByOffset().find(1), rules.dataByOffset().end());
    EXPECT_NE(rules.dataByOffset().find(47), rules.dataByOffset().end());
    EXPECT_NE(rules.dataByOffset().find(93), rules.dataByOffset().end());
}


TEST_F(RuleManagement, DuplicitTagGeneratesSemanticError) {
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_NO_THROW(entry_file->getLocalRule(0)->addTag("abc"));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


TEST_F(RuleManagement, DuplicitTagGeneratesThrowsExceptionInStrictMode) {
    y->config().setErrorMode(YaramodConfig::ErrorMode::Strict);

    y_src = y->parseString(yara);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_ANY_THROW(y_src->getEntryFile()->getLocalRule(0)->addTag("abc"));
}



TEST_F(RuleManagement, AdditionOfRule) {
    std::unique_ptr new_rule = std::make_unique<Rule>();
    new_rule->setId("new_rule");
    new_rule->setOffset(300);
    new_rule->setLen(10);

    EXPECT_NO_THROW(y_src->getEntryFile()->addRule(std::move(new_rule)));

    const auto &rules = y_src->getEntryFile()->getLocalRules();

    EXPECT_EQ((--rules.dataByOffset().end())->second->getId(), "new_rule");
}


TEST_F(RuleManagement, DuplicitRuleGeneratesSemanticError) {
    std::unique_ptr new_rule = std::make_unique<Rule>();
    new_rule->setId("z");
    new_rule->setOffset(300);
    new_rule->setLen(10);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_NO_THROW(y_src->getEntryFile()->addRule(std::move(new_rule)));
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


TEST_F(RuleManagement, DuplicitRuleThrowsExceptionInStrictMode) {
    y->config().setErrorMode(YaramodConfig::ErrorMode::Strict);

    y_src = y->parseString(yara);

    std::unique_ptr new_rule = std::make_unique<Rule>();
    new_rule->setId("z");
    new_rule->setOffset(300);
    new_rule->setLen(10);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_ANY_THROW(y_src->getEntryFile()->addRule(std::move(new_rule)));

}


class RuleSemanticErrors : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule z : abc {
    condition:
        true
}

rule z : def {
    condition:
        true
}
        )";

        std::string yara2 = R"(
global global rule z : abc {
    condition:
        true
}

rule y : def {
    condition:
        true
}
        )";

        RuleSemanticErrors() {
        }

        ~RuleSemanticErrors() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        std::shared_ptr<YaraSource> y_src;
};



TEST_F(RuleSemanticErrors, DuplicitRuleIdentifierInParsedString) {
    y_src = y->parseString(yara1);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().data().begin()->second->getOffset(), 52);

    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}



TEST_F(RuleSemanticErrors, DuplicitRuleModifierifier) {
    y_src = y->parseString(yara2);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().data().begin()->second->getOffset(), 8);

    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}



class CircularIncludeParsing : public ::testing::Test {
    protected:
        const char *tmp1_filename = "_parsing_test1_.tmp";

        std::string yara1 = R"(
import "cuckoo"
import "console"

include "_parsing_test2_.tmp"

rule a : abc {
    condition:
        true
}

rule b : def {
    condition:
        true
}

        )";

        const char *tmp2_filename = "_parsing_test2_.tmp";

        std::string yara2 = R"(
rule c : abc {
    condition:
        true
}

include "_parsing_test1_.tmp"

        )";

        CircularIncludeParsing() {
        }

        ~CircularIncludeParsing() {
        }

        void SetUp() override {
            writeStringToFile(tmp1_filename, yara1);
            writeStringToFile(tmp2_filename, yara2);

            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};


TEST_F(CircularIncludeParsing, CircularIncludeShouldNotMakeInfLoop) {
    std::shared_ptr<YaraSource> y_src = y->parseFile(tmp1_filename);
}



class Complex : public ::testing::Test {
    protected:
        const char *tmp1_filename = "_parsing_test1_.tmp";

        std::string yara1 = R"(
import "cuckoo"
import "console"

include "_parsing_test2_.tmp"

rule a : abc {
    condition:
        true
}

rule b : def {
    condition:
        true
}

        )";

        const char *tmp2_filename = "_parsing_test2_.tmp";

        std::string yara2 = R"(
rule c : abc {
    condition:
        true
}

        )";

        Complex() {
        }

        ~Complex() {
        }

        void SetUp() override {
            writeStringToFile(tmp1_filename, yara1);
            writeStringToFile(tmp2_filename, yara2);

            y = std::make_unique<Yaramod>();
            y_src = y->parseFile(tmp1_filename);
            entry_file = y_src->getEntryFile();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        std::shared_ptr<YaraSource> y_src;
        YaraFilePtr entry_file;
};


TEST_F(Complex, ModulesAreImported) {
    Import *import;
    EXPECT_NO_THROW(import = *(entry_file->getImports().begin()););
    EXPECT_EQ(import->getModuleName(), "cuckoo");

    EXPECT_EQ(entry_file->getLocalImport(1)->getModuleName(), "console");
}


TEST_F(Complex, EntryFileContainsCorrectInclude) {
    YaraFilePtr included;

    EXPECT_NO_THROW(included = entry_file->getLocalFileInclude(0)->getFile());
    EXPECT_EQ(included->getName(), tmp2_filename);

    EXPECT_EQ(included->getLocalRule(0)->getId(), "c");
}


TEST_F(Complex, GettingIncludedRulesFromIncludedFile) {
    std::vector<Rule *> all_rules = entry_file->getRules();

    EXPECT_EQ(all_rules.at(0)->getId(), "c");
    EXPECT_EQ(all_rules.at(1)->getId(), "a");
    EXPECT_EQ(all_rules.at(2)->getId(), "b");
}


TEST_F(Complex, CircularIncludeGeneratesSemanticError) {
    auto included = entry_file->getLocalFileInclude(0)->getFile();

    std::unique_ptr<FileInclude> new_include = std::make_unique<FileInclude>();
    new_include->setOffset(0);
    new_include->setPath(std::filesystem::weakly_canonical(included->getName()).filename(), "");
    new_include->setFile(included);

    entry_file->addFileInclude(std::move(new_include));

    EXPECT_EQ(entry_file->getSemanticErrors().size(), 1);
}



class ComplexWithReparsing : public ::testing::Test {
    protected:
        const char *tmp_filename = "_parsing_test_.tmp";

        std::string yara = R"(
rule c : abc {
    condition:
        true
}

        )";

        std::string yara1 = R"(
include "_parsing_test_.tmp"

rule b : a {
    condition:
        false
}
//abc
rule d : a {
    condition:
        false
}

        )";

        std::string yara2 = R"(


rule b : a {
    condition:
        false
}
//abc
rule d : a {
    condition:
        false
}

        )";

        std::string yara3 = R"(
rule c : a {
    condition:
        false
}

rule b : a {
    condition:
        false
}
//abc
rule d : a {
    condition:
        false
}

        )";

        ComplexWithReparsing() {
        }

        ~ComplexWithReparsing() {
        }

        void SetUp() override {
            writeStringToFile(tmp_filename, yara);

            y = std::make_unique<Yaramod>();
            y_src = y->parseString(yara1);
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        YaraSourcePtr y_src;
};


TEST_F(ComplexWithReparsing, IncludedFileIsDeletedAfterReparsing) {
    y_src->edit(1, 0, 28);

    y->parseString(yara2, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getLocalFileIncludes().size(), 0);
    EXPECT_FALSE(y_src->hasFile(tmp_filename));

    // Other symbol tables should be unchanged
    auto y_file = y_src->getEntryFile();
    EXPECT_EQ(y_file->getLocalRules().size(), 2);
    EXPECT_EQ(y_file->getComments().size(), 1);
    EXPECT_EQ(y_file->getCommentByIndex(0)->getText(), "//abc");
}


TEST_F(ComplexWithReparsing, ReparsingAfterMultipleEdits) {
    y_src->edit({{1, 0}, {1, 28}}, {});
    y_src->edit({{1, 0}, {1, 0}}, "rule c : a {\n    condition:\n        false\n}");

    y->parseString(yara3, y_src);

    // EXPECT_EQ(y_src->getEntryFile()->getLocalRules().dataByOffset().find(47)->second->getId(), "b");
    // EXPECT_EQ(y_src->getEntryFile()->getLocalRules().dataByOffset().find(97)->second->getId(), "d");

    const auto &comments = y_src->getEntryFile()->getComments();
    EXPECT_EQ(comments.data().begin()->first, 90);
    EXPECT_EQ(comments.data().begin()->second->getText(), "//abc");
}



class ReparsingWithErrorsExtraCharacters : public ::testing::Test {
    protected:
        std::string yara = R"(

rule c : abc {
    condition:
        true
} abc
rule d : def {
    condition:
        false
}
        )";

        std::string yara1 = R"(
//TODO: Fix the error
rule c : abc {
    condition:
        true
} abc
rule d : def {
    condition:
        false
}
        )";

        std::string yara2 = R"(
//TODO: Fix the error
rule c : abc {
    condition:
        true
}
rule d : def {
    condition:
        false
}
        )";

        ReparsingWithErrorsExtraCharacters() {
        }

        ~ReparsingWithErrorsExtraCharacters() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
            y_src = y->parseString(yara);
            entry_file = y_src->getEntryFile();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        YaraSourcePtr y_src;
        YaraFilePtr entry_file;
};


TEST_F(ReparsingWithErrorsExtraCharacters, BeforeFirstEdit) {
    EXPECT_EQ(y_src->getEntryFile()->getLocalRules().size(), 2);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
}


TEST_F(ReparsingWithErrorsExtraCharacters, AfterFirstEdit) {
    y_src->edit(1, 21, 0);

    y->parseString(yara1, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getLocalRules().size(), 2);
    EXPECT_EQ(y_src->getEntryFile()->getComments().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().data().begin()->second->getOffset(), 68);
}


TEST_F(ReparsingWithErrorsExtraCharacters, AfterSecondEdit) {
    y_src->edit(1, 21, 0);

    y->parseString(yara1, y_src);

    y_src->edit(67, 0, 4);

    y->parseString(yara2, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getLocalRules().size(), 2);
    EXPECT_EQ(y_src->getEntryFile()->getComments().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}



class ReparsingWithErrorsMissingRightBrace : public ::testing::Test {
    protected:
        std::string yara = R"(
rule c : abc {
    condition:
        true

rule d : def {
    condition:
        false
}
        )";

        std::string yara1 = R"(
rule c : abc {
    condition:
        true
}
rule d : def {
    condition:
        false
}
        )";

        ReparsingWithErrorsMissingRightBrace() {
        }

        ~ReparsingWithErrorsMissingRightBrace() {
        }

        void SetUp() override {
            y = std::make_unique<Yaramod>();
            y_src = y->parseString(yara);
            entry_file = y_src->getEntryFile();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        YaraSourcePtr y_src;
        YaraFilePtr entry_file;
};


TEST_F(ReparsingWithErrorsMissingRightBrace, BeforeFix) {
    EXPECT_EQ(y_src->getEntryFile()->getLocalRules().size(), 1);
    EXPECT_GT(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}


TEST_F(ReparsingWithErrorsMissingRightBrace, AfterFix) {
    y_src->edit(44, 1, 0);

    y->parseString(yara1, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getLocalRules().size(), 2);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
}



class CrossFileSemanticErrors : public ::testing::Test {
    protected:
        const char *tmp_entry1_filename = "_parsing_test_entry_file1.tmp";

        std::string yara_entry1 = R"(
import "cuckoo"
import "console"

include "_parsing_test1_.tmp"

rule a : abc {
    condition:
        true
}

        )";

        const char *tmp_entry2_filename = "_parsing_test_entry_file2.tmp";

        std::string yara_entry2 = R"(
import "cuckoo"
import "console"

include "_parsing_test3_.tmp"

rule a : abc {
    condition:
        true
}
        )";

        const char *tmp_entry3_filename = "_parsing_test_entry_file3.tmp";

        std::string yara_entry3 = R"(
import "cuckoo"
import "console"

include "_parsing_test1_.tmp"
include "_parsing_test2_.tmp"

        )";



        const char *tmp1_filename = "_parsing_test1_.tmp";

        std::string yara1 = R"(
rule a : def {
    condition:
        true
}

rule c : def {
    condition: false
}
        )";

        const char *tmp2_filename = "_parsing_test2_.tmp";

        std::string yara2 = R"(
rule b : def {
    condition:
        true
}

rule c : def {
    condition: false
}
        )";

        const char *tmp3_filename = "_parsing_test3_.tmp";

        std::string yara3 = R"(
include "_parsing_test1_.tmp"
        )";


        CrossFileSemanticErrors() {
        }

        ~CrossFileSemanticErrors() {
        }

        void SetUp() override {
            writeStringToFile(tmp_entry1_filename, yara_entry1);
            writeStringToFile(tmp_entry2_filename, yara_entry2);
            writeStringToFile(tmp_entry3_filename, yara_entry3);

            writeStringToFile(tmp1_filename, yara1);
            writeStringToFile(tmp2_filename, yara2);
            writeStringToFile(tmp3_filename, yara3);

            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
        std::shared_ptr<YaraSource> y_src;
        YaraFilePtr entry_file;
};


TEST_F(CrossFileSemanticErrors, SameRuleInEntryFileAndInIncludedFile) {
    y_src = y->parseFile(tmp_entry1_filename);
    entry_file = y_src->getEntryFile();

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


TEST_F(CrossFileSemanticErrors, SameRuleInEntryFileAndInIndirectlyIncludedFile) {
    y_src = y->parseFile(tmp_entry2_filename);
    entry_file = y_src->getEntryFile();

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}


TEST_F(CrossFileSemanticErrors, SameRuleInBothIncludedFiles) {
    y_src = y->parseFile(tmp_entry3_filename);
    entry_file = y_src->getEntryFile();

    EXPECT_EQ(y_src->getFile(tmp2_filename)->getSemanticErrors().size(), 1);
}


class CrossFileSemanticErrorsAfterReparsing : public ::testing::Test {
    protected:
        const char *tmp_entry_filename = "_parsing_test_entry_file1.tmp";

        std::string yara_entry1 = R"(
import "cuckoo"
import "console"

include "_parsing_test1_.tmp"



rule d : abc {
    condition:
        true
}

rule a : abc {
    condition:
        true
}

        )";

        std::string yara_entry2 = R"(
import "cuckoo"
import "console"





rule d : abc {
    condition:
        true
}

rule a : abc {
    condition:
        true
}

        )";


        const char *tmp_filename = "_parsing_test1_.tmp";

        std::string yara1 = R"(
rule c : def {
    condition: false
}
        )";

        std::string yara2 = R"(
rule c : def {
    condition: false
}

rule a : def {
    condition: false
}
          )";

        CrossFileSemanticErrorsAfterReparsing() {
        }

        ~CrossFileSemanticErrorsAfterReparsing() {
        }

        void SetUp() override {
            writeStringToFile(tmp_entry_filename, yara_entry1);
            writeStringToFile(tmp_filename, yara1);

            y = std::make_unique<Yaramod>();
        }

        void TearDown() override {
        }

        std::unique_ptr<Yaramod> y;
};

TEST_F(CrossFileSemanticErrorsAfterReparsing, ReparsingOfIncludedFile) {
    YaraSourcePtr y_src = y->parseFile(tmp_entry_filename);
    y_src->edit(38, 38, 0, tmp_filename);

    writeStringToFile(tmp_filename, yara2);

    y->parseFile(tmp_entry_filename, y_src);

    EXPECT_EQ(y_src->getFile(tmp_filename)->getLocalRules().size(), 2);
}

TEST_F(CrossFileSemanticErrorsAfterReparsing, SemanticErrorOccurred) {
    YaraSourcePtr y_src = y->parseFile(tmp_entry_filename);
    y_src->edit(38, 38, 0, tmp_filename);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);

    writeStringToFile(tmp_filename, yara2);

    y->parseFile(tmp_entry_filename, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
}

TEST_F(CrossFileSemanticErrorsAfterReparsing, SemanticErrorDisappearWhenIncludeDeleted) {
    YaraSourcePtr y_src = y->parseFile(tmp_entry_filename);

    y_src->edit(38, 38, 0, tmp_filename);
    writeStringToFile(tmp_filename, yara2);
    y->parseFile(tmp_entry_filename, y_src);

    y_src->edit(35, 0, 29);
    writeStringToFile(tmp_entry_filename, yara_entry2);
    y->parseFile(tmp_entry_filename, y_src);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
}

TEST_F(CrossFileSemanticErrorsAfterReparsing, YaramodShouldnNotParseIncludedFileInManualMode) {
    y->config().setParsingMode(YaramodConfig::ParsingMode::Manual);

    writeStringToFile(tmp_filename, yara2);
    std::shared_ptr<YaraSource> y_src = y->parseFile(tmp_entry_filename);

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getLocalFileIncludes().size(), 1);
    EXPECT_FALSE(y_src->hasFile("_parsing_test1_.tmp"));
}



int main(int argc, char **argv) {
    ::testing::InitGoogleTest(&argc, argv);
    return RUN_ALL_TESTS();
}
