/**
 * condition_management.cpp - Unit tests focused on semantic analysis and
 * parsing of expressions
 * 
 * @author Vojtěch Dvořák 
 */

#include "gtest/gtest.h"
#include "yaramod.h"

#include "common.cpp"


class Arithmetics : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule a {
    condition:
        1 + 2
}

rule b {
    condition:
        1 * 2.4
}

rule c {
    condition:
        1 \ 2
}
        )";

        std::string yara2 = R"(
rule a {
    condition:
        1 + "a"
}

        )";

        std::string yara3 = R"(
rule a {
    condition:
        5.5 % 4
}

        )";

        std::string yara4 = R"(
rule a {
    condition:
        ((1 + 3) * 4) \ 6.6
}

        )";

        Arithmetics() {
        }

        ~Arithmetics() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
};

TEST_F(Arithmetics, ParsingOfValidArithmeticExpression) {
    YaraSourcePtr y_src;
    EXPECT_NO_THROW(y_src = y.parseString(yara1););

    EXPECT_NE(y_src->getEntryFile()->getLocalRule("a")->getCondition(), nullptr);
    EXPECT_NE(y_src->getEntryFile()->getLocalRule("b")->getCondition(), nullptr);
    EXPECT_NE(y_src->getEntryFile()->getLocalRule("c")->getCondition(), nullptr);
};

TEST_F(Arithmetics, ParsedConditionsHaveCorrectDataType) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara1);
    
    EXPECT_EQ(y_src->getEntryFile()->getLocalRule("a")->getCondition()->getType(), Expression::Type::Int);
    EXPECT_EQ(y_src->getEntryFile()->getLocalRule("b")->getCondition()->getType(), Expression::Type::Float);
    EXPECT_EQ(y_src->getEntryFile()->getLocalRule("c")->getCondition()->getType(), Expression::Type::Int);
};

TEST_F(Arithmetics, TypeMismatch) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara2);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(Arithmetics, BadTypesOfRemainderOperands) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara3);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(Arithmetics, ComplexExpression) {
    YaraSourcePtr y_src;
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class RelationalOperators : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule a {
    condition:
        1 + 2 > 2
}

rule b {
    condition:
        "a" == "b"
}

rule c {
    condition:
        5.5 != 6
}
        )";

        std::string yara2 = R"(
rule a {
    condition:
        5.5 <= 5
}

rule b {
    condition:
        -1 >= 5
}

rule c {
    condition:
        4 < 1
}

        )";

        std::string yara3 = R"(
rule a {
    condition:
        5.5 == "a"
}

        )";

        std::string yara4 = R"(
rule a {
    condition:
        "abc" startswith "a"
}

rule b {
    condition:
        "abc" iendswith "C"
}

rule c {
    condition:
        "abc" endswith "C"
}

rule d {
    condition:
        "def" iequals "DEF"
}

        )";

        std::string yara5 = R"(
rule a {
    condition:
        5.5 iequals "a"
}

        )";


        RelationalOperators() {
        }

        ~RelationalOperators() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
};

TEST_F(RelationalOperators, Basics1) {
    YaraSourcePtr y_src;

    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RelationalOperators, Basics2) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara2);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RelationalOperators, EqualOperatorTypeMismatch) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara3);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RelationalOperators, StringRelationalOperators) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara4);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RelationalOperators, StringRelationalOperatorsTypeMismatch) {
    YaraSourcePtr y_src;
    y_src = y.parseString(yara5);
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class StringsRefExpressions : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule a {
    strings:
        $s00 = "a"
        $s01 = "b"
        $h00 = { BB ?? AA }
        $r00 = /abcdef/i
    condition:
        $s00 and $s01 or $h00 and not $r00
}

        )";

        std::string yara2 = R"(
rule a {
    strings:
        $s00 = "a"
        $s01 = "b"
    condition:
        $s00 in (0..100) and not $s01 in (100..200) 
}
        )";

        std::string yara3 = R"(
rule a {
    strings:
        $s00 = "a"
        $s01 = "b"
        $h00 = { BB ?? AA }
        $r00 = /abcdef/i
    condition:
        all of ($s00, $s01, $h00)
}


rule b {
    strings:
        $s00 = "a"
        $s01 = "b"
        $h00 = { BB ?? AA }
        $r00 = /abcdef/i
    condition:
        2 of ($s00, $s01, $r00, $h00)
}

        )";

        std::string yara4 = R"(
rule a {
    strings:
        $s00 = "a"
        $s01 = "b"
        $h00 = { BB ?? AA }
        $r00 = /abcdef/i
    condition:
        #s00 == 3 and (@s01 == 3 or @s01[1] > 3) and !h00[0] < 3 or !r00 > 3  
}

        )";

        std::string yara5 = R"(
rule a {
    strings:
        $s00 = "abc"
    condition:
        $s00 at 300
}

        )";


        StringsRefExpressions() {
        }

        ~StringsRefExpressions() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};


TEST_F(StringsRefExpressions, BasicsBooleanExpression) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(StringsRefExpressions, StringsWithInOperator) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(StringsRefExpressions, StringsWithOfOperator) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(StringsRefExpressions, StringPropertyExpression) {
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(StringsRefExpressions, StringsWithAtOperator) {
    EXPECT_NO_THROW(y_src = y.parseString(yara5););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class RuleRefExpressions : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule ref1 {
    condition: true
}

rule ref2 {
    condition: false
}

rule ref3 {
    condition: false
}

rule ref4 {
    condition: false
}

rule main {
    condition:
        ref1 and ref2 or ref3 and not ref4
}
        )";

        std::string yara2 = R"(
rule ref1 {
    condition: true
}

rule ref2 {
    condition: false
}

rule ref3 {
    condition: false
}

rule ref4 {
    condition: false
}

rule main {
    condition:
        2 of (ref2, ref3, ref4)
}

        )";

        std::string yara3 = R"(
rule ref1 {
    condition: true
}

rule ref2 {
    condition: false
}

rule ref3 {
    condition: false
}

rule ref4 {
    condition: false
}

rule main {
    condition:
        2 of (ref*)
}

        )";

        std::string yara4 = R"(
rule ref1 {
    condition: true
}

rule ref2 {
    condition: false
}

rule ref3 {
    condition: false
}

rule ref4 {
    condition: false
}

rule main {
    condition:
        50% of (ref*)
}

        )";


        RuleRefExpressions() {
        }

        ~RuleRefExpressions() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(RuleRefExpressions, RuleReferencesInBooleanExpressions) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RuleRefExpressions, RuleReferencesWithOf) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RuleRefExpressions, RuleReferencesWildcard) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(RuleRefExpressions, PercentQuantifier) {
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};




class ModuleSymbolsExpression : public ::testing::Test {
    protected:
        std::string yara1 = R"(
import "console"

rule a {
    condition:
        console.log("Hello!")
}
        )";

        std::string yara2 = R"(
import "console"
import "cuckoo"

rule a {
    condition:
        console.log("Hello!") and cuckoo.network.dns_lookup(/evil\\.com/)
}

        )";

        std::string yara3 = R"(
import "pe"

rule a {
    condition:
        (pe.MACHINE_AMD64 or pe.MACHINE_IA64) and pe.export_details[0].offset
}
        )";

        std::string yara4 = R"(
import "pe"

rule a {
    condition:
        cuckoo.processes[1].parent.parent.parent.parent.parent.parent.parent.pid == 123
}
        )";


        ModuleSymbolsExpression() {
        }

        ~ModuleSymbolsExpression() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};


TEST_F(ModuleSymbolsExpression, TrivialUSageOfConsoleModule) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ModuleSymbolsExpression, CuckooNetworkDNSLookup) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ModuleSymbolsExpression, PEModule) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ModuleSymbolsExpression, ReferenceKind) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class ForExpressionParsing : public ::testing::Test {
    protected:
        std::string yara1 = R"(
import "console"

rule a {
    strings:
        $s00 = "a"
        $s01 = "b"
        $s02 = "c"
    condition:
        for all of ($s*) : ( # > 3 ) 
}
        )";

        std::string yara2 = R"(
import "pe"

rule a {
    //This test case was taken from YARA documentation (https://yara.readthedocs.io/en/stable/writingrules.html#using-anonymous-strings-with-of-and-for-of)
    condition:
        for any s in ("71b36345516e076a0663e0bea97759e4", "1e7f7edeb06de02f2c2a9319de99e033") : ( pe.imphash() == s )
}

        )";

        std::string yara3 = R"(
import "pe"

rule a {
    //This test case was taken from YARA documentation (https://yara.readthedocs.io/en/stable/writingrules.html#using-anonymous-strings-with-of-and-for-of)
    condition:
        for any section in pe.sections : ( section.name == ".text" )
}

        )";

        std::string yara4 = R"(
import "console"

rule a {
    condition:
        for all i in (1..100) : ( 
            for all j in (1..100) : ( 
                console.log(i) and console.log(" ") and console.log(j)
            ) 
        ) 
}
        )";


        ForExpressionParsing() {
        }

        ~ForExpressionParsing() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(ForExpressionParsing, ForWithStrings) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ForExpressionParsing, PEWithImpHash) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ForExpressionParsing, PESections) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(ForExpressionParsing, NestedForLoops) {
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class SemanticErrorsTest : public ::testing::Test {
    protected:
        std::string yara1 = R"(
rule a {
    condition:
        "this should have been number" == 1
}
        )";

        std::string yara2 = R"(
import "cuckoo"

rule a {
    condition:
        cuckoo.network
}

        )";

        std::string yara3 = R"(
rule a {
    condition:
        $s00
}

        )";

        std::string yara4 = R"(
import "cuckoo"

rule a {
    condition:
        cuckoo.network.dns_lookup(100)
}

        )";

        std::string yara5 = R"(
rule a {
    condition:
        abc
}

        )";


        std::string yara6 = R"(
rule a {
    condition:
        1 matches /abc/
}

        )";


        std::string yara7 = R"(
rule a {
    condition:
        3 of ($a*)
}

        )";


        std::string yara8 = R"(
import "console"

rule a {
    condition:
        for all i in (1..10) : ( console.log(i) ) and i == 1
}

        )";

        std::string yara9 = R"(
rule a {
    condition:
        console.log("a")
}

        )";

        std::string yara10 = R"(
import "console"

rule defined_before {
    condition:
        true
}

rule a {
    condition:
        1 of (defined_before, defined_after)
}


rule defined_after {
    condition:
        true
}

        )";


        SemanticErrorsTest() {
        }

        ~SemanticErrorsTest() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(SemanticErrorsTest, EqOperatorTypeMismatch) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, InvalidSymbolUsage) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, UndefinedString) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, BadFunctionArguments) {
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, UndefinedIdentifier) {
    EXPECT_NO_THROW(y_src = y.parseString(yara5););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, MatchesOperatorTypeMismatch) {
    EXPECT_NO_THROW(y_src = y.parseString(yara6););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, UndefinedStringWildcard) {
    EXPECT_NO_THROW(y_src = y.parseString(yara7););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, UsageOfVariableOutsideContext) {
    EXPECT_NO_THROW(y_src = y.parseString(yara8););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, UndefinedModuleSymbol) {
    EXPECT_NO_THROW(y_src = y.parseString(yara9););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(SemanticErrorsTest, RuleReferenceBeforeDefinition) {
    EXPECT_NO_THROW(y_src = y.parseString(yara10););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class Reparsing : public ::testing::Test {
    protected:
        std::string yara1 = R"(
import "cuckoo"

rule a {
    condition:
        cuckoo.network.dns_lookup(/evil\.com/i)
}
        )";

        std::string yara1_updated = R"(


rule a {
    condition:
        cuckoo.network.dns_lookup(/evil\.com/i)
}
        )";


        std::string yara2 = R"(


rule a {
    condition:
        cuckoo.network.dns_lookup(/evil\.com/i)
}

        )";

        std::string yara2_updated = R"(
import "cuckoo"

rule a {
    condition:
        cuckoo.network.dns_lookup(/evil\.com/i)
}

        )";


        std::string yara3 = R"(
rule ref { condition: true }

rule b {
    condition:
        true
}

rule a {
    condition:
        ref
}

        )";

        std::string yara3_updated = R"(


rule b {
    condition:
        true
}

rule a {
    condition:
        ref
}
        )";

        Reparsing() {
        }

        ~Reparsing() {
        }

        void SetUp() override {
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(Reparsing, DeletedModuleImportShouldRaiseError) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 0}, {1, 15}}, "");

    EXPECT_NO_THROW(y.parseString(yara1_updated, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(Reparsing, AddedModuleImportShouldDismissError) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 0}, {1, 0}}, "import \"cuckoo\"");

    EXPECT_NO_THROW(y.parseString(yara2_updated, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};

TEST_F(Reparsing, ReferencedRuleDeletionAndAddition) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 0}, {1, 28}}, "");

    EXPECT_NO_THROW(y.parseString(yara3_updated, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 0}, {1, 0}}, "rule ref { condition: true }");

    EXPECT_NO_THROW(y.parseString(yara3, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


class ReparsingWithInclude : public ::testing::Test {
    protected:
        std::string yara1 = R"(
include  "_parsing_test1_.tmp"

rule a {
    condition:
        b
}
        )";

        std::string yara1_updated = R"(


rule a {
    condition:
        b
}
        )";

        const char *tmp1_filename = "_parsing_test1_.tmp";

        std::string yara1_included = R"(

rule b {
    condition:
        true
}
        )";


        std::string yara2 = R"(
include  "_parsing_test2_.tmp"

rule a {
    condition:
        b
}
        )";


        const char *tmp2_filename = "_parsing_test2_.tmp";

        std::string yara2_included = R"(

rule b {
    condition:
        1 + 2 + 3
}
        )";

        std::string yara2_included_updated = R"(

rule changed_rule_b {
    condition:
        1 - 2 - 3
}
        )";


        std::string yara3 = R"(
include  "_parsing_test3_.tmp"

rule a {
    condition:
        console.log("Hello!")
}
        )";

        const char *tmp3_filename = "_parsing_test3_.tmp";

        std::string yara3_included = R"(
import "console"
        )";

        std::string yara3_included_updated = R"(
import "cuckoo"
        )";


        std::string yara4 = R"(
import "console"
include  "_parsing_test4_.tmp"
rule parent {
    condition:
        referenced
}
        )";

        std::string yara4_updated = R"(
import "cuckoo"
include  "_parsing_test4_.tmp"
rule parent {
    condition:
        referenced
}
        )";

        const char *tmp4_filename = "_parsing_test4_.tmp";

        std::string yara4_included = R"(
rule referenced {
    condition:
        console.log("Hello!")
}


        )";

        std::string yara4_included_updated = R"(
rule ref {
    condition:
        console.log("Hello!")
}


        )";


        std::string yara5 = R"(
import "console"
include "_parsing_test5_1_.tmp"
include "_parsing_test5_2_.tmp"
        )";

        std::string yara5_updated = R"(
import "console"

include "_parsing_test5_2_.tmp"
        )";

        std::string yara5_included1 = R"(
rule referenced {
    condition:
        true
}
        )";

        std::string yara5_included2 = R"(
rule parent {
    condition:
        console.log("Hello!") and referenced
}
        )";

        const char *tmp5_filename1 = "_parsing_test5_1_.tmp";


        const char *tmp5_filename2 = "_parsing_test5_2_.tmp";


        ReparsingWithInclude() {
        }

        ~ReparsingWithInclude() {
        }

        void SetUp() override {
            writeStringToFile(tmp1_filename, yara1_included);
            writeStringToFile(tmp2_filename, yara2_included);
            writeStringToFile(tmp3_filename, yara3_included);
            writeStringToFile(tmp4_filename, yara4_included);
            writeStringToFile(tmp5_filename1, yara5_included1);
            writeStringToFile(tmp5_filename2, yara5_included2);
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(ReparsingWithInclude, DeleteFileInclude) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 0}, {1, 30}}, "");

    EXPECT_NO_THROW(y.parseString(yara1_updated, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


TEST_F(ReparsingWithInclude, ReferencedDeletedRuleInIncludedFile) {
    EXPECT_NO_THROW(y_src = y.parseString(yara2););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{2, 5}, {2, 6}}, "changed_rule_b", tmp2_filename);
    y_src->edit({{4, 10}, {4, 11}}, "-", tmp2_filename);
    y_src->edit({{4, 14}, {4, 15}}, "-", tmp2_filename);

    writeStringToFile(tmp2_filename, yara2_included_updated);

    EXPECT_NO_THROW(y.parseString(yara2, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


TEST_F(ReparsingWithInclude, ImportInIncludedFile) {
    EXPECT_NO_THROW(y_src = y.parseString(yara3););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 9}, {1, 14}}, "uckoo", tmp3_filename);

    writeStringToFile(tmp3_filename, yara3_included_updated);

    EXPECT_NO_THROW(y.parseString(yara3, y_src););

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
};


TEST_F(ReparsingWithInclude, TwoWayCrossFileDependecy) {
    EXPECT_NO_THROW(y_src = y.parseString(yara4););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 8}, {1, 15}}, "", tmp4_filename);
    writeStringToFile(tmp4_filename, yara4_included_updated);

    EXPECT_NO_THROW(y.parseString(yara4, y_src));

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSyntaxErrors().size(), 0);

    y_src->edit({{1, 9}, {1, 15}}, "uckoo");
    EXPECT_NO_THROW(y.parseString(yara4_updated, y_src));

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getFile(tmp4_filename)->getSyntaxErrors().size(), 0);
};


TEST_F(ReparsingWithInclude, DependencyOfNeighboringIncludes) {
    EXPECT_NO_THROW(y_src = y.parseString(yara5););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    EXPECT_EQ(y_src->getFile(tmp5_filename1)->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp5_filename1)->getSyntaxErrors().size(), 0);

    EXPECT_EQ(y_src->getFile(tmp5_filename2)->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getFile(tmp5_filename2)->getSyntaxErrors().size(), 0);

    y_src->edit({{2, 0}, {2, 31}}, "");
    EXPECT_NO_THROW(y.parseString(yara5_updated, y_src));

    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    EXPECT_FALSE(y_src->hasFile(tmp5_filename1));

    EXPECT_EQ(y_src->getFile(tmp5_filename2)->getSemanticErrors().size(), 1);
    EXPECT_EQ(y_src->getFile(tmp5_filename2)->getSyntaxErrors().size(), 0);
};


class ExternalVariables : public ::testing::Test {
    protected:
        std::string yara1 = R"(

rule a {
    condition:
        ext_var + 2
}
        )";

        ExternalVariables() {
        }

        ~ExternalVariables() {
        }

        void SetUp() override {
            y.config().addVar("ext_var", 3);
        }

        void TearDown() override {
        }

        Yaramod y;
        YaraSourcePtr y_src;
};

TEST_F(ExternalVariables, ConditionWithExternalVariablesShouldBeParsedCorrectly) {
    EXPECT_NO_THROW(y_src = y.parseString(yara1););
    
    EXPECT_EQ(y_src->getEntryFile()->getSemanticErrors().size(), 0);
    EXPECT_EQ(y_src->getEntryFile()->getSyntaxErrors().size(), 0);

    EXPECT_EQ(y_src->getEntryFile()->getLocalRule(0)->getCondition()->getType(), Expression::Type::Int);
};


int main(int argc, char **argv) {
    ::testing::InitGoogleTest(&argc, argv);
    return RUN_ALL_TESTS();
}
