/**
 * @file yaramod.cpp
 * @brief Implementation of Yaramod methods
 * 
 * @author Vojtěch Dvořák    
 */


#include "../headers/common.h"
#include "../headers/module.h"
#include "../headers/yara_source.h"
#include "../headers/yaramod_config.h"

#include "../headers/yaramod.h"



Yaramod::Yaramod() {
    config_ = std::make_shared<YaramodConfig>();
    yara_parser_ = std::make_unique<TSParserAdapter>(config_);

    loadModules();
}


Yaramod::Yaramod(
    const std::string &base_dir,
    const std::string &module_dir,
    YaramodConfig::ParsingMode p_mode,
    YaramodConfig::ErrorMode e_mode) {

    config_ = std::make_shared<YaramodConfig>();
    config_->setBaseDir(base_dir);
    config_->setModuleDir(module_dir);
    config_->setParsingMode(p_mode);
    config_->setErrorMode(e_mode);

    yara_parser_ = std::make_unique<TSParserAdapter>(config_);

    loadModules();
}


Yaramod::~Yaramod() {
}


YaraSourcePtr Yaramod::parseFile(std::string_view path) const {
    YaraSourcePtr yara_src;

    yara_src = yara_parser_->createYaraSourceFromFile(path.data());

    return yara_src;
}


void Yaramod::parseFile(std::string_view path, const YaraSourcePtr &old_src) const {
    parseFile(path.data(), old_src.get());
}


void Yaramod::parseFile(std::string_view path, YaraSource *old_src) const {
    yara_parser_->updateYaraSourceFromFile(path.data(), old_src);
}


YaraSourcePtr Yaramod::parseString(const std::string &string) const {
    YaraSourcePtr yara_src = std::make_unique<YaraSource>(); 

    yara_src = yara_parser_->createYaraSource(string);

    return yara_src;
}


YaraSourcePtr Yaramod::parseString(const std::string &string, const std::string &entry_file_name) const {
    YaraSourcePtr yara_src = std::make_unique<YaraSource>(); 

    yara_src = yara_parser_->createYaraSource(string, entry_file_name);

    return yara_src;
}


void Yaramod::parseString(const std::string &string, const YaraSourcePtr &old_src) const {
    parseString(string, old_src.get());
}


void Yaramod::parseString(const std::string &string, YaraSource *old_src) const {
    yara_parser_->updateYaraSource(string, old_src);
}


void Yaramod::parseString(const std::string &string, const YaraSourcePtr &old_src, const std::string &entry_file_name) const {
    parseString(string, old_src.get(), entry_file_name);
}


void Yaramod::parseString(const std::string &string, YaraSource *old_src, const std::string &entry_file_name) const {
    yara_parser_->updateYaraSource(string, entry_file_name, old_src);
}


YaramodConfig &Yaramod::config() {
    return *config_;
}


void Yaramod::loadModulesFromPath(std::string_view path) {
    if(!std::filesystem::exists(path)) {
        return;
    }

    if(std::filesystem::is_directory(path)) {
        for(auto &content: std::filesystem::directory_iterator(path)) {
            std::shared_ptr mod = std::make_shared<Module>(content.path());
            mod->load();
            ModuleProvider::getInstance().addModule(mod);
        }
    }
    else if(std::filesystem::is_regular_file(path)) {
        std::shared_ptr mod = std::make_shared<Module>(path);
        mod->load();
        ModuleProvider::getInstance().addModule(mod);
    }
}


void Yaramod::loadModulesFromEnvironment() {
    const char *env_var = std::getenv(MODULE_PATH_ENV_VAR);
    if(!env_var) {
        return;
    }

    #if _WIN32
        char delim = ';';
    #else 
        char delim = ':';
    #endif

    std::string_view env_modules_path(env_var);
    while(!env_modules_path.empty()) {
        size_t index = env_modules_path.find(delim);

        std::string_view cur_path = env_modules_path.substr(0, index);
        loadModulesFromPath(cur_path);

        if(index == std::string_view::npos) {
            break;
        }
        else {
            env_modules_path = env_modules_path.substr(index + 1, env_modules_path.length());
        }
    }
}


void Yaramod::loadModules() {
    loadModulesFromEnvironment();

    if(!std::filesystem::exists(config().getModuleDir())) {
        DEBUG_LOG("Module directory (%s) does not exists!\n", config().getModuleDir().c_str());
        return;
    }
    loadModulesFromPath(config().getModuleDir());
}
