/**
 * @file yara_source.cpp
 * @brief Implementation of YaraSource methods
 * 
 * @author Vojtěch Dvořák    
 */


#include "headers/common.h"
#include "headers/yara_file.h"

#include "headers/yara_source.h"


YaraSource::~YaraSource() {
}


const std::shared_ptr<YaraFile> &YaraSource::getEntryFile() const {
    return entry_file_;
}


std::shared_ptr<YaraFile> YaraSource::getFile(std::string_view path) const {
    std::string name(path);
    if(!path.empty()) {
        name = std::filesystem::weakly_canonical(path);
    }

    auto result_it = files_.find(name);
    if(result_it == files_.end()) {
        return nullptr;
    }
    else {
        return result_it->second;
    }
}


bool YaraSource::hasFile(std::string_view path) {
    std::string name(path);
    if(!path.empty()) {
        name = std::filesystem::weakly_canonical(path);
    }

    return files_.find(name) != files_.end();
}


void YaraSource::addFile(const std::shared_ptr<YaraFile> &file_ptr) {
    std::string file_name = normalizeFSPath(file_ptr->getName());

    if(files_.empty()) {
        entry_file_ = file_ptr;
    }

    files_[file_name] = file_ptr;
}


void YaraSource::setEntryFile(std::string_view path) {
    auto file = getFile(path);
    if(file != nullptr) {
        entry_file_ = file; ///< File must be already registered in this YaraSource
        DEBUG_LOG("Setting entry file...\n");
    }
}


void YaraSource::removeFile(const std::string &path) {
    std::string name(path);
    if(!path.empty()) {
        name = std::filesystem::weakly_canonical(path);
    }

    auto file_it = files_.find(name);
    if(file_it == files_.end()) {
        return;
    }

    file_it->second->clearAll();
    files_.erase(file_it);
}


bool YaraSource::renameFile(std::string_view old_path, std::string_view new_path) {
    std::string name(old_path);
    if(!old_path.empty()) {
        name = std::filesystem::weakly_canonical(old_path);
    }

    if(hasFile(new_path)) {
        return false;
    }

    auto file_it = files_.find(name);
    if(file_it == files_.end()) {
        return false;
    }

    auto file_node = files_.extract(file_it);

    file_node.mapped()->setName(new_path);
    file_node.key() = new_path;

    files_.insert(std::move(file_node));

    return true;
}


const YaraSource::FileMap &YaraSource::getFiles() const {
    return files_;
}


void YaraSource::clearInactiveFiles() {
    for(auto file_it = files_.cbegin(); file_it != files_.cend();) { ///< Clear all files that have not set activity flag (they are not included in yara source)
            
        if(!file_it->second->isActive()) {
            file_it->second->clearAll();
            file_it = files_.erase(file_it);
        }
        else {
            ++file_it;
        }
    }
}


void YaraSource::deactivateFiles() {
    for(auto file_it = files_.cbegin(); file_it != files_.cend(); ++file_it) {

        file_it->second->setActivity(false);
    }
}


void YaraSource::edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n) {
    edit(offset, ins_n, del_n, getEntryFile()->getName());
}


void YaraSource::edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n, const std::string &path) {
    YaraFilePtr yara_file = getFile(path);

    if(yara_file != nullptr) {
        DEBUG_LOG("*Editing of %s (by edit params)\n", path.empty() ? "unnamed" : path.c_str());
        yara_file->edit(offset, ins_n, del_n);
    }
}


void YaraSource::edit(const range_t &edited_range, const std::string &text) {
    edit(edited_range, text, getEntryFile()->getName());
}


void YaraSource::edit(const range_t &edited_range, const std::string &text, const std::string &path) {
    YaraFilePtr yara_file = getFile(path);
    if(yara_file != nullptr) {
        DEBUG_LOG("*Editing of %s (by range [<%d, %d>,(%d, %d)] )\n", 
            path.empty() ? "unnamed" : path.c_str(), 
            edited_range.start.row, 
            edited_range.start.col, 
            edited_range.end.row, 
            edited_range.end.col
        );
        yara_file->edit(edited_range, text);
    }
}


void YaraSource::undo() {
    for(auto &f: files_) {
        f.second->clearEdits();
    }
}


std::stringstream YaraSource::getTextFormatted() const {
    return getEntryFile()->getTextFormatted();
}


const std::string &YaraSource::getBaseDir() {
    return base_path_;
}


void YaraSource::setBaseDir(std::string_view path) {
    std::filesystem::path fs_path(path);

    base_path_ = fs_path.parent_path().string();
}


void YaraSource::notifyChange(YaraFileElement *element) {
    if(bindings_.find(element) != bindings_.end()) {
        DEBUG_LOG("Notifying...\n");

        for(auto &b: bindings_[element]) { ///< Iterate over all children of element
            DEBUG_LOG("Notifying element at (%d, %d)\n", b->getPosition().row, b->getPosition().col);
            offset_range_t range = { ///< Get their range
                b->getOffset(), 
                b->getOffset() + static_cast<offset_t>(b->getLen())
            };

            if(b->isFixed()) { ///< If dependent element has fixed position that depends on this element, request its deletion 
                DEBUG_LOG("Deletion requested!\n");
                b->setParent(nullptr); // Prevent segmentation fault by setting parent element of dependent element to nullptr
                b->setOffset(b->getOffset() + element->getOffset()); // Freeze dependent offset at the latest value
                b->requestDeletion();
            }
            else {
                if(auto file = b->getParentFile().lock()) {
                    DEBUG_LOG("Added check range to file '%s'\n", file->getName().c_str());
                    file->checkRange(range); ///< Add range to check_ranges (these ranges will be reparsed)
                }      
            }
        }

    }
}


void YaraSource::bind(YaraFileElement *parent, YaraFileElement *dependency) {
    bindings_[parent].insert(dependency); ///< Creates bind between parent and dependency
    bindings_rev_[dependency].insert(parent); ///< Create reversed binding to easily delete forward binding
}


void YaraSource::deleteBindings(YaraFileElement *element) {
    // dumpBindChildren();
    // dumpBindParents();
    auto children_it = bindings_.find(element);
    if(children_it != bindings_.end()) { ///< Destroy all bindings where element acts as child
        for(auto &ch: children_it->second) {
            auto children_dependency_set = bindings_rev_.find(ch);
            if(children_dependency_set != bindings_rev_.end()) {
                children_dependency_set->second.erase(element);
            }

        }
    }

    auto parents_it = bindings_rev_.find(element);
    if(parents_it != bindings_rev_.end()) { ///< Destroy all bindings where element acts as child
        for(auto &p: parents_it->second) {
            auto parent_dependency_set = bindings_.find(p);
            if(parent_dependency_set != bindings_.end()) {
                parent_dependency_set->second.erase(element);
            }

        }
    }

    if(children_it != bindings_.end()) {
        bindings_.erase(children_it); ///< Destroy all bindings where element acts as parent
    }

    if(parents_it != bindings_rev_.end()) {
        bindings_rev_.erase(parents_it); ///< Remove reverse binding of element
    }
    
    DEBUG_LOG("Deletion of bindings is done\n");
}


void YaraSource::moveBindings(YaraFileElement *original_parent, YaraFileElementBindable *new_parent) {
    auto children_it = bindings_.find(original_parent);
    if(children_it != bindings_.end()) {
        for(auto &ch: children_it->second) {
            auto children_dependency_set = bindings_rev_.find(ch);
            if(children_dependency_set != bindings_rev_.end()) {
                children_dependency_set->second.erase(original_parent);
                children_dependency_set->second.insert(new_parent);
            }
        }
    }

    auto parents_it = bindings_rev_.find(original_parent);
    if(parents_it != bindings_rev_.end()) {
        for(auto &p: parents_it->second) {
            auto parent_dependency_set = bindings_.find(p);
            if(parent_dependency_set != bindings_.end()) {
                parent_dependency_set->second.erase(original_parent);
                parent_dependency_set->second.insert(new_parent);
            }
        }
    }

    if(children_it != bindings_.end()) {
        for(auto &ch: children_it->second) {
            bindings_[new_parent].insert(ch);
        }

        bindings_.erase(children_it);
    }

    if(parents_it != bindings_rev_.end()) {
        for(auto &p: parents_it->second) {
            bindings_rev_[new_parent].insert(p);
        }

        bindings_rev_.erase(parents_it);
    }
}


void YaraSource::dumpBindChildren() {
    for(auto &binding_set: bindings_) {
        std::cerr << "Element at " << binding_set.first->getOffset();

        range_t r = binding_set.first->getRange();
        std::cerr << ", <" << r.start.row << ", " << r.start.col << ">, ";
        std::cerr << "(" << r.end.row << ", " << r.end.col << ") is PARENT for: " << std::endl;

        for(auto &binded: binding_set.second) {
            std::cerr << "Element at " << binded->getOffset();

            range_t r = binded->getRange();
            std::cerr << ", <" << r.start.row << ", " << r.start.col << ">, ";
            std::cerr << "(" << r.end.row << ", " << r.end.col << ")" << std::endl;
        }
    }
}


void YaraSource::dumpBindParents() {
    for(auto &binding_set: bindings_rev_) {
        std::cerr << "Element at " << binding_set.first->getOffset();

        range_t r = binding_set.first->getRange();
        std::cerr << ", <" << r.start.row << ", " << r.start.col << ">, ";
        std::cerr << "(" << r.end.row << ", " << r.end.col << ") is CHILD for: " << std::endl;

        for(auto &binded: binding_set.second) {
            std::cerr << "Element at " << binded->getOffset();

            range_t r = binded->getRange();
            std::cerr << ", <" << r.start.row << ", " << r.start.col << ">, ";
            std::cerr << "(" << r.end.row << ", " << r.end.col << ")" << std::endl;
        }
    }
}


std::vector<Rule *> YaraSource::getAllRules() {
    return entry_file_->getRules();
}


std::vector<SyntaxError *> YaraSource::getAllSyntaxErrors() {
    std::vector<SyntaxError *> result;

    for(auto &f: files_) {
        for(auto e: f.second->getSyntaxErrors()) {
            result.insert(result.end(), e);
        }
    }

    return result;
}


std::vector<SemanticError *> YaraSource::getAllSemanticErrors() {
    std::vector<SemanticError *> result;
    
    for(auto &f: files_) {
        for(auto e: f.second->getSemanticErrors()) {
            result.insert(result.end(), e);
        }
    }

    return result;
}


Import *YaraSource::getImport(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) const {
    if(entry_file_) {
        return entry_file_->getImport(name, stop_file, includes);
    }
    else {
        return nullptr;
    }
}


Rule *YaraSource::getRule(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) const {
    if(entry_file_) {
        return entry_file_->getRule(name, stop_file, includes);
    }
    else {
        return nullptr;
    }
}


bool YaraSource::hasPendingChanges() const {
    for(auto &f: files_) {
        if(f.second->wasEdited() || f.second->mustBeReparsed()) {
            return true;
        }
    }

    return false;
}

