/**
 * @file yara_file_element.cpp 
 * @brief Implementation of YaraFileElement methods 
 * 
 * @author Vojtěch Dvořák
 */

#include "../headers/common.h"
#include "../headers/yara_file.h"

#include "../headers/yara_file_element.h"


// YaraFileElement

YaraFileElement::YaraFileElement() {
}


YaraFileElement::YaraFileElement(const offset_t &offset, const size_t &len) : offset_(offset), len_(len) {
}


size_t YaraFileElement::getLen() const {
    return len_;
}


void YaraFileElement::setLen(const size_t &new_len) {
    len_ = new_len;
}


offset_t YaraFileElement::getOffset() const {
    if(parent_element_) {
        return parent_element_->getOffset() + offset_;
    }
    else {
        return offset_;
    }
}


void YaraFileElement::setOffset(const offset_t &new_offset) {
    offset_ = new_offset;
}


point_t YaraFileElement::getPosition() const {
    auto parent = parent_file_.lock();

    if(!parent) {
        return {
            static_cast<uint32_t>(-1), 
            static_cast<uint32_t>(-1)
        };
    }

    return parent->ctx().offsetToPointCached(getOffset());
}


range_t YaraFileElement::getRange() const {
    auto parent = getParentFile().lock();
    if(!parent) {
        return { 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)}, 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)} 
        };
    }

    point_t start = parent->ctx().offsetToPointCached(getOffset());
    point_t end = parent->ctx().offsetToPointCached(getOffset() + getLen());

    return {start, end};
}


void YaraFileElement::setParentFile(const std::shared_ptr<YaraFile> &parent) {
    parent_file_ = parent;
}


std::weak_ptr<YaraFile> YaraFileElement::getParentFile() const {
    return parent_file_;
}


bool YaraFileElement::isDeletionRequested() const {
    return is_deletion_requested_;
}


void YaraFileElement::requestDeletion() {
    is_deletion_requested_ = true;
}


bool YaraFileElement::isFixed() const {
    return parent_element_ != nullptr;
}


void YaraFileElement::setParent(YaraFileElement *parent) {
    parent_element_ = parent;
}


// YaraFileElementBindable

YaraFileElementBindable::YaraFileElementBindable() {
}


YaraFileElementBindable::YaraFileElementBindable(const offset_t &offset, const size_t &len) : YaraFileElement(offset, len) {
}


YaraFileElementBindable::~YaraFileElementBindable() {
    DEBUG_LOG("Deleting YaraFileElementBindable...\n");
    if(auto file = parent_file_.lock()) {
        file->notifyChange(this);

        DEBUG_LOG("Deleting bindings...\n");
        file->deleteBindings(this);
    }
}


YaraFileElementBindable *YaraFileElementBindable::bind(YaraFileElementBindable *dependency) {
    if(auto file = parent_file_.lock()) {
        file->bind(this, dependency);
    }

    return this;
}


YaraFileElementBindable *YaraFileElementBindable::bind_to(YaraFileElementBindable *parent) {
    if(auto file = parent_file_.lock()) {
        file->bind(parent, this);
    }

    return this;
}


// TopLevelYaraFileElement
TopLevelYaraFileElement::TopLevelYaraFileElement() {
}


TopLevelYaraFileElement::TopLevelYaraFileElement(const offset_t &offset, const size_t &len) : YaraFileElementBindable(offset, len) {
}


TopLevelYaraFileElement::~TopLevelYaraFileElement() {
}


point_t TopLevelYaraFileElement::getPosition() {
    if(is_cached_position_valid_ || is_range_cache_valid_) {
        is_cached_position_valid_ = true;
        return cached_range_.start;
    }

    auto parent = getParentFile().lock();
    if(!parent) {
        return { 
            static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)
        };
    }

    point_t position = parent->ctx().offsetToPointCached(getOffset());
    cached_range_.start = position;
    is_cached_position_valid_ = true;

    return position;
}


range_t TopLevelYaraFileElement::getRange() {
    if(is_range_cache_valid_) {
        return cached_range_;
    }

    auto parent = getParentFile().lock();
    if(!parent) {
        return { 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)}, 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)} 
        };
    }

    point_t start;
    if(is_cached_position_valid_) {
        start = cached_range_.start;
    }
    else {
        start = parent->ctx().offsetToPointCached(getOffset());
        is_cached_position_valid_ = true;
    }
     
    point_t end = parent->ctx().offsetToPointCached(
        getOffset() + getLen(), 
        getOffset(), 
        start
    );

    cached_range_ = {start, end};
    is_range_cache_valid_ = true;

    return cached_range_;
}


void TopLevelYaraFileElement::invalidateCache() {
    DEBUG_LOG("Invalidating cache...\n");
    is_range_cache_valid_ = false;
    is_cached_position_valid_ = false;
}


void TopLevelYaraFileElement::setRangeCache(const range_t &range) {
    cached_range_ = range;
    is_range_cache_valid_ = true;
    is_cached_position_valid_ = true;
}


void TopLevelYaraFileElement::setPositionCache(const point_t &position) {
    cached_range_.start = position;
    is_cached_position_valid_ = true;
}


bool TopLevelYaraFileElement::hasValidCache() {
    return is_range_cache_valid_ || is_cached_position_valid_;
}

