/**
 * @file yara_file.cpp 
 * @brief Implementation of YaraFile methods
 * 
 * @author Vojtěch Dvořák
 */


#include "../headers/common.h"
#include "../headers/yara_file_element.h"
#include "../headers/rule.h"
#include "../headers/language.h"
#include "../headers/error_collector.h"
#include "../headers/other.h"
#include "../headers/symtab.h"
#include "../headers/yara_source.h"

#include "../headers/yara_file.h"



YaraFile::YaraFile(YaraSource *parent_src) {
    parent_src_ = parent_src;
}


YaraFile::~YaraFile() {
    DEBUG_LOG("Deleting YaraFile %s\n", getName().c_str());
}


const std::string &YaraFile::getName() const {
    return name_;
}

void YaraFile::setName(const std::string &new_name) {
    name_ = new_name;
}

void YaraFile::setName(std::string_view new_name) {
    name_ = new_name;
}


// Syntax errors management

const SymTabOffset<SyntaxError> &YaraFile::getSyntaxErrors() const {
    return syntax_.getConst();
}

void YaraFile::clearSyntaxErrors() {
    syntax_.clear();
}

template<typename Fn>
uint32_t YaraFile::removeSyntaxErrors(const Fn &f) {
   return syntax_.remove(f);
}

template<typename Fn>
void YaraFile::updateSyntaxErrors(const Fn &f) {
    syntax_.update(f);
}

void YaraFile::dumpSyntaxErrors() {
    syntax_.dump(ctx_.getString());
}


// Semantic errors management

const SymTabOffset<SemanticError> &YaraFile::getSemanticErrors() const {
    return semantic_.getConst();
}

void YaraFile::clearSemanticErrors() {
    semantic_.clear();
}

template<typename Fn>
uint32_t YaraFile::removeSemanticErrors(const Fn &f) {
   return semantic_.remove(f);
}

template<typename Fn>
void YaraFile::updateSemanticErrors(const Fn &f) {
    semantic_.update(f);
}

void YaraFile::dumpSemanticErrors() const {
    syntax_.dump(ctx_.getString());
}


// Rule management

Rule *YaraFile::ruleLookUp(const std::string &rule_id, std::vector<YaraFileElementBindable *> &includes) {
    Rule *same_id_rule = parent_src_->getRule(rule_id, this, includes); ///< Try to find same id rule in parent source
    Rule *local_same_id_rule = same_id_rule = getRule(rule_id); ///< Try To find local same id rule
    
    if(!same_id_rule) {
        same_id_rule = local_same_id_rule;
    }
    else if(local_same_id_rule && same_id_rule) {
        if(includes.empty() || ((*includes.begin())->getOffset() > local_same_id_rule->getOffset())) {
            same_id_rule = local_same_id_rule;
        }
    }

    return same_id_rule;
}


RuleError *YaraFile::createRuleError(const std::string &err_msg, Rule *wrong_rule, const std::vector<YaraFileElementBindable *> &includes) {
    RuleError *err_ptr = nullptr;
    if(includes.empty()) {
        err_ptr = addSemanticError<RuleError>(
            err_msg, wrong_rule->getIdOffset(), wrong_rule->getId().length()
        );
    }
    else { ///< Rule is in another file
        if(auto file = wrong_rule->getParentFile().lock()) {
            err_ptr = wrong_rule->getParentFile().lock()->addSemanticError<RuleError>(
                err_msg, wrong_rule->getIdOffset(), wrong_rule->getId().length()
            );
        }
    }

    return err_ptr;
}


Rule *YaraFile::addRule(RulePtr &&rule) {
    if(rule->getId() == std::string({})) {
        wrong_elements_.insert(std::move(rule));
        return nullptr;
    }

    std::vector<YaraFileElementBindable *> includes;
    Rule *same_id_rule = ruleLookUp(rule->getId(), includes);

    // There is already rule with given id
    if(same_id_rule) {
        Rule *wrong_rule = rule.get(), *correct_rule = same_id_rule;

        // Ensure, that the wrong rule will be rule with greater offset (to be consistent)
        bool incl_reverse_order = !includes.empty() && (*includes.begin())->getOffset() > rule->getOffset();
        bool reverse_order = includes.empty() && same_id_rule->getOffset() > rule->getOffset();
        if(incl_reverse_order || reverse_order) {
            wrong_rule = same_id_rule;
            correct_rule = rule.get();
        }

        std::string err_msg;
        auto parent = same_id_rule->getParentFile().lock();
        if(parent && !includes.empty()) {
            err_msg = "There is already rule '" + rule->getId() + "'! (in " +  normalizeFSPath(parent->getName()) + ")";
        }
        else {
            err_msg = "There is already rule '" + rule->getId() + "'!";
        }

        RuleError *semantic_err_ptr = createRuleError(err_msg, wrong_rule, includes);

        // Bind semantic errors to ensure, that they will be deleted when correct rule will be deleted
        if(parent && semantic_err_ptr) {
            parent->bind(includes, semantic_err_ptr);
            parent->bind(correct_rule, semantic_err_ptr);
        }

        if(incl_reverse_order || reverse_order) {
            parent->moveBindings(wrong_rule, correct_rule); ///< Redirect bindings of wring rule to the correct rule
            rules_tab_.remove(wrong_rule->getId());
        }
        else {
            wrong_elements_.insert(std::move(rule)); ///< Save the wrong rule to wrong element tab
            return nullptr;
        }
    }

    DEBUG_LOG("Inserting %s to rule tab (offset %d)\n", 
        rule->getId().c_str(), 
        rule->getOffset()
    );

    rule->setParentFile(shared_from_this());  

    Rule *raw_ptr = rule.get();  
    rules_tab_.insert(std::move(rule));
    return raw_ptr;
}


void YaraFile::checkDuplicatedRules(const YaraFilePtr &other, FileInclude *include) {
    auto rules = other->getRules();

    for(auto &r: rules) {

        auto same = getRules(r->getId());
        if(same.size() > 1) { ///< Every rule should be at least once in YaraFile (or in included YaraFile)
            if(auto target_file = same.back()->getParentFile().lock()) {
                target_file->addSemanticError<RuleError>(
                    "Rule '" + r->getId() + "' was already defined!",
                    same.back()->getIdOffset(), 
                    r->getId().length()
                )->bind_to(include)->bind_to((*same.begin()));
            }
        }

    }
}


void YaraFile::clearLocalRules() {
    rules_tab_.clear(); ///< Remove rules from the list
}


const SymTab<Rule> &YaraFile::getLocalRules() const {
    return rules_tab_;
}


Rule *YaraFile::getLocalRule(uint32_t index) const {
    auto rules_it = rules_tab_.dataByOffset().begin();
    std::advance(rules_it, index);
    return rules_it->second;
}


Rule *YaraFile::getLocalRuleByRow(uint32_t row) {
    if(rules_tab_.empty()) {
        return nullptr;
    }

    point_t point = {row, 0};
    offset_t row_offset = ctx_.pointToOffsetCached(point);

    auto rule_it = rules_tab_.dataByOffset().upper_bound(row_offset);

    if(rule_it == rules_tab_.dataByOffset().begin()) {
        return nullptr;
    }
    else {
        Rule *rule = (--rule_it)->second;
        if(rule->getRange().end < point) {
            return nullptr;
        }

        return rule;
    }
}


std::vector<Rule *> YaraFile::getRules() const {
    std::vector<Rule *> result;
    auto own_rules = rules_tab_.dataByOffset();
    auto file_includes = includes_tab_.dataByOffset();

    auto rule_it = own_rules.begin();
    for(auto include_it = file_includes.begin(); 
        include_it != file_includes.end(); 
        ++include_it) {
        
        auto included_file = include_it->second->getFile();
        if(!included_file) {
            continue;
        }

        auto included_rules = included_file->getRules();
        
        // It pushes rules until rule with higher offset than firs include is found - function preserves order
        while(rule_it != own_rules.end() && rule_it->first < include_it->first) {

            result.push_back(rule_it->second);
            ++rule_it;
        }

        for(auto r: included_rules) {      
            result.push_back(r);
        }
    }

    // Push the rest of the rules to the result vector
    for(;rule_it != own_rules.end(); ++rule_it) {      
        result.push_back(rule_it->second);
    }

    return result;
}


std::vector<Rule *> YaraFile::getRules(std::string const &name) const {
    std::vector<Rule *> result;

    Rule *local_rule = nullptr;
    if(rules_tab_.exists(name)) { ///< Firstly, try to find rule name in current yara file 
        local_rule = rules_tab_.search(name).get();
    }

    for(auto dep: includes_tab_.dataByOffset()) {

        if(local_rule && local_rule->getOffset() < dep.first) {
            result.push_back(local_rule);
            local_rule = nullptr;
        }

        if(dep.second->getFile()) { ///< Finally, try to find rule name in file includes
            auto dep_result = dep.second->getFile()->getRules(name);

            ///< Insert returned vector to the end of the result vector
            result.insert(
                result.end(), 
                dep_result.begin(),
                dep_result.end()
            );
        }
    }

    if(local_rule) {
        result.push_back(local_rule);
    }

    return result;
}


bool YaraFile::hasRule(std::string_view name) const {
    if(!rules_tab_.exists(name)) {

        // If rule is not directly located in the file itself, try included files
        for(auto i: includes_tab_) {

            if(i->getFile()) {
                if(i->getFile()->hasRule(name)) {
                    return true;
                }
            }

        }

        return false;
    }
    else {
        return true;
    }
}


bool YaraFile::hasRule(std::string_view name, offset_t before) const {
    if(!rules_tab_.exists(name)) {

        // Try included files (but only until before offset is reached)
        for(auto i: includes_tab_) {
            if(i->getOffset() >= before) {
                break;
            }

            if(i->getFile()) {
                if(i->getFile()->hasRule(name)) {
                    return true;
                }
            }

        }

        return false;
    }
    else {
        const auto &rule = rules_tab_.search(name);
        if(rule->getOffset() < before) { ///< Check if rule has offset lower than given offset
            return true;
        }
        else {
            return false;
        }
    }
}


Rule *YaraFile::getRule(std::string_view name) const {
    if(!rules_tab_.exists(name)) {
        for(auto i: includes_tab_) {

            if(i->getFile()) {
                return i->getFile()->getRule(name);
            }

        }

        return nullptr;
    }
    else {
        return rules_tab_.search(name).get();
    }
}


Rule *YaraFile::getRule(std::string_view name, offset_t before) const {
    // Similar to hasRule method

    if(!rules_tab_.exists(name)) {

        // Try included files (but only until before offset is reached)
        for(auto i: includes_tab_) {
            if(i->getOffset() >= before) {
                break;
            }

            if(i->getFile()) {
                return i->getFile()->getRule(name);
            }

        }

        return nullptr;
    }
    else {
        const auto &rule = rules_tab_.search(name);
        if(rule->getOffset() < before) {
            return rule.get();
        }
        else {
            return nullptr;
        }
    }
}


Rule *YaraFile::getRule(std::string_view name, offset_t before, std::vector<YaraFileElementBindable *> &includes, bool make_lookup_from_top) const {
    includes.clear(); //< Clear dependent file includes
    
    if(!rules_tab_.exists(name)) {
        // Try included files
        for(auto i: includes_tab_) {
            if(i->getOffset() >= before) {
                break;
            }

            if(i->getFile()) {
                Rule *included_rule = i->getFile()->getRule(name); ///< Try to find rule in included files
                if(included_rule) {
                    includes.push_back(i); ///< Store dependent fle include in the output vector
                    return included_rule;
                }
            }

        }

        // If there is no such rule in included file (and make_lookup_from_top is true), try to search from the start of the YaraSource
        if(parent_src_ && make_lookup_from_top) {
            return parent_src_->getRule(name, this, includes);
        }
        else {
            return nullptr;
        }
    }
    else { ///< Rule is located in the current yara file
        const auto &rule = rules_tab_.search(name);
        if(rule->getOffset() < before) {
            return rule.get();
        }
        else {
            return nullptr;
        }
    }
}


Rule *YaraFile::getRule(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) {
    for(auto i: includes_tab_) {
        if(i->getFile()) {
            if(i->getFile().get() == stop_file) { ///< Stop file is reached, interrupt searching
                //DEBUG_LOG("STOP FILE REACHED!\n");
                return getRule(name, i->getOffset(), includes);
            }
            else { ///< Included file is not stop file, so try to find the rule
                Rule *result = i->getFile()->getRule(name, stop_file, includes);
                if(result) {
                    includes.push_back(i);
                    return result;
                }
            }
        }
    }

    return getRule(name);
}


const RulePtr &YaraFile::getLocalRule(const std::string &name) const {
    return rules_tab_.search(name);
}


template <typename Fn>
uint32_t YaraFile::removeLocalRules(const Fn &f) {
    return rules_tab_.remove(f);
}



template <typename Fn>
void YaraFile::updateLocalRules(const Fn &f) {
    rules_tab_.update(f);
}


// FileInclude management

FileInclude *YaraFile::addFileInclude(std::unique_ptr<FileInclude> include) {

    if(includes_tab_.exists(include->getPath())) { ///< There is already include with the same path
        addSemanticError<IncludeError>(
            "File is already included in this file!",
            include->getOffset(),
            include->getLen(),
            include->getRange()
        );

        wrong_elements_.insert(std::move(include));
        return nullptr;
    }


    DEBUG_LOG("Inserting %s to include tab (offset %d)\n", 
        include->getPath().c_str(), 
        include->getOffset()
    );

    FileInclude *raw_ptr = include.get();
    includes_tab_.insert(std::move(include));

    return raw_ptr;
}


void YaraFile::clearLocalFileIncludes() {
    includes_tab_.clear();
}


FileInclude *YaraFile::getLocalFileInclude(uint32_t index) const {
    auto include_it = includes_tab_.dataByOffset().begin();
    std::advance(include_it, index);
    return include_it->second;
}


const SymTab<FileInclude> &YaraFile::getLocalFileIncludes() const {
    return includes_tab_;
}


template<typename Fn>
uint32_t YaraFile::removeLocalFileIncludes(const Fn &f) {
    return includes_tab_.remove(f);
}


template <typename Fn>
void YaraFile::updateLocalFileIncludes(const Fn &f) {
    includes_tab_.update(f);
}


std::vector<FileInclude *> YaraFile::getFileIncludes() const {
    std::vector<FileInclude *> result;
    auto file_includes = includes_tab_.dataByOffset();

    // Interspacing of local includes by the included includes

    for(auto include_it = file_includes.begin(); 
        include_it != file_includes.end(); 
        ++include_it) {

        result.push_back(include_it->second);
        
        auto included_file = include_it->second->getFile();
        if(!included_file) {
            continue;
        }

        auto included_includes = included_file->getFileIncludes();

        for(auto i: included_includes) {      
            result.push_back(i);
        }
    }

    return result;
}



// Import management

Import *YaraFile::addImport(std::unique_ptr<Import> import) {
    DEBUG_LOG("Inserting %s to import tab (offset %d)\n", 
        import->getModuleName().c_str(), 
        import->getOffset()
    );

    Import *raw_ptr = import.get();
    imports_.insert(std::move(import));

    return raw_ptr;
}


Import *YaraFile::getImport(std::string_view name) const {
    if(!imports_.exists(name)) {  

        // If there is no such import in current file, try included files 
        for(auto i: includes_tab_) {

            if(i->getFile()) {
                return i->getFile()->getImport(name);
            }

        }

        return nullptr;
    }
    else {
        return imports_.get(name); ///< Return the first import that imports given module
    }
}


Import *YaraFile::getImport(std::string_view name, offset_t before) const {
    if(!imports_.exists(name)) {

        // If there is not such import in current file try to find it in included files (until before offset is reached)
        for(auto i: includes_tab_) {
            if(i->getOffset() >= before) {
                break;
            }

            if(i->getFile()) {
                return i->getFile()->getImport(name); ///< Search for import in included file
            }

        }

        return nullptr;
    }
    else {
        const auto &import = imports_.getAll(name);
        for(auto i: import) { ///< If there is import with given module name, check if any of them occurs before 'before' offset
            if(i->getOffset() < before) {
                return i;
            }
        }

        return nullptr;
    }
}



Import *YaraFile::getImport(std::string_view name, offset_t before, std::vector<YaraFileElementBindable *> &includes, bool make_lookup_from_top) const {
    includes.clear(); ///< Clear dependent includes
    
    if(!imports_.exists(name)) {
        DEBUG_LOG("Import does not exists...\n");
        for(auto i: includes_tab_) {
            if(i->getOffset() >= before) {
                break;
            }

            if(i->getFile()) { ///< Try to find import in included file
                Import *included_import = i->getFile()->getImport(name, before, includes);
                if(included_import) {
                    includes.push_back(i); ///< Store dependent file includes or other YaraFile elements
                    return included_import;
                }
            }

        }

        if(parent_src_ && make_lookup_from_top) { ///< If there is no such import in included file, optionally try to find it from the start of parent YaraSource
            DEBUG_LOG("Checking complete src...\n");
            return parent_src_->getImport(name, this, includes);
        }
        else {
            return nullptr;
        }
    }
    else {
        const auto &import = imports_.getAll(name);
        for(auto i: import) {
            if(i->getOffset() < before) {
                return i;
            }
        }

        return nullptr;
    }
}


Import *YaraFile::getImport(std::string_view name, const YaraFile *stop_file, std::vector<YaraFileElementBindable *> &includes) {
    DEBUG_LOG("Gettting imports from %s\n", getName().c_str());
    for(auto i: includes_tab_) {
        if(i->getFile()) {
            DEBUG_LOG("Searching in %s\n", i->getPath().c_str());
            if(i->getFile().get() == stop_file) { ///< If stop file is reached, interrupt searching
                return getImport(name, i->getOffset(), includes);
            }
            else {
                Import *result = i->getFile()->getImport(name, stop_file, includes);
                if(result) {
                    includes.push_back(i); ///< Store dependent file includes or other YaraFile elements
                    return result;
                }
            }
        }
    }

    return getImport(name);
}


void YaraFile::clearLocalImports() {
    imports_.clear();
}


std::vector<Import *> YaraFile::getImports() const {
    std::vector<Import *> result;
    auto own_imports = imports_.dataByOffset();
    auto file_includes = includes_tab_.dataByOffset();

    // Interspacing of local imports by the included imports

    auto import_it = own_imports.begin();
    for(auto include_it = file_includes.begin(); 
        include_it != file_includes.end(); 
        ++include_it) {
        
        auto included_file = include_it->second->getFile();
        if(!included_file) {
            continue;
        }

        auto included_imports = included_file->getImports();

        // Preserve the order of imports, so push import until offset of include is reached...
        while(import_it != own_imports.end() && import_it->first < include_it->first) {

            result.push_back(import_it->second);
            ++import_it;
        }

        for(auto i: included_imports) {      
            result.push_back(i);
        }
    }

    // Push the rest of imports to the result vector
    for(;import_it != own_imports.end(); ++import_it) {      
        result.push_back(import_it->second);
    }

    return result;
}


const SymTabDuplId<Import> &YaraFile::getLocalImports() const {
    return imports_;
}


Import *YaraFile::getLocalImport(uint32_t index) const {
    auto import_it = imports_.dataByOffset().begin();
    std::advance(import_it, index);
    return import_it->second;
}


template <typename Fn>
uint32_t YaraFile::removeLocalImports(const Fn &f) {
    return imports_.remove(f);
}



template <typename Fn>
void YaraFile::updateLocalImports(const Fn &f) {
    return imports_.update(f);
}


// Comment management

void YaraFile::addComment(std::unique_ptr<Comment> comment) {
    DEBUG_LOG("Inserting new comment to comments tab (offset %d)\n",
        comment->getOffset()
    );

    comments_.insert(std::move(comment));
}

void YaraFile::clearComments() {
    comments_.clear();
}

const std::unique_ptr<Comment> &YaraFile::getCommentByIndex(uint32_t index) const {
    auto comm_it = comments_.data().begin();
    std::advance(comm_it, index);
    return comm_it->second;
}

const SymTabOffset<Comment> &YaraFile::getComments() const {
    return comments_;
}

template <typename Fn>
uint32_t YaraFile::removeComments(const Fn &f) {
    return comments_.remove(f);
}

template <typename Fn>
void YaraFile::updateComments(const Fn &f) {
    return comments_.update(f);
}


void YaraFile::addWrong(std::unique_ptr<TopLevelYaraFileElement> &&element) {
    wrong_elements_.insert(std::move(element));
}

void YaraFile::clearWrong() {
    wrong_elements_.clear();
}


void YaraFile::clearAll() {
    clearLocalRules();
    clearLocalFileIncludes();
    clearLocalImports();
    clearComments();
    clearWrong();

    clearSyntaxErrors();
    clearSemanticErrors();
} 


// Edit management

void YaraFile::addEdit(const edit_t &edit) {
    if(hasFastEdit()) {
        throw YaramodErrorException("If fast edits are used, there can be only one before each reparsing!");
    }

    edits_.push_back(edit);
}


bool YaraFile::hasFastEdit() {
    return !edits_.empty() && edits_.back().fast_edit;
}


void YaraFile::edit(const offset_t &offset, const size_t &ins_n, const size_t &del_n) {
    DEBUG_LOG("*Added numerical (fast) edit");

    edit_t edit(offset, ins_n, del_n);
    edit.fast_edit = true;

    // toTSEdit is called after string updated, because now we don't know how the inserted/deleted text looks  

    addEdit(edit); ///< Save the edit to the vector for reparsing
}


void YaraFile::edit(const range_t &edited_range, const std::string &text) {
    offset_t start_offset = pointToOffset(ctx_.getTempString(), edited_range.start);
    offset_t end_offset = pointToOffset(
        ctx_.getTempString(), 
        edited_range.end, 
        edited_range.start, 
        start_offset
    );

    size_t del_size = end_offset - start_offset;

    DEBUG_LOG("*Added edit by range (translated to offsets: %d-%d)\n", start_offset, end_offset);
    
    edit_t edit(start_offset, text.size(), del_size, text);

    ctx_.stringEdit(edit.offset, edit.del_n, edit.new_text);
    edit.ts_edit = ctx_.toTSEdit(edit.offset, edit.ins_n, edit.del_n, edited_range);

    //DEBUG_LOG("New file content:\n%s\n", ctx_.getString().c_str());
    
    addEdit(edit);
}


void YaraFile::undo() {
    ctx_.undoEdits();
    clearEdits();
}


std::vector<edit_t> &YaraFile::getEdits() {
    return edits_;
}


void YaraFile::clearEdits() {
    edits_.clear();
}


bool YaraFile::wasEdited() {
    return edits_.size() > 0;
}


bool YaraFile::mustBeReparsed() {
    return to_be_checked_.size() > 0;
}


void YaraFile::removeGlobalErrors() {
    // Remove all semantic errors with set global flag

    removeSemanticErrors([](const SemanticError &e) {
        return e.isGlobal();
    });
}

template <typename T>
bool hitsRange(const T &element, const offset_range_t &range, offset_range_t &max_range) {
    offset_t el_start = element.getOffset();
    offset_t el_end = element.getOffset() + element.getLen();

    bool hits_element = (el_end > range.start && el_start < range.end);

    if(hits_element) {
        max_range.start = max_range.start > el_start ? el_start : max_range.start;
        max_range.end = max_range.end > el_end ? el_end : max_range.end;
    }
    
    return hits_element;
}


offset_range_t YaraFile::removeElementsInRange(const offset_t &start, const offset_t &end) {
    // Remove all yara file elements in given range (from start to end)
    
    offset_range_t range = {start, end};
    offset_range_t max_range = {start, end};

    // First must be removed rules, imports, includes... and then errors, because their deletion can be requested

    wrong_elements_.revRemoveUntil([&range, &max_range](const YaraFileElement &e) {
        return hitsRange(e, range, max_range);
    }, range.end);

    rules_tab_.revRemoveUntil([&range, &max_range](const Rule &r) {
        return hitsRange(r, range, max_range);
    }, range.end);

    includes_tab_.revRemoveUntil([&range, &max_range](const FileInclude &i) {
        return hitsRange(i, range, max_range);
    }, range.end);

    imports_.revRemoveUntil([&range, &max_range](const Import &i) {
        return hitsRange(i, range, max_range);
    }, range.end);

    comments_.revRemoveUntil([&range, &max_range](const Comment &c) {
        return hitsRange(c, range, max_range);
    }, range.end);


    // Now delete the errors

    syntax_.revRemoveUntil([&range, &max_range](const SyntaxError &e) {
        return e.isDeletionRequested() || 
            (!e.isFixed() && hitsRange(e, range, max_range));
    }, range.end);

    semantic_.revRemoveUntil([&range, &max_range](const SemanticError &e) {
        return e.isDeletionRequested() || 
            (!e.isFixed() && hitsRange(e, range, max_range));
    }, range.end);

    DEBUG_LOG("Deletion OF elements is done!\n");

    return max_range;
}


template <typename T>
void shift(const T &element, const int32_t &delta) {
    element->setOffset(element->getOffset() + delta);
}


template <typename T>
void shiftCachedRange(const T &element, const uint32_t &start_row, const int32_t &row_delta, const int32_t &col) {
    range_t range = element->getRange();
    if(range.start.row == start_row && row_delta == 0) {
        element->setRangeCache({
            {range.start.row + row_delta, range.start.col + col}, 
            {range.end.row + row_delta, range.end.col + col}
        });
    }
    else {
        element->setRangeCache({
            {range.start.row + row_delta, range.start.col}, 
            {range.end.row + row_delta, range.end.col}
        });
    }
}


void YaraFile::shiftElements(const offset_t &start, const uint32_t &start_row, const int32_t &delta, const int32_t &row_delta, const int32_t &col) {
    // Updated position of all YaraFileElements in this YaraFile

    rules_tab_.update([delta, start_row, row_delta, col](const RulePtr &r) {
        shift(r, delta);
        shiftCachedRange(r, start_row, row_delta, col);
    }, start, static_cast<offset_t>(-1));

    includes_tab_.update([delta, start_row, row_delta, col](const std::unique_ptr<FileInclude> &i) {
        shift(i, delta);
        shiftCachedRange(i, start_row, row_delta, col);
    }, start, static_cast<offset_t>(-1));

    imports_.update([delta, start_row, row_delta, col](Import *i) {
        shift(i, delta);
        shiftCachedRange(i, start_row, row_delta, col);
    }, start, static_cast<offset_t>(-1));

    comments_.update([delta, start_row, row_delta, col](const std::unique_ptr<Comment> &c) {
        shift(c, delta);
        shiftCachedRange(c, start_row, row_delta, col);
    }, start, static_cast<offset_t>(-1));

    wrong_elements_.update([delta, start_row, row_delta, col](const std::unique_ptr<TopLevelYaraFileElement> &e) {
        shift(e, delta);
        shiftCachedRange(e, start_row, row_delta, col);
    }, start, static_cast<offset_t>(-1));

    syntax_.update([delta, start_row, row_delta, col](const std::unique_ptr<SyntaxError> &e) {
        if(!e->isFixed()) {
            shift(e, delta);
            shiftCachedRange(e, start_row, row_delta, col);
        }
        else {
            e->invalidateCache();
        }
    }, start, static_cast<offset_t>(-1));

    semantic_.update([delta, start_row, row_delta, col](const std::unique_ptr<SemanticError> &e) {
        if(!e->isFixed()) {
            shift(e, delta);
            shiftCachedRange(e, start_row, row_delta, col);
        }
        else {
            e->invalidateCache();
        }
    }, start, static_cast<offset_t>(-1));


    DEBUG_LOG("Element updates are done!\n");
}


void updateModifiedRanges(
    const edit_t &e, 
    std::vector<offset_edit_range_t> &ranges) {

    size_t new_char_n = e.ins_n - e.del_n;
    offset_t old_end = e.offset + e.del_n;
    offset_t new_end = e.offset + e.ins_n, start = e.offset;

    // Old ranges can be influenced by the new edit, so they must be updated
    for(auto range_it = ranges.begin(); range_it != ranges.end(); ++range_it) {
        offset_edit_range_t *prev_range = &(*range_it);

        if(prev_range->type == offset_edit_range_t::EditType::DEL) { ///< If previous edit was deletion, it is processed differently, because there are no characters in deleted range
            if(prev_range->end < start) { ///< New edit occurs completely after older edit (older edit is not affect by the new edit)
                // Nothing
            }
            else if(start <= prev_range->start) {
                if(prev_range->start < old_end && prev_range->end < old_end) {
                    prev_range->start = 0;
                    prev_range->end = 0;
                }
                else if(old_end <= prev_range->start) {
                    prev_range->start += new_char_n;
                    prev_range->end += new_char_n;
                }
            }
        }
        else {
            if(prev_range->start <= start && start < prev_range->end) { ///< New edit occurs inside older edit range -> extend or shrink edited interval
                if(prev_range->end < old_end) {
                    prev_range->end = start;
                }

                prev_range->end += new_char_n;
            }
            else if(prev_range->end < start) { ///< New edit occurs completely after older edit (older edit is not affect by the new edit)
                // Nothing
            }
            else if(start <= prev_range->start) { ///< New edit start occurs before older edit
                if(prev_range->start < old_end && old_end < prev_range->end) {
                    prev_range->start = new_end;
                    prev_range->end += new_char_n;
                }
                else if(prev_range->start < old_end && prev_range->end < old_end) {
                    prev_range->start = 0;
                    prev_range->end = 0;
                }
                else { ///< New edit completely occurs before older edit
                    prev_range->start += new_char_n;
                    prev_range->end += new_char_n;
                }
            }
        }
    }
}



void YaraFile::mergeOverlappingRanges_(std::vector<offset_edit_range_t> &ranges) {
    if(ranges.empty()) {
        return;
    }
    
    sort(ranges.begin(), ranges.end()); ///< Sort ranges

    #ifdef DEBUG_LOG
        DEBUG_LOG("Merging edited ranges...\n");
        DEBUG_LOG("Before merge: \n");
        for(auto e_it = ranges.begin(); e_it != ranges.end();) {
            DEBUG_LOG("[%d--%d]\n", e_it->start, e_it->end);
            ++e_it;
        }
    #endif

    for(auto e_it = ranges.begin(); e_it < ranges.end() - 1;) {

        for(auto next_e_it = e_it + 1; next_e_it != ranges.end();) {
            if(e_it->end >= next_e_it->start) { ///< If range overlaps, remove it
                
                if(next_e_it->end > e_it->end) {
                    e_it->end = next_e_it->end;
                }
                
                next_e_it = ranges.erase(next_e_it);
            }
            else {
                break;
            }
        }

        ++e_it;
    }

    #ifdef DEBUG_LOG
        DEBUG_LOG("After merge: \n");
        for(auto e_it = ranges.begin(); e_it != ranges.end();) {
            DEBUG_LOG("[%d--%d]\n", e_it->start, e_it->end);
            ++e_it;
        }
    #endif
}


void computePositionShift(const edit_t &e, int32_t &row_diff, int32_t &col) {
    DEBUG_LOG("New end point row: %d, old end point row: %d\n", 
        e.ts_edit.new_end_point.row, 
        e.ts_edit.old_end_point.row
    );

    row_diff = e.ts_edit.new_end_point.row - e.ts_edit.old_end_point.row;

    if(row_diff == 0) {
        col = e.ts_edit.new_end_point.column - e.ts_edit.old_end_point.column;
    }
    else if(row_diff < 0) {
        col = e.ts_edit.old_end_point.column;
    }
    else {
        col = e.ts_edit.new_end_point.column;
    }
}


std::vector<offset_edit_range_t> YaraFile::modify(TSTree *new_tree) {
    std::vector<offset_edit_range_t> edited_ranges;

    // Identify modification sites by tracking edits

    for(auto &e: edits_) { ///< Traverse all performed edits
        offset_edit_range_t ins_range(
            e.offset, 
            e.offset + e.ins_n, 
            offset_edit_range_t::EditType::INS
        );

        // Update (remove or shift) all top level elements of YaraFile (for now it updates only top level elements - rules, imports...)
        offset_t old_end = e.offset + e.del_n, start = e.offset;
        size_t new_char_n = e.ins_n - e.del_n;

        // Delete regions in symbol tabs, that were explicitly deleted
        offset_range_t del_offset_range = removeElementsInRange(start, old_end);
        offset_edit_range_t del_range(
            del_offset_range, 
            offset_edit_range_t::EditType::DEL
        );

        int32_t row_diff, col;
        size_t old_end_pt_row = e.ts_edit.old_end_point.row;
        computePositionShift(e, row_diff, col);    
        DEBUG_LOG("Shifting by offset:%ld row:%d col:%d\n", new_char_n, row_diff, col);

        // Shift affected elements
        shiftElements(start, old_end_pt_row, new_char_n, row_diff, col);
        
        // Get ranges to be checked (they were added in notifyChange function) -- they must be modified too
        edited_ranges.insert(edited_ranges.end(), to_be_checked_.begin(), to_be_checked_.end());
        clearToBeChecked();

        updateModifiedRanges(e, edited_ranges); ///< Update ranges in edited_ranges vector (new edit can affect them)
       
        if(del_range.len()) {
            edited_ranges.push_back(del_range); ///< Push the new edit to the edited_ranges
        }
        if(ins_range.len()) {
            edited_ranges.push_back(ins_range);
        }
    }

    // Get modified ranges identified by tree-sitter

    ctx_.diffTree(new_tree, edited_ranges);

    // Remove global errors and add the ranges to be checked to the result vector

    removeGlobalErrors();
    edited_ranges.insert(
        edited_ranges.end(), 
        to_be_checked_.begin(), 
        to_be_checked_.end()
    );


    mergeOverlappingRanges_(edited_ranges); ///< Merge ranges to avoid repeated construction of high level representation
    return edited_ranges;
}


void YaraFile::setActivity(bool state) {
    active_ = state;
}


bool YaraFile::isActive() {
    return active_;
}


void YaraFile::setErrorMode(YaramodConfig::ErrorMode new_mode) {
    err_mode_ = new_mode;
}


void YaraFile::notifyChange(YaraFileElement *element) {
    if(parent_src_) {
        parent_src_->notifyChange(element);
    }
}


void YaraFile::bind(YaraFileElement *parent, YaraFileElement *dependency) {
    if(parent_src_) {
        parent_src_->bind(parent, dependency);

        //parent_src_->dumpBindParents();
    }
}


void YaraFile::bind(const std::vector<YaraFileElementBindable *> &parents, YaraFileElement *dependency) {
    if(parent_src_) { 
        for(auto &p: parents) { ///< Create binding for all elements in vector
            parent_src_->bind(p, dependency);
        }
    
    }
}


void YaraFile::deleteBindings(YaraFileElement *element) {
    if(parent_src_) {
        parent_src_->deleteBindings(element);
    }
}


void YaraFile::moveBindings(YaraFileElement *original_parent, YaraFileElementBindable *new_parent) {
    if(parent_src_) {
        parent_src_->moveBindings(original_parent, new_parent);
    }
}


void YaraFile::checkRange(const offset_range_t &range_to_be_checked) {
    to_be_checked_.push_back(range_to_be_checked);
}


void YaraFile::clearToBeChecked() {
    to_be_checked_.clear();
}


FileContext &YaraFile::ctx() {
    return ctx_;
} 


bool YaraFile::isIsolated() const {
    return is_isolated_;
}


void YaraFile::setIsolated(bool new_state) {
    is_isolated_ = new_state;
}


void YaraFile::addMissingSymbol(const std::string &missing, const offset_range_t &range) {
    missing_symbols_.emplace(missing, range);
}


void YaraFile::removeMissingSymbol(const std::string &missing) {
    missing_symbols_.erase(missing);
}


void YaraFile::checkMissing(const std::string &new_symbol) {
    auto missing_it_range = missing_symbols_.equal_range(new_symbol);
    for(auto it = missing_it_range.first; it != missing_it_range.second; ++it) {
        to_be_checked_.push_back(it->second);
    }

    removeMissingSymbol(new_symbol);
}


void YaraFile::notifyAddedSymbol(const std::string &id) {
    checkMissing(id);
    for(auto &f: getFileIncludes()) {
        if(f->getFile()) {
            f->getFile()->checkMissing(id);
        }
    }
}



std::stringstream YaraFile::getTextFormatted() const {
    std::stringstream stream;

    for(auto i: getImports()) {
        stream << i->getTextFormatted().rdbuf() << std::endl;
    }

    for(auto &r: getRules()) {
        stream << r->getTextFormatted().rdbuf() << std::endl << std::endl;
    }

    return stream;
}

