/**
 * @file observing_visitor.cpp 
 * @brief Contains default implementation of ObservingVisitor methods
 * 
 * @author Vojtěch Dvořák 
 */


#include "../headers/expression.h"

#include "../headers/observing_visitor.h"

// Helper methods

void ObservingVisitor::visitBinaryOperatorDefault(BinaryExpression *e) {
    if(e->getLop()) {
        e->getLop()->accept(this);
    }
    
    if(e->getRop()) {
        e->getRop()->accept(this);
    }
}


void ObservingVisitor::visitUnaryOperatorDefault(UnaryExpression *e) {
    if(e->getOp()) {
        e->getOp()->accept(this);
    }
}


void ObservingVisitor::visitSetDefault(SetExpression *e) {
    for(auto &e: e->getElements()) {
        if(e) {
            e->accept(this);
        }
    }
}


//---


void ObservingVisitor::start(Expression *e) {
    e->accept(this);
}


void ObservingVisitor::visitRange(Range *r) {
    if(r->getFrom()) {
        r->getFrom()->accept(this);
    }

    if(r->getTo()) {
        r->getTo()->accept(this);
    }
}


void ObservingVisitor::visitEnum(Enum *e) {
    for(auto &e: e->getValues()) {
        if(e) {
            e->accept(this);
        }
    }
}


void ObservingVisitor::visitParentheses(ParenthesesExpression *e) {
    if(e->getInnerExpression()) {
        e->getInnerExpression()->accept(this);
    }
}

void ObservingVisitor::visitIn(InExpression *e) {
    if(e->getElementExpression()) {
        e->getElementExpression()->accept(this);
    }

    if(e->getRangeExpression()) {
        e->getRangeExpression()->accept(this);
    }
}

void ObservingVisitor::visitMatches(MatchesExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitAt(AtExpression *e) {
    visitBinaryOperatorDefault(e);
}


// Arithmetics

void ObservingVisitor::visitAdd(AddExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitSub(SubExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitMul(MulExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitRemainder(RemainderExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitDiv(DivExpression *e) {
    visitBinaryOperatorDefault(e);
}


// Bitwise operators

void ObservingVisitor::visitLeftShift(LeftShiftExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitRightShift(RightShiftExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitBitwiseOr(BitwiseOrExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitBitwiseAnd(BitwiseAndExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitBitwiseXor(BitwiseXorExpression *e) {
    visitBinaryOperatorDefault(e);
}


// Relational operators 

void ObservingVisitor::visitLt(LtExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitLte(LteExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitGt(GtExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitGte(GteExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitEq(EqExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitNeq(NeqExpression *e) {
    visitBinaryOperatorDefault(e);
}


// String relational operators

void ObservingVisitor::visitContains(ContainsExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitIContains(IContainsExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitStartsWith(StartsWithExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitIStartsWith(IStartsWithExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitEndsWith(EndsWithExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitIEndsWith(IEndsWithExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitIEq(IEqExpression *e) {
    visitBinaryOperatorDefault(e);
}


// Bool operators

void ObservingVisitor::visitAnd(AndExpression *e) {
    visitBinaryOperatorDefault(e);
}

void ObservingVisitor::visitOr(OrExpression *e) {
    visitBinaryOperatorDefault(e);
}


// Unary operators

void ObservingVisitor::visitDefined(DefinedExpression *e) {
    visitUnaryOperatorDefault(e);
}

void ObservingVisitor::visitUnaryMinus(UnaryMinusExpression *e) {
    visitUnaryOperatorDefault(e);
}

void ObservingVisitor::visitBitwiseNot(BitwiseNotExpression *e) {
    visitUnaryOperatorDefault(e);
}

void ObservingVisitor::visitNot(NotExpression *e) {
    visitUnaryOperatorDefault(e);
}


// Terminals

void ObservingVisitor::visitLiteral(LiteralExpression *) {
}

void ObservingVisitor::visitSize(SizeExpression *) {
}

void ObservingVisitor::visitRegexp(RegexpExpression *) {
}

void ObservingVisitor::visitString(StringExpression *) {
}

void ObservingVisitor::visitStringWildcard(StringWildcardExpression *) {
}

void ObservingVisitor::visitStringCount(StringCountExpression *) {
}


// String properties

void ObservingVisitor::visitStringOffset(StringOffsetExpression *e) {
    if(e->getIndex()) {
        e->getIndex()->accept(this);
    }
}

void ObservingVisitor::visitStringMatchLength(StringMatchLengthExpression *e) {
    if(e->getIndex()) {
        e->getIndex()->accept(this);
    }
}


// Symbols

void ObservingVisitor::visitPlainSymbol(PlainSymbol *) {
}

void ObservingVisitor::visitArray(ArrayExpression *e) {
    if(e->getKey()) {
        e->getKey()->accept(this);
    }
}

void ObservingVisitor::visitStruct(StructExpression *) {
}

void ObservingVisitor::visitFunctionCall(FunctionCallExpression *e) {
    for(auto &e: e->getArgs()) {
        if(e) {
            e->accept(this);
        }
    }
}

void ObservingVisitor::visitVariable(VariableExpression *) {

}

void ObservingVisitor::visitRuleWildCard(RuleWildcardExpression *) {
}


// Quantifiers

void ObservingVisitor::visitNone(NoneExpression *) {
}

void ObservingVisitor::visitAny(AnyExpression *) {
}

void ObservingVisitor::visitAll(AllExpression *) {
}

void ObservingVisitor::visitPercentQuantifier(PercentQuantifierExpression *e) {
    if(e->getValue()) {
        e->getValue()->accept(this);
    }
}

void ObservingVisitor::visitThem(ThemExpression *) {
}


// Sets

void ObservingVisitor::visitStringSet(StringSetExpression *e) {
    visitSetDefault(e);
}

void ObservingVisitor::visitRuleSet(RuleSetExpression *e) {
    visitSetDefault(e);
}

void ObservingVisitor::visitExpressionSet(ExpressionSetExpression *e) {
    visitSetDefault(e);
}


// Of operator

void ObservingVisitor::visitOf(OfExpression *e) {
    if(e->getQuantifier()) {
        e->getQuantifier()->accept(this);
    }

    if(e->getSet()) {
        e->getSet()->accept(this);
    }
}


// For operators

void ObservingVisitor::visitFor(ForExpression *e) {
    if(e->getQuantifier()) {
        e->getQuantifier()->accept(this);
    }

    if(e->getSet()) {
        e->getSet()->accept(this);
    }

    if(e->getExpression()) {
        e->getExpression()->accept(this);
    }
}

void ObservingVisitor::visitForInt(ForIntExpression *e) {
    if(e->getQuantifier()) {
        e->getQuantifier()->accept(this);
    }

    if(e->getVarList()) {
        e->getVarList()->accept(this);
    }

    if(e->getIterable()) {
        e->getIterable()->accept(this);
    }

    if(e->getExpression()) {
        e->getExpression()->accept(this);
    }
}

void ObservingVisitor::visitVarList(VarListExpression *) {
}

