/**
 * @file string.cpp 
 * @brief Implementation of String and StringModifier methods
 * 
 * @author Vojtěch Dvořák 
 */


#include "../headers/string.h"
#include "../headers/yara_file.h"

// Database of string types with valid types of stirng modifiers
const String::ValidModsTab String::valid_mods_ = {
    {Type::Plain, 
        {
            StringModifier::Type::NoCase,
            StringModifier::Type::Wide,
            StringModifier::Type::Ascii,
            StringModifier::Type::Xor,
            StringModifier::Type::Base64,
            StringModifier::Type::Base64Wide,
            StringModifier::Type::Fullword,
            StringModifier::Type::Private
        }
    }, 
    {String::Type::Hex, 
        {
            StringModifier::Type::Private
        }
    }, 
    {String::Type::Regexp, 
        {
            StringModifier::Type::NoCase,
            StringModifier::Type::Wide,
            StringModifier::Type::Ascii,
            StringModifier::Type::Fullword,
            StringModifier::Type::Private
        }
    }
};


// String to StringModifier::Type mapping
const std::unordered_map<std::string_view, StringModifier::Type> StringModifier::string_to_type_ = {
    {"nocase", StringModifier::Type::NoCase},
    {"wide", StringModifier::Type::Wide},
    {"ascii", StringModifier::Type::Ascii},
    {"xor", StringModifier::Type::Xor},
    {"base64", StringModifier::Type::Base64},
    {"base64wide", StringModifier::Type::Base64Wide},
    {"fullword", StringModifier::Type::Fullword},
    {"private", StringModifier::Type::Private}
};


// StringModifier::Type to string mapping
const std::unordered_map<StringModifier::Type, std::string_view> StringModifier::type_to_string_ = {
    {StringModifier::Type::NoCase, "nocase"},
    {StringModifier::Type::Wide, "wide"},
    {StringModifier::Type::Ascii, "ascii"},
    {StringModifier::Type::Xor, "xor"},
    {StringModifier::Type::Base64, "base64"},
    {StringModifier::Type::Base64Wide, "base64wide"},
    {StringModifier::Type::Fullword, "fullword"},
    {StringModifier::Type::Private, "private"}
};


StringModifier::StringModifier(StringModifier::Type type) : type_(type) {
}


StringModifier::Type StringModifier::getType() const {
    return type_;
}


StringModifier::Type StringModifier::stringToType(std::string_view string) {
    return string_to_type_.at(string);
}


std::string_view StringModifier::typeToString(StringModifier::Type type) {
    return type_to_string_.at(type);
}


void StringModifier::setArg(const Key &key) {
    arg_ = key;
}

void StringModifier::setArg(const Range &range) {
    arg_ = range;
}

void StringModifier::setArg(const Alphabet &alphabet) {
    arg_ = alphabet;
}

void StringModifier::setArg(const std::string &alphabet_str) {
    if(alphabet_str.length() != BASE64_ALPHA_LEN) {
        throw InternalErrorException("Alphabet string does not have correct length!");
    }

    Alphabet alpha;
    std::copy(alphabet_str.begin(), alphabet_str.end(), alpha.begin());

    arg_ = alpha;
}


bool StringModifier::hasAnyArg() const {
    return !std::holds_alternative<std::monostate>(arg_);
}

const StringModifier::Arg &StringModifier::getArg() const {
    return arg_;
}

std::stringstream StringModifier::getTextFormatted() const {
    std::stringstream sstream;

    sstream << typeToString(type_);

    if(hasAnyArg()) {
        sstream << "(";

        if(hasArg<Alphabet>()) {
            sstream << "\"";  
            for(auto &ch: std::get<Alphabet>(getArg())) {
                sstream << ch;
            }
            sstream << "\"";
        }
        else if(hasArg<Key>()) {
            sstream << static_cast<unsigned int>(std::get<Key>(getArg()));
        }
        else if(hasArg<Range>()) {
            sstream << static_cast<unsigned int>(std::get<Range>(getArg()).first);
            sstream << "-";
            sstream << static_cast<unsigned int>(std::get<Range>(getArg()).second);
        }

        sstream << ")";
    }

    return sstream;
}



// StringModifierContainer iterator

StringModifierContainer::iterator::iterator(const StringModifierContainer *container) {
    if(container) {
        container_ = container;
        
        it_ = container_->modifiers_.cbegin();
        if(it_ == container_->modifiers_.end()) {
            container_ = nullptr;
        }
    }
}


StringModifier* StringModifierContainer::iterator::operator*() const {
    if(!container_ || it_ == container_->modifiers_.end()) {
        return nullptr;
    }
    else {
        return it_->get();
    }
}


StringModifier* StringModifierContainer::iterator::operator->() const {
    if(!container_ || it_ == container_->modifiers_.end()) {
        return nullptr;
    }
    else {
        return it_->get();
    }
}


StringModifierContainer::iterator& StringModifierContainer::iterator::operator++() {
    if(container_ && it_ != container_->modifiers_.end()) {
        ++it_;
        
        if(it_ == container_->modifiers_.end()) {
            container_ = nullptr;
        }
    }
    else {
        container_ = nullptr;
    }

    return *this;
}


StringModifierContainer::iterator StringModifierContainer::iterator::operator++(int) {
    StringModifierContainer::iterator result = *this;
    ++(*this);
    return result;
}


// StringModifierContainer

size_t StringModifierContainer::size() const {
    return modifiers_.size();
}


bool StringModifierContainer::empty() const {
    return modifiers_.empty();
}


bool StringModifierContainer::has(StringModifier::Type type) const {
    for(auto modifier_it = modifiers_.begin(); 
        modifier_it != modifiers_.end();
        ++modifier_it) {

        if((*modifier_it)->getType() == type) {
            return true;
        }
    }

    return false;
}


StringModifier *StringModifierContainer::get(StringModifier::Type type) const {
    for(auto modifier_it = modifiers_.begin(); 
        modifier_it != modifiers_.end();
        ++modifier_it) {

        if((*modifier_it)->getType() == type) {
            return modifier_it->get();
        }
    }

    return nullptr;
}


StringModifierContainer::iterator StringModifierContainer::begin() const {
    return iterator(this);
}


StringModifierContainer::iterator StringModifierContainer::end() const {
    return iterator(nullptr);
}


void StringModifierContainer::add(std::unique_ptr<StringModifier> &&new_mod) {
    StringModifier::Type type = new_mod->getType();
    
    erase(type);
    
    modifiers_.push_back(std::move(new_mod));
}


void StringModifierContainer::erase(StringModifier::Type type) {
    for(auto modifier_it = modifiers_.begin(); 
        modifier_it != modifiers_.end();
        ++modifier_it) {

        if((*modifier_it)->getType() == type) {
            modifiers_.erase(modifier_it);
            break;
        }
    }
}



// String

String::String(String::Type type) : type_(type) {
}


void String::setId(const std::string &id) {
    id_ = id;
}


const std::string &String::getId(bool ignore_internal_id) const {
    if(isAnonymous() && ignore_internal_id) {
        static const std::string anonymous_id = std::string({});
        return anonymous_id;
    }
    else {
        return id_;
    }
}


String::Type String::getType() const {
    return type_;
}


void String::setContent(const std::string &content) {
    content_ = content;
}


const std::string &String::getContent() const {
    return content_;
}


void String::addModifier(std::unique_ptr<StringModifier> &&new_mod) {
    if(mods_.has(new_mod->getType())) { ///< Check if there is already this type of modifier
        if(auto parent = getParentFile().lock()) {
            parent->addSemanticError<StringModifierError>(
                "Duplicated string modifier!",
                new_mod->getOffset(),
                new_mod->getLen()
            );
        }

        return;
    }

    if(!isModValid_(new_mod)) {
        if(auto parent = getParentFile().lock()) { ///< Check if modifier type is valid for current type of the string
            parent->addSemanticError<StringModifierError>(
                "Modifier is invalid for this type of string!",
                new_mod->getOffset(),
                new_mod->getLen()
            );
        }

        return;
    }

    mods_.add(std::move(new_mod));
}


const StringModifierContainer &String::getModifiers() const {
    return mods_;
}


bool String::isModValid_(const std::unique_ptr<StringModifier> &new_mod) {
    auto current_row = valid_mods_.at(type_);
    return current_row.find(new_mod->getType()) != current_row.end();
}


std::string_view String::typeToString(String::Type type) {
    switch (type)
    {
    case String::Type::Regexp:
        return "regexp";
    case String::Type::Hex:
        return "hex";
    case String::Type::Plain:
        return "plain";
    default:
        return "unknown";
    }
}


bool String::isAnonymous() const {
    return is_anonymous_;
}


void String::setAnonymous(bool is_anonymous) {
    is_anonymous_ = is_anonymous;
}



std::stringstream String::getTextFormatted() const {
    std::stringstream sstream;

    // Print string id and =
    if(!isAnonymous()) {
        sstream << "$" << getId() << " = ";
    }
    else {
        sstream << "$ = ";
    }

    // Print string value
    if(getType() == String::Type::Plain) {
        sstream << "\"" << getContent() << "\"";
    }
    else if(getType() == String::Type::Hex) {
        sstream << "{ " << getContent() << " }";
    }
    else if(getType() == String::Type::Regexp) {
        sstream << getContent();
    }

    // Print string modifiers
    for(auto m: getModifiers()) {
        sstream << " ";
        sstream << m->getTextFormatted().rdbuf();
    }

    return sstream;
}
