/**
 * @file rule_element.cpp 
 * @brief Implementation of RuleElement methods
 * 
 * @author Vojtěch Dvořák
 */


#include "../headers/rule.h"
#include "../headers/yara_file.h"

#include "../headers/rule_element.h"


offset_t RuleElement::getOffset() const {
    if(parent_rule_) {
        return offset_ + parent_rule_->getOffset(); ///< Convert local offset to global offset by adding rule offset
    }
    else {
        return offset_;
    }
}


void RuleElement::setOffset(const offset_t &offset) {
    if(parent_rule_) {
        offset_ = offset - parent_rule_->getOffset(); ///< Only local offset is stored
    }
}


offset_t RuleElement::getLocalOffset() {
    return offset_;
}


void RuleElement::setLocalOffset(const offset_t &offset) {
    offset_ = offset;
}


range_t RuleElement::getRange() const {
    auto parent = getParentFile().lock();
    if(!parent || !parent_rule_) {
        return { 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)}, 
            {static_cast<uint32_t>(-1), static_cast<uint32_t>(-1)} 
        };
    }

    range_t result = range_;
    point_t rule_pos = parent_rule_->getPosition();

    result = result + rule_pos;
    if(range_.start.row == 0) {
        result.start.col = range_.start.col + rule_pos.col;
    }

    if(range_.end.row == 0) {
        result.end.col = range_.end.col + rule_pos.col;
    }

    return result;
}


void RuleElement::setRange(const range_t &range) {
    if(!parent_rule_) {
        return;
    }

    range_ = range;
    point_t rule_pos = parent_rule_->getPosition();
    
    range_.start = range_.start - rule_pos;
    range_.end = range_.end - rule_pos;

    if(range_.start.row == 0) {
        range_.start.col = range_.start.col - rule_pos.col;
    }

    if(range_.end.row == 0) {
        range_.end.col = range_.end.col - rule_pos.col;
    }
}


Rule *RuleElement::getParentRule() const {
    return parent_rule_;
}


void RuleElement::setParentRule(const RulePtr &parent_rule) {
    parent_rule_ = parent_rule.get();
}


void RuleElement::setParentRule(Rule *parent_rule) {
    parent_rule_ = parent_rule;
}
