/**
 * @file rule.h
 * @brief Contains class, that represents yara rule
 * 
 * @author Vojtěch Dvořák    
 */


#include "../headers/common.h"
#include "../headers/language.h"
#include "../headers/yara_file.h"
#include "../headers/string.h"
#include "../headers/yara_file_element.h"
#include "../headers/meta.h"
#include "../headers/variable.h"

#include "../headers/rule.h"


RuleModifier::RuleModifier(RuleModifier::Type type) : type_(type) {
}


// Provides string to RuleModifier::Type mapping
RuleModifier::Type RuleModifier::stringToType(std::string_view str) {
    if(str == "private") {
        return RuleModifier::Type::Private;
    }
    else if(str == "global") {
        return RuleModifier::Type::Global;
    }
    else {
        throw InternalErrorException("Unable to convert string to rule modifier type!");
    }
}


// RuleModifier::Type to string mapping
std::string_view RuleModifier::typeToString(RuleModifier::Type type) {
    if(type == RuleModifier::Type::Private) {
        return "private";
    }
    else {
        return "global";
    }
}


RuleModifier::Type RuleModifier::getType() {
    return type_;
}


// RuleModifierContainer iterator

RuleModifierContainer::iterator::iterator(const RuleModifierContainer *container) : container_(container) {
    if(container) {
        if(container->hasPrivate()) {
            type_ = RuleModifier::Type::Private;
        }
        else if(container->hasGlobal()) {
            type_ = RuleModifier::Type::Global;
        }
        else {
            container_ = nullptr;
        }
    }
}


RuleModifier* RuleModifierContainer::iterator::operator*() const {
    if(container_ && type_ == RuleModifier::Type::Global) {
        return container_->getGlobal(); 
    }
    else if(container_ && type_ == RuleModifier::Type::Private) {
        return container_->getPrivate(); 
    }
    else {
        return nullptr; 
    }
}


RuleModifier* RuleModifierContainer::iterator::operator->() const {
    if(container_ && type_ == RuleModifier::Type::Global) {
        return container_->getGlobal(); 
    }
    else if(container_ && type_ == RuleModifier::Type::Private) {
        return container_->getPrivate(); 
    }
    else {
        return nullptr; 
    }
}


RuleModifierContainer::iterator& RuleModifierContainer::iterator::operator++() {
    if(container_ && type_ == RuleModifier::Type::Private) {
        if(container_->hasGlobal()) {
            type_ = RuleModifier::Type::Global;
        }
        else {
            container_ = nullptr;
        }
    }
    else if(container_ && type_ == RuleModifier::Type::Global) {
        container_ = nullptr; 
    }

    return *this;
}


RuleModifierContainer::iterator RuleModifierContainer::iterator::operator++(int) {
    RuleModifierContainer::iterator result = *this;
    ++(*this);
    return result;
}


// RuleModifierContainer

size_t RuleModifierContainer::size() const {
    if(hasGlobal() || hasPrivate()) {
        if(hasGlobal() && hasPrivate()) {
            return 2;
        }
        else {
            return 1;
        }
    }
    else {
        return 0;
    }
}


bool RuleModifierContainer::empty() const {
    return hasGlobal() || hasPrivate();
}


bool RuleModifierContainer::has(RuleModifier::Type type) const {
    return (type == RuleModifier::Type::Global && hasGlobal()) ||
        (type == RuleModifier::Type::Private && hasPrivate());
}   


bool RuleModifierContainer::hasGlobal() const {
    return global_ != nullptr;
}


bool RuleModifierContainer::hasPrivate() const {
    return private_ != nullptr;
}


RuleModifier *RuleModifierContainer::getGlobal() const {
    if(global_) {
        return global_.get();
    }
    else {
        return nullptr;
    }
}


RuleModifier *RuleModifierContainer::getPrivate() const {
    if(private_) {
        return private_.get();
    }
    else {
        return nullptr;
    }
}


void RuleModifierContainer::add(std::unique_ptr<RuleModifier> &&new_mod) {
    if(new_mod->getType() == RuleModifier::Type::Global) {
        global_ = std::move(new_mod);
    }
    else if(new_mod->getType() == RuleModifier::Type::Private) {
        private_ = std::move(new_mod);
    }
}


void RuleModifierContainer::erase(RuleModifier::Type type) {
    if(type == RuleModifier::Type::Global) {
        global_ = nullptr;
    }
    else if(type == RuleModifier::Type::Private) {
        private_ = nullptr;
    }
}


RuleModifierContainer::iterator RuleModifierContainer::begin() const {
    return RuleModifierContainer::iterator(this);
}


RuleModifierContainer::iterator RuleModifierContainer::end() const {
    return RuleModifierContainer::iterator(nullptr);
}


// Rule

Rule::~Rule() {
    DEBUG_LOG("Deleting rule %s...\n", getId().c_str());
}


const std::string &Rule::getId() const {
    return id_;
}


void Rule::setId(const std::string &id) {
    id_ = id;
}

// Tag management

const std::vector<std::string> &Rule::getTags() const {
    return tags_;
}


bool Rule::hasTag(const std::string &tag) {
    for(auto it = tags_.begin(); it != tags_.end(); ++it) {
        if(*it == tag) {
            return true;
        }
    }

    return false;
}


void Rule::addTag(const std::string &new_tag) {
    if(hasTag(new_tag)) { ///< Tag is already in tag list
        if(auto parent = getParentFile().lock()) {
            parent->addSemanticError<RuleError>(
                "Rule have duplicated tags!",
                getOffset(),
                getLen(),
                getRange()
            );
        }

        return;
    }

    tags_.push_back(new_tag);
}


void Rule::removeTag(const std::string &tag) {
    for(auto it = tags_.begin(); it != tags_.end(); ++it) {
        if(*it == tag) {
            tags_.erase(it);
            break;
        }
    }
}


// Modifier management

const RuleModifierContainer &Rule::getModifiers() const {
    return mods_;
}


void Rule::addModifier(std::unique_ptr<RuleModifier> &&new_mod) {
    if(mods_.has(new_mod->getType())) { ///< Modifier is already in the modifier list
        if(auto parent = getParentFile().lock()) {
            parent->addSemanticError<RuleModifierError>(
                "Rule have duplicated modifiers!",
                new_mod->getOffset(),
                new_mod->getLen()
            );
        }

        return;
    }

    mods_.add(std::move(new_mod));
}


void Rule::removeModifier(RuleModifier::Type type) {
    mods_.erase(type);
}


bool Rule::isGlobal() {
    return mods_.hasGlobal();
}


bool Rule::isPrivate() {
    return mods_.hasPrivate();
}


// Meta management

const SymTabDuplId<Meta> &Rule::getMetas() const {
    return metas_;
}


const SymTabDuplId<Meta>::Pool &Rule::getMetasById(const std::string &id) const {
    return metas_ .getAll(id);
}


void Rule::addMeta(std::unique_ptr<Meta> &&new_meta) {
    if(new_meta->getId() == std::string({})) {
        return;
    }

    metas_.insert(std::move(new_meta));
}


void Rule::removeMeta(const std::string &id) {
    metas_.remove(id);
}

// String management

const SymTab<String> &Rule::getStrings() const {
    return strings_;
}


const std::unique_ptr<String> &Rule::getStringById(const std::string &id) const {
    return strings_.search(id);
}


void Rule::addString(std::unique_ptr<String> &&new_string) {
    if(new_string->isAnonymous()) { ///< If string is anonymous, generate unique ID for storing it in symbol table
        do {
            std::string internal_id = std::string("$__").append(std::to_string(anonym_str_cnt++));
            new_string->setId(internal_id);
        }
        while(strings_.exists(new_string->getId()));

    }
    else {
        if(strings_.exists(new_string->getId())) {

            if(auto parent = getParentFile().lock()) { 
                parent->addSemanticError<StringError>( ///< String with  this id is already declared
                    "The string with id '" + new_string->getId() + "' is already declared!",
                    new_string->getOffset(),
                    new_string->getId().length() + 1 ///< Increment length by 1 because dollar sign is shifted from Id
                );
            }

            return;
        }
    }

    strings_.insert(std::move(new_string));
}


void Rule::removeString(const std::string &id) {
    strings_.remove(id);
}

// Variable management

const SymTab<IntVariable> &Rule::getVars() const {
    return vars_;
}


const std::unique_ptr<IntVariable> &Rule::getVarById(const std::string &id) const {
    return vars_.search(id);
}


void Rule::addVar(std::unique_ptr<IntVariable> &&new_var) {
    if(new_var->getId() == std::string({})) {
        return;
    }

    if(vars_.exists(new_var->getId())) { ///< Check if there is already variable with this id in symbol tab
        if(auto parent = getParentFile().lock()) {
            parent->addSemanticError<VariableError>(
                "The variable with id '" + new_var->getId() + "' is already declared!",
                new_var->getOffset(),
                new_var->getId().length(),
                new_var->getRange()
            );
        }

        return;
    }

    
    vars_.insert(std::move(new_var));
}


void Rule::removeVar(const std::string &id) {
    vars_.remove(id);
}


void Rule::setCondition(ExpressionPtr &&condition_expr) {
    condition_ = std::move(condition_expr);
}


void Rule::setCondition(const ExpressionPtr &condition_expr) {
    condition_ = std::move(condition_expr);
}


const ExpressionPtr &Rule::getCondition() const {
    return condition_;
}


offset_t Rule::getIdOffset() const {
    return offset_ + id_start_;
}


void Rule::setIdOffset(const offset_t &offset) {
    id_start_ = offset - offset_;
}



std::stringstream Rule::getTextFormatted() const {
    std::stringstream sstream;

    // Print rule modifiers
    for(auto m: getModifiers()) {
        sstream << RuleModifier::typeToString(m->getType()) << " ";
    }

    sstream << "rule " << id_ << " ";

    // Print rule tags
    if(!getTags().empty()) {
        sstream << ": ";
        for(auto &t: getTags()) {
            sstream << t << " ";
        }
    }

    sstream << "{" << std::endl;

    // Print metas
    if(!getMetas().empty()) {
        sstream << "\tmeta:" << std::endl;
        for(auto m: getMetas()) {
            sstream << "\t\t" << m->getTextFormatted().rdbuf() << std::endl;
        }
    }

    // Print internal variables
    if(!getVars().empty()) {
        sstream << "\tvariables:" << std::endl;
        for(auto v: getVars()) {
            sstream << "\t\t" << v->getTextFormatted().rdbuf() << std::endl;
        }
    }

    // Print strings
    if(!getStrings().empty()) {
        sstream << "\tstrings:" << std::endl;
        for(auto s: getStrings()) {
            sstream << "\t\t" << s->getTextFormatted().rdbuf() << std::endl;
        }
    }

    sstream << "\tcondition:" << std::endl << "\t\t";

    if(getCondition()) {
        sstream << getCondition()->getTextFormatted().rdbuf();
    }
    else {
        sstream << "true";
    }
    
    sstream << std::endl << "}";

    return sstream;
}
