/**
 * @file expression.cpp 
 * @brief Implementation of Expression class methods 
 * 
 * @author Vojtěch Dvořák 
 */

#include "../headers/module.h"

#include "../headers/expression.h"


// Expression

Expression::Expression() {
}


Expression::~Expression() {
}


std::string_view Expression::typeToString(Expression::Type type) {
    switch (type)
    {
        case Expression::Type::Object:
            return "object";
        case Expression::Type::Bool:
            return "bool";
        case Expression::Type::Int:
            return "int";
        case Expression::Type::Float:
            return "float";
        case Expression::Type::String:
            return "str";
        case Expression::Type::Regexp:
            return "regexp";
        default:
            return "undefined";
    }
}

bool Expression::isArithmeticCompatible(const ExpressionPtr &l, const ExpressionPtr &r) {
    return ((l->isInt() || l->isFloat()) && (r->isInt() || r->isFloat()));
}

bool Expression::isUndefined() const {
    return getType() == Expression::Type::Undefined;
}

bool Expression::isObject() const {
    return getType() == Expression::Type::Object;
}

bool Expression::isBool() const {
    return getType() == Expression::Type::Bool;
}

bool Expression::isInt() const {
    return getType() == Expression::Type::Int;
}

bool Expression::isFloat() const {
    return getType() == Expression::Type::Float;
}

bool Expression::isString() const {
    return getType() == Expression::Type::String;
}

bool Expression::isRegexp() const {
    return getType() == Expression::Type::Regexp;
}


// Range 

bool Range::areOperandsValid() const {
    return true;
}


bool Range::isComplete() const {
    return from_ != nullptr && to_ != nullptr;
}


bool Range::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!from_->isValid(msg) || !to_->isValid(msg)) {
            return false;
        }
        else if(from_->isInt() && to_->isInt()) {
            return true;
        }
        else {
            return false;
        }
    }
}


std::stringstream Range::getTextFormatted() const {
    std::stringstream sstream;

    sstream << "(";
    if(from_) {
        sstream << from_->getTextFormatted().rdbuf();
    }
    sstream << "..";
    if(to_) {
        sstream << to_->getTextFormatted().rdbuf();
    }
    sstream << ")";

    return sstream;
}


Expression::Type Range::getType() const {
    return Expression::Type::Int;
}


const ExpressionPtr &Range::getFrom() const {
    return from_;
}


const ExpressionPtr &Range::getTo() const {
    return to_;
}


// Enum

bool Enum::areOperandsValid() const {
    std::string_view _;

    for(auto &v: values_) {
        if(v && v->isValid(_)) {
            return false;
        }
    }

    return true;
}


bool Enum::isComplete() const  {
    return true;
}


bool Enum::isValid(std::string_view &msg) const {
    auto data_type = Expression::Type::Undefined;
    for(auto &v: values_) {
        if(data_type == Expression::Type::Undefined) {
            data_type = v->getType();
        }
        else {
            if(data_type != v->getType()) {
                msg = "Invalid types of enum items! (all types must be the same)";
                return false;
            }
        }
    }

    return true;
}


Expression::Type Enum::getType() const {
    if(values_.empty()) {
        return Expression::Type::Undefined;
    }
    else {
        if(values_.at(0)) {
            return values_.at(0)->getType();
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


std::stringstream Enum::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "(";

    for(auto val_it = values_.begin(); 
        val_it != values_.end(); 
        ++val_it) {

        if(val_it != values_.begin()) {
            sstream << ", ";
        }
        if(*val_it) {
            sstream << (*val_it)->getTextFormatted().str();
        }

    }

    sstream << ")";

    return sstream;
}


const std::vector<ExpressionPtr> &Enum::getValues() const {
    return values_;
}


// ParenthesesExpression

bool ParenthesesExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        return inner_expr_->isValid(msg);
    }
}


bool ParenthesesExpression::areOperandsValid() const {
    std::string_view _;
    if(!isComplete()) {
        return false;
    }
    else {
        return inner_expr_->isValid(_);
    }
}


bool ParenthesesExpression::isComplete() const {
    return inner_expr_ != nullptr;
}


Expression::Type ParenthesesExpression::getType() const {
    if(inner_expr_) {
        return inner_expr_->getType();
    }
    else {
        return Expression::Type::Undefined;
    }
}


std::stringstream ParenthesesExpression::getTextFormatted() const {
    std::stringstream sstream;

    sstream << "(";
    if(inner_expr_) {
        sstream << inner_expr_->getTextFormatted().rdbuf();
    }

    sstream << ")";

    return sstream;
}


const ExpressionPtr &ParenthesesExpression::getInnerExpression() const {
    return inner_expr_;
}


// InExpression

bool InExpression::isComplete() const {
    if(!element_ || !range_ || !range_->isComplete()) {
        return false;
    }
    else {
        return true;
    }
}


bool InExpression::areOperandsValid() const {
    std::string_view _;
    if(!element_ || !range_) {
        return false;
    }
    else {
        if(!element_->isValid(_) || !range_->isValid(_)) {
            return false;
        }
        else {
            return true;
        }
    }
}


bool InExpression::isValid(std::string_view &msg) const {
    if(!range_ || !element_) {
        return false;
    }
    else {
        if(!element_->isValid(msg) || !range_->isValid(msg)) {
            return false;
        }
        else if(element_->isInt() || element_->isBool()) {
            return true;
        }
        else {
            return false;
        }
    }
}


Expression::Type InExpression::getType() const {
    if(!range_ || !element_) {
        return Expression::Type::Undefined;
    }
    else {
        if(element_->isInt() || element_->isBool()) {
            return element_->getType();
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


std::stringstream InExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(element_) {
        sstream << element_->getTextFormatted().rdbuf();
    }
    sstream << " " << opsign() << " ";
    if(element_) {
        sstream << range_->getTextFormatted().rdbuf();
    }

    return sstream;
}


const ExpressionPtr &InExpression::getElementExpression() const {
    return element_;
}


const std::shared_ptr<Range> &InExpression::getRangeExpression() const {
    return range_;
}


// BinaryExpression

bool BinaryExpression::isComplete() const {
    return lop_ != nullptr && rop_ != nullptr;
}


bool BinaryExpression::areOperandsValid() const {
    std::string_view _;
    if(!isComplete()) {
        return false;
    }
    else {
        return lop_->isValid(_) && rop_->isValid(_);
    }
}


std::stringstream BinaryExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(lop_) {
        sstream << lop_->getTextFormatted().rdbuf() << " ";
    }
    sstream << opsign();
    if(rop_) {
        sstream << " " << rop_->getTextFormatted().rdbuf();
    }

    return sstream;
}


const ExpressionPtr &BinaryExpression::getLop() const {
    return lop_;
}


const ExpressionPtr &BinaryExpression::getRop() const {
    return rop_;
}


// MatchesExpression

bool MatchesExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(lop_->isString() && rop_->isRegexp()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected string and regular expr. as operands for 'matches')";
            return false;
        }
    }
}


Expression::Type MatchesExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isString() && rop_->isString()) {
            return Expression::Type::Bool;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// AtExpression

bool AtExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(lop_->isBool() && rop_->isInt()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected string id and integer as operands for 'at')";
            return false;
        }
    }
}


Expression::Type AtExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isBool() && rop_->isInt()) {
            return Expression::Type::Bool;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}



// BinaryArithmeticExpression

bool BinaryArithmeticExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(isArithmeticCompatible(lop_, rop_)) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected int or float operands for arithmetic operators)";
            return false;
        }
    }
}


Expression::Type BinaryArithmeticExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isFloat() || rop_->isFloat()) {
            return Expression::Type::Float;
        }
        else if(lop_->isInt() && lop_->isInt()) {
            return Expression::Type::Int;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// RemainderExpression

bool RemainderExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(lop_->isInt() && rop_->isInt()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected int as operand for '%')";
            return false;
        }
    }
}


Expression::Type RemainderExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isInt() && rop_->isInt()) {
            return Expression::Type::Int;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// BinaryBitwiseExpression

bool BinaryBitwiseExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(lop_->isInt() && rop_->isInt()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected int operands for bitwise operator)";
            return false;
        }
    }
}


Expression::Type BinaryBitwiseExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isInt() && lop_->isInt()) {
            return Expression::Type::Int;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// BinaryRelationalExpression

bool BinaryRelationalExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(isArithmeticCompatible(lop_, rop_) ||
            lop_->getType() == rop_->getType()) {
            return true;
        }
        else {
            msg = "Type mismatch! (types must be compatible to be compared)";
            return false;
        }
    }
}


Expression::Type BinaryRelationalExpression::getType() const {
    std::string_view _;
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(isArithmeticCompatible(lop_, rop_) ||
            lop_->getType() == rop_->getType()) {
            return Expression::Type::Bool;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// BinaryStrRelationalExpression

bool BinaryStrRelationalExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else if(lop_->isString() && rop_->isString()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected string operands)";
            return false;
        }
    }
}


Expression::Type BinaryStrRelationalExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        if(lop_->isString() && rop_->isString()) {
            return Expression::Type::Bool;
        }
        else {
            return Expression::Type::Undefined;
        }
    }
}


// BinaryBoolExpression

bool BinaryBoolExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!lop_->isValid(msg) || !rop_->isValid(msg)) {
            return false;
        }
        else {
            return true;
        }
    }
}


Expression::Type BinaryBoolExpression::getType() const {
    std::string_view _;
    if(isValid(_)) {
        return Expression::Type::Bool;
    } 
    else {
        return Expression::Type::Undefined;
    }
}


// UnaryExpression

bool UnaryExpression::isComplete() const {
    return op_ != nullptr;
}


bool UnaryExpression::areOperandsValid() const {
    std::string_view _;
    if(!isComplete()) {
        return false;
    }
    else {
        return op_->isValid(_);
    }
}


std::stringstream UnaryExpression::getTextFormatted() const {
    std::stringstream sstream;

    sstream << opsign() << " ";
    if(op_) {
        sstream << op_->getTextFormatted().rdbuf();
    }

    return sstream;
}


const ExpressionPtr &UnaryExpression::getOp() const {
    return op_;
}


// DefinedExpression

bool DefinedExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!op_->isValid(msg)) {
            return false;
        }
        else {
            return true;
        }
    }
}


Expression::Type DefinedExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        return Expression::Type::Bool;
    }
}


// UnaryArithmeticExpression

bool UnaryArithmeticExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!op_->isValid(msg)) {
            return false;
        }
        else if(op_->isInt() || op_->isFloat()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected float or int as operand for arithmetic operator)";
            return false;
        }
    }
}


Expression::Type UnaryArithmeticExpression::getType() const {
    if(op_) {
        if(op_->isFloat() || op_->isInt()) {
            return op_->getType(); 
        }
        else {
            return Expression::Type::Undefined;
        }
    }
    else {
        return Expression::Type::Undefined;
    }
}


// UnaryBitwiseExpression

bool UnaryBitwiseExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!op_->isValid(msg)) {
            return false;
        }
        else if(op_->isInt()) {
            return true;
        }
        else {
            msg = "Type mismatch! (expected int as operand for bitwise operator)";
            return false;
        }
    }
}


Expression::Type UnaryBitwiseExpression::getType() const {
    if(isComplete()) {
        if(op_->isInt()) {
            return Expression::Type::Int; 
        }
        else {
            return Expression::Type::Undefined;
        }
    }
    else {
        return Expression::Type::Undefined;
    }
}


// UnaryBoolExpression

bool UnaryBoolExpression::isValid(std::string_view &msg) const {
    if(isComplete()) {
        return op_->isValid(msg);
    }
    else {
        return false;
    }
}


Expression::Type UnaryBoolExpression::getType() const {
    std::string_view _;
    if(isValid(_)) {
        return Expression::Type::Bool;
    } 
    else {
        return Expression::Type::Undefined;
    }
}



// TerminalExpression

bool TerminalExpression::isValid(std::string_view &) const {
    return true;
}


bool TerminalExpression::areOperandsValid() const {
    return true;
}


// LiteralExpression


bool LiteralExpression::isComplete() const {
    return true;
}


std::stringstream LiteralExpression::getTextFormatted() const {
    std::stringstream sstream;
    if(l_) {
        sstream << l_->getTextFormatted().rdbuf();
    }
    return sstream;
}


Expression::Type LiteralExpression::getType() const {
    if(l_->isType<std::string>()) {
        return Expression::Type::String;
    }
    else if(l_->isType<int64_t>()) {
        return Expression::Type::Int;
    }
    else if(l_->isType<float>()) {
        return Expression::Type::Float;
    }
    else if(l_->isType<bool>()) {
        return Expression::Type::Bool;
    }
    else {
        return Expression::Type::Undefined;
    }
}


const std::unique_ptr<Literal> &LiteralExpression::getLiteral() const {
    return l_;
}


// SizeExpression

bool SizeExpression::isComplete() const {
    return true;
}


Expression::Type SizeExpression::getType() const {
    return Expression::Type::Int;
}


std::stringstream SizeExpression::getTextFormatted() const {
    std::stringstream sstream;

    sstream << value_ << unitToString(unit_);

    return sstream;
}


const SizeExpression::Unit &SizeExpression::getUnit() const {
    return unit_;
}


uint32_t SizeExpression::getValue() const {
    return value_;
}


std::string_view SizeExpression::unitToString(SizeExpression::Unit unit) {
    switch (unit) {
        case SizeExpression::Unit::MB:
            return "MB";
        case SizeExpression::Unit::KB:
            return "KB";
        default:
            return "";
    } 
}


// RegexpExpression

bool RegexpExpression::isComplete() const {
    return true;
}


Expression::Type RegexpExpression::getType() const{
    return Expression::Type::Regexp;
}


std::stringstream RegexpExpression::getTextFormatted() const{
    std::stringstream sstream;

    sstream << content_;

    return sstream;
}


const std::string &RegexpExpression::getContent() const {
    return content_;
}


// StringExpression

bool StringExpression::isComplete() const {
    return true;
}


Expression::Type StringExpression::getType() const{
    return Expression::Type::Bool;
}


std::stringstream StringExpression::getTextFormatted() const{
    std::stringstream sstream;
    sstream << "$" << string_id_;

    return sstream;
}


const std::string &StringExpression::getId() const {
    return string_id_;
}


// StringCountExpression

bool StringCountExpression::isComplete() const {
    return true;
}


Expression::Type StringCountExpression::getType() const{
    return Expression::Type::Int;
}


std::stringstream StringCountExpression::getTextFormatted() const{
    std::stringstream sstream;

    sstream << "#" << string_id_;

    return sstream;
}


const std::string &StringCountExpression::getId() const {
    return string_id_;
}


// StringPropertyExpression

bool StringPropertyExpression::areOperandsValid() const {
    std::string_view _;
    if(index_) {
        return index_->isValid(_);
    }
    else {
        return true;
    }
}


bool StringPropertyExpression::isValid(std::string_view &msg) const {
    if(index_) {
        return index_->isValid(msg);
    }
    else {
        return true;
    }
}


bool StringPropertyExpression::isComplete() const {
    return true;
}


const std::string &StringPropertyExpression::getId() const {
    return string_id_;
}


const ExpressionPtr &StringPropertyExpression::getIndex() const {
    return index_;
}


// StringOffsetExpression

Expression::Type StringOffsetExpression::getType() const {
    return Expression::Type::Int;
}

std::stringstream StringOffsetExpression::getTextFormatted() const {
    std::stringstream sstream;

    sstream << "@" << string_id_;
    
    if(index_) {
        sstream << "[" << index_->getTextFormatted().rdbuf() << "]";
    }

    return sstream;
}


// StringMatchLengthExpression

Expression::Type StringMatchLengthExpression::getType() const {
    return Expression::Type::Int;
}

std::stringstream StringMatchLengthExpression::getTextFormatted() const {
    std::stringstream sstream;

    sstream << "!" << string_id_;
    
    if(index_) {
        sstream << "[" << index_->getTextFormatted().rdbuf() << "]";
    }

    return sstream;
}


// Symbol

bool Symbol::isComplete() const {
    return true;
}


bool Symbol::areOperandsValid() const {
    return true;
}


bool Symbol::isValid(std::string_view &msg) const {
    if(!isDefined()) {
        msg = "Undefined identifier!";
        return false;
    }
    else {
        return true;
    }
}


void Symbol::setType(Expression::Type type) {
    dtype_ = type;
}


Expression::Type Symbol::getType() const {
    return dtype_;
}


void Symbol::setSymType(Symbol::Type type) {
    stype_ = type;
}


Symbol::Type Symbol::getSymType() {
    return stype_;
}


void Symbol::setContext(const json *attr_array) {
    ctx_ = attr_array;
}


const json *Symbol::getContext() const {
    return ctx_;
}


void Symbol::setDefined(bool defined) {
    defined_ = defined;
}


bool Symbol::isDefined() const {
    return defined_;
}


bool Symbol::isFunction() const {
    return stype_ == Symbol::Type::Function;
}

bool Symbol::isArray() const {
    return stype_ == Symbol::Type::Array;
}

bool Symbol::isDict() const {
    return stype_ == Symbol::Type::Dict;
}

bool Symbol::isStruct() const {
    return stype_ == Symbol::Type::Struct;
}

bool Symbol::isValue() const {
    return stype_ == Symbol::Type::Value;
}


std::stringstream Symbol::getTextFormatted() const {
    std::stringstream sstream;
    return sstream;
}


// ArrayExpression

bool ArrayExpression::isComplete() const {
    return array_ != nullptr && key_ != nullptr;
}


bool ArrayExpression::areOperandsValid() const {
    return true;
}


bool ArrayExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!array_->isValid(msg) || !key_->isValid(msg)) {
            return false;
        }
        else {
            return true;
        }
    }
}


std::stringstream ArrayExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(array_ && key_) {
        sstream << array_->getTextFormatted().rdbuf();
        sstream << "[";
        sstream << key_->getTextFormatted().rdbuf();
        sstream << "]";
    }

    return sstream;
}


const std::shared_ptr<Symbol> &ArrayExpression::getArray() const {
    return array_;
}


const ExpressionPtr &ArrayExpression::getKey() const {
    return key_;
}


// StructExpression

bool StructExpression::isComplete() const {
    return structure_ != nullptr && member_ != nullptr;
}


bool StructExpression::areOperandsValid() const {
    return true;
}


bool StructExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        if(!structure_->isValid(msg) || !member_->isValid(msg)) {
            return false;
        }
        else {
            return true;
        }
    }
}


Expression::Type StructExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        return member_->getType();
    }
}


std::stringstream StructExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(structure_ && member_) {
        sstream << structure_->getTextFormatted().rdbuf();
        sstream << ".";
        sstream << member_->getTextFormatted().rdbuf();
    }

    return sstream;
}


const std::shared_ptr<Symbol> &StructExpression::getStructure() const {
    return structure_;
}


const std::shared_ptr<Symbol> &StructExpression::getMember() const {
    return member_;
}


// FunctionCallExpression

bool FunctionCallExpression::isComplete() const {
    return id_ != nullptr;
}


bool FunctionCallExpression::areOperandsValid() const {
    std::string_view _;
    for(auto &arg: args_) {
        if(!arg) {
            return false;
        }
        else if(!arg->isValid(_)) {
            return false;
        }
    }

    return true;
}


bool FunctionCallExpression::isValid(std::string_view &msg) const {
    if(!isComplete()) {
        return false;
    }
    else {
        for(auto &arg: args_) {
            if(!arg) {
                return false;
            }
            else if(!arg->isValid(msg)) {
                return false;
            }
        }
       
        return true;
    }
}


Expression::Type FunctionCallExpression::getType() const {
    if(!isComplete()) {
        return Expression::Type::Undefined;
    }
    else {
        return id_->getType();
    }
}


std::stringstream FunctionCallExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(id_) {
        sstream << id_->getTextFormatted().rdbuf();
    }

    sstream << "(";
    for(auto arg_it = args_.begin(); arg_it != args_.end(); ++arg_it) {
        if(arg_it != args_.begin()) {
            sstream << ", ";
        }

        if(!(*arg_it)) {
            continue;
        }
        else {
            sstream << (*arg_it)->getTextFormatted().rdbuf();
        }
    }
    
    sstream << ")";

    return sstream;
}


bool FunctionCallExpression::hasValidArgs() const {
    if(!id_ || !id_->getContext()) {
        return false;
    }

    auto ctx = id_->getContext(); ///< The check is done against JSON context of the function
    const auto &overloads = Module::findField(ctx, "overloads");
    
    if(!overloads || !overloads->is_array()) { ///< Field with overloads must have array type
        throw YaramodErrorException("Module error! Overloads field should be array!");
    }

    // Iteration over all overloads
    auto overload_it = overloads->begin();
    for(; overload_it != overloads->end(); ++overload_it) {

        const auto &arguments = Module::findField(&(*overload_it), "arguments");
        if(!arguments || !arguments->is_array()) { ///< Field with arguments must have also array type in JSON
            throw YaramodErrorException("Module error! Arguments field should be array!");
        }

        auto arg_it = arguments->begin(); ///< (Expected) Argument iterator
        auto arg_end_it = arguments->end();
        auto rarg_it = args_.begin(); ///< Real arguments iterator
        auto rarg_end_it = args_.end();

        for(; arg_it != arg_end_it && rarg_it != rarg_end_it; ++arg_it, ++rarg_it) {
            auto type_field = Module::findField(&(*arg_it), "type"); ///< Comparison of real argument and expected argument
            if(!type_field || !(*rarg_it)) {
                continue;
            }

            if(Module::jsonToExprType(type_field) != (*rarg_it)->getType()) {
                break;
            }
        }
 
        if(arg_it != arg_end_it || rarg_it != rarg_end_it) { 
            continue;
        }
        else { ///< Comparison of all real and expected arguments was successful 
            break;
        }
    }

    if(overload_it == overloads->end()) { ///< There is no other overload of the function
        return false;
    }
    else {
        return true;
    }
}


const std::shared_ptr<Symbol> &FunctionCallExpression::getFunction() const {
    return id_;
}


const std::vector<ExpressionPtr> &FunctionCallExpression::getArgs() const {
    return args_;
}


// PlainSymbol

const std::string &PlainSymbol::get() const {
    return id_;
}


std::stringstream PlainSymbol::getTextFormatted() const {
    std::stringstream sstream;
    sstream << id_;
    return sstream;
}


const std::string &PlainSymbol::getId() const {
    return id_;
}


// QuantifierExpression

bool QuantifierExpression::isValid(std::string_view &) const {
    return true;
}


bool QuantifierExpression::areOperandsValid() const {
    return true;
}


bool QuantifierExpression::isComplete() const {
    return true;
}


Expression::Type QuantifierExpression::getType() const {
    return Expression::Type::Undefined;
}


// NoneExpression

std::stringstream NoneExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "none";
    return sstream;
}


// AnyExpression

std::stringstream AnyExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "any";
    return sstream;
}


// AllExpression

std::stringstream AllExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "all";
    return sstream;
}


// PercentQuantifierExpression

bool PercentQuantifierExpression::isValid(std::string_view &msg) const {
    if(value_) {
        return value_->isValid(msg);
    }
    else {
        return false;
    }
}


bool PercentQuantifierExpression::areOperandsValid() const {
    std::string_view _;
    if(value_) {
        return value_->isValid(_);
    }
    else {
        return false;
    }
}


bool PercentQuantifierExpression::isComplete() const {
    return value_ != nullptr;
}


Expression::Type PercentQuantifierExpression::getType() const {
    return Expression::Type::Bool;
}


std::stringstream PercentQuantifierExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(value_) {
        sstream << value_->getTextFormatted().str();
    }

    sstream << "%";
    return sstream;
}


const ExpressionPtr &PercentQuantifierExpression::getValue() const {
    return value_;
}


// ThemExpression

bool ThemExpression::isValid(std::string_view &) const {
    return true;
}


bool ThemExpression::areOperandsValid() const {
    return true;
}


bool ThemExpression::isComplete() const {
    return true;
}


Expression::Type ThemExpression::getType() const {
    return Expression::Type::Undefined;
}


std::stringstream ThemExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "them";
    return sstream;
}


// SetExpression

bool SetExpression::isValid(std::string_view &) const {
    return true;
}


bool SetExpression::areOperandsValid() const {
    return true;
}


bool SetExpression::isComplete() const {
    return true;
}


Expression::Type SetExpression::getType() const {
    return Expression::Type::Undefined;
}


const std::vector<ExpressionPtr> &SetExpression::getElements() const {
    return elements_;
}


// StringSetExpression

std::stringstream StringSetExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "(";

    for(auto element_it = elements_.begin(); 
        element_it != elements_.end(); 
        ++element_it) {

        if(element_it != elements_.begin()) {
            sstream << ", ";
        }
        if(*element_it) {
            sstream << (*element_it)->getTextFormatted().str();
        }

    }

    sstream << ")";
    return sstream;
}


// RuleSetExpression

std::stringstream RuleSetExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "(";

    for(auto element_it = elements_.begin(); 
        element_it != elements_.end(); 
        ++element_it) {

        if(element_it != elements_.begin()) {
            sstream << ", ";
        }
        if(*element_it) {
            sstream << (*element_it)->getTextFormatted().str();
        }

    }

    sstream << ")";
    return sstream;
}


// ExpressionSetExpression

std::stringstream ExpressionSetExpression::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "[";

    for(auto element_it = elements_.begin(); 
        element_it != elements_.end(); 
        ++element_it) {

        if(element_it != elements_.begin()) {
            sstream << ", ";
        }
        if(*element_it) {
            sstream << (*element_it)->getTextFormatted().str();
        }

    }

    sstream << "]";
    return sstream;
}


// OfExpression

bool OfExpression::isValid(std::string_view &) const {
    return true;
}


bool OfExpression::areOperandsValid() const {
    return true;
}


bool OfExpression::isComplete() const {
    return true;
}


std::stringstream OfExpression::getTextFormatted() const {
    std::stringstream sstream;

    if(quantifier_) {
        sstream << quantifier_->getTextFormatted().str();
    }

    sstream << " " << "of" << " ";

    if(set_) {
        sstream << set_->getTextFormatted().str();
    }

    return sstream;
}


Expression::Type OfExpression::getType() const {
    return Expression::Type::Bool;
}


const ExpressionPtr &OfExpression::getQuantifier() const {
    return quantifier_;
}


const ExpressionPtr &OfExpression::getSet() const {
    return set_;
}


// ForExpression 

bool ForExpression::isValid(std::string_view &) const {
    return true;
}


bool ForExpression::areOperandsValid() const {
    return true;
}


bool ForExpression::isComplete() const {
    return true;
}


std::stringstream ForExpression::getTextFormatted() const {
    std::stringstream sstream;
    
    sstream << "for ";

    if(quantifier_) {
        sstream << quantifier_->getTextFormatted().str();
    }

    sstream << " " << "of" << " ";

    if(set_) {
        sstream << set_->getTextFormatted().str();
    }

    sstream << ": ( ";

    if(expression_) {
        sstream << expression_->getTextFormatted().str();
    }

    sstream << " )";

    return sstream;
}


Expression::Type ForExpression::getType() const {
    return Expression::Type::Bool;
}


const ExpressionPtr &ForExpression::getQuantifier() const {
    return quantifier_;
}


const ExpressionPtr &ForExpression::getSet() const {
    return set_;
}


const ExpressionPtr &ForExpression::getExpression() const {
    return expression_;
}


// VarListExpression

bool VarListExpression::isValid(std::string_view &) const {
    return true;
}


bool VarListExpression::areOperandsValid() const {
    return true;
}


bool VarListExpression::isComplete() const {
    return true;
}


std::stringstream VarListExpression::getTextFormatted() const {
    std::stringstream sstream;
    
    for(auto var_it = vars_.begin(); var_it != vars_.end(); ++var_it) {
        if(var_it != vars_.begin()) {
            sstream << ", ";
        }
    
        sstream << *var_it;
    }

    return sstream;
}


Expression::Type VarListExpression::getType() const {
    return Expression::Type::Undefined;
}


const std::vector<std::string> &VarListExpression::getVariables() const {
    return vars_;
}


// ForIntExpression

bool ForIntExpression::isValid(std::string_view &) const {
    return true;
}


bool ForIntExpression::areOperandsValid() const {
    return true;
}


bool ForIntExpression::isComplete() const {
    return true;
}


std::stringstream ForIntExpression::getTextFormatted() const {
    std::stringstream sstream;
    
    sstream << "for ";

    if(quantifier_) {
        sstream << quantifier_->getTextFormatted().str();
    }
    if(list_) {
        sstream << " " << list_->getTextFormatted().str();
    }

    sstream << " " << "in" << " ";

    if(iterable_) {
        sstream << iterable_->getTextFormatted().str();
    }

    sstream << ": ( ";

    if(expression_) {
        sstream << expression_->getTextFormatted().str();
    }

    sstream << " )";

    return sstream;
}


Expression::Type ForIntExpression::getType() const {
    return Expression::Type::Bool;
}


const ExpressionPtr &ForIntExpression::getQuantifier() const {
    return quantifier_;
}


const std::shared_ptr<VarListExpression>  &ForIntExpression::getVarList() const {
    return list_;
}


const ExpressionPtr &ForIntExpression::getIterable() const {
    return iterable_;
}


const ExpressionPtr &ForIntExpression::getExpression() const {
    return expression_;
}
