/**
 * @file parser_adapter_rule.cpp 
 * @brief Implementation of TSParserAdapter members, that 
 * are responsible for converting TSNode with rules and rule elements to
 * high level representation  
 * 
 * @author Vojtěch Dvořák 
 */


#include "../headers/yara_file.h"
#include "../headers/rule.h"
#include "../headers/string.h"
#include "../headers/meta.h"
#include "../headers/variable.h"

#include "../headers/parser_adapter.h"


RulePtr TSParserAdapter::buildRule_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    RulePtr rule = std::make_unique<Rule>(); 

    rule->setParentFile(yara_file);
    rule->setOffset(getNodeOffset_(node));
    rule->setLen(getNodeLen_(node));
    rule->setRangeCache(getNodeRange_(node));
    
    TSNode head = getNodeChildByFieldName_(node, "head");
    if(isValidNode_(head)) {
        buildRuleHead_(yara_file, rule, head, sv);
    }

    TSNode body = getNodeChildByFieldName_(node, "body");
    if(isValidNode_(body)) {
        buildRuleBody_(yara_file, rule, body, sv);
    }

    return rule;
}


void TSParserAdapter::buildRuleHead_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &head_node, std::string_view sv) {
    (void)yara_file;
    
    // Rule modifiers
    TSNode mod_list = getNodeChildByFieldName_(head_node, "mods");
    if(isValidNode_(mod_list)) {
        buildRuleModifierList_(yara_file, rule, mod_list, sv);
    }

    // Rule identifier
    TSNode id = getNodeChildByFieldName_(head_node, "id");
    if(isValidNode_(id) && isValidId_(yara_file, id, sv)) {
        rule->setId(getNodeString_(id, sv));
        rule->setIdOffset(getNodeOffset_(id));
    }

    // Tags
    TSNode tag_list = getNodeChildByFieldName_(head_node, "tags");
    if(isValidNode_(tag_list)) {
        buildRuleTagList_(yara_file, rule, tag_list, sv);
    }
}


void TSParserAdapter::buildRuleModifierList_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &list_node, std::string_view sv) {
    (void)yara_file;
        
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) { ///< All children should be mods (but actually there can be just two mods)
        TSNode mod_node = ts_node_named_child(list_node, i);
        if(!isValidNode_(mod_node) || isGlobalNode_(mod_node)) { ///< Skip invalid and global nodes
            continue;
        }

        RuleModifier::Type type = RuleModifier::stringToType(getNodeStringV_(mod_node, sv));
        std::unique_ptr<RuleModifier> new_mod = std::make_unique<RuleModifier>(type);
        
        new_mod->setParentFile(yara_file);
        new_mod->setLen(getNodeLen_(mod_node));
        new_mod->setOffset(getNodeOffset_(mod_node));

        rule->addModifier(std::move(new_mod));
    }
}



void TSParserAdapter::buildRuleTagList_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &list_node, std::string_view sv) {
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) { /// All children should be tags
        TSNode tag = ts_node_named_child(list_node, i);
        if(!isValidNode_(tag) || isGlobalNode_(tag)) {
            continue;
        }

        TSNode tag_id = getNodeChildByFieldName_(tag, "id");
        
        if(isValidNode_(tag_id) && isValidId_(yara_file, tag_id, sv)) {
            rule->addTag(getNodeString_(tag_id, sv));
        }

    } 
}


void TSParserAdapter::buildRuleBody_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &body_node, std::string_view sv) {
    TSNode meta_list = getNodeChildByFieldName_(body_node, "metas");
    if(isValidNode_(meta_list)) {

        // Building metas 
        buildMetaList_(yara_file, rule, meta_list, sv);
    }

    TSNode strings_list = getNodeChildByFieldName_(body_node, "strings");
    if(isValidNode_(strings_list)) {

        // Building strings
        buildStringList_(yara_file, rule, strings_list, sv);
    }

    TSNode var_list = getNodeChildByFieldName_(body_node, "variables");
    if(isValidNode_(var_list)) {

        // Building internal variables
        buildVarList_(yara_file, rule, var_list, sv);
    }

    TSNode condition_node = getNodeChildByFieldName_(body_node, "condition");
    if(isValidNode_(condition_node)) {

        // Building conditions
        buildCondition_(yara_file, rule, condition_node, sv);
    }
}


void TSParserAdapter::buildMetaList_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &list_node, std::string_view sv) {
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) {
        TSNode meta_node = ts_node_named_child(list_node, i);
        if(!isValidNode_(meta_node) || isGlobalNode_(meta_node)) {
            continue;
        }

        std::unique_ptr<Meta> meta = std::make_unique<Meta>();

        meta->setParentRule(rule);
        meta->setParentFile(yara_file);
        meta->setOffset(getNodeOffset_(meta_node));
        meta->setLen(getNodeLen_(meta_node));
        meta->setRange(getNodeRange_(meta_node));

        // Building ID
        TSNode id = getNodeChildByFieldName_(meta_node, "id");
        if(isValidNode_(id) && isValidId_(yara_file, id, sv)) {
            meta->setId(std::move(getNodeString_(id, sv)));
        }

        // Building value of meta
        TSNode value = getNodeChildByFieldName_(meta_node, "value");
        if(isValidNode_(value)) {
            meta->setValue(std::move(buildLiteral_(yara_file, value, sv)));
        }

        rule->addMeta(std::move(meta));
    }
}


void TSParserAdapter::buildStringList_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &list_node, std::string_view sv) {
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) {
        TSNode string_node = ts_node_named_child(list_node, i);
        if(!isValidNode_(string_node) || isGlobalNode_(string_node)) {
            continue;
        }

        // Building value of string
        std::unique_ptr<String> string;
        TSNode string_val_node = getNodeChildByFieldName_(string_node, "value");
        if(isValidNode_(string_val_node)) {
            if(isType_(string_val_node, "hex_str")) {
                string = std::make_unique<String>(String::Type::Hex);
                string->setContent(buildHexStr_(yara_file, string_val_node, sv));
            }
            else if(isType_(string_val_node, "regexp")) {
                string = std::make_unique<String>(String::Type::Regexp);
                string->setContent(buildRegexp_(yara_file, string_val_node, sv));
            }
            else {
                string = std::make_unique<String>(String::Type::Plain);
                string->setContent(buildStr_(yara_file, string_val_node, sv));
            }
        }

        // Set the properties of the rule
        string->setParentRule(rule);
        string->setParentFile(yara_file);
        string->setOffset(getNodeOffset_(string_node));
        string->setLen(getNodeLen_(string_node));
        string->setRange(getNodeRange_(string_node));

        // Building ID
        TSNode id_node = getNodeChildByFieldName_(string_node, "id");
        std::string id = std::string({});
        if(isValidNode_(id_node)) { ///< Check if string is valid id, is not necessary because any keyword does not start with $
            id = getStringId_(id_node, sv);
        }

        // Set the id to the newly created id or set the anonymous flag
        if(id.empty()) {
            string->setAnonymous(true);
        }
        else {
            string->setId(std::move(getStringId_(id_node, sv)));
        }


        // Building modifiers
        TSNode mod_list = getNodeChildByFieldName_(string_node, "mods");
        if(isValidNode_(mod_list)) {
            buildStringModifierList_(yara_file, string, mod_list, sv);
        }

        rule->addString(std::move(string));
    }
}


void TSParserAdapter::buildVarList_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &list_node, std::string_view sv) {
    auto var_expr_ctx = expression_ctx_t(yara_file, rule);
    
    // Iterating over the list of internal variables
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) {
        TSNode var_node = ts_node_named_child(list_node, i);
        if(!isValidNode_(var_node) || isGlobalNode_(var_node)) {
            continue;
        }

        std::unique_ptr<IntVariable> var = std::make_unique<IntVariable>();

        var->setParentRule(rule);
        var->setParentFile(yara_file);
        var->setOffset(getNodeOffset_(var_node));
        var->setLen(getNodeLen_(var_node));
        var->setRange(getNodeRange_(var_node));

        // Building ID
        TSNode id = getNodeChildByFieldName_(var_node, "id");
        if(isValidNode_(id) && isValidId_(yara_file, id, sv)) {
            var->setId(std::move(getNodeString_(id, sv)));
        }

        // Building value of value
        TSNode value = getNodeChildByFieldName_(var_node, "value");
        var->setValue(buildExpression_(var_expr_ctx, value, sv));
        
        rule->addVar(std::move(var));
    }
}



void TSParserAdapter::buildCondition_(const YaraFilePtr &yara_file, const RulePtr &rule, const TSNode &condition_node, std::string_view sv) {
    auto condition_ctx = expression_ctx_t(yara_file, rule);
    rule->setCondition(buildExpression_(condition_ctx, condition_node, sv));

    #ifdef DEBUG
        if(rule->getCondition()) {
            DEBUG_LOG("Condition type of %s is %s.\n", 
                rule->getId().c_str(), 
                Expression::typeToString(rule->getCondition()->getType()).data()
            );
        }
    #endif
}

