/**
 * @file parser_adapter_expression.cpp
 * @brief Definitions of methods, that are responsible
 * for converting concrete syntax tree returned by tree-sitter to high level 
 * representation
 * 
 * @author Vojtěch Dvořák 
 */


#include "../headers/yara_file.h"
#include "../headers/expression.h"
#include "../headers/rule.h"
#include "../headers/other.h"
#include "../headers/string.h"
#include "../headers/language.h"
#include "../headers/variable.h"
#include "../headers/module.h"

#include "../headers/parser_adapter.h"

/**
 * Provides mapping of TSNode type to corresponding function, that converts 
 * TSNode to high level representation  
 */
const std::unordered_map<std::string_view, TSParserAdapter::expression_method_ptr> TSParserAdapter::expression_method_tab_ = {
    {"addition", &TSParserAdapter::buildBinaryExpression_<AddExpression>},
    {"subtraction", &TSParserAdapter::buildBinaryExpression_<SubExpression>},
    {"division", &TSParserAdapter::buildBinaryExpression_<DivExpression>},
    {"remainder", &TSParserAdapter::buildBinaryExpression_<RemainderExpression>},
    {"multiplication", &TSParserAdapter::buildBinaryExpression_<MulExpression>},

    {"left_shift", &TSParserAdapter::buildBinaryExpression_<LeftShiftExpression>},
    {"right_shift", &TSParserAdapter::buildBinaryExpression_<RightShiftExpression>},
    {"bitwise_and", &TSParserAdapter::buildBinaryExpression_<BitwiseAndExpression>},
    {"bitwise_xor", &TSParserAdapter::buildBinaryExpression_<BitwiseXorExpression>},
    {"bitwise_or", &TSParserAdapter::buildBinaryExpression_<BitwiseOrExpression>},

    {"less_than", &TSParserAdapter::buildBinaryExpression_<LtExpression>},
    {"less_than_or_equal", &TSParserAdapter::buildBinaryExpression_<LteExpression>},
    {"greater_than", &TSParserAdapter::buildBinaryExpression_<GtExpression>},
    {"greater_than_or_equal", &TSParserAdapter::buildBinaryExpression_<GteExpression>},
    {"equal", &TSParserAdapter::buildBinaryExpression_<EqExpression>},
    {"not_equal", &TSParserAdapter::buildBinaryExpression_<NeqExpression>},

    {"contains", &TSParserAdapter::buildBinaryExpression_<ContainsExpression>},
    {"icontains", &TSParserAdapter::buildBinaryExpression_<IContainsExpression>},
    {"startswith", &TSParserAdapter::buildBinaryExpression_<StartsWithExpression>},
    {"istartswith", &TSParserAdapter::buildBinaryExpression_<IStartsWithExpression>},
    {"endswith", &TSParserAdapter::buildBinaryExpression_<EndsWithExpression>},
    {"iendswith", &TSParserAdapter::buildBinaryExpression_<IEndsWithExpression>},
    {"iequals", &TSParserAdapter::buildBinaryExpression_<IEqExpression>},

    {"and", &TSParserAdapter::buildBinaryExpression_<AndExpression>},
    {"or", &TSParserAdapter::buildBinaryExpression_<OrExpression>},

    {"not", &TSParserAdapter::buildUnaryExpression_<NotExpression>},
    {"unary_minus", &TSParserAdapter::buildUnaryExpression_<UnaryMinusExpression>},
    {"bitwise_not", &TSParserAdapter::buildUnaryExpression_<BitwiseNotExpression>},
    {"defined", &TSParserAdapter::buildUnaryExpression_<DefinedExpression>},

    {"matches", &TSParserAdapter::buildBinaryExpression_<MatchesExpression>},
    {"at", &TSParserAdapter::buildBinaryExpression_<AtExpression>},

    {"in", &TSParserAdapter::buildInExpression_},

    {"structure_access", &TSParserAdapter::buildSymbolExpression_},
    {"array_access", &TSParserAdapter::buildSymbolExpression_},
    {"function_call", &TSParserAdapter::buildSymbolExpression_},
    {"identifier", &TSParserAdapter::buildSymbolExpression_},

    {"of", &TSParserAdapter::buildOfExpression_},
    {"rule_set", &TSParserAdapter::buildRuleSet_},
    {"string_set", &TSParserAdapter::buildStringSet_},
    {"expression_set", &TSParserAdapter::buildExpressionSet_},
    {"enum", &TSParserAdapter::buildEnumExpression_},

    {"for", &TSParserAdapter::buildForExpression_},
    {"for_int", &TSParserAdapter::buildForIntExpression_},
    {"primary_parentheses", &TSParserAdapter::buildParenthesesExpression_},
    {"parentheses", &TSParserAdapter::buildParenthesesExpression_},

    {"string_count", &TSParserAdapter::buildStringPropExpression_<StringCountExpression>},
    {"string_match_length", &TSParserAdapter::buildStringPropExpression_<StringMatchLengthExpression>},
    {"string_offset", &TSParserAdapter::buildStringPropExpression_<StringOffsetExpression>},

    {"indexed_match_length", &TSParserAdapter::buildIndexedStringPropExpression_<StringMatchLengthExpression>},
    {"indexed_offset", &TSParserAdapter::buildIndexedStringPropExpression_<StringOffsetExpression>},

    {"uint_literal", &TSParserAdapter::buildLiteralExpression_},
    {"bool_literal", &TSParserAdapter::buildLiteralExpression_},
    {"string_literal", &TSParserAdapter::buildLiteralExpression_},
    {"float_literal", &TSParserAdapter::buildLiteralExpression_},

    {"string_identifier", &TSParserAdapter::buildStringExpression_},

    {"regexp", &TSParserAdapter::buildRegexpExpression_},

    {"size_literal", &TSParserAdapter::buildSizeLiteral_},
};

ExpressionPtr TSParserAdapter::buildExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    if(!isValidNode_(expr_node)) {
        return nullptr;
    }

    DEBUG_LOG("Building expr. from: %s...\n", ts_node_type(expr_node));
    
    auto result_it = expression_method_tab_.find(ts_node_type(expr_node)); ///< Try to find mapped function

    ExpressionPtr result = nullptr;
    if(result_it != expression_method_tab_.end()) {
        result = (this->*(result_it->second))(ctx, expr_node, sv); ///< Do conversion of TSNode to high level representation
    }

    if(result != nullptr) {

        // Set expression properties
        result->setParentRule(ctx.rule);
        result->setParentFile(ctx.file);
        result->setOffset(getNodeOffset_(expr_node));
        result->setLen(getNodeLen_(expr_node));

        std::string_view err_msg;
        if(result->isComplete() && !result->isValid(err_msg)) { ///< Check completion (to avoid interference of semantic and syntax errors) and validity

            if(result->areOperandsValid()) { ///< If operands (subexpressions) are invalid, error was already created by them (avoid duplicated errors) 
                ctx.file->addSemanticError<ExpressionError>(
                    err_msg, 
                    getNodeOffset_(expr_node), 
                    getNodeLen_(expr_node),
                    getNodeRange_(expr_node)
                )->setGlobal(true)->bind(ctx.rule.get());
            }

        }
    }

    return result;
}


template <typename T>
ExpressionPtr TSParserAdapter::buildBinaryExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode lop_node = getNodeChildByFieldName_(expr_node, "lop");
    TSNode rop_node = getNodeChildByFieldName_(expr_node, "rop");

    // Building both operands (they can be nullptr)
    ExpressionPtr lop = buildExpression_(ctx, lop_node, sv);
    ExpressionPtr rop = buildExpression_(ctx, rop_node, sv);

    return std::make_shared<T>(lop, rop);
}


template <typename T>
ExpressionPtr TSParserAdapter::buildUnaryExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode op_node = getNodeChildByFieldName_(expr_node, "op");

    // Building operands(they can be nullptr)
    ExpressionPtr op = buildExpression_(ctx, op_node, sv);

    return std::make_shared<T>(op);
}


ExpressionPtr TSParserAdapter::buildInExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode str_node = getNodeChildByFieldName_(expr_node, "str");
    TSNode range_node = getNodeChildByFieldName_(expr_node, "range");

    auto str_expr = buildExpression_(ctx, str_node, sv);
    auto range = buildRange_(ctx, range_node, sv);

    return std::make_shared<InExpression>(str_expr, range);
}


ExpressionPtr TSParserAdapter::buildSymbolExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::shared_ptr<Symbol> symbol = buildSymbol_(ctx, expr_node, sv);
    if(symbol) {
        if(!symbol->isValue() && symbol->isDefined()) { ///< Standalone symbol must have value type (it cannot be struct or array etc...)
            ctx.file->addSemanticError<SymbolError>(
                "Wrong usage of symbol '" + getNodeString_(expr_node, sv) + "'! (it is not value)",
                getNodeOffset_(expr_node),
                getNodeLen_(expr_node),
                getNodeRange_(expr_node)
            );
        }
    }

    return symbol;
}


std::shared_ptr<Symbol> TSParserAdapter::buildSymbol_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    if(isType_(expr_node, "identifier")) {

        return buildPlainSymbol_(ctx, expr_node, sv);
    }
    else if(isType_(expr_node, "structure_access")) {

        return buildStructureAccess_(ctx, expr_node, sv);
    }
    else if(isType_(expr_node, "array_access")) {

        return buildArrayAccess_(ctx, expr_node, sv);
    }
    else if(isType_(expr_node, "function_call")) {

        return buildFunctionCall_(ctx, expr_node, sv);
    }
    else {
        return nullptr;
    }
}


std::shared_ptr<ArrayExpression> TSParserAdapter::buildArrayAccess_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode array_node = getNodeChildByFieldName_(expr_node, "array");
    TSNode key_node = getNodeChildByFieldName_(expr_node, "key");

    std::shared_ptr<Symbol> array = nullptr;
    ExpressionPtr key = nullptr;

    if(isValidNode_(array_node)) {
        array = buildSymbol_(ctx, array_node, sv); ///< Build array symbol
    }

    if(array) {
        if(!array->isDefined()) {
            ctx.file->addSemanticError<SymbolReferenceError>( ///< Array symbol is not defined
                "Undefined symbol '" + getNodeString_(array_node, sv) + "'! (expected array)",
                getNodeOffset_(array_node),
                getNodeLen_(array_node),
                getNodeRange_(array_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
        else if(!array->isArray() && !array->isDict()) {
            ctx.file->addSemanticError<SymbolError>( ///< Array symbol has wrong type
                "Invalid usage of symbol '" + getNodeString_(array_node, sv) + "'! (it is not array nor dictionary)",
                getNodeOffset_(array_node),
                getNodeLen_(array_node),
                getNodeRange_(array_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
    }
    else {
        return nullptr;
    }
    
    if(isValidNode_(key_node)) {
        key = buildExpression_(ctx, key_node, sv); ///< Build key expression
    }

    auto result = std::make_shared<ArrayExpression>(array, key);

    // Propagate context
    if(array) {
        if(array->isArray()) {
            const json *structure = Module::findField(array->getContext(), "structure");
            Module::buildModuleContext(result, structure);
        }
        else if(array->isDict()) {
            result->setType(array->getType());
            result->setSymType(Symbol::Type::Value);
            result->setContext(nullptr);
        }

        result->setDefined(true);
    }
    
    return result;
}


std::shared_ptr<StructExpression> TSParserAdapter::buildStructFromCache_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    const ModuleProvider &mp = ModuleProvider::getInstance();
    std::string_view full_name = getNodeStringV_(expr_node, sv);
    const json *cached_ctx = mp.findCached(full_name);

    if(cached_ctx) {
        std::vector<YaraFileElementBindable *> includes;
        auto import = ctx.file->getImport(
            full_name.substr(0, full_name.find(".")), 
            getNodeOffset_(expr_node), 
            includes, 
            true
        );

        if(import) {
            auto structure = std::make_shared<PlainSymbol>();
            auto member = std::make_shared<PlainSymbol>();
            auto result = std::make_shared<StructExpression>(structure, member);

            structure->setDefined(true);
            member->setDefined(true);

            Module::buildModuleContext(member, cached_ctx);

            result->setContext(cached_ctx);
            result->setType(member->getType());
            result->setSymType(member->getSymType());
            result->setDefined(true);

            ctx.file->bind(import, ctx.rule.get());
            ctx.file->bind(includes, ctx.rule.get());

            return result;
        }
    }

    return nullptr;
}


std::shared_ptr<StructExpression> TSParserAdapter::buildStructureAccess_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    #ifdef USE_MODULE_CACHE
        if(auto from_cache = buildStructFromCache_(ctx, expr_node, sv)) {
            return from_cache;
        }
    #endif

    TSNode struct_node = getNodeChildByFieldName_(expr_node, "struct");
    TSNode member_node = getNodeChildByFieldName_(expr_node, "member");
    std::shared_ptr<Symbol> structure = nullptr, member = nullptr;

    if(isValidNode_(struct_node)) {
        structure = buildSymbol_(ctx, struct_node, sv); ///< Build struct symbol
    }

    if(structure) {
        // Make semantics check of struct symbol
        if(!structure->isDefined()) {
            ctx.file->addSemanticError<SymbolReferenceError>(
                "Undefined symbol '" + getNodeString_(struct_node, sv) + "'! (expected structure)",
                getNodeOffset_(struct_node),
                getNodeLen_(struct_node),
                getNodeRange_(struct_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
        else if(!structure->isStruct()) {
            ctx.file->addSemanticError<SymbolError>(
                "Invalid usage of symbol '" + getNodeString_(struct_node, sv) + "'! (it is not struct)",
                getNodeOffset_(struct_node),
                getNodeLen_(struct_node),
                getNodeRange_(struct_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
    }
    else {
        return nullptr;
    }

    // Create new parsing context with updated JSON context of struct symbol
    auto new_ctx = expression_ctx_t(ctx.file, ctx.rule, structure ? structure->getContext() : ctx.module_ctx);
    new_ctx.vars = ctx.vars;
    if(isValidNode_(member_node)) {
        member = buildSymbol_(new_ctx, member_node, sv); // Build member symbol
    }

    auto result = std::make_shared<StructExpression>(structure, member);

    if(member) {
        // Propagate the context
        result->setContext(member->getContext());
        result->setType(member->getType());
        result->setSymType(member->getSymType());
        result->setDefined(true);
    }
    
    return result;
}


ExpressionPtr TSParserAdapter::buildParenthesesExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode inner_expr;
    for(uint32_t i = 0; i < ts_node_named_child_count(expr_node); i++) {
        inner_expr = ts_node_named_child(expr_node, 0);
        if(isValidNode_(inner_expr) && !isGlobalNode_(inner_expr)) { ///< Skip all invalid and global nodes
            break;
        }
    }

    return std::make_shared<ParenthesesExpression>(
        buildExpression_(ctx, inner_expr, sv)
    ); 
}



ExpressionPtr TSParserAdapter::buildRegexpExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    return std::make_shared<RegexpExpression>(buildRegexp_(ctx.file, expr_node, sv));
}


template <typename T>
ExpressionPtr TSParserAdapter::buildStringPropExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {    
    std::string string_id;
    if(isValidNode_(expr_node)) {
        string_id = getStringId_(expr_node, sv); ///< Get string id (pure id without @ or ! character)
    }

    if(ctx.is_in_loop && string_id.empty()) {
        // Anonymous string offset or match length in loop is Ok
    }
    else if(string_id.empty()) { ///< But outside the loop it is not
        ctx.file->addSemanticError<StringReferenceError>(
            "Wrong use of anonymous string",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }
    else if(!ctx.rule->getStrings().exists(string_id)) {
        ctx.file->addSemanticError<StringReferenceError>( ///< String is not defined inside the rule
            "Undefined string '$" + string_id + "'!",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }

    return std::make_shared<T>(string_id);
}


template <typename T>
ExpressionPtr TSParserAdapter::buildIndexedStringPropExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string string_id;
   
    TSNode str_node = getNodeChildByFieldName_(expr_node, "str");
    TSNode index_node = getNodeChildByFieldName_(expr_node, "index");
    if(isValidNode_(str_node)) {
        string_id = getStringId_(str_node, sv);
    }

    ExpressionPtr index = buildExpression_(ctx, index_node, sv); ///< Build index expression
    if(index && !index->isInt()) {
        ctx.file->addSemanticError<IndexError>(
            "Index must be specified by integer expression!",
            getNodeOffset_(index_node),
            getNodeLen_(index_node),
            getNodeRange_(index_node)
        );
    }


    if(ctx.is_in_loop && string_id.empty()) {
        // Anonymous string offset or match length in loop is Ok
    }
    else if(string_id.empty()) { ///< But outside the loop it is not
        ctx.file->addSemanticError<StringReferenceError>(
            "Wrong use of anonymous string",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }
    else if(!ctx.rule->getStrings().exists(string_id)) {
        ctx.file->addSemanticError<StringReferenceError>( ///< String is not defined inside the rule
            "Undefined string '$" + string_id + "'!",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }

    return std::make_shared<T>(string_id, index);
}


ExpressionPtr TSParserAdapter::buildLiteralExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    return std::make_shared<LiteralExpression>(buildLiteral_(ctx.file, expr_node, sv)); 
}


ExpressionPtr TSParserAdapter::buildStringCountExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string str_id = getStringId_(expr_node, sv); 

    if(ctx.is_in_loop && str_id.empty()) {
        // Anonymous string count in loop is OK
    }
    else if(str_id.empty()) {
        ctx.file->addSemanticError<StringReferenceError>(
            "Wrong use of anonymous string",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }
    else if(!ctx.rule->getStrings().exists(str_id)) {
        ctx.file->addSemanticError<StringReferenceError>(
            "Undefined string '$" + str_id + "'",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }

    return std::make_shared<StringCountExpression>(str_id);
}


ExpressionPtr TSParserAdapter::buildStringExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string id_str = getStringId_(expr_node, sv); 

    if(id_str.empty()) {
        // Anonymous string
    }
    else if(!ctx.rule->getStrings().exists(id_str)) {
        ctx.file->addSemanticError<StringReferenceError>(
            "Undefined string '$" + id_str + "'",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node)
        );
    }

    return std::make_shared<StringExpression>(id_str);
}


ExpressionPtr TSParserAdapter::buildStringWildcardExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string id_str = getStringId_(expr_node, sv); 

    bool defined = false;
    std::string_view id_sv = std::string_view(id_str).substr(0, id_str.length() - 1);
    
    for(auto str: ctx.rule->getStrings()) {
        if(str->getId().starts_with(id_sv)) { ///< There must be at least one string that corresponds to given wildcard symbol
            defined = true;
            break;
        }
    }

    if(!defined) {
        ctx.file->addSemanticError<StringReferenceError>( ///< No string was found
            "Undefined string '$" + id_str + "'",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        );
    }

    return std::make_shared<StringWildcardExpression>(std::move(id_str));
}


ExpressionPtr TSParserAdapter::buildRuleWildcardExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string id_str = getNodeString_(expr_node, sv); 

    bool defined = false;
    std::string_view id_sv = id_str.substr(0, id_str.length() - 1);
    for(auto &r: ctx.file->getRules()) {
        if(r->getId().starts_with(id_sv)) { ///< There must be at least one rule that corresponds to given wildcard symbol
            defined = true;
            ctx.rule->bind_to(r);
        }
    }

    if(!defined) {
        ctx.file->addSemanticError<RuleReferenceError>(
            "Undefined identifier '" + id_str + "'",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        )->setGlobal(true)->bind(ctx.rule.get());
    }

    return std::make_shared<RuleWildcardExpression>(std::move(id_str));
}


std::shared_ptr<FunctionCallExpression> TSParserAdapter::buildFunctionCall_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode id_node = getNodeChildByFieldName_(expr_node, "id");
    TSNode args_node = getNodeChildByFieldName_(expr_node, "args");

    std::shared_ptr<Symbol> func_name = nullptr;

    if(isValidNode_(id_node)) {
        func_name = buildSymbol_(ctx, id_node, sv); ///< Build function symbol
    }

    if(func_name) {
        if(!func_name->isDefined()) {
            ctx.file->addSemanticError<SymbolReferenceError>(
                "Undefined function '" + getNodeString_(id_node, sv) + "'!",
                getNodeOffset_(id_node),
                getNodeLen_(id_node),
                getNodeRange_(id_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
        else if(!func_name->isFunction()) { ///< Symbol is not function
            ctx.file->addSemanticError<SymbolError>(
                "Invalid usage of symbol '" + getNodeString_(id_node, sv) + "'! (it is not function)",
                getNodeOffset_(id_node),
                getNodeLen_(id_node),
                getNodeRange_(id_node)
            )->setGlobal(true)->bind(ctx.rule.get());

            return nullptr;
        }
    }
    else {
        return nullptr;
    }

    std::vector<ExpressionPtr> args;
    if(isValidNode_(args_node)) {
        for(uint32_t i = 0; i < ts_node_named_child_count(args_node); i++) {
            TSNode arg_node = ts_node_named_child(args_node, i);
            if(!isValidNode_(arg_node) || isGlobalNode_(arg_node)) {
                continue;
            }

            args.push_back(buildExpression_(ctx, arg_node, sv)); ///< Build arguments of function
        }
    }
    
    auto f_call = std::make_shared<FunctionCallExpression>(func_name, std::move(args));
    if(f_call->isComplete() && !f_call->hasValidArgs()) {
        ctx.file->addSemanticError<FunctionCallError>(
            "Wrong arguments for function '" + getNodeString_(id_node, sv)  + "'!",
            getNodeOffset_(expr_node),
            getNodeLen_(expr_node),
            getNodeRange_(expr_node)
        )->setGlobal(true)->bind(ctx.rule.get());
    }

    if(func_name) { ///< Propagate some contextual information from func_name symbol to function call expression object
        f_call->setDefined(true);
        f_call->setType(func_name->getType());
        f_call->setSymType(Symbol::Type::Value);
    }
    
    return f_call;
}


std::shared_ptr<Range> TSParserAdapter::buildRange_(expression_ctx_t &ctx, const TSNode &range_node, std::string_view sv) {
    if(!isValidNode_(range_node)) {
        return nullptr;
    }

    TSNode from_node = getNodeChildByFieldName_(range_node, "from");
    TSNode to_node = getNodeChildByFieldName_(range_node, "to");

    ExpressionPtr from = buildExpression_(ctx, from_node, sv);
    ExpressionPtr to = buildExpression_(ctx, to_node, sv);

    // Only integer range is allowed
    if(from && !from->isInt()) {
        ctx.file->addSemanticError<RangeError>(
            "Expected integer as range left boundary!",
            getNodeOffset_(from_node),
            getNodeLen_(from_node),
            getNodeRange_(from_node)
        );
    }

    if(to && !to->isInt()) {
        ctx.file->addSemanticError<RangeError>(
            "Expected integer as range right boundary!",
            getNodeOffset_(to_node),
            getNodeLen_(to_node),
            getNodeRange_(to_node)
        );
    }

    return std::make_shared<Range>(from, to);
}


ExpressionPtr TSParserAdapter::buildEnumExpression_(expression_ctx_t &ctx, const TSNode &enum_node, std::string_view sv) {
    std::vector<ExpressionPtr> values;
    
    for(uint32_t i = 0; i < ts_node_named_child_count(enum_node); i++) {
        TSNode cur_item_node = ts_node_named_child(enum_node, i);
        if(!isValidNode_(cur_item_node)) {
            continue;
        }

        values.push_back(buildExpression_(ctx, cur_item_node, sv)); ///< Builds elements of enumeration
    }

    return std::make_shared<Enum>(std::move(values));
}



ExpressionPtr TSParserAdapter::buildSizeLiteral_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode value_node = getNodeChildByFieldName_(expr_node, "value");
    TSNode unit_node = getNodeChildByFieldName_(expr_node, "unit");

    int64_t value;
    SizeExpression::Unit unit; 

    auto value_opt = buildInt_(ctx.file, value_node, sv); ///< In Size literal cannot be integer expression as value (ther must be directly integer literal)
    auto unit_sv = getNodeStringV_(unit_node, sv);

    if(value_opt.has_value()) {
        value = value_opt.value();
        unit = unit_sv == "KB" ? SizeExpression::Unit::KB : SizeExpression::Unit::MB;

        return std::make_shared<SizeExpression>(value, unit);
    }
    else {
        return nullptr;
    }
}


void TSParserAdapter::buildSymbolContextFrom(const std::shared_ptr<PlainSymbol> &symbol, const var_def_t &var) {
    symbol->setSymType(var.type);
    symbol->setType(var.dtype);
    symbol->setContext(var.module_ctx);
    symbol->setDefined(true);
}


void TSParserAdapter::buildSymbolContextFrom(const std::shared_ptr<PlainSymbol> &symbol, Import *import, const expression_ctx_t &ctx, const std::vector<YaraFileElementBindable *> &includes)  {
    DEBUG_LOG("Creating binding between import and rule %s<->%s\n", import->getId().c_str(), ctx.rule->getId().c_str());
    ctx.file->bind(import, ctx.rule.get());
    ctx.file->bind(includes, ctx.rule.get());

    if(import->getModule()) {
        const auto &mod_data = import->getModule()->getData();
        Module::buildModuleContext(symbol, mod_data.get());
    }

    symbol->setDefined(true);
}


void TSParserAdapter::buildSymbolContextFrom(const std::shared_ptr<PlainSymbol> &symbol, const std::unique_ptr<IntVariable> &int_var) {
    if(int_var->getValue()) {
        symbol->setType(int_var->getValue()->getType());

        // Downcasting of Expression * to Symbol *
        auto sym_expr = dynamic_cast<Symbol *>(int_var->getValue().get());
        if(sym_expr) { ///< If there is Symbol stored in internal variable (e. g. it is used as alias for module)
            symbol->setContext(sym_expr->getContext());
            symbol->setSymType(sym_expr->getSymType());
        }
        else {
            symbol->setSymType(Symbol::Type::Value);
        }
    }

    symbol->setDefined(true);
}


void TSParserAdapter::buildSymbolContextFrom(const std::shared_ptr<PlainSymbol> &symbol, Rule *rule, const expression_ctx_t &ctx, const std::vector<YaraFileElementBindable *> &includes) {
    DEBUG_LOG("Creating binding between rules %s<->%s\n", rule->getId().c_str(), ctx.rule->getId().c_str());
    ctx.file->bind(rule, ctx.rule.get());
    ctx.file->bind(includes, ctx.rule.get());
    symbol->setSymType(Symbol::Type::Value);
    symbol->setType(Expression::Type::Bool);
    symbol->setDefined(true);
}


void TSParserAdapter::buildSymbolContextFrom(const std::shared_ptr<PlainSymbol> &symbol, ExtVariable *ext_var) {
    symbol->setSymType(Symbol::Type::Value);

    if(ext_var->getValue()->isType<int64_t>()) {
        symbol->setType(Expression::Type::Int);
    }
    else if(ext_var->getValue()->isType<bool>()) {
        symbol->setType(Expression::Type::Bool);
    }
    else if(ext_var->getValue()->isType<std::string>()) {
        symbol->setType(Expression::Type::String);
    }
    else {
        symbol->setType(Expression::Type::Undefined);
    }
    
    symbol->setDefined(true);
}


void TSParserAdapter::buildSymbolContext(std::shared_ptr<PlainSymbol> symbol, expression_ctx_t &ctx) {
    const std::string &id = symbol->get();
    std::vector<YaraFileElementBindable *> includes;

    if(ctx.module_ctx) { ///< There is any module context (the symbol is member of struct)
        const json *attributes = Module::findField(ctx.module_ctx, "attributes");
        const json *obj = Module::findInArray(attributes, id);
        if(obj) {
            Module::buildModuleContext(symbol, obj);
            symbol->setDefined(true);
        }
    }
    else {
        const json* builtin_symbol = getBuiltinSymbol(id);
        if(builtin_symbol) { ///< Try to find symbol among builtin definitions
            Module::buildModuleContext(symbol, builtin_symbol);
            symbol->setDefined(true);
        }
        else if(ctx.hasVariable(id)) { ///< Ty to find symbol in "loop variables"
            buildSymbolContextFrom(symbol, ctx.getVariable(id));
        }
        else if(auto import = ctx.file->getImport(id, symbol->getOffset(), includes, true)) { ///< Try to find symbol in imported module names
            buildSymbolContextFrom(symbol, import, ctx, includes);
        }
        else if(ctx.rule->getVars().exists(id)) { ///< Ty to find symbol in internal variables of the current rule
            buildSymbolContextFrom(symbol, ctx.rule->getVarById(id));
        }
        else if(auto rule = ctx.file->getRule(id, symbol->getOffset(), includes, true)) { ///< Try to find 
            buildSymbolContextFrom(symbol, rule, ctx, includes);
        }
        else if(auto ext_var = config_->getVar(id)) {
            buildSymbolContextFrom(symbol, ext_var);
        }
    }
}


std::shared_ptr<PlainSymbol> TSParserAdapter::buildPlainSymbol_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::string_view id_sv = getNodeStringV_(expr_node, sv);

    std::shared_ptr<PlainSymbol> identifier = std::make_shared<PlainSymbol>(id_sv);

    identifier->setParentFile(ctx.file);
    identifier->setParentRule(ctx.rule);
    identifier->setOffset(getNodeOffset_(expr_node));
    identifier->setLen(getNodeOffset_(expr_node));

    buildSymbolContext(identifier, ctx);

    if(!identifier->isDefined()) {
        ctx.file->addMissingSymbol(
            identifier->getId(), 
            {ctx.rule->getOffset(), ctx.rule->getOffset() + static_cast<offset_t>(ctx.rule->getLen())}
        );
    }

    return identifier;
}


ExpressionPtr TSParserAdapter::buildStringSet_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::vector<ExpressionPtr> elements;
    for(uint32_t i = 0; i < ts_node_named_child_count(expr_node); i++) {
        TSNode cur_child = ts_node_named_child(expr_node, i);
        if(!isValidNode_(cur_child) || isGlobalNode_(cur_child)) { ///< Skip invalid nodes and global nodes
            continue;
        }
        
        if(isType_(cur_child, "string_identifier")) {
            elements.push_back(buildStringExpression_(ctx, cur_child, sv));
        }
        else if(isType_(cur_child, "string_wildcard")) {
            elements.push_back(buildStringWildcardExpression_(ctx, cur_child, sv));
        }

    }

    return std::make_shared<StringSetExpression>(elements);
}


ExpressionPtr TSParserAdapter::buildExpressionSet_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::vector<ExpressionPtr> elements;
    for(uint32_t i = 0; i < ts_node_named_child_count(expr_node); i++) {
        TSNode cur_child = ts_node_named_child(expr_node, i);
        if(!isValidNode_(cur_child) || isGlobalNode_(cur_child)) { ///< Skip invalid nodes and global nodes
            continue;
        }
        
        elements.push_back(buildExpression_(ctx, cur_child, sv));
    }

    return std::make_shared<ExpressionSetExpression>(elements);
}


ExpressionPtr TSParserAdapter::buildRuleSet_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    std::vector<ExpressionPtr> elements;
    for(uint32_t i = 0; i < ts_node_named_child_count(expr_node); i++) {
        TSNode cur_child = ts_node_named_child(expr_node, i);
        if(!isValidNode_(cur_child) || isGlobalNode_(cur_child)) {
            continue;
        }

        auto id = getNodeString_(cur_child, sv);
        if(isType_(cur_child, "identifier")) {
            if(!ctx.file->hasRule(id, ctx.rule->getOffset())) { ///< Try to find rule
                ctx.file->addSemanticError<RuleReferenceError>(
                    "Undefined identifier '" + id + "' !",
                    getNodeOffset_(cur_child),
                    getNodeLen_(cur_child),
                    getNodeRange_(cur_child)
                )->setGlobal(true)->bind(ctx.rule.get());
            }

            elements.push_back(std::make_shared<PlainSymbol>(id));
        }
        else if(isType_(cur_child, "rule_wildcard")) {
            elements.push_back(buildRuleWildcardExpression_(ctx, cur_child, sv));
        }

    }

    return std::make_shared<RuleSetExpression>(elements);
}


ExpressionPtr TSParserAdapter::buildThemExpression_(expression_ctx_t &, const TSNode &, std::string_view) {
    return std::make_shared<ThemExpression>();
}


ExpressionPtr TSParserAdapter::buildSetExpression_(expression_ctx_t &ctx, const TSNode &set_node, std::string_view sv) {
    if(isType_(set_node, "them")) {
        return buildThemExpression_(ctx, set_node, sv);
    }
    else if(isType_(set_node, "rule_set")) {
        return buildRuleSet_(ctx, set_node, sv);
    }
    else if(isType_(set_node, "string_set")) {
        return buildStringSet_(ctx, set_node, sv);
    }
    else if(isType_(set_node, "expression_enum")) {
        return buildExpressionSet_(ctx, set_node, sv);
    }
    else {
        return buildExpression_(ctx, set_node, sv);
    }
}


ExpressionPtr TSParserAdapter::buildOfExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode quantifier_node = getNodeChildByFieldName_(expr_node, "quant");
    TSNode set_node = getNodeChildByFieldName_(expr_node, "set");
    TSNode range_node = getNodeChildByFieldName_(expr_node, "range");

    ExpressionPtr quantifier, set;
    std::shared_ptr<Range> range = nullptr;

    // Build quantifier expression
    if(isValidNode_(quantifier_node)) {
        quantifier = buildQuantifierExpression_(ctx, quantifier_node, sv);
    }

    // Build set expression
    if(isValidNode_(set_node)) {
        set = buildSetExpression_(ctx, set_node, sv);
    }

    // Build range (optionally - if there is no range range_node is null)
    if(isValidNode_(range_node)) {
        range = buildRange_(ctx, range_node, sv);
    }


    return std::make_shared<OfExpression>(quantifier, set, range);
}


ExpressionPtr TSParserAdapter::buildPercentQuantifier_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode value_node = getNodeChildByFieldName_(expr_node, "value");

    ExpressionPtr value;
    if(isValidNode_(value_node)) {
        value = buildExpression_(ctx, value_node, sv);
    }

    if(value && !value->isInt()) { ///< Value of percent quantifier must be integer
        ctx.file->addSemanticError<ExpressionError>(
            "Wrong type for percent quantifier!",
            getNodeOffset_(value_node),
            getNodeLen_(value_node)  
        );
    }

    return std::make_shared<PercentQuantifierExpression>(value);
}


ExpressionPtr TSParserAdapter::buildQuantifierExpression_(expression_ctx_t &ctx, const TSNode &q_node, std::string_view sv) {
    if(isType_(q_node, "any")) {
        return std::make_shared<AnyExpression>();
    }
    else if(isType_(q_node, "none")) {
        return std::make_shared<NoneExpression>();
    }
    else if(isType_(q_node, "all")) {
        return std::make_shared<AllExpression>();
    }
    else if(isType_(q_node, "percent_quantifier")) {
        return buildPercentQuantifier_(ctx, q_node, sv);
    }
    else {
        ExpressionPtr quantifier = buildExpression_(ctx, q_node, sv);
        
        if(quantifier && !quantifier->isInt()) { ///< If quantifier is another expression it must have integer type
            ctx.file->addSemanticError<ExpressionError>(
                "Expected integer, any, none, all or %% as quantifier!",
                getNodeOffset_(q_node),
                getNodeLen_(q_node)
            )->setGlobal(true)->bind(ctx.rule.get());
        }

        return quantifier;
    }
}


ExpressionPtr TSParserAdapter::buildForExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode quantifier_node = getNodeChildByFieldName_(expr_node, "quant");
    TSNode set_node = getNodeChildByFieldName_(expr_node, "set");
    TSNode internal_expr_node = getNodeChildByFieldName_(expr_node, "expression");

    ExpressionPtr quantifier, set, internal_expr;

    // Build quantifier
    if(isValidNode_(quantifier_node)) {
        quantifier = buildQuantifierExpression_(ctx, quantifier_node, sv);
    }

    // Them keyword or string set can appear here
    if(isValidNode_(set_node)) {
        if(isType_(set_node, "them")) {
            set = buildThemExpression_(ctx, set_node, sv);
        }
        else if(isType_(set_node, "string_set")) {
            set = buildStringSet_(ctx, set_node, sv);
        }
    }

    expression_ctx_t new_ctx = expression_ctx_t(ctx.file, ctx.rule, ctx.module_ctx, true); ///< Update expression context, to propagate information about this loop
    new_ctx.vars = ctx.vars; ///< Propagate vector of declared variables

    // Build internal expression
    if(isValidNode_(internal_expr_node)) {
        internal_expr = buildExpression_(new_ctx, internal_expr_node, sv);
    }


    return std::make_shared<ForExpression>(quantifier, set, internal_expr);
}


std::shared_ptr<VarListExpression> TSParserAdapter::buildVarList_(expression_ctx_t &ctx, const var_def_t &var_ctx, const TSNode &list_node, std::string_view sv) {
    std::vector<std::string> vars;
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) {
        TSNode cur_child = ts_node_named_child(list_node, i);
        if(!isValidNode_(cur_child) || isGlobalNode_(cur_child)) { ///< Skip invalid and global rules
            continue;
        }

        std::string new_id = getNodeString_(cur_child, sv);

        if(ctx.hasVariable(new_id)) {
            ctx.file->addSemanticError<ExpressionError>( ///< Check if there is already loop identifier with this id defined
                "Duplicated loop identifier '" + new_id + "'!",
                getNodeOffset_(cur_child),
                getNodeLen_(cur_child)
            );
        }
        else {
            // Add var_def_t struct to var list to preserve contextual information of newly defined variable
            var_def_t var_def = var_ctx;
            var_def.id = new_id;
            ctx.vars.push_back(var_def);
            vars.push_back(new_id);
        }
    }

    return std::make_shared<VarListExpression>(std::move(vars));
}


ExpressionPtr TSParserAdapter::buildIterable_(expression_ctx_t &ctx, var_def_t &var_ctx, const TSNode &it_node, std::string_view sv) {
    ExpressionPtr iterable;
    
    if(isType_(it_node, "range")) {
        iterable = buildRange_(ctx, it_node, sv);
        var_ctx.dtype = iterable->getType();
        var_ctx.type = Symbol::Type::Value;
    }
    else if(isType_(it_node, "enum")) {
        iterable = buildEnumExpression_(ctx, it_node, sv);
        var_ctx.dtype = iterable->getType();
        var_ctx.type = Symbol::Type::Value;
    }
    else {
        auto iterable_symbol = buildSymbol_(ctx, it_node, sv);
        if(iterable_symbol) {
            if(!iterable_symbol->isArray() && !iterable_symbol->isDict()) { ///< Check if symbol is really iterable
                ctx.file->addSemanticError<SymbolError>(
                    "Symbol '" + getNodeString_(it_node, sv) + "' is not iterable!",
                    getNodeOffset_(it_node),
                    getNodeLen_(it_node)
                )->setGlobal(true)->bind(ctx.rule.get());
            }
            else {
                var_ctx.module_ctx = &(iterable_symbol->getContext()->at("structure"));
                var_ctx.dtype = Module::getElementDataType(iterable_symbol->getContext());
                var_ctx.type = Module::getElementType(iterable_symbol->getContext());
                iterable = iterable_symbol;
            }
        }

    }

    return iterable;
}


ExpressionPtr TSParserAdapter::buildForIntExpression_(expression_ctx_t &ctx, const TSNode &expr_node, std::string_view sv) {
    TSNode quantifier_node = getNodeChildByFieldName_(expr_node, "quant");
    TSNode list_node = getNodeChildByFieldName_(expr_node, "var_list");
    TSNode iterable_node = getNodeChildByFieldName_(expr_node, "iterable");
    TSNode internal_expr_node = getNodeChildByFieldName_(expr_node, "expression");

    ExpressionPtr quantifier, iterable, internal_expr;
    var_def_t var_ctx;
    std::shared_ptr<VarListExpression> list;

    // Build quantifier
    if(isValidNode_(quantifier_node)) {
        quantifier = buildQuantifierExpression_(ctx, quantifier_node, sv);
    }

    // Build iterable
    if(isValidNode_(iterable_node)) {
        iterable = buildIterable_(ctx, var_ctx, iterable_node, sv);
    }

    // Build variables list
    expression_ctx_t new_ctx = ctx; ///< Create new context structure to preserve content of original context
    if(isValidNode_(list_node)) {
        list = buildVarList_(new_ctx, var_ctx, list_node, sv);
    }

    // Build internal expression
    if(isValidNode_(internal_expr_node)) {
        internal_expr = buildExpression_(new_ctx, internal_expr_node, sv);
    }

    return std::make_shared<ForIntExpression>(quantifier, list, iterable, internal_expr);
}


bool TSParserAdapter::expression_ctx_t::hasVariable(std::string_view id) {
    for(auto v_it = vars.rbegin(); v_it != vars.rend(); ++v_it) {
        if(v_it->id == id) {
            return true;
        }
    }
    
    return false;
}


const var_def_t &TSParserAdapter::expression_ctx_t::getVariable(std::string_view id) {
    for(auto v_it = vars.rbegin(); v_it != vars.rend(); ++v_it) { ///< Search vector from the end to find the newest variable
        if(v_it->id == id) {
            return *v_it;
        }
    }
    
    throw InternalErrorException("Variable was not found in expression context!");
}

