/**
 * @file parser_adapter.cpp
 * @brief Implementation of TSParserAdapter class members 
 * 
 * Class TSParserAdapter is responsible for converting concrete syntax tree 
 * from tree sitter to high level AST
 * 
 * Note: Originally it was planned to use tree-sitter builtin mechanism for
 * analysis of concrete syntax tree - TS queries, but it looks like, that
 * they are not so effective like more adhoc and less universal methods (simple
 * traversal of tree)
 * 
 * @author Vojtěch Dvořák 
 */


#include "../headers/language.h"
#include "../headers/common.h"
#include "../headers/meta.h"
#include "../headers/rule.h"
#include "../headers/yara_file.h"
#include "../headers/yara_source.h"
#include "../headers/other.h"
#include "../headers/module.h"
#include "../headers/string.h"
#include "../headers/expression.h"

#include "../headers/parser_adapter.h"



/**
 * Auxiliary structure for associating node with its depth
 * (used in methods for traversing)
 */
struct node_w_depth_t {
    size_t depth;
    TSNode node;
};



TSParserAdapter::TSParserAdapter(std::shared_ptr<YaramodConfig> config) : IParserAdapter(config) {
    // Initialization of TS parser 
    ts_parser_ = ts_parser_new();
    if(!ts_parser_) {
        throw InternalErrorException("Unable to allocate TS parser!");
    }

    if(!ts_parser_set_language(ts_parser_, tree_sitter_yara())) {
        throw InternalErrorException("Unable to set lang to the TS parser!");
    }

    node_stack_.reserve(INITIAL_NODE_STACK_SIZE);
}



TSParserAdapter::~TSParserAdapter() {
    // Cleanup of TS parser
    ts_parser_delete(ts_parser_);
}



YaraSourcePtr TSParserAdapter::createYaraSourceFromFile(std::string_view path) {
    YaraSourcePtr yara_src = std::make_unique<YaraSource>(); 

    yara_src->setBaseDir(path);

    auto yara_file = parseFile(yara_src.get(), path);

    yara_src->setEntryFile(yara_file->getName());

    return yara_src;
}



void TSParserAdapter::updateYaraSourceFromFile(std::string_view path, YaraSource *old_src) {
    old_src->deactivateFiles();
    auto entry_yara_file = old_src->getEntryFile();

    if(path != entry_yara_file->getName()) {
        if(old_src->hasFile(path)) {
            entry_yara_file = old_src->getFile(path);
        }
        else {
            // Performs renaming of entry file if it is necessary
            // It can be used for debugging purposes - versions of files can be stored at different paths
            if(!old_src->renameFile(entry_yara_file->getName(), path)) {
                throw YaramodErrorException("Unable to rename entry file!");
            }

            old_src->setBaseDir(path);
        }
    }

    reparseFile(old_src, path, entry_yara_file);

    old_src->clearInactiveFiles();
}



YaraFilePtr TSParserAdapter::parseFile(YaraSource *yara_src, std::string_view path) {
    std::string str(readFileToString(path));
    
    return parse(yara_src, str, path.data());
}



void TSParserAdapter::reparseFile(YaraSource *parent_src, std::string_view path, const YaraFilePtr &old_yara_file) {
    if(old_yara_file->wasEdited() && old_yara_file->hasFastEdit()) {
        std::string string(readFileToString(path));

        reparse(parent_src, string, old_yara_file);
    }
    else {
        reparse(parent_src, old_yara_file->ctx().getString(), old_yara_file);
    }
}



YaraSourcePtr TSParserAdapter::createYaraSource(const std::string &string) {
    YaraSourcePtr yara_src = std::make_unique<YaraSource>();

    auto yara_file = parse(yara_src.get(), string);
    yara_src->setEntryFile(yara_file->getName()); 

    #ifdef DEBUG
        //yara_src->dumpBindParents();
        //yara_src->dumpBindChildren();
    #endif

    return yara_src;
}


YaraSourcePtr TSParserAdapter::createYaraSource(const std::string &string, const std::string &entry_file_path) {
    YaraSourcePtr yara_src = std::make_unique<YaraSource>();

    yara_src->setBaseDir(entry_file_path);
    auto yara_file = parse(yara_src.get(), string, entry_file_path);
    yara_src->setEntryFile(yara_file->getName()); 

    #ifdef DEBUG
        //yara_src->dumpBindParents();
        //yara_src->dumpBindChildren();
    #endif

    return yara_src;
}



void TSParserAdapter::updateYaraSource(const std::string &string, YaraSource *old_src) {
    old_src->deactivateFiles();
    auto entry_yara_file = old_src->getEntryFile();

    reparse(old_src, string, entry_yara_file);
    old_src->clearInactiveFiles();


    #ifdef DEBUG
        //old_src->dumpBindParents();
        //old_src->dumpBindChildren();
    #endif
}


void TSParserAdapter::updateYaraSource(const std::string &string, const std::string &entry_file_path, YaraSource *old_src) {
    old_src->deactivateFiles();
    auto entry_yara_file = old_src->getEntryFile();

    if(entry_file_path != entry_yara_file->getName()) {
        if(old_src->hasFile(entry_file_path)) {
            entry_yara_file = old_src->getFile(entry_file_path);
        }
        else {
            if(!old_src->renameFile(entry_yara_file->getName(), entry_file_path)) {
                throw YaramodErrorException("Unable to rename entry file!");
            }

            old_src->setBaseDir(entry_file_path);
        }
    }

    reparse(old_src, string, entry_yara_file);
    old_src->clearInactiveFiles();

    #ifdef DEBUG
        //old_src->dumpBindParents();
        //old_src->dumpBindChildren();
    #endif
}



/**
 * Performs parsing of string creates its high level representation
 */
YaraFilePtr TSParserAdapter::parse(YaraSource *yara_src, const std::string &string, const std::string &name) {
    YaraFilePtr yara_file;

    if(yara_src->hasFile(name)) {
        yara_file = yara_src->getFile(name);
        yara_file->setErrorMode(config_->getErrorMode());
    }
    else {
        TSTree* tree = ts_parser_parse_string(
            ts_parser_, 
            NULL, 
            string.c_str(), 
            string.length()
        );
        if(!tree) {
            throw InternalErrorException("Unable to parse input!");
        }

        yara_file = std::make_shared<YaraFile>(yara_src);

        yara_file->setErrorMode(config_->getErrorMode());
        yara_file->setName(name);
        yara_src->addFile(yara_file);
        
        yara_file->ctx().setTree(tree);
        yara_file->ctx().setString(string);

        buildYaraFile_(yara_src, yara_file);
    }

    return yara_file;
}



/**
 * Performs reparsing of string and updates its high level representation 
 */
void TSParserAdapter::reparse(YaraSource *parent_src, const std::string &string, const YaraFilePtr &old_yara_file) {
    old_yara_file->setActivity(true); ///< Set activity to true, because reparsing of file means, that it is still referenced from somewhere 
    old_yara_file->setErrorMode(config_->getErrorMode());

    updateString_(string, old_yara_file); ///< Update string
    while(update_(parent_src, old_yara_file)); ///< Update tree and high level representation until there are any changes (to handle semantic errors)
}


void TSParserAdapter::updateYaraFile_(YaraSource *parent_src, const YaraFilePtr &old_yara_file) {
    
    // Performs edits above the TS tree

    for(auto &e: old_yara_file->getEdits()) {
        TSInputEdit ts_edit;
        if(e.fast_edit) {
            // If it is fast edit, now we must create TSEdit from it, because string is updated
            ts_edit = old_yara_file->ctx().toTSEdit(e.offset, e.ins_n, e.del_n);
            e.ts_edit = ts_edit;
        }
        else {
            ts_edit = e.ts_edit;
        }

        DEBUG_LOG("Performing edit offset:%d, start:[%d, %d], change:(%d, %d) -> (%d %d)\n", 
            e.offset, ts_edit.start_point.row, ts_edit.start_point.column,
            ts_edit.old_end_point.row, ts_edit.old_end_point.column,
            ts_edit.new_end_point.row, ts_edit.new_end_point.column
        );

        old_yara_file->ctx().treeEdit(ts_edit);
    }
    
    TSTree *tree = ts_parser_parse_string( ///< Get updated TSTree
        ts_parser_, 
        old_yara_file->ctx().getTree(), 
        old_yara_file->ctx().getString().c_str(), 
        old_yara_file->ctx().getString().length()
    );
    if(!tree) {
        throw InternalErrorException("Unable to reparse input!");
    }

    #ifndef FULL_REPARSE
        // Modified areas are selected and only corresponding objects are rebuild again

        std::vector<offset_edit_range_t> edited_ranges; 
        edited_ranges = old_yara_file->modify(tree);
        old_yara_file->clearEdits();
        old_yara_file->clearToBeChecked();

        TSNode new_root = ts_tree_root_node(tree);
        for(auto &r: edited_ranges) {
            offset_t end = r.end, start = r.start;

            // Rebuild modified areas (that were deleted in previous step)
            TSNode node = ts_node_first_child_for_byte(new_root, start);

            while(!ts_node_is_null(node) && 
                ((ts_node_end_byte(node) >= start && 
                ts_node_start_byte(node) < end))) {

                DEBUG_LOG("START: %d -- END: %d\n", start, end);
                old_yara_file->removeElementsInRange(
                    ts_node_start_byte(node), 
                    ts_node_end_byte(node)
                );

                DEBUG_LOG("Node: %s (%d, %d) is beeing rebuild\n", 
                    ts_node_type(node), 
                    ts_node_start_byte(node), 
                    ts_node_end_byte(node)
                );

                rebuildYaraFileSubtree_(parent_src, old_yara_file, node);
                node = ts_node_next_sibling(node);
            }
        }

        old_yara_file->ctx().clearTree();
        old_yara_file->ctx().setTree(tree);

    #else
        // The whole yara file object is rebuild again

        old_yara_file->ctx().clearTree();
        old_yara_file->ctx().setTree(tree);
        
        forceUpdateYaraFile_(parent_src, old_yara_file);
    #endif
}


bool TSParserAdapter::update_(YaraSource *parent_src, const YaraFilePtr &old_yara_file) {
    bool was_any_edited = false;

    DEBUG_LOG("--- Looking for edits in file %s\n", 
        old_yara_file->getName().empty() ? "unnamed" : old_yara_file->getName().c_str()
    );
    
    if(old_yara_file->wasEdited() || old_yara_file->mustBeReparsed()) { // Check if there was at least one textual edit
        
        DEBUG_LOG("--- Start of %s reparsing ---\n", 
            old_yara_file->getName().empty() ? "unnamed" : old_yara_file->getName().c_str()
        );
        
        updateYaraFile_(parent_src, old_yara_file);

        DEBUG_LOG("--- End of %s reparsing ---\n", 
            old_yara_file->getName().empty() ? "unnamed" : old_yara_file->getName().c_str()
        );

    }

    if(old_yara_file->wasEdited()) {
        was_any_edited = true;
    }

    // Reparse included files
    if(config_->getParsingMode() == YaramodConfig::ParsingMode::Auto) {
        for(auto &include: old_yara_file->getLocalFileIncludes().dataByOffset()) {
            auto file = include.second->getFile();
            if(file == nullptr) {
                continue;
            }

            bool was_included_file_edited = file->wasEdited();
            bool was_included_file_checked = file->mustBeReparsed();
            reparseFile(parent_src, include.second->getPath(), file); ///< Thanks to condition in reparse method, YaraFiles that are not edited are not reparsed 

            if(was_included_file_edited || was_included_file_checked) { ///< If there was change check duplicity against currently reparsed YaraFile
                old_yara_file->checkDuplicatedRules(file, include.second);
            }

            if(was_included_file_edited) {
                was_any_edited = true;
            }
        }
    }

    #ifdef DEBUG
        old_yara_file->dumpSyntaxErrors();
    #endif

    return was_any_edited;
}


void TSParserAdapter::forceUpdateYaraFile_(YaraSource *parent_src, const YaraFilePtr &yara_file) {
    yara_file->clearAll();
    yara_file->clearEdits();
    yara_file->clearToBeChecked();

    TSNode root = ts_tree_root_node(yara_file->ctx().getTree());
    if(!isValidNode_(root)) {
        return;
    }

    for(uint32_t i = 0; i < ts_node_child_count(root); i++) {
        TSNode child = ts_node_child(root, i);
        rebuildYaraFileSubtree_(parent_src, yara_file, child);
    }
}



void TSParserAdapter::updateString_(const std::string &new_string, const YaraFilePtr &old_yara_file) {
    if(old_yara_file->wasEdited()) {
        if(old_yara_file->hasFastEdit()) {
            old_yara_file->ctx().setString(new_string);
        }
        else {
            old_yara_file->ctx().stringUpdate();
        }
    }
}



/**
 * Performs BFS above the subtree represented by given node, all nodes that has 
 * error type are appended to given vector 
 */
void TSParserAdapter::findErrorNodes_(
    TSNode root, 
    std::vector<TSNode> &err_nodes, 
    size_t max_depth) {

    if(!ts_node_has_error(root)) { ///< If root node does not have error flag, there are no errors in subtree (error flag is propagated to root)
        return;
    }
    if(!strcmp(ts_node_type(root), "ERROR")) {
        err_nodes.push_back(root);
        return;
    }
    
    std::vector<node_w_depth_t> node_stack;
    node_stack.push_back({0, root});

    while(!node_stack.empty()) {
        node_w_depth_t cur_node = node_stack.back();
        node_stack.pop_back();

        for(uint32_t i = 0; i < ts_node_child_count(cur_node.node); i++) {
            TSNode child = ts_node_child(cur_node.node, i);

            if(!ts_node_has_error(child) || (cur_node.depth + 1 > max_depth)) {
                continue;
            }
            if(!strcmp(ts_node_type(child), "ERROR")) {
                err_nodes.push_back(child);
                continue;
            }

            node_stack.push_back({cur_node.depth + 1, child});
        }
    }
}



/**
 * Finds all nodes, that satisfy given condition and puts them to gitven vector
 */
template <typename Fn>
void TSParserAdapter::findNodes_(
    TSNode start, 
    std::vector<TSNode> &out_vec, 
    const Fn &f, 
    size_t max_depth) { 

    TSParserAdapter::traverseSubtree_(
        start,
        [f, &out_vec](TSNode n) {
            if(f(n)) {
                out_vec.push_back(n);
            }
        },
        max_depth
    );
}



/**
 * Finds all nodes, that satisfy given condition  
 */
template <typename Fn>
void TSParserAdapter::traverseSubtree_(
    TSNode start, 
    const Fn &f, 
    size_t max_depth) { 

    std::vector<node_w_depth_t> node_stack;
    node_stack.push_back({0, start});

    while(!node_stack.empty()) {
        node_w_depth_t cur_node = node_stack.back();
        node_stack.pop_back();

        f(cur_node.node);

        // Here is ts_node_child_count (for unnamed nodes) neccessary - e. g. MISSING nodes are not named
        for(uint32_t i = 0; i < ts_node_child_count(cur_node.node); i++) {
            TSNode child = ts_node_child(cur_node.node, i);
            //DEBUG_LOG("%s\n", ts_node_type(child));

            if(cur_node.depth + 1 > max_depth) { ///< Max depth was reached
                continue;
            }
            else {
                node_stack.push_back({cur_node.depth + 1, child});
            }
        }
    }
}


/**
 * Creates STRING from TSNode structure 
 */
std::string TSParserAdapter::getNodeString_(const TSNode &node, std::string_view file_sv) {
    size_t len = ts_node_end_byte(node) - ts_node_start_byte(node);
    return std::string(file_sv.substr(ts_node_start_byte(node), len));
}


/**
 * Creates STRING from TSNode structure 
 */
std::string TSParserAdapter::getStringId_(const TSNode &id_node, std::string_view file_sv) {
    size_t len = ts_node_end_byte(id_node) - ts_node_start_byte(id_node);

    if(len == 0) {
        return std::string({});
    }
    else {
        return std::string(file_sv.substr(ts_node_start_byte(id_node) + 1, len - 1));
    }
}



/**
 * Creates string view from TSNode structure 
 */
std::string_view TSParserAdapter::getNodeStringV_(const TSNode &node, std::string_view file_sv) {
    size_t len = ts_node_end_byte(node) - ts_node_start_byte(node);
    return file_sv.substr(ts_node_start_byte(node), len);
}



void TSParserAdapter::buildSyntaxErrors_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    // Build error nodes
    findErrorNodes_(node, node_stack_);
    for(auto &err_node: node_stack_) {
        offset_t pos = getNodeOffset_(err_node);
        yara_file->addSyntaxError<UnexpectedTokenError>(
            getNodeString_(err_node, sv), 
            pos, 
            getNodeLen_(err_node),
            getNodeRange_(err_node)
        );
    }

    node_stack_.clear();
}


void TSParserAdapter::buildGlobals_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv, size_t max_depth, YaraFileElementBindable *parent) {
     // Build missing (syntax errors) and comments - they can be everywhere
    traverseSubtree_(node, [this, yara_file, &sv, parent](TSNode n) {
        size_t missing_node_len = 2; ///< Constant size for missing nodes

        if(ts_node_is_missing(n)) {
            offset_t offset = getNodeOffset_(n) - 1; ///< Offset decremented by one because we want to make error overlapping with its parent element 
            if(parent) {
                offset -= parent->getOffset();
            }

            auto error = std::make_unique<MissingTokenError>(
                std::string(ts_node_type(n)), 
                offset, 
                missing_node_len
            );

            error->setParentFile(yara_file);
            error->bind_to(parent)->setParent(parent);
            yara_file->addSyntaxError(std::move(error));
        }
        else if(isType_(n, "comment")) {
            std::unique_ptr<Comment> comment = std::make_unique<Comment>(
                    getNodeString_(n, sv), 
                    getNodeOffset_(n) 
            );
            comment->setParentFile(yara_file);
            yara_file->addComment(std::move(comment));
        }
    }, max_depth);
}


/**
 * Instantiates the new YaraFile and sets it members due to parsing 
 */
void TSParserAdapter::buildYaraFile_(YaraSource *parent_src, const YaraFilePtr &yara_file) {
    DEBUG_LOG("--- Parsing file %s ---\n", yara_file->getName().empty() ? "unnamed" : yara_file->getName().c_str());
    TSNode root = ts_tree_root_node(yara_file->ctx().getTree());

    std::string_view source_sv = std::string_view(
        yara_file->ctx().getString()
    );

    // Build the global elements of the yara file
    buildSyntaxErrors_(yara_file, root, source_sv);
    
    for(size_t i = 0; i < ts_node_named_child_count(root); i++) {
        TSNode child = ts_node_named_child(root, i);

        if(isType_(child, "rule")) { // The rule node was found
            auto rule = buildRule_(yara_file, child, source_sv);
            buildGlobals_(yara_file, child, source_sv, (unsigned)-1, rule.get());
            yara_file->addRule(std::move(rule));
        }
        else if(isType_(child, "include")) { // The include node was found
            auto include = buildInclude_(parent_src, yara_file, child, source_sv);
            buildGlobals_(yara_file, child, source_sv, (unsigned)-1, include.get());
            addFileInclude(std::move(include), parent_src, yara_file);
        }
        else if(isType_(child, "import")) { // The import node was found
            auto import = buildImport_(yara_file, child, source_sv);
            buildGlobals_(yara_file, child, source_sv, (unsigned)-1, import.get());
            if(import->getModule()) {
                yara_file->addImport(std::move(import));
            }
            else {
                yara_file->addWrong(std::move(import));
            }
        }
        else if(isType_(child, "comment")) {
            auto comment = std::make_unique<Comment>(
                    getNodeString_(child, source_sv), 
                    getNodeOffset_(child) 
            );
            comment->setParentFile(yara_file);
            yara_file->addComment(std::move(comment));
        }
    }
    
    #ifdef DEBUG
        yara_file->dumpSyntaxErrors();
    #endif

    DEBUG_LOG("--- End of %s parsing ---\n", yara_file->getName().empty() ? "unnamed" : yara_file->getName().c_str());
}



bool TSParserAdapter::isValidId_(const YaraFilePtr &yara_file, const TSNode &id_node, std::string_view sv) {
    if(isYaraKeyword(getNodeStringV_(id_node, sv))) {
        yara_file->addSyntaxError<SyntaxError>(
            "Unexpected keyword " + getNodeString_(id_node, sv) + ", expected identifier!",
            getNodeOffset_(id_node),
            getNodeLen_(id_node),
            getNodeRange_(id_node)
        );

        return false;
    }

    return true;
}



std::string TSParserAdapter::buildRegexp_(const YaraFilePtr &yara_file, const TSNode &regexp_node, std::string_view sv) {
    TSNode regexp_body = getNodeChildByFieldName_(regexp_node, "body");
    if(!isValidNode_(regexp_body)) {
        return std::string({});
    }

    TSNode regexp_init_node = ts_node_named_child(regexp_body, 0);
    if(!isValidNode_(regexp_init_node)) {
        return std::string({});
    }

    buildRegexpNode_(yara_file, regexp_init_node, sv);

    return getNodeString_(regexp_node, sv); ///< For now, return just string with the whole regular expression (less infromation is stored -> better memory performance)
}



void TSParserAdapter::buildRegexpNode_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    // Check its type
    if(isType_(node, "regexp_cat")) {
        buildRegexpCat_(yara_file, node, sv);
    }
    else if(isType_(node, "regexp_alt")) {
        buildRegexpAlt_(yara_file, node, sv);
    }
    else if(isType_(node, "regexp_class")) { ///< Regexp class contains N regexp class chars or ranges
        for(uint32_t i = 0; i < ts_node_named_child_count(node); i++) {
            TSNode cur_child = ts_node_named_child(node, i);
            if(!isValidNode_(cur_child)) {
                continue;
            }

            buildRegexpNode_(yara_file, cur_child, sv);
        }
    }
    else if(isType_(node, "regexp_class_range")) {
        buildRegexpClassRange_(yara_file, node, sv);
    }
    else if(isType_(node, "regexp_rep")) {
        buildRegexpRep_(yara_file, node, sv);
    }
    else if(isType_(node, "regexp_char") || isType_(node, "regexp_class_char")) {

        // All characters in regular expressions in YARA must be ASCII chars
        if(!isAsciiChar_(node, sv)) {
            yara_file->addSyntaxError<SyntaxError>(
                "Non ascii character in regular expression!",
                getNodeOffset_(node),
                getNodeLen_(node),
                getNodeRange_(node)
            );
        }

    }
}



void TSParserAdapter::buildRegexpCat_(const YaraFilePtr &yara_file, const TSNode &cat_node, std::string_view sv) {
    // For now it is the same as buildRegexpAlt
    TSNode lop_node = ts_node_named_child(cat_node, 0);
    TSNode rop_node = ts_node_named_child(cat_node, 1);

    // Push the operands to the stack, no semantic check must be done here
    if(isValidNode_(rop_node)) { ///< Because comments cant appear in regular expression, there is no need to use isGlobalNode_
        buildRegexpNode_(yara_file, rop_node, sv);
    }

    if(isValidNode_(lop_node)) {
        buildRegexpNode_(yara_file, lop_node, sv);
    }
}



void TSParserAdapter::buildRegexpAlt_(const YaraFilePtr &yara_file, const TSNode &alt_node, std::string_view sv) {
    TSNode lop_node = ts_node_named_child(alt_node, 0);
    TSNode rop_node = ts_node_named_child(alt_node, 1);

    if(isValidNode_(rop_node)) {
        buildRegexpNode_(yara_file, rop_node, sv);
    }

    if(isValidNode_(lop_node)) {
        buildRegexpNode_(yara_file, lop_node, sv);
    }
}



void TSParserAdapter::buildRegexpClassRange_(const YaraFilePtr &yara_file, const TSNode &range_node, std::string_view sv) {
    TSNode from_node = getNodeChildByFieldName_(range_node, "from");
    TSNode to_node = getNodeChildByFieldName_(range_node, "to");
    
    // Perform check only if both bound nodes are valid (there can be syntax error)
    if(isValidNode_(from_node) && isValidNode_(to_node)) {
        if(!isAsciiChar_(from_node, sv) || !isAsciiChar_(to_node, sv)) {
            yara_file->addSyntaxError<SyntaxError>(
                "Non ascii character in regular expression!",
                getNodeOffset_(range_node),
                getNodeLen_(range_node),
                getNodeRange_(range_node)
            );

            return;
        }

        std::string_view from_node_sv = getNodeStringV_(from_node, sv);
        std::string_view to_node_sv = getNodeStringV_(to_node, sv);

        // Convert bound nodes to ASCII values (to be easily compared)
        u_char from_val, to_val;
        if(isValidEscapeSequence(from_node_sv)) {
            from_val = escapeSequenceToChar(from_node_sv);
        }
        else {
            from_val = from_node_sv.at(0);
        }

        if(isValidEscapeSequence(to_node_sv)) {
            to_val = escapeSequenceToChar(to_node_sv);
        }
        else {
            to_val = to_node_sv.at(0);
        }

        if(from_val > to_val) { ///< Lower bound cannot be greater then upper bound
            yara_file->addSemanticError<RegexpError>(
                "Bad character range!",
                getNodeOffset_(range_node),
                getNodeLen_(range_node)
            );
        }
    }
}



void TSParserAdapter::buildRegexpRep_(const YaraFilePtr &yara_file, const TSNode &rep_node, std::string_view sv) {
    TSNode base_node = getNodeChildByFieldName_(rep_node, "base");
    if(isValidNode_(base_node)) {
        buildRegexpNode_(yara_file, base_node, sv);
    }
    
    TSNode quant_node = getNodeChildByFieldName_(rep_node, "quant");
    if(!isValidNode_(quant_node)) {
        return;
    }

    TSNode count_node = getNodeChildByFieldName_(quant_node, "count");
    TSNode lbound_node = getNodeChildByFieldName_(quant_node, "lower");
    TSNode ubound_node = getNodeChildByFieldName_(quant_node, "upper");

    // There can be count node or two (optional) bounds {\d*,\d*} or one count node {\d+}
    if(isValidNode_(count_node)) {
        std::optional<int64_t> count = buildInt_(yara_file, count_node, sv);

        if(count.has_value()) {
            if(count.value() > REGEXP_MAXIMUM_QUANTIFIER_NUM) {
                yara_file->addSemanticError<RegexpError>(
                    "Repeat interval too large!",
                    getNodeOffset_(count_node),
                    getNodeLen_(count_node),
                    getNodeRange_(count_node)
                );
            }
        }
    }
    else {
        // Convert boundaries to integers and check the validity of repeat interval
        std::optional<int64_t> lower_val, upper_val, val_to_check;
        if(isValidNode_(lbound_node)) {
            lower_val = buildInt_(yara_file, lbound_node, sv);
        }

        if(isValidNode_(ubound_node)) {
            upper_val = buildInt_(yara_file, ubound_node, sv);
        }

        if(lower_val.has_value() && upper_val.has_value()) {
            if(lower_val.value() > upper_val.value()) { ///< Lower bound should be lower than upper bound
                yara_file->addSemanticError<RegexpError>(
                    "Bad repeat interval!",
                    getNodeOffset_(quant_node),
                    getNodeLen_(quant_node),
                    getNodeRange_(quant_node)
                );
            }
            else {
                val_to_check = upper_val;
            }
        }
        else if(lower_val.has_value() || upper_val.has_value()) {
            val_to_check = lower_val.has_value() ? lower_val : upper_val;
        }

        if(val_to_check.has_value() && val_to_check.value() > REGEXP_MAXIMUM_QUANTIFIER_NUM) {
            yara_file->addSemanticError<RegexpError>(
                "Repeat interval too large!",
                getNodeOffset_(quant_node),
                getNodeLen_(quant_node),
                getNodeRange_(quant_node)
            );
        }


    }
}



std::string TSParserAdapter::buildHexStr_(const YaraFilePtr &yara_file, const TSNode &hex_str_node, std::string_view sv) {
    TSNode init_hex_str_seq;
    bool init_node_found = false;

    // In hex strings can comments in curly braces 
    for(uint32_t i = 0; i < ts_node_named_child_count(hex_str_node); i++) {
        TSNode cur_child = ts_node_named_child(hex_str_node, i);
        if(!isGlobalNode_(cur_child) && isValidNode_(cur_child)) {
            init_hex_str_seq = cur_child;
            init_node_found = true;
            break;
        }
    }
    if(!init_node_found) { ///< Check if initial node was found
        return std::string({});
    }

    buildHexStrNode_(yara_file, init_hex_str_seq, sv); ///< Analyze the hexstring

    std::string content = getNodeString_(init_hex_str_seq, sv);

    // Clear hex string from comments (workaround to keep grammar simple)
    offset_t hex_str_start = getNodeOffset_(init_hex_str_seq);
    offset_t hex_str_end = hex_str_start + getNodeLen_(init_hex_str_seq);

    // Remove comments from hex string content
    auto end_it = yara_file->getComments().data().upper_bound(hex_str_end);
    for(auto it = std::make_reverse_iterator(end_it); 
        it != yara_file->getComments().data().rend(); 
        ++it) {

        offset_t comment_start = it->second->getOffset();
        size_t comment_len = it->second->getLen();

        if(comment_start + comment_len > hex_str_start) {
            content.erase(comment_start - hex_str_start, comment_len);
        }
        else {
            break;
        }
    }

    return content;
}



void TSParserAdapter::buildHexStrNode_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    if(isType_(node, "hex_str_sequence")) {
        for(uint32_t i = 0; i < ts_node_named_child_count(node); i++) {
            TSNode cur_child = ts_node_named_child(node, i);
            if(!isValidNode_(cur_child) || isGlobalNode_(cur_child)) {
                continue;
            }

            // Push or children of hex_str_sequence to stack to be examined
            buildHexStrNode_(yara_file, cur_child, sv);
        }
    }
    else if(isType_(node, "hex_str_parentheses")) {
        TSNode par_child = ts_node_named_child(node, 0);
        if(isValidNode_(par_child)) {
            buildHexStrNode_(yara_file, par_child, sv);
        }
    }
    else if(isType_(node, "hex_str_alt")) {
        buildHexStrAlt_(yara_file, node, sv);
    }
    else if(isType_(node, "hex_str_jump")) {
        buildHexJump_(yara_file, node, sv);
    }
}



void TSParserAdapter::buildHexStrAlt_(const YaraFilePtr &yara_file, const TSNode &hex_str_alt, std::string_view sv) {
    TSNode lop_node = getNodeChildByFieldName_(hex_str_alt, "lop");
    TSNode rop_node = getNodeChildByFieldName_(hex_str_alt, "rop");
    
    if(isValidNode_(rop_node)) {
        buildHexStrNode_(yara_file, rop_node, sv);
    }

    if(isValidNode_(lop_node)) {
        buildHexStrNode_(yara_file, rop_node, sv);
    }
}



void TSParserAdapter::buildHexJump_(const YaraFilePtr &yara_file, const TSNode &jump_node, std::string_view sv) {
    TSNode lbound = getNodeChildByFieldName_(jump_node, "lower");
    TSNode ubound = getNodeChildByFieldName_(jump_node, "upper");

    if(isValidNode_(lbound) && isValidNode_(ubound)) {
        auto lbound_val = buildInt_(yara_file, lbound, sv);
        auto ubound_val = buildInt_(yara_file, ubound, sv);
    
        if(lbound_val.has_value() && ubound_val.has_value()) {
            if(lbound_val > ubound_val) { // Check the validity of hexstring jump 
                yara_file->addSemanticError<HexStrError>(
                    "Invalid jump range!",
                    getNodeOffset_(jump_node),
                    getNodeLen_(jump_node),
                    getNodeRange_(jump_node)
                );
            }
        }



    }
}



void TSParserAdapter::buildStringModifierList_(const YaraFilePtr &yara_file, const std::unique_ptr<String> &string, const TSNode &list_node, std::string_view sv) {
    for(uint32_t i = 0; i < ts_node_named_child_count(list_node); i++) {
        TSNode mod_node = ts_node_named_child(list_node, i);
        if(!isValidNode_(mod_node) || isGlobalNode_(mod_node)) {
            continue;
        }

        TSNode mod_name = ts_node_child(mod_node, 0);
        if(!isValidNode_(mod_name)) {
            continue;
        }

        std::string_view mod_sv = getNodeStringV_(mod_name, sv);
        StringModifier::Type mod_type = StringModifier::stringToType(mod_sv);

        std::unique_ptr<StringModifier> mod = std::make_unique<StringModifier>(mod_type);

        // Only base64 and xor modifiers can have arguments
        if(mod_type == StringModifier::Type::Base64) {
            TSNode alphabet_node = getNodeChildByFieldName_(mod_node, "alphabet");
            if(isValidNode_(alphabet_node)) {
                buildModAlphaArg_(yara_file, mod, alphabet_node, sv);
            }
        }
        else if(mod_type == StringModifier::Type::Xor) {
            TSNode range_node = getNodeChildByFieldName_(mod_node, "range");
            if(isValidNode_(range_node)) {
                buildModRangeArg_(yara_file, mod, range_node, sv);
            }
        }

        // Set properties of string modifier
        mod->setParentRule(string->getParentRule());
        mod->setParentFile(yara_file);
        mod->setLen(getNodeLen_(mod_name));
        mod->setOffset(getNodeOffset_(mod_name));

        string->addModifier(std::move(mod));
    }
}



void TSParserAdapter::buildModAlphaArg_(const YaraFilePtr &yara_file, const std::unique_ptr<StringModifier> &mod, const TSNode &alphabet_node, std::string_view sv) {
    std::string alphabet_string = buildStr_(yara_file, alphabet_node, sv);
                
    if(alphabet_string.length() != BASE64_ALPHA_LEN) { ///< Alphabet must have BASE64_ALPHA_LEN bytes
        yara_file->addSemanticError<StringModifierError>(
            "Length of base64 alphabet must be " TO_STR(BASE64_ALPHA_LEN) "!",
            getNodeOffset_(alphabet_node),
            getNodeLen_(alphabet_node),
            getNodeRange_(alphabet_node)
        );
    }
    else {
        mod->setArg(alphabet_string);
    }
}



void TSParserAdapter::buildModRangeArg_(const YaraFilePtr &yara_file, const std::unique_ptr<StringModifier> &mod, const TSNode &range_node, std::string_view sv) {
    TSNode key_node = getNodeChildByFieldName_(range_node, "key");
    TSNode lbound_node = getNodeChildByFieldName_(range_node, "lower");
    TSNode ubound_node = getNodeChildByFieldName_(range_node, "upper");

    // In xor modifier arguments can be range or one key
    if(isValidNode_(key_node)) {
        std::optional<int64_t> key_val = buildInt_(yara_file, key_node, sv);

        if(key_val.has_value()) {
            if(key_val.value() > XOR_UPPER_RANGE_BOUND) {
                yara_file->addSemanticError<StringModifierError>(
                    "Invalid xor range!",
                    getNodeOffset_(key_node),
                    getNodeLen_(key_node),
                    getNodeRange_(key_node)
                );
            }
            else {
                mod->setArg(key_val.value());
            }
        }
    }
    else if(isValidNode_(lbound_node) && isValidNode_(ubound_node)) {
        std::optional<int64_t> lower_val = buildInt_(yara_file, lbound_node, sv);
        std::optional<int64_t> upper_val = buildInt_(yara_file, ubound_node, sv);

        if(lower_val.has_value() && upper_val.has_value()) {

            if(lower_val.value() > upper_val.value()) {
                yara_file->addSemanticError<StringModifierError>(
                    "Xor lower bound exceeds upper bound!",
                    getNodeOffset_(range_node),
                    getNodeLen_(range_node),
                    getNodeRange_(range_node)
                );
            }
            else if(lower_val.value() > XOR_UPPER_RANGE_BOUND || 
                upper_val.value() > XOR_UPPER_RANGE_BOUND) {

                yara_file->addSemanticError<StringModifierError>(
                    "Upper bound for xor range exceeded (max: " TO_STR(XOR_UPPER_RANGE_BOUND) ")!",
                    getNodeOffset_(range_node),
                    getNodeLen_(range_node),
                    getNodeRange_(range_node)
                );
            }
            else {
                mod->setArg(StringModifier::Range(lower_val.value(), upper_val.value()));
            }
        }
        

    }
}



std::unique_ptr<Literal>  TSParserAdapter::buildLiteral_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    // Determine the type of literal
    if(isType_(literal_node, "string_literal")) {
        return buildStrLiteral_(yara_file, literal_node, sv);
    }
    else if(isType_(literal_node, "int_literal") || 
        isType_(literal_node, "uint_literal")) {

        return buildIntLiteral_(yara_file, literal_node, sv);
    }
    else if(isType_(literal_node, "bool_literal")) {
        return buildBoolLiteral_(yara_file, literal_node, sv);
    }
    else if(isType_(literal_node, "float_literal")) {
        return buildFloatLiteral_(yara_file, literal_node, sv);
    }
    else {
        throw InternalErrorException("Unknown literal type!");
    }
}



std::unique_ptr<Literal>  TSParserAdapter::buildStrLiteral_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    std::unique_ptr<Literal> result = std::make_unique<Literal>();

    result->set(buildStr_(yara_file, literal_node, sv));

    return result;
}



std::string TSParserAdapter::buildStr_(const YaraFilePtr &yara_file, const TSNode &str_literal, std::string_view sv) {
    TSNode str_node = getNodeChildByFieldName_(str_literal, "str");
    if(ts_node_is_null(str_node)) { ///< If str node is missing it implies empty string
        return std::string({});
    }
    else if(isValidNode_(str_node)) {

        // Check internal nodes of string (for now there is only esc seq)
        for(uint32_t i = 0; i < ts_node_named_child_count(str_node); i++) {
            TSNode str_child = ts_node_named_child(str_node, i);
            
            if(isType_(str_child, "string_esc_seq")) {
                if(!isValidEscapeSequence(getNodeStringV_(str_child, sv))) {
                    yara_file->addSyntaxError<SyntaxError>(
                        "Invalid escape sequence!",
                        getNodeOffset_(str_child),
                        getNodeLen_(str_child),
                        getNodeRange_(str_child)
                    );
                }
            }


        }

        return getNodeString_(str_node, sv);
    }
    else { 
        throw InternalErrorException("Unable to make string literal from " + getNodeString_(str_node, sv));
    }
}



void TSParserAdapter::getIntProps(const TSNode &int_node, int &base, int &sign, int &start) {
    start = 0;
    sign = 1;
    base = 10;

    TSNode child;
    uint32_t child_count = ts_node_child_count(int_node);
    for(uint32_t i = 0; i < child_count; i++) {
        child = ts_node_child(int_node, i);
        if(!isValidNode_(child) || isGlobalNode_(child)) {
            continue;
        }

        if(isType_(child, "-")) { ///< If there is - it is negative integer
            sign = -1;
            child = ts_node_next_sibling(child);
        }
        else if(isType_(child, "oct_uint_literal_value")) {
            base = 8;
            start += std::char_traits<char>::length("0o");
            break;
        }
        else if(isType_(child, "hex_uint_literal_value")) {
            base = 16;
            start += std::char_traits<char>::length("0x");
            break;
        }
        else if(isType_(child, "dec_uint_literal_value")) {
            base = 10;
            break;
        }
    }

    if(child_count) {
        start += ts_node_start_byte(child) - ts_node_start_byte(int_node); ///< Compute the offset from the start of literal node
    }
}



std::optional<int64_t> TSParserAdapter::buildInt_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    if(!isValidNode_(literal_node)) {
        return std::optional<int64_t>();
    }

    int base, sign, start;
    int64_t result_val;

    getIntProps(literal_node, base, sign, start);

    std::string_view int_str_v = getNodeStringV_(literal_node, sv);
    auto [ptr, err] = std::from_chars(int_str_v.begin() + start, int_str_v.end(), result_val, base);
    if(err == std::errc()) {
        // Success
        return std::optional<int64_t>(result_val*sign);
    }
    else if(err == std::errc::invalid_argument) {
        throw InternalErrorException("Unable to make int literal from" + std::string(int_str_v));
    }
    else if(err == std::errc::result_out_of_range) {
        yara_file->addSemanticError<OverflowError>(
            "Integer overflow!",
            getNodeOffset_(literal_node),
            getNodeLen_(literal_node),
            getNodeRange_(literal_node)
        );
    }

    return std::optional<int64_t>();
}



std::unique_ptr<Literal> TSParserAdapter::buildIntLiteral_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    std::unique_ptr<Literal> result = std::make_unique<Literal>();

    std::optional<int64_t> result_val = buildInt_(yara_file, literal_node, sv);

    if(result_val.has_value()) {
        result->set(result_val.value());
    }
    else {
        result->set(std::monostate());
    }

    return result;
}



std::optional<float> TSParserAdapter::buildFloat_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    if(!isValidNode_(literal_node)) {
        return std::optional<float>();
    }

    float result_val;

    std::string_view float_str_v = getNodeStringV_(literal_node, sv);
    auto [ptr, err] = std::from_chars(float_str_v.begin(), float_str_v.end(), result_val);
    if(err == std::errc()) {
        // Success
        return std::optional<float>(result_val);
    }
    else if(err == std::errc::invalid_argument) {
        throw InternalErrorException("Unable to make float literal from" + std::string(float_str_v));
    }
    else if(err == std::errc::result_out_of_range) {
        yara_file->addSemanticError<OverflowError>(
            "Float overflow!",
            getNodeOffset_(literal_node),
            getNodeLen_(literal_node),
            getNodeRange_(literal_node)
        );
    }

    return std::optional<float>();
}



std::unique_ptr<Literal> TSParserAdapter::buildFloatLiteral_(const YaraFilePtr &yara_file, const TSNode &literal_node, std::string_view sv) {
    std::unique_ptr<Literal> result = std::make_unique<Literal>();
    std::optional<float> result_val = buildFloat_(yara_file, literal_node, sv);

    if(result_val.has_value()) {
        result->set(result_val.value());
    }
    else {
        result->set(std::monostate());
    }

    return result;
}



std::unique_ptr<Literal> TSParserAdapter::buildBoolLiteral_(const YaraFilePtr &, const TSNode &literal_node, std::string_view sv) {
    std::unique_ptr<Literal> result = std::make_unique<Literal>();

    std::string_view bool_str_v = getNodeStringV_(literal_node, sv);
    if(bool_str_v == "true") {
        result->set(true);
    }
    else if(bool_str_v == "false") {
        result->set(false);
    }
    else {
        throw InternalErrorException("Unable to make bool literal from " + getNodeString_(literal_node, sv));
    }

    return result;
}



std::unique_ptr<FileInclude> TSParserAdapter::buildInclude_(YaraSource *parent_src, const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    std::unique_ptr<FileInclude> include = std::make_unique<FileInclude>();

    include->setParentFile(yara_file);
    include->setLen(getNodeLen_(node));
    include->setOffset(getNodeOffset_(node));
    include->setRangeCache(getNodeRange_(node));

    TSNode path_node = getNodeChildByFieldName_(node, "path");
    if(isValidNode_(path_node)) {

        TSNode str_node = getNodeChildByFieldName_(path_node, "str");
        if(isValidNode_(str_node)) {

            if(parent_src->getBaseDir() != std::string({})) {
                include->setPath(getNodeString_(str_node, sv), parent_src->getBaseDir());
            }
            else {
                include->setPath(getNodeString_(str_node, sv), config_->getBaseDir());
            }

        
        }
    }

    return include;
}



std::unique_ptr<Import> TSParserAdapter::buildImport_(const YaraFilePtr &yara_file, const TSNode &node, std::string_view sv) {
    std::unique_ptr<Import> import = std::make_unique<Import>();

    import->setParentFile(yara_file);
    import->setOffset(getNodeOffset_(node));
    import->setLen(getNodeLen_(node));
    import->setRangeCache(getNodeRange_(node));

    TSNode module_node = getNodeChildByFieldName_(node, "module");
    if(!isValidNode_(module_node)) {
        return import;
    }

    TSNode str_node = getNodeChildByFieldName_(module_node, "str");
    if(isValidNode_(str_node)) {
        import->setModuleName(getNodeString_(str_node, sv));
    }

    ModuleProvider &module_provider = ModuleProvider::getInstance();
    auto mod_ptr = module_provider.getModule(import->getModuleName());
    if(!mod_ptr) {
        DEBUG_LOG("Module '%s' does not exists!\n", import->getModuleName().c_str());
        yara_file->addSemanticError<ModuleError>(
            "Unrecognized module '" + import->getModuleName() + "' imported!",
            import->getOffset(),
            import->getLen()
        );
    }
    else {
        //DEBUG_LOG("Module exists!\n");
        import->setModule(module_provider.getModule(import->getModuleName()));
    }

    return import;
}


void TSParserAdapter::addFileInclude(std::unique_ptr<FileInclude> &&include, YaraSource *parent_src, const YaraFilePtr &yara_file) {
    // Building of newly included file (if it not in cache - in parent yara src object)
    auto path = include->getPath();
    
    auto dep = parent_src->getFile(path);

    if(dep && !dep->isIsolated()) {
        yara_file->addSemanticError<IncludeError>(
            "File cannot be included twice!",
            include->getOffset(),
            include->getLen(),
            include->getRange()
        );

        yara_file->addWrong(std::move(include));
    }
    else {
        bool file_exists = std::filesystem::exists(path);
        if(!file_exists) {
            yara_file->addSemanticError<IncludeError>(
                "Included file does not exists!",
                include->getOffset(),
                include->getLen(),
                include->getRange()
            );
        }

        auto saved = yara_file->addFileInclude(std::move(include));
        if(config_->getParsingMode() == YaramodConfig::ParsingMode::Auto && 
            file_exists &&
            saved) {

            if(!dep) {
                dep = parseFile(parent_src, path);
            }
        
            saved->setFile(dep);
            yara_file->checkDuplicatedRules(dep, saved);
            dep->setIsolated(false);
        }
    }
}


void TSParserAdapter::rebuildYaraFileSubtree_(YaraSource *parent_src, const YaraFilePtr &yara_file, const TSNode &root) {
    std::string_view sv = std::string_view(yara_file->ctx().getString());

    buildSyntaxErrors_(yara_file, root, sv); ///< Globals and syntax errors must be rebuild completely
    //buildGlobals_(yara_file, root, sv);

    if(!isValidNode_(root)) {
        return;
    }

    if(isType_(root, "rule")) {
        auto rule = buildRule_(yara_file, root, sv);
        buildGlobals_(yara_file, root, sv, (unsigned)-1, rule.get());
        if(auto saved = yara_file->addRule(std::move(rule))) {
            yara_file->notifyAddedSymbol(saved->getId());
        }
    }
    else if(isType_(root, "include")) {
        auto include = buildInclude_(parent_src, yara_file, root, sv);
        buildGlobals_(yara_file, root, sv, (unsigned)-1, include.get());
        addFileInclude(std::move(include), parent_src, yara_file);        
    }
    else if(isType_(root, "import")) {
        auto import = buildImport_(yara_file, root, sv);
        buildGlobals_(yara_file, root, sv, (unsigned)-1, import.get());

        if(import->getModule()) {
            if(auto saved = yara_file->addImport(std::move(import))) {
                yara_file->notifyAddedSymbol(saved->getModuleName());
            }
        }
        else {
            yara_file->addWrong(std::move(import));
        }
    }
    else if(isType_(root, "comment")) {
        auto comment = std::make_unique<Comment>(
                getNodeString_(root, sv), 
                getNodeOffset_(root) 
        );
        comment->setParentFile(yara_file);
        yara_file->addComment(std::move(comment));
    }
}
