/**
 * @file other.cpp 
 * 
 * @brief Implementation of methods that belongs to "other" elements of
 * YaraFile (includes, imports and comments)
 * 
 * @author Vojtěch Dvořák    
 */


#include "headers/common.h"
#include "headers/module.h"
#include "headers/yara_file.h"
#include "headers/rule.h"
#include "headers/yara_file_element.h"
#include "headers/yaramod_config.h"
#include "headers/other.h"


// Comment

Comment::Comment(const std::string &text, const offset_t &pos) : TopLevelYaraFileElement(pos, text.size()), text_(text) {
}


Comment::~Comment() {
   DEBUG_LOG("Deleting comment (at %d)\n", getOffset());
}


const std::string &Comment::getText() const {
    return text_;
}


std::stringstream Comment::getTextFormatted() const {
    std::stringstream sstream;
    sstream << text_;
    return sstream;
}


// FileInclude

FileInclude::~FileInclude() {
    DEBUG_LOG("Deleting file include of %s\n", getPath().c_str());

    if(yara_file_) {
        yara_file_->setIsolated();
    }
}

const std::string &FileInclude::getId() const {
    return path_;
}


void FileInclude::setPath(const std::string &path, const std::string &base_path) {
    std::filesystem::path fs_path(path);

    if(fs_path.is_relative()) {
        std::filesystem::path fs_base_dir(base_path);
        fs_base_dir /= fs_path;
        this->path_ = fs_base_dir.string();
    }
    else {
        this->path_ = path;
    }
}


const std::string &FileInclude::getPath() const {
    return path_;
}


void FileInclude::setFile(const std::shared_ptr<YaraFile> &yara_file) {
    yara_file_ = yara_file;
}


std::shared_ptr<YaraFile> FileInclude::getFile() const {
    return yara_file_;
}


std::stringstream FileInclude::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "include \"" << path_ << "\"";
    return sstream;
}


// Import

Import::Import() {
}


Import::~Import() {
    DEBUG_LOG("Deleting import (at %d)\n", getOffset());
}


const std::string &Import::getId() const {
    return mod_name_;
}


void Import::setModuleName(const std::string &mod_name) {
    mod_name_ = mod_name;
}


const std::string &Import::getModuleName() const {
    return mod_name_;
}


void Import::setModule(const std::shared_ptr<Module> &module) {
    module_ = module;
}


std::shared_ptr<Module> Import::getModule() const {
    return module_;
}


std::stringstream Import::getTextFormatted() const {
    std::stringstream sstream;
    sstream << "import \"" << mod_name_ << "\"";
    return sstream;
}
